/*! DataTables 1.12.1
 * ©2008-2022 SpryMedia Ltd - datatables.net/license
 */

/**
 * @summary     DataTables
 * @description Paginate, search and order HTML tables
 * @version     1.12.1
 * @author      SpryMedia Ltd
 * @contact     www.datatables.net
 * @copyright   SpryMedia Ltd.
 *
 * This source file is free software, available under the following license:
 *   MIT license - http://datatables.net/license
 *
 * This source file is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
 *
 * For details please refer to: http://www.datatables.net
 */

/*jslint evil: true, undef: true, browser: true */
/*globals $,require,jQuery,define,_selector_run,_selector_opts,_selector_first,_selector_row_indexes,_ext,_Api,_api_register,_api_registerPlural,_re_new_lines,_re_html,_re_formatted_numeric,_re_escape_regex,_empty,_intVal,_numToDecimal,_isNumber,_isHtml,_htmlNumeric,_pluck,_pluck_order,_range,_stripHtml,_unique,_fnBuildAjax,_fnAjaxUpdate,_fnAjaxParameters,_fnAjaxUpdateDraw,_fnAjaxDataSrc,_fnAddColumn,_fnColumnOptions,_fnAdjustColumnSizing,_fnVisibleToColumnIndex,_fnColumnIndexToVisible,_fnVisbleColumns,_fnGetColumns,_fnColumnTypes,_fnApplyColumnDefs,_fnHungarianMap,_fnCamelToHungarian,_fnLanguageCompat,_fnBrowserDetect,_fnAddData,_fnAddTr,_fnNodeToDataIndex,_fnNodeToColumnIndex,_fnGetCellData,_fnSetCellData,_fnSplitObjNotation,_fnGetObjectDataFn,_fnSetObjectDataFn,_fnGetDataMaster,_fnClearTable,_fnDeleteIndex,_fnInvalidate,_fnGetRowElements,_fnCreateTr,_fnBuildHead,_fnDrawHead,_fnDraw,_fnReDraw,_fnAddOptionsHtml,_fnDetectHeader,_fnGetUniqueThs,_fnFeatureHtmlFilter,_fnFilterComplete,_fnFilterCustom,_fnFilterColumn,_fnFilter,_fnFilterCreateSearch,_fnEscapeRegex,_fnFilterData,_fnFeatureHtmlInfo,_fnUpdateInfo,_fnInfoMacros,_fnInitialise,_fnInitComplete,_fnLengthChange,_fnFeatureHtmlLength,_fnFeatureHtmlPaginate,_fnPageChange,_fnFeatureHtmlProcessing,_fnProcessingDisplay,_fnFeatureHtmlTable,_fnScrollDraw,_fnApplyToChildren,_fnCalculateColumnWidths,_fnThrottle,_fnConvertToWidth,_fnGetWidestNode,_fnGetMaxLenString,_fnStringToCss,_fnSortFlatten,_fnSort,_fnSortAria,_fnSortListener,_fnSortAttachListener,_fnSortingClasses,_fnSortData,_fnSaveState,_fnLoadState,_fnSettingsFromNode,_fnLog,_fnMap,_fnBindAction,_fnCallbackReg,_fnCallbackFire,_fnLengthOverflow,_fnRenderer,_fnDataSource,_fnRowAttributes*/

(function( factory ) {
	"use strict";

	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				// CommonJS environments without a window global must pass a
				// root. This will give an error otherwise
				root = window;
			}

			if ( ! $ ) {
				$ = typeof window !== 'undefined' ? // jQuery's factory checks for a global window
					require('jquery') :
					require('jquery')( root );
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		window.DataTable = factory( jQuery, window, document );
	}
}
(function( $, window, document, undefined ) {
	"use strict";

	
	var DataTable = function ( selector, options )
	{
		// When creating with `new`, create a new DataTable, returning the API instance
		if (this instanceof DataTable) {
			return $(selector).DataTable(options);
		}
		else {
			// Argument switching
			options = selector;
		}
	
		/**
		 * Perform a jQuery selector action on the table's TR elements (from the tbody) and
		 * return the resulting jQuery object.
		 *  @param {string|node|jQuery} sSelector jQuery selector or node collection to act on
		 *  @param {object} [oOpts] Optional parameters for modifying the rows to be included
		 *  @param {string} [oOpts.filter=none] Select TR elements that meet the current filter
		 *    criterion ("applied") or all TR elements (i.e. no filter).
		 *  @param {string} [oOpts.order=current] Order of the TR elements in the processed array.
		 *    Can be either 'current', whereby the current sorting of the table is used, or
		 *    'original' whereby the original order the data was read into the table is used.
		 *  @param {string} [oOpts.page=all] Limit the selection to the currently displayed page
		 *    ("current") or not ("all"). If 'current' is given, then order is assumed to be
		 *    'current' and filter is 'applied', regardless of what they might be given as.
		 *  @returns {object} jQuery object, filtered by the given selector.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Highlight every second row
		 *      oTable.$('tr:odd').css('backgroundColor', 'blue');
		 *    } );
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Filter to rows with 'Webkit' in them, add a background colour and then
		 *      // remove the filter, thus highlighting the 'Webkit' rows only.
		 *      oTable.fnFilter('Webkit');
		 *      oTable.$('tr', {"search": "applied"}).css('backgroundColor', 'blue');
		 *      oTable.fnFilter('');
		 *    } );
		 */
		this.$ = function ( sSelector, oOpts )
		{
			return this.api(true).$( sSelector, oOpts );
		};
		
		
		/**
		 * Almost identical to $ in operation, but in this case returns the data for the matched
		 * rows - as such, the jQuery selector used should match TR row nodes or TD/TH cell nodes
		 * rather than any descendants, so the data can be obtained for the row/cell. If matching
		 * rows are found, the data returned is the original data array/object that was used to
		 * create the row (or a generated array if from a DOM source).
		 *
		 * This method is often useful in-combination with $ where both functions are given the
		 * same parameters and the array indexes will match identically.
		 *  @param {string|node|jQuery} sSelector jQuery selector or node collection to act on
		 *  @param {object} [oOpts] Optional parameters for modifying the rows to be included
		 *  @param {string} [oOpts.filter=none] Select elements that meet the current filter
		 *    criterion ("applied") or all elements (i.e. no filter).
		 *  @param {string} [oOpts.order=current] Order of the data in the processed array.
		 *    Can be either 'current', whereby the current sorting of the table is used, or
		 *    'original' whereby the original order the data was read into the table is used.
		 *  @param {string} [oOpts.page=all] Limit the selection to the currently displayed page
		 *    ("current") or not ("all"). If 'current' is given, then order is assumed to be
		 *    'current' and filter is 'applied', regardless of what they might be given as.
		 *  @returns {array} Data for the matched elements. If any elements, as a result of the
		 *    selector, were not TR, TD or TH elements in the DataTable, they will have a null
		 *    entry in the array.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Get the data from the first row in the table
		 *      var data = oTable._('tr:first');
		 *
		 *      // Do something useful with the data
		 *      alert( "First cell is: "+data[0] );
		 *    } );
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Filter to 'Webkit' and get all data for
		 *      oTable.fnFilter('Webkit');
		 *      var data = oTable._('tr', {"search": "applied"});
		 *
		 *      // Do something with the data
		 *      alert( data.length+" rows matched the search" );
		 *    } );
		 */
		this._ = function ( sSelector, oOpts )
		{
			return this.api(true).rows( sSelector, oOpts ).data();
		};
		
		
		/**
		 * Create a DataTables Api instance, with the currently selected tables for
		 * the Api's context.
		 * @param {boolean} [traditional=false] Set the API instance's context to be
		 *   only the table referred to by the `DataTable.ext.iApiIndex` option, as was
		 *   used in the API presented by DataTables 1.9- (i.e. the traditional mode),
		 *   or if all tables captured in the jQuery object should be used.
		 * @return {DataTables.Api}
		 */
		this.api = function ( traditional )
		{
			return traditional ?
				new _Api(
					_fnSettingsFromNode( this[ _ext.iApiIndex ] )
				) :
				new _Api( this );
		};
		
		
		/**
		 * Add a single new row or multiple rows of data to the table. Please note
		 * that this is suitable for client-side processing only - if you are using
		 * server-side processing (i.e. "bServerSide": true), then to add data, you
		 * must add it to the data source, i.e. the server-side, through an Ajax call.
		 *  @param {array|object} data The data to be added to the table. This can be:
		 *    <ul>
		 *      <li>1D array of data - add a single row with the data provided</li>
		 *      <li>2D array of arrays - add multiple rows in a single call</li>
		 *      <li>object - data object when using <i>mData</i></li>
		 *      <li>array of objects - multiple data objects when using <i>mData</i></li>
		 *    </ul>
		 *  @param {bool} [redraw=true] redraw the table or not
		 *  @returns {array} An array of integers, representing the list of indexes in
		 *    <i>aoData</i> ({@link DataTable.models.oSettings}) that have been added to
		 *    the table.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    // Global var for counter
		 *    var giCount = 2;
		 *
		 *    $(document).ready(function() {
		 *      $('#example').dataTable();
		 *    } );
		 *
		 *    function fnClickAddRow() {
		 *      $('#example').dataTable().fnAddData( [
		 *        giCount+".1",
		 *        giCount+".2",
		 *        giCount+".3",
		 *        giCount+".4" ]
		 *      );
		 *
		 *      giCount++;
		 *    }
		 */
		this.fnAddData = function( data, redraw )
		{
			var api = this.api( true );
		
			/* Check if we want to add multiple rows or not */
			var rows = Array.isArray(data) && ( Array.isArray(data[0]) || $.isPlainObject(data[0]) ) ?
				api.rows.add( data ) :
				api.row.add( data );
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return rows.flatten().toArray();
		};
		
		
		/**
		 * This function will make DataTables recalculate the column sizes, based on the data
		 * contained in the table and the sizes applied to the columns (in the DOM, CSS or
		 * through the sWidth parameter). This can be useful when the width of the table's
		 * parent element changes (for example a window resize).
		 *  @param {boolean} [bRedraw=true] Redraw the table or not, you will typically want to
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable( {
		 *        "sScrollY": "200px",
		 *        "bPaginate": false
		 *      } );
		 *
		 *      $(window).on('resize', function () {
		 *        oTable.fnAdjustColumnSizing();
		 *      } );
		 *    } );
		 */
		this.fnAdjustColumnSizing = function ( bRedraw )
		{
			var api = this.api( true ).columns.adjust();
			var settings = api.settings()[0];
			var scroll = settings.oScroll;
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw( false );
			}
			else if ( scroll.sX !== "" || scroll.sY !== "" ) {
				/* If not redrawing, but scrolling, we want to apply the new column sizes anyway */
				_fnScrollDraw( settings );
			}
		};
		
		
		/**
		 * Quickly and simply clear a table
		 *  @param {bool} [bRedraw=true] redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Immediately 'nuke' the current rows (perhaps waiting for an Ajax callback...)
		 *      oTable.fnClearTable();
		 *    } );
		 */
		this.fnClearTable = function( bRedraw )
		{
			var api = this.api( true ).clear();
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
		};
		
		
		/**
		 * The exact opposite of 'opening' a row, this function will close any rows which
		 * are currently 'open'.
		 *  @param {node} nTr the table row to 'close'
		 *  @returns {int} 0 on success, or 1 if failed (can't find the row)
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnClose = function( nTr )
		{
			this.api( true ).row( nTr ).child.hide();
		};
		
		
		/**
		 * Remove a row for the table
		 *  @param {mixed} target The index of the row from aoData to be deleted, or
		 *    the TR element you want to delete
		 *  @param {function|null} [callBack] Callback function
		 *  @param {bool} [redraw=true] Redraw the table or not
		 *  @returns {array} The row that was deleted
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Immediately remove the first row
		 *      oTable.fnDeleteRow( 0 );
		 *    } );
		 */
		this.fnDeleteRow = function( target, callback, redraw )
		{
			var api = this.api( true );
			var rows = api.rows( target );
			var settings = rows.settings()[0];
			var data = settings.aoData[ rows[0][0] ];
		
			rows.remove();
		
			if ( callback ) {
				callback.call( this, settings, data );
			}
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return data;
		};
		
		
		/**
		 * Restore the table to it's original state in the DOM by removing all of DataTables
		 * enhancements, alterations to the DOM structure of the table and event listeners.
		 *  @param {boolean} [remove=false] Completely remove the table from the DOM
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      // This example is fairly pointless in reality, but shows how fnDestroy can be used
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnDestroy();
		 *    } );
		 */
		this.fnDestroy = function ( remove )
		{
			this.api( true ).destroy( remove );
		};
		
		
		/**
		 * Redraw the table
		 *  @param {bool} [complete=true] Re-filter and resort (if enabled) the table before the draw.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Re-draw the table - you wouldn't want to do it here, but it's an example :-)
		 *      oTable.fnDraw();
		 *    } );
		 */
		this.fnDraw = function( complete )
		{
			// Note that this isn't an exact match to the old call to _fnDraw - it takes
			// into account the new data, but can hold position.
			this.api( true ).draw( complete );
		};
		
		
		/**
		 * Filter the input based on data
		 *  @param {string} sInput String to filter the table on
		 *  @param {int|null} [iColumn] Column to limit filtering to
		 *  @param {bool} [bRegex=false] Treat as regular expression or not
		 *  @param {bool} [bSmart=true] Perform smart filtering or not
		 *  @param {bool} [bShowGlobal=true] Show the input global filter in it's input box(es)
		 *  @param {bool} [bCaseInsensitive=true] Do case-insensitive matching (true) or not (false)
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sometime later - filter...
		 *      oTable.fnFilter( 'test string' );
		 *    } );
		 */
		this.fnFilter = function( sInput, iColumn, bRegex, bSmart, bShowGlobal, bCaseInsensitive )
		{
			var api = this.api( true );
		
			if ( iColumn === null || iColumn === undefined ) {
				api.search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
			else {
				api.column( iColumn ).search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
		
			api.draw();
		};
		
		
		/**
		 * Get the data for the whole table, an individual row or an individual cell based on the
		 * provided parameters.
		 *  @param {int|node} [src] A TR row node, TD/TH cell node or an integer. If given as
		 *    a TR node then the data source for the whole row will be returned. If given as a
		 *    TD/TH cell node then iCol will be automatically calculated and the data for the
		 *    cell returned. If given as an integer, then this is treated as the aoData internal
		 *    data index for the row (see fnGetPosition) and the data for that row used.
		 *  @param {int} [col] Optional column index that you want the data of.
		 *  @returns {array|object|string} If mRow is undefined, then the data for all rows is
		 *    returned. If mRow is defined, just data for that row, and is iCol is
		 *    defined, only data for the designated cell is returned.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    // Row data
		 *    $(document).ready(function() {
		 *      oTable = $('#example').dataTable();
		 *
		 *      oTable.$('tr').click( function () {
		 *        var data = oTable.fnGetData( this );
		 *        // ... do something with the array / object of data for the row
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Individual cell data
		 *    $(document).ready(function() {
		 *      oTable = $('#example').dataTable();
		 *
		 *      oTable.$('td').click( function () {
		 *        var sData = oTable.fnGetData( this );
		 *        alert( 'The cell clicked on had the value of '+sData );
		 *      } );
		 *    } );
		 */
		this.fnGetData = function( src, col )
		{
			var api = this.api( true );
		
			if ( src !== undefined ) {
				var type = src.nodeName ? src.nodeName.toLowerCase() : '';
		
				return col !== undefined || type == 'td' || type == 'th' ?
					api.cell( src, col ).data() :
					api.row( src ).data() || null;
			}
		
			return api.data().toArray();
		};
		
		
		/**
		 * Get an array of the TR nodes that are used in the table's body. Note that you will
		 * typically want to use the '$' API method in preference to this as it is more
		 * flexible.
		 *  @param {int} [iRow] Optional row index for the TR element you want
		 *  @returns {array|node} If iRow is undefined, returns an array of all TR elements
		 *    in the table's body, or iRow is defined, just the TR element requested.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Get the nodes from the table
		 *      var nNodes = oTable.fnGetNodes( );
		 *    } );
		 */
		this.fnGetNodes = function( iRow )
		{
			var api = this.api( true );
		
			return iRow !== undefined ?
				api.row( iRow ).node() :
				api.rows().nodes().flatten().toArray();
		};
		
		
		/**
		 * Get the array indexes of a particular cell from it's DOM element
		 * and column index including hidden columns
		 *  @param {node} node this can either be a TR, TD or TH in the table's body
		 *  @returns {int} If nNode is given as a TR, then a single index is returned, or
		 *    if given as a cell, an array of [row index, column index (visible),
		 *    column index (all)] is given.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      $('#example tbody td').click( function () {
		 *        // Get the position of the current data from the node
		 *        var aPos = oTable.fnGetPosition( this );
		 *
		 *        // Get the data array for this row
		 *        var aData = oTable.fnGetData( aPos[0] );
		 *
		 *        // Update the data array and return the value
		 *        aData[ aPos[1] ] = 'clicked';
		 *        this.innerHTML = 'clicked';
		 *      } );
		 *
		 *      // Init DataTables
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnGetPosition = function( node )
		{
			var api = this.api( true );
			var nodeName = node.nodeName.toUpperCase();
		
			if ( nodeName == 'TR' ) {
				return api.row( node ).index();
			}
			else if ( nodeName == 'TD' || nodeName == 'TH' ) {
				var cell = api.cell( node ).index();
		
				return [
					cell.row,
					cell.columnVisible,
					cell.column
				];
			}
			return null;
		};
		
		
		/**
		 * Check to see if a row is 'open' or not.
		 *  @param {node} nTr the table row to check
		 *  @returns {boolean} true if the row is currently open, false otherwise
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnIsOpen = function( nTr )
		{
			return this.api( true ).row( nTr ).child.isShown();
		};
		
		
		/**
		 * This function will place a new row directly after a row which is currently
		 * on display on the page, with the HTML contents that is passed into the
		 * function. This can be used, for example, to ask for confirmation that a
		 * particular record should be deleted.
		 *  @param {node} nTr The table row to 'open'
		 *  @param {string|node|jQuery} mHtml The HTML to put into the row
		 *  @param {string} sClass Class to give the new TD cell
		 *  @returns {node} The row opened. Note that if the table row passed in as the
		 *    first parameter, is not found in the table, this method will silently
		 *    return.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnOpen = function( nTr, mHtml, sClass )
		{
			return this.api( true )
				.row( nTr )
				.child( mHtml, sClass )
				.show()
				.child()[0];
		};
		
		
		/**
		 * Change the pagination - provides the internal logic for pagination in a simple API
		 * function. With this function you can have a DataTables table go to the next,
		 * previous, first or last pages.
		 *  @param {string|int} mAction Paging action to take: "first", "previous", "next" or "last"
		 *    or page number to jump to (integer), note that page 0 is the first page.
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnPageChange( 'next' );
		 *    } );
		 */
		this.fnPageChange = function ( mAction, bRedraw )
		{
			var api = this.api( true ).page( mAction );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw(false);
			}
		};
		
		
		/**
		 * Show a particular column
		 *  @param {int} iCol The column whose display should be changed
		 *  @param {bool} bShow Show (true) or hide (false) the column
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Hide the second column after initialisation
		 *      oTable.fnSetColumnVis( 1, false );
		 *    } );
		 */
		this.fnSetColumnVis = function ( iCol, bShow, bRedraw )
		{
			var api = this.api( true ).column( iCol ).visible( bShow );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.columns.adjust().draw();
			}
		};
		
		
		/**
		 * Get the settings for a particular table for external manipulation
		 *  @returns {object} DataTables settings object. See
		 *    {@link DataTable.models.oSettings}
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      var oSettings = oTable.fnSettings();
		 *
		 *      // Show an example parameter from the settings
		 *      alert( oSettings._iDisplayStart );
		 *    } );
		 */
		this.fnSettings = function()
		{
			return _fnSettingsFromNode( this[_ext.iApiIndex] );
		};
		
		
		/**
		 * Sort the table by a particular column
		 *  @param {int} iCol the data index to sort on. Note that this will not match the
		 *    'display index' if you have hidden data entries
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sort immediately with columns 0 and 1
		 *      oTable.fnSort( [ [0,'asc'], [1,'asc'] ] );
		 *    } );
		 */
		this.fnSort = function( aaSort )
		{
			this.api( true ).order( aaSort ).draw();
		};
		
		
		/**
		 * Attach a sort listener to an element for a given column
		 *  @param {node} nNode the element to attach the sort listener to
		 *  @param {int} iColumn the column that a click on this node will sort on
		 *  @param {function} [fnCallback] callback function when sort is run
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sort on column 1, when 'sorter' is clicked on
		 *      oTable.fnSortListener( document.getElementById('sorter'), 1 );
		 *    } );
		 */
		this.fnSortListener = function( nNode, iColumn, fnCallback )
		{
			this.api( true ).order.listener( nNode, iColumn, fnCallback );
		};
		
		
		/**
		 * Update a table cell or row - this method will accept either a single value to
		 * update the cell with, an array of values with one element for each column or
		 * an object in the same format as the original data source. The function is
		 * self-referencing in order to make the multi column updates easier.
		 *  @param {object|array|string} mData Data to update the cell/row with
		 *  @param {node|int} mRow TR element you want to update or the aoData index
		 *  @param {int} [iColumn] The column to update, give as null or undefined to
		 *    update a whole row.
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @param {bool} [bAction=true] Perform pre-draw actions or not
		 *  @returns {int} 0 on success, 1 on error
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnUpdate( 'Example update', 0, 0 ); // Single cell
		 *      oTable.fnUpdate( ['a', 'b', 'c', 'd', 'e'], $('tbody tr')[0] ); // Row
		 *    } );
		 */
		this.fnUpdate = function( mData, mRow, iColumn, bRedraw, bAction )
		{
			var api = this.api( true );
		
			if ( iColumn === undefined || iColumn === null ) {
				api.row( mRow ).data( mData );
			}
			else {
				api.cell( mRow, iColumn ).data( mData );
			}
		
			if ( bAction === undefined || bAction ) {
				api.columns.adjust();
			}
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
			return 0;
		};
		
		
		/**
		 * Provide a common method for plug-ins to check the version of DataTables being used, in order
		 * to ensure compatibility.
		 *  @param {string} sVersion Version string to check for, in the format "X.Y.Z". Note that the
		 *    formats "X" and "X.Y" are also acceptable.
		 *  @returns {boolean} true if this version of DataTables is greater or equal to the required
		 *    version, or false if this version of DataTales is not suitable
		 *  @method
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      alert( oTable.fnVersionCheck( '1.9.0' ) );
		 *    } );
		 */
		this.fnVersionCheck = _ext.fnVersionCheck;
		
	
		var _that = this;
		var emptyInit = options === undefined;
		var len = this.length;
	
		if ( emptyInit ) {
			options = {};
		}
	
		this.oApi = this.internal = _ext.internal;
	
		// Extend with old style plug-in API methods
		for ( var fn in DataTable.ext.internal ) {
			if ( fn ) {
				this[fn] = _fnExternApiFunc(fn);
			}
		}
	
		this.each(function() {
			// For each initialisation we want to give it a clean initialisation
			// object that can be bashed around
			var o = {};
			var oInit = len > 1 ? // optimisation for single table case
				_fnExtend( o, options, true ) :
				options;
	
			/*global oInit,_that,emptyInit*/
			var i=0, iLen, j, jLen, k, kLen;
			var sId = this.getAttribute( 'id' );
			var bInitHandedOff = false;
			var defaults = DataTable.defaults;
			var $this = $(this);
			
			
			/* Sanity check */
			if ( this.nodeName.toLowerCase() != 'table' )
			{
				_fnLog( null, 0, 'Non-table node initialisation ('+this.nodeName+')', 2 );
				return;
			}
			
			/* Backwards compatibility for the defaults */
			_fnCompatOpts( defaults );
			_fnCompatCols( defaults.column );
			
			/* Convert the camel-case defaults to Hungarian */
			_fnCamelToHungarian( defaults, defaults, true );
			_fnCamelToHungarian( defaults.column, defaults.column, true );
			
			/* Setting up the initialisation object */
			_fnCamelToHungarian( defaults, $.extend( oInit, $this.data() ), true );
			
			
			
			/* Check to see if we are re-initialising a table */
			var allSettings = DataTable.settings;
			for ( i=0, iLen=allSettings.length ; i<iLen ; i++ )
			{
				var s = allSettings[i];
			
				/* Base check on table node */
				if (
					s.nTable == this ||
					(s.nTHead && s.nTHead.parentNode == this) ||
					(s.nTFoot && s.nTFoot.parentNode == this)
				) {
					var bRetrieve = oInit.bRetrieve !== undefined ? oInit.bRetrieve : defaults.bRetrieve;
					var bDestroy = oInit.bDestroy !== undefined ? oInit.bDestroy : defaults.bDestroy;
			
					if ( emptyInit || bRetrieve )
					{
						return s.oInstance;
					}
					else if ( bDestroy )
					{
						s.oInstance.fnDestroy();
						break;
					}
					else
					{
						_fnLog( s, 0, 'Cannot reinitialise DataTable', 3 );
						return;
					}
				}
			
				/* If the element we are initialising has the same ID as a table which was previously
				 * initialised, but the table nodes don't match (from before) then we destroy the old
				 * instance by simply deleting it. This is under the assumption that the table has been
				 * destroyed by other methods. Anyone using non-id selectors will need to do this manually
				 */
				if ( s.sTableId == this.id )
				{
					allSettings.splice( i, 1 );
					break;
				}
			}
			
			/* Ensure the table has an ID - required for accessibility */
			if ( sId === null || sId === "" )
			{
				sId = "DataTables_Table_"+(DataTable.ext._unique++);
				this.id = sId;
			}
			
			/* Create the settings object for this table and set some of the default parameters */
			var oSettings = $.extend( true, {}, DataTable.models.oSettings, {
				"sDestroyWidth": $this[0].style.width,
				"sInstance":     sId,
				"sTableId":      sId
			} );
			oSettings.nTable = this;
			oSettings.oApi   = _that.internal;
			oSettings.oInit  = oInit;
			
			allSettings.push( oSettings );
			
			// Need to add the instance after the instance after the settings object has been added
			// to the settings array, so we can self reference the table instance if more than one
			oSettings.oInstance = (_that.length===1) ? _that : $this.dataTable();
			
			// Backwards compatibility, before we apply all the defaults
			_fnCompatOpts( oInit );
			_fnLanguageCompat( oInit.oLanguage );
			
			// If the length menu is given, but the init display length is not, use the length menu
			if ( oInit.aLengthMenu && ! oInit.iDisplayLength )
			{
				oInit.iDisplayLength = Array.isArray( oInit.aLengthMenu[0] ) ?
					oInit.aLengthMenu[0][0] : oInit.aLengthMenu[0];
			}
			
			// Apply the defaults and init options to make a single init object will all
			// options defined from defaults and instance options.
			oInit = _fnExtend( $.extend( true, {}, defaults ), oInit );
			
			
			// Map the initialisation options onto the settings object
			_fnMap( oSettings.oFeatures, oInit, [
				"bPaginate",
				"bLengthChange",
				"bFilter",
				"bSort",
				"bSortMulti",
				"bInfo",
				"bProcessing",
				"bAutoWidth",
				"bSortClasses",
				"bServerSide",
				"bDeferRender"
			] );
			_fnMap( oSettings, oInit, [
				"asStripeClasses",
				"ajax",
				"fnServerData",
				"fnFormatNumber",
				"sServerMethod",
				"aaSorting",
				"aaSortingFixed",
				"aLengthMenu",
				"sPaginationType",
				"sAjaxSource",
				"sAjaxDataProp",
				"iStateDuration",
				"sDom",
				"bSortCellsTop",
				"iTabIndex",
				"fnStateLoadCallback",
				"fnStateSaveCallback",
				"renderer",
				"searchDelay",
				"rowId",
				[ "iCookieDuration", "iStateDuration" ], // backwards compat
				[ "oSearch", "oPreviousSearch" ],
				[ "aoSearchCols", "aoPreSearchCols" ],
				[ "iDisplayLength", "_iDisplayLength" ]
			] );
			_fnMap( oSettings.oScroll, oInit, [
				[ "sScrollX", "sX" ],
				[ "sScrollXInner", "sXInner" ],
				[ "sScrollY", "sY" ],
				[ "bScrollCollapse", "bCollapse" ]
			] );
			_fnMap( oSettings.oLanguage, oInit, "fnInfoCallback" );
			
			/* Callback functions which are array driven */
			_fnCallbackReg( oSettings, 'aoDrawCallback',       oInit.fnDrawCallback,      'user' );
			_fnCallbackReg( oSettings, 'aoServerParams',       oInit.fnServerParams,      'user' );
			_fnCallbackReg( oSettings, 'aoStateSaveParams',    oInit.fnStateSaveParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoadParams',    oInit.fnStateLoadParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoaded',        oInit.fnStateLoaded,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCallback',        oInit.fnRowCallback,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCreatedCallback', oInit.fnCreatedRow,        'user' );
			_fnCallbackReg( oSettings, 'aoHeaderCallback',     oInit.fnHeaderCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoFooterCallback',     oInit.fnFooterCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoInitComplete',       oInit.fnInitComplete,      'user' );
			_fnCallbackReg( oSettings, 'aoPreDrawCallback',    oInit.fnPreDrawCallback,   'user' );
			
			oSettings.rowIdFn = _fnGetObjectDataFn( oInit.rowId );
			
			/* Browser support detection */
			_fnBrowserDetect( oSettings );
			
			var oClasses = oSettings.oClasses;
			
			$.extend( oClasses, DataTable.ext.classes, oInit.oClasses );
			$this.addClass( oClasses.sTable );
			
			
			if ( oSettings.iInitDisplayStart === undefined )
			{
				/* Display start point, taking into account the save saving */
				oSettings.iInitDisplayStart = oInit.iDisplayStart;
				oSettings._iDisplayStart = oInit.iDisplayStart;
			}
			
			if ( oInit.iDeferLoading !== null )
			{
				oSettings.bDeferLoading = true;
				var tmp = Array.isArray( oInit.iDeferLoading );
				oSettings._iRecordsDisplay = tmp ? oInit.iDeferLoading[0] : oInit.iDeferLoading;
				oSettings._iRecordsTotal = tmp ? oInit.iDeferLoading[1] : oInit.iDeferLoading;
			}
			
			/* Language definitions */
			var oLanguage = oSettings.oLanguage;
			$.extend( true, oLanguage, oInit.oLanguage );
			
			if ( oLanguage.sUrl )
			{
				/* Get the language definitions from a file - because this Ajax call makes the language
				 * get async to the remainder of this function we use bInitHandedOff to indicate that
				 * _fnInitialise will be fired by the returned Ajax handler, rather than the constructor
				 */
				$.ajax( {
					dataType: 'json',
					url: oLanguage.sUrl,
					success: function ( json ) {
						_fnCamelToHungarian( defaults.oLanguage, json );
						_fnLanguageCompat( json );
						$.extend( true, oLanguage, json, oSettings.oInit.oLanguage );
			
						_fnCallbackFire( oSettings, null, 'i18n', [oSettings]);
						_fnInitialise( oSettings );
					},
					error: function () {
						// Error occurred loading language file, continue on as best we can
						_fnInitialise( oSettings );
					}
				} );
				bInitHandedOff = true;
			}
			else {
				_fnCallbackFire( oSettings, null, 'i18n', [oSettings]);
			}
			
			/*
			 * Stripes
			 */
			if ( oInit.asStripeClasses === null )
			{
				oSettings.asStripeClasses =[
					oClasses.sStripeOdd,
					oClasses.sStripeEven
				];
			}
			
			/* Remove row stripe classes if they are already on the table row */
			var stripeClasses = oSettings.asStripeClasses;
			var rowOne = $this.children('tbody').find('tr').eq(0);
			if ( $.inArray( true, $.map( stripeClasses, function(el, i) {
				return rowOne.hasClass(el);
			} ) ) !== -1 ) {
				$('tbody tr', this).removeClass( stripeClasses.join(' ') );
				oSettings.asDestroyStripes = stripeClasses.slice();
			}
			
			/*
			 * Columns
			 * See if we should load columns automatically or use defined ones
			 */
			var anThs = [];
			var aoColumnsInit;
			var nThead = this.getElementsByTagName('thead');
			if ( nThead.length !== 0 )
			{
				_fnDetectHeader( oSettings.aoHeader, nThead[0] );
				anThs = _fnGetUniqueThs( oSettings );
			}
			
			/* If not given a column array, generate one with nulls */
			if ( oInit.aoColumns === null )
			{
				aoColumnsInit = [];
				for ( i=0, iLen=anThs.length ; i<iLen ; i++ )
				{
					aoColumnsInit.push( null );
				}
			}
			else
			{
				aoColumnsInit = oInit.aoColumns;
			}
			
			/* Add the columns */
			for ( i=0, iLen=aoColumnsInit.length ; i<iLen ; i++ )
			{
				_fnAddColumn( oSettings, anThs ? anThs[i] : null );
			}
			
			/* Apply the column definitions */
			_fnApplyColumnDefs( oSettings, oInit.aoColumnDefs, aoColumnsInit, function (iCol, oDef) {
				_fnColumnOptions( oSettings, iCol, oDef );
			} );
			
			/* HTML5 attribute detection - build an mData object automatically if the
			 * attributes are found
			 */
			if ( rowOne.length ) {
				var a = function ( cell, name ) {
					return cell.getAttribute( 'data-'+name ) !== null ? name : null;
				};
			
				$( rowOne[0] ).children('th, td').each( function (i, cell) {
					var col = oSettings.aoColumns[i];
			
					if ( col.mData === i ) {
						var sort = a( cell, 'sort' ) || a( cell, 'order' );
						var filter = a( cell, 'filter' ) || a( cell, 'search' );
			
						if ( sort !== null || filter !== null ) {
							col.mData = {
								_:      i+'.display',
								sort:   sort !== null   ? i+'.@data-'+sort   : undefined,
								type:   sort !== null   ? i+'.@data-'+sort   : undefined,
								filter: filter !== null ? i+'.@data-'+filter : undefined
							};
			
							_fnColumnOptions( oSettings, i );
						}
					}
				} );
			}
			
			var features = oSettings.oFeatures;
			var loadedInit = function () {
				/*
				 * Sorting
				 * @todo For modularisation (1.11) this needs to do into a sort start up handler
				 */
			
				// If aaSorting is not defined, then we use the first indicator in asSorting
				// in case that has been altered, so the default sort reflects that option
				if ( oInit.aaSorting === undefined ) {
					var sorting = oSettings.aaSorting;
					for ( i=0, iLen=sorting.length ; i<iLen ; i++ ) {
						sorting[i][1] = oSettings.aoColumns[ i ].asSorting[0];
					}
				}
			
				/* Do a first pass on the sorting classes (allows any size changes to be taken into
				 * account, and also will apply sorting disabled classes if disabled
				 */
				_fnSortingClasses( oSettings );
			
				if ( features.bSort ) {
					_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
						if ( oSettings.bSorted ) {
							var aSort = _fnSortFlatten( oSettings );
							var sortedColumns = {};
			
							$.each( aSort, function (i, val) {
								sortedColumns[ val.src ] = val.dir;
							} );
			
							_fnCallbackFire( oSettings, null, 'order', [oSettings, aSort, sortedColumns] );
							_fnSortAria( oSettings );
						}
					} );
				}
			
				_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
					if ( oSettings.bSorted || _fnDataSource( oSettings ) === 'ssp' || features.bDeferRender ) {
						_fnSortingClasses( oSettings );
					}
				}, 'sc' );
			
			
				/*
				 * Final init
				 * Cache the header, body and footer as required, creating them if needed
				 */
			
				// Work around for Webkit bug 83867 - store the caption-side before removing from doc
				var captions = $this.children('caption').each( function () {
					this._captionSide = $(this).css('caption-side');
				} );
			
				var thead = $this.children('thead');
				if ( thead.length === 0 ) {
					thead = $('<thead/>').appendTo($this);
				}
				oSettings.nTHead = thead[0];
			
				var tbody = $this.children('tbody');
				if ( tbody.length === 0 ) {
					tbody = $('<tbody/>').insertAfter(thead);
				}
				oSettings.nTBody = tbody[0];
			
				var tfoot = $this.children('tfoot');
				if ( tfoot.length === 0 && captions.length > 0 && (oSettings.oScroll.sX !== "" || oSettings.oScroll.sY !== "") ) {
					// If we are a scrolling table, and no footer has been given, then we need to create
					// a tfoot element for the caption element to be appended to
					tfoot = $('<tfoot/>').appendTo($this);
				}
			
				if ( tfoot.length === 0 || tfoot.children().length === 0 ) {
					$this.addClass( oClasses.sNoFooter );
				}
				else if ( tfoot.length > 0 ) {
					oSettings.nTFoot = tfoot[0];
					_fnDetectHeader( oSettings.aoFooter, oSettings.nTFoot );
				}
			
				/* Check if there is data passing into the constructor */
				if ( oInit.aaData ) {
					for ( i=0 ; i<oInit.aaData.length ; i++ ) {
						_fnAddData( oSettings, oInit.aaData[ i ] );
					}
				}
				else if ( oSettings.bDeferLoading || _fnDataSource( oSettings ) == 'dom' ) {
					/* Grab the data from the page - only do this when deferred loading or no Ajax
					 * source since there is no point in reading the DOM data if we are then going
					 * to replace it with Ajax data
					 */
					_fnAddTr( oSettings, $(oSettings.nTBody).children('tr') );
				}
			
				/* Copy the data index array */
				oSettings.aiDisplay = oSettings.aiDisplayMaster.slice();
			
				/* Initialisation complete - table can be drawn */
				oSettings.bInitialised = true;
			
				/* Check if we need to initialise the table (it might not have been handed off to the
				 * language processor)
				 */
				if ( bInitHandedOff === false ) {
					_fnInitialise( oSettings );
				}
			};
			
			/* Must be done after everything which can be overridden by the state saving! */
			_fnCallbackReg( oSettings, 'aoDrawCallback', _fnSaveState, 'state_save' );
			
			if ( oInit.bStateSave )
			{
				features.bStateSave = true;
				_fnLoadState( oSettings, oInit, loadedInit );
			}
			else {
				loadedInit();
			}
			
		} );
		_that = null;
		return this;
	};
	
	
	/*
	 * It is useful to have variables which are scoped locally so only the
	 * DataTables functions can access them and they don't leak into global space.
	 * At the same time these functions are often useful over multiple files in the
	 * core and API, so we list, or at least document, all variables which are used
	 * by DataTables as private variables here. This also ensures that there is no
	 * clashing of variable names and that they can easily referenced for reuse.
	 */
	
	
	// Defined else where
	//  _selector_run
	//  _selector_opts
	//  _selector_first
	//  _selector_row_indexes
	
	var _ext; // DataTable.ext
	var _Api; // DataTable.Api
	var _api_register; // DataTable.Api.register
	var _api_registerPlural; // DataTable.Api.registerPlural
	
	var _re_dic = {};
	var _re_new_lines = /[\r\n\u2028]/g;
	var _re_html = /<.*?>/g;
	
	// This is not strict ISO8601 - Date.parse() is quite lax, although
	// implementations differ between browsers.
	var _re_date = /^\d{2,4}[\.\/\-]\d{1,2}[\.\/\-]\d{1,2}([T ]{1}\d{1,2}[:\.]\d{2}([\.:]\d{2})?)?$/;
	
	// Escape regular expression special characters
	var _re_escape_regex = new RegExp( '(\\' + [ '/', '.', '*', '+', '?', '|', '(', ')', '[', ']', '{', '}', '\\', '$', '^', '-' ].join('|\\') + ')', 'g' );
	
	// http://en.wikipedia.org/wiki/Foreign_exchange_market
	// - \u20BD - Russian ruble.
	// - \u20a9 - South Korean Won
	// - \u20BA - Turkish Lira
	// - \u20B9 - Indian Rupee
	// - R - Brazil (R$) and South Africa
	// - fr - Swiss Franc
	// - kr - Swedish krona, Norwegian krone and Danish krone
	// - \u2009 is thin space and \u202F is narrow no-break space, both used in many
	// - Ƀ - Bitcoin
	// - Ξ - Ethereum
	//   standards as thousands separators.
	var _re_formatted_numeric = /['\u00A0,$£€¥%\u2009\u202F\u20BD\u20a9\u20BArfkɃΞ]/gi;
	
	
	var _empty = function ( d ) {
		return !d || d === true || d === '-' ? true : false;
	};
	
	
	var _intVal = function ( s ) {
		var integer = parseInt( s, 10 );
		return !isNaN(integer) && isFinite(s) ? integer : null;
	};
	
	// Convert from a formatted number with characters other than `.` as the
	// decimal place, to a Javascript number
	var _numToDecimal = function ( num, decimalPoint ) {
		// Cache created regular expressions for speed as this function is called often
		if ( ! _re_dic[ decimalPoint ] ) {
			_re_dic[ decimalPoint ] = new RegExp( _fnEscapeRegex( decimalPoint ), 'g' );
		}
		return typeof num === 'string' && decimalPoint !== '.' ?
			num.replace( /\./g, '' ).replace( _re_dic[ decimalPoint ], '.' ) :
			num;
	};
	
	
	var _isNumber = function ( d, decimalPoint, formatted ) {
		var strType = typeof d === 'string';
	
		// If empty return immediately so there must be a number if it is a
		// formatted string (this stops the string "k", or "kr", etc being detected
		// as a formatted number for currency
		if ( _empty( d ) ) {
			return true;
		}
	
		if ( decimalPoint && strType ) {
			d = _numToDecimal( d, decimalPoint );
		}
	
		if ( formatted && strType ) {
			d = d.replace( _re_formatted_numeric, '' );
		}
	
		return !isNaN( parseFloat(d) ) && isFinite( d );
	};
	
	
	// A string without HTML in it can be considered to be HTML still
	var _isHtml = function ( d ) {
		return _empty( d ) || typeof d === 'string';
	};
	
	
	var _htmlNumeric = function ( d, decimalPoint, formatted ) {
		if ( _empty( d ) ) {
			return true;
		}
	
		var html = _isHtml( d );
		return ! html ?
			null :
			_isNumber( _stripHtml( d ), decimalPoint, formatted ) ?
				true :
				null;
	};
	
	
	var _pluck = function ( a, prop, prop2 ) {
		var out = [];
		var i=0, ien=a.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[i] && a[i][ prop ] ) {
					out.push( a[i][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[i] ) {
					out.push( a[i][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	// Basically the same as _pluck, but rather than looping over `a` we use `order`
	// as the indexes to pick from `a`
	var _pluck_order = function ( a, order, prop, prop2 )
	{
		var out = [];
		var i=0, ien=order.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ][ prop ] ) {
					out.push( a[ order[i] ][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				out.push( a[ order[i] ][ prop ] );
			}
		}
	
		return out;
	};
	
	
	var _range = function ( len, start )
	{
		var out = [];
		var end;
	
		if ( start === undefined ) {
			start = 0;
			end = len;
		}
		else {
			end = start;
			start = len;
		}
	
		for ( var i=start ; i<end ; i++ ) {
			out.push( i );
		}
	
		return out;
	};
	
	
	var _removeEmpty = function ( a )
	{
		var out = [];
	
		for ( var i=0, ien=a.length ; i<ien ; i++ ) {
			if ( a[i] ) { // careful - will remove all falsy values!
				out.push( a[i] );
			}
		}
	
		return out;
	};
	
	
	var _stripHtml = function ( d ) {
		return d.replace( _re_html, '' );
	};
	
	
	/**
	 * Determine if all values in the array are unique. This means we can short
	 * cut the _unique method at the cost of a single loop. A sorted array is used
	 * to easily check the values.
	 *
	 * @param  {array} src Source array
	 * @return {boolean} true if all unique, false otherwise
	 * @ignore
	 */
	var _areAllUnique = function ( src ) {
		if ( src.length < 2 ) {
			return true;
		}
	
		var sorted = src.slice().sort();
		var last = sorted[0];
	
		for ( var i=1, ien=sorted.length ; i<ien ; i++ ) {
			if ( sorted[i] === last ) {
				return false;
			}
	
			last = sorted[i];
		}
	
		return true;
	};
	
	
	/**
	 * Find the unique elements in a source array.
	 *
	 * @param  {array} src Source array
	 * @return {array} Array of unique items
	 * @ignore
	 */
	var _unique = function ( src )
	{
		if ( _areAllUnique( src ) ) {
			return src.slice();
		}
	
		// A faster unique method is to use object keys to identify used values,
		// but this doesn't work with arrays or objects, which we must also
		// consider. See jsperf.com/compare-array-unique-versions/4 for more
		// information.
		var
			out = [],
			val,
			i, ien=src.length,
			j, k=0;
	
		again: for ( i=0 ; i<ien ; i++ ) {
			val = src[i];
	
			for ( j=0 ; j<k ; j++ ) {
				if ( out[j] === val ) {
					continue again;
				}
			}
	
			out.push( val );
			k++;
		}
	
		return out;
	};
	
	// Surprisingly this is faster than [].concat.apply
	// https://jsperf.com/flatten-an-array-loop-vs-reduce/2
	var _flatten = function (out, val) {
		if (Array.isArray(val)) {
			for (var i=0 ; i<val.length ; i++) {
				_flatten(out, val[i]);
			}
		}
		else {
			out.push(val);
		}
	  
		return out;
	}
	
	var _includes = function (search, start) {
		if (start === undefined) {
			start = 0;
		}
	
		return this.indexOf(search, start) !== -1;	
	};
	
	// Array.isArray polyfill.
	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/isArray
	if (! Array.isArray) {
	    Array.isArray = function(arg) {
	        return Object.prototype.toString.call(arg) === '[object Array]';
	    };
	}
	
	if (! Array.prototype.includes) {
		Array.prototype.includes = _includes;
	}
	
	// .trim() polyfill
	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/trim
	if (!String.prototype.trim) {
	  String.prototype.trim = function () {
	    return this.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g, '');
	  };
	}
	
	if (! String.prototype.includes) {
		String.prototype.includes = _includes;
	}
	
	/**
	 * DataTables utility methods
	 * 
	 * This namespace provides helper methods that DataTables uses internally to
	 * create a DataTable, but which are not exclusively used only for DataTables.
	 * These methods can be used by extension authors to save the duplication of
	 * code.
	 *
	 *  @namespace
	 */
	DataTable.util = {
		/**
		 * Throttle the calls to a function. Arguments and context are maintained
		 * for the throttled function.
		 *
		 * @param {function} fn Function to be called
		 * @param {integer} freq Call frequency in mS
		 * @return {function} Wrapped function
		 */
		throttle: function ( fn, freq ) {
			var
				frequency = freq !== undefined ? freq : 200,
				last,
				timer;
	
			return function () {
				var
					that = this,
					now  = +new Date(),
					args = arguments;
	
				if ( last && now < last + frequency ) {
					clearTimeout( timer );
	
					timer = setTimeout( function () {
						last = undefined;
						fn.apply( that, args );
					}, frequency );
				}
				else {
					last = now;
					fn.apply( that, args );
				}
			};
		},
	
	
		/**
		 * Escape a string such that it can be used in a regular expression
		 *
		 *  @param {string} val string to escape
		 *  @returns {string} escaped string
		 */
		escapeRegex: function ( val ) {
			return val.replace( _re_escape_regex, '\\$1' );
		},
	
		/**
		 * Create a function that will write to a nested object or array
		 * @param {*} source JSON notation string
		 * @returns Write function
		 */
		set: function ( source ) {
			if ( $.isPlainObject( source ) ) {
				/* Unlike get, only the underscore (global) option is used for for
				 * setting data since we don't know the type here. This is why an object
				 * option is not documented for `mData` (which is read/write), but it is
				 * for `mRender` which is read only.
				 */
				return DataTable.util.set( source._ );
			}
			else if ( source === null ) {
				// Nothing to do when the data source is null
				return function () {};
			}
			else if ( typeof source === 'function' ) {
				return function (data, val, meta) {
					source( data, 'set', val, meta );
				};
			}
			else if ( typeof source === 'string' && (source.indexOf('.') !== -1 ||
					  source.indexOf('[') !== -1 || source.indexOf('(') !== -1) )
			{
				// Like the get, we need to get data from a nested object
				var setData = function (data, val, src) {
					var a = _fnSplitObjNotation( src ), b;
					var aLast = a[a.length-1];
					var arrayNotation, funcNotation, o, innerSrc;
		
					for ( var i=0, iLen=a.length-1 ; i<iLen ; i++ ) {
						// Protect against prototype pollution
						if (a[i] === '__proto__' || a[i] === 'constructor') {
							throw new Error('Cannot set prototype values');
						}
		
						// Check if we are dealing with an array notation request
						arrayNotation = a[i].match(__reArray);
						funcNotation = a[i].match(__reFn);
		
						if ( arrayNotation ) {
							a[i] = a[i].replace(__reArray, '');
							data[ a[i] ] = [];
		
							// Get the remainder of the nested object to set so we can recurse
							b = a.slice();
							b.splice( 0, i+1 );
							innerSrc = b.join('.');
		
							// Traverse each entry in the array setting the properties requested
							if ( Array.isArray( val ) ) {
								for ( var j=0, jLen=val.length ; j<jLen ; j++ ) {
									o = {};
									setData( o, val[j], innerSrc );
									data[ a[i] ].push( o );
								}
							}
							else {
								// We've been asked to save data to an array, but it
								// isn't array data to be saved. Best that can be done
								// is to just save the value.
								data[ a[i] ] = val;
							}
		
							// The inner call to setData has already traversed through the remainder
							// of the source and has set the data, thus we can exit here
							return;
						}
						else if ( funcNotation ) {
							// Function call
							a[i] = a[i].replace(__reFn, '');
							data = data[ a[i] ]( val );
						}
		
						// If the nested object doesn't currently exist - since we are
						// trying to set the value - create it
						if ( data[ a[i] ] === null || data[ a[i] ] === undefined ) {
							data[ a[i] ] = {};
						}
						data = data[ a[i] ];
					}
		
					// Last item in the input - i.e, the actual set
					if ( aLast.match(__reFn ) ) {
						// Function call
						data = data[ aLast.replace(__reFn, '') ]( val );
					}
					else {
						// If array notation is used, we just want to strip it and use the property name
						// and assign the value. If it isn't used, then we get the result we want anyway
						data[ aLast.replace(__reArray, '') ] = val;
					}
				};
		
				return function (data, val) { // meta is also passed in, but not used
					return setData( data, val, source );
				};
			}
			else {
				// Array or flat object mapping
				return function (data, val) { // meta is also passed in, but not used
					data[source] = val;
				};
			}
		},
	
		/**
		 * Create a function that will read nested objects from arrays, based on JSON notation
		 * @param {*} source JSON notation string
		 * @returns Value read
		 */
		get: function ( source ) {
			if ( $.isPlainObject( source ) ) {
				// Build an object of get functions, and wrap them in a single call
				var o = {};
				$.each( source, function (key, val) {
					if ( val ) {
						o[key] = DataTable.util.get( val );
					}
				} );
		
				return function (data, type, row, meta) {
					var t = o[type] || o._;
					return t !== undefined ?
						t(data, type, row, meta) :
						data;
				};
			}
			else if ( source === null ) {
				// Give an empty string for rendering / sorting etc
				return function (data) { // type, row and meta also passed, but not used
					return data;
				};
			}
			else if ( typeof source === 'function' ) {
				return function (data, type, row, meta) {
					return source( data, type, row, meta );
				};
			}
			else if ( typeof source === 'string' && (source.indexOf('.') !== -1 ||
					  source.indexOf('[') !== -1 || source.indexOf('(') !== -1) )
			{
				/* If there is a . in the source string then the data source is in a
				 * nested object so we loop over the data for each level to get the next
				 * level down. On each loop we test for undefined, and if found immediately
				 * return. This allows entire objects to be missing and sDefaultContent to
				 * be used if defined, rather than throwing an error
				 */
				var fetchData = function (data, type, src) {
					var arrayNotation, funcNotation, out, innerSrc;
		
					if ( src !== "" ) {
						var a = _fnSplitObjNotation( src );
		
						for ( var i=0, iLen=a.length ; i<iLen ; i++ ) {
							// Check if we are dealing with special notation
							arrayNotation = a[i].match(__reArray);
							funcNotation = a[i].match(__reFn);
		
							if ( arrayNotation ) {
								// Array notation
								a[i] = a[i].replace(__reArray, '');
		
								// Condition allows simply [] to be passed in
								if ( a[i] !== "" ) {
									data = data[ a[i] ];
								}
								out = [];
		
								// Get the remainder of the nested object to get
								a.splice( 0, i+1 );
								innerSrc = a.join('.');
		
								// Traverse each entry in the array getting the properties requested
								if ( Array.isArray( data ) ) {
									for ( var j=0, jLen=data.length ; j<jLen ; j++ ) {
										out.push( fetchData( data[j], type, innerSrc ) );
									}
								}
		
								// If a string is given in between the array notation indicators, that
								// is used to join the strings together, otherwise an array is returned
								var join = arrayNotation[0].substring(1, arrayNotation[0].length-1);
								data = (join==="") ? out : out.join(join);
		
								// The inner call to fetchData has already traversed through the remainder
								// of the source requested, so we exit from the loop
								break;
							}
							else if ( funcNotation ) {
								// Function call
								a[i] = a[i].replace(__reFn, '');
								data = data[ a[i] ]();
								continue;
							}
		
							if ( data === null || data[ a[i] ] === undefined ) {
								return undefined;
							}
	
							data = data[ a[i] ];
						}
					}
		
					return data;
				};
		
				return function (data, type) { // row and meta also passed, but not used
					return fetchData( data, type, source );
				};
			}
			else {
				// Array or flat object mapping
				return function (data, type) { // row and meta also passed, but not used
					return data[source];
				};
			}
		}
	};
	
	
	
	/**
	 * Create a mapping object that allows camel case parameters to be looked up
	 * for their Hungarian counterparts. The mapping is stored in a private
	 * parameter called `_hungarianMap` which can be accessed on the source object.
	 *  @param {object} o
	 *  @memberof DataTable#oApi
	 */
	function _fnHungarianMap ( o )
	{
		var
			hungarian = 'a aa ai ao as b fn i m o s ',
			match,
			newKey,
			map = {};
	
		$.each( o, function (key, val) {
			match = key.match(/^([^A-Z]+?)([A-Z])/);
	
			if ( match && hungarian.indexOf(match[1]+' ') !== -1 )
			{
				newKey = key.replace( match[0], match[2].toLowerCase() );
				map[ newKey ] = key;
	
				if ( match[1] === 'o' )
				{
					_fnHungarianMap( o[key] );
				}
			}
		} );
	
		o._hungarianMap = map;
	}
	
	
	/**
	 * Convert from camel case parameters to Hungarian, based on a Hungarian map
	 * created by _fnHungarianMap.
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 *  @memberof DataTable#oApi
	 */
	function _fnCamelToHungarian ( src, user, force )
	{
		if ( ! src._hungarianMap ) {
			_fnHungarianMap( src );
		}
	
		var hungarianKey;
	
		$.each( user, function (key, val) {
			hungarianKey = src._hungarianMap[ key ];
	
			if ( hungarianKey !== undefined && (force || user[hungarianKey] === undefined) )
			{
				// For objects, we need to buzz down into the object to copy parameters
				if ( hungarianKey.charAt(0) === 'o' )
				{
					// Copy the camelCase options over to the hungarian
					if ( ! user[ hungarianKey ] ) {
						user[ hungarianKey ] = {};
					}
					$.extend( true, user[hungarianKey], user[key] );
	
					_fnCamelToHungarian( src[hungarianKey], user[hungarianKey], force );
				}
				else {
					user[hungarianKey] = user[ key ];
				}
			}
		} );
	}
	
	
	/**
	 * Language compatibility - when certain options are given, and others aren't, we
	 * need to duplicate the values over, in order to provide backwards compatibility
	 * with older language files.
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnLanguageCompat( lang )
	{
		// Note the use of the Hungarian notation for the parameters in this method as
		// this is called after the mapping of camelCase to Hungarian
		var defaults = DataTable.defaults.oLanguage;
	
		// Default mapping
		var defaultDecimal = defaults.sDecimal;
		if ( defaultDecimal ) {
			_addNumericSort( defaultDecimal );
		}
	
		if ( lang ) {
			var zeroRecords = lang.sZeroRecords;
	
			// Backwards compatibility - if there is no sEmptyTable given, then use the same as
			// sZeroRecords - assuming that is given.
			if ( ! lang.sEmptyTable && zeroRecords &&
				defaults.sEmptyTable === "No data available in table" )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sEmptyTable' );
			}
	
			// Likewise with loading records
			if ( ! lang.sLoadingRecords && zeroRecords &&
				defaults.sLoadingRecords === "Loading..." )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sLoadingRecords' );
			}
	
			// Old parameter name of the thousands separator mapped onto the new
			if ( lang.sInfoThousands ) {
				lang.sThousands = lang.sInfoThousands;
			}
	
			var decimal = lang.sDecimal;
			if ( decimal && defaultDecimal !== decimal ) {
				_addNumericSort( decimal );
			}
		}
	}
	
	
	/**
	 * Map one parameter onto another
	 *  @param {object} o Object to map
	 *  @param {*} knew The new parameter name
	 *  @param {*} old The old parameter name
	 */
	var _fnCompatMap = function ( o, knew, old ) {
		if ( o[ knew ] !== undefined ) {
			o[ old ] = o[ knew ];
		}
	};
	
	
	/**
	 * Provide backwards compatibility for the main DT options. Note that the new
	 * options are mapped onto the old parameters, so this is an external interface
	 * change only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatOpts ( init )
	{
		_fnCompatMap( init, 'ordering',      'bSort' );
		_fnCompatMap( init, 'orderMulti',    'bSortMulti' );
		_fnCompatMap( init, 'orderClasses',  'bSortClasses' );
		_fnCompatMap( init, 'orderCellsTop', 'bSortCellsTop' );
		_fnCompatMap( init, 'order',         'aaSorting' );
		_fnCompatMap( init, 'orderFixed',    'aaSortingFixed' );
		_fnCompatMap( init, 'paging',        'bPaginate' );
		_fnCompatMap( init, 'pagingType',    'sPaginationType' );
		_fnCompatMap( init, 'pageLength',    'iDisplayLength' );
		_fnCompatMap( init, 'searching',     'bFilter' );
	
		// Boolean initialisation of x-scrolling
		if ( typeof init.sScrollX === 'boolean' ) {
			init.sScrollX = init.sScrollX ? '100%' : '';
		}
		if ( typeof init.scrollX === 'boolean' ) {
			init.scrollX = init.scrollX ? '100%' : '';
		}
	
		// Column search objects are in an array, so it needs to be converted
		// element by element
		var searchCols = init.aoSearchCols;
	
		if ( searchCols ) {
			for ( var i=0, ien=searchCols.length ; i<ien ; i++ ) {
				if ( searchCols[i] ) {
					_fnCamelToHungarian( DataTable.models.oSearch, searchCols[i] );
				}
			}
		}
	}
	
	
	/**
	 * Provide backwards compatibility for column options. Note that the new options
	 * are mapped onto the old parameters, so this is an external interface change
	 * only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatCols ( init )
	{
		_fnCompatMap( init, 'orderable',     'bSortable' );
		_fnCompatMap( init, 'orderData',     'aDataSort' );
		_fnCompatMap( init, 'orderSequence', 'asSorting' );
		_fnCompatMap( init, 'orderDataType', 'sortDataType' );
	
		// orderData can be given as an integer
		var dataSort = init.aDataSort;
		if ( typeof dataSort === 'number' && ! Array.isArray( dataSort ) ) {
			init.aDataSort = [ dataSort ];
		}
	}
	
	
	/**
	 * Browser feature detection for capabilities, quirks
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBrowserDetect( settings )
	{
		// We don't need to do this every time DataTables is constructed, the values
		// calculated are specific to the browser and OS configuration which we
		// don't expect to change between initialisations
		if ( ! DataTable.__browser ) {
			var browser = {};
			DataTable.__browser = browser;
	
			// Scrolling feature / quirks detection
			var n = $('<div/>')
				.css( {
					position: 'fixed',
					top: 0,
					left: $(window).scrollLeft()*-1, // allow for scrolling
					height: 1,
					width: 1,
					overflow: 'hidden'
				} )
				.append(
					$('<div/>')
						.css( {
							position: 'absolute',
							top: 1,
							left: 1,
							width: 100,
							overflow: 'scroll'
						} )
						.append(
							$('<div/>')
								.css( {
									width: '100%',
									height: 10
								} )
						)
				)
				.appendTo( 'body' );
	
			var outer = n.children();
			var inner = outer.children();
	
			// Numbers below, in order, are:
			// inner.offsetWidth, inner.clientWidth, outer.offsetWidth, outer.clientWidth
			//
			// IE6 XP:                           100 100 100  83
			// IE7 Vista:                        100 100 100  83
			// IE 8+ Windows:                     83  83 100  83
			// Evergreen Windows:                 83  83 100  83
			// Evergreen Mac with scrollbars:     85  85 100  85
			// Evergreen Mac without scrollbars: 100 100 100 100
	
			// Get scrollbar width
			browser.barWidth = outer[0].offsetWidth - outer[0].clientWidth;
	
			// IE6/7 will oversize a width 100% element inside a scrolling element, to
			// include the width of the scrollbar, while other browsers ensure the inner
			// element is contained without forcing scrolling
			browser.bScrollOversize = inner[0].offsetWidth === 100 && outer[0].clientWidth !== 100;
	
			// In rtl text layout, some browsers (most, but not all) will place the
			// scrollbar on the left, rather than the right.
			browser.bScrollbarLeft = Math.round( inner.offset().left ) !== 1;
	
			// IE8- don't provide height and width for getBoundingClientRect
			browser.bBounding = n[0].getBoundingClientRect().width ? true : false;
	
			n.remove();
		}
	
		$.extend( settings.oBrowser, DataTable.__browser );
		settings.oScroll.iBarWidth = DataTable.__browser.barWidth;
	}
	
	
	/**
	 * Array.prototype reduce[Right] method, used for browsers which don't support
	 * JS 1.6. Done this way to reduce code size, since we iterate either way
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnReduce ( that, fn, init, start, end, inc )
	{
		var
			i = start,
			value,
			isSet = false;
	
		if ( init !== undefined ) {
			value = init;
			isSet = true;
		}
	
		while ( i !== end ) {
			if ( ! that.hasOwnProperty(i) ) {
				continue;
			}
	
			value = isSet ?
				fn( value, that[i], i, that ) :
				that[i];
	
			isSet = true;
			i += inc;
		}
	
		return value;
	}
	
	/**
	 * Add a column to the list used for the table with default values
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} nTh The th element for this column
	 *  @memberof DataTable#oApi
	 */
	function _fnAddColumn( oSettings, nTh )
	{
		// Add column to aoColumns array
		var oDefaults = DataTable.defaults.column;
		var iCol = oSettings.aoColumns.length;
		var oCol = $.extend( {}, DataTable.models.oColumn, oDefaults, {
			"nTh": nTh ? nTh : document.createElement('th'),
			"sTitle":    oDefaults.sTitle    ? oDefaults.sTitle    : nTh ? nTh.innerHTML : '',
			"aDataSort": oDefaults.aDataSort ? oDefaults.aDataSort : [iCol],
			"mData": oDefaults.mData ? oDefaults.mData : iCol,
			idx: iCol
		} );
		oSettings.aoColumns.push( oCol );
	
		// Add search object for column specific search. Note that the `searchCols[ iCol ]`
		// passed into extend can be undefined. This allows the user to give a default
		// with only some of the parameters defined, and also not give a default
		var searchCols = oSettings.aoPreSearchCols;
		searchCols[ iCol ] = $.extend( {}, DataTable.models.oSearch, searchCols[ iCol ] );
	
		// Use the default column options function to initialise classes etc
		_fnColumnOptions( oSettings, iCol, $(nTh).data() );
	}
	
	
	/**
	 * Apply options for a column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iCol column index to consider
	 *  @param {object} oOptions object with sType, bVisible and bSearchable etc
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnOptions( oSettings, iCol, oOptions )
	{
		var oCol = oSettings.aoColumns[ iCol ];
		var oClasses = oSettings.oClasses;
		var th = $(oCol.nTh);
	
		// Try to get width information from the DOM. We can't get it from CSS
		// as we'd need to parse the CSS stylesheet. `width` option can override
		if ( ! oCol.sWidthOrig ) {
			// Width attribute
			oCol.sWidthOrig = th.attr('width') || null;
	
			// Style attribute
			var t = (th.attr('style') || '').match(/width:\s*(\d+[pxem%]+)/);
			if ( t ) {
				oCol.sWidthOrig = t[1];
			}
		}
	
		/* User specified column options */
		if ( oOptions !== undefined && oOptions !== null )
		{
			// Backwards compatibility
			_fnCompatCols( oOptions );
	
			// Map camel case parameters to their Hungarian counterparts
			_fnCamelToHungarian( DataTable.defaults.column, oOptions, true );
	
			/* Backwards compatibility for mDataProp */
			if ( oOptions.mDataProp !== undefined && !oOptions.mData )
			{
				oOptions.mData = oOptions.mDataProp;
			}
	
			if ( oOptions.sType )
			{
				oCol._sManualType = oOptions.sType;
			}
	
			// `class` is a reserved word in Javascript, so we need to provide
			// the ability to use a valid name for the camel case input
			if ( oOptions.className && ! oOptions.sClass )
			{
				oOptions.sClass = oOptions.className;
			}
			if ( oOptions.sClass ) {
				th.addClass( oOptions.sClass );
			}
	
			var origClass = oCol.sClass;
	
			$.extend( oCol, oOptions );
			_fnMap( oCol, oOptions, "sWidth", "sWidthOrig" );
	
			// Merge class from previously defined classes with this one, rather than just
			// overwriting it in the extend above
			if (origClass !== oCol.sClass) {
				oCol.sClass = origClass + ' ' + oCol.sClass;
			}
	
			/* iDataSort to be applied (backwards compatibility), but aDataSort will take
			 * priority if defined
			 */
			if ( oOptions.iDataSort !== undefined )
			{
				oCol.aDataSort = [ oOptions.iDataSort ];
			}
			_fnMap( oCol, oOptions, "aDataSort" );
		}
	
		/* Cache the data get and set functions for speed */
		var mDataSrc = oCol.mData;
		var mData = _fnGetObjectDataFn( mDataSrc );
		var mRender = oCol.mRender ? _fnGetObjectDataFn( oCol.mRender ) : null;
	
		var attrTest = function( src ) {
			return typeof src === 'string' && src.indexOf('@') !== -1;
		};
		oCol._bAttrSrc = $.isPlainObject( mDataSrc ) && (
			attrTest(mDataSrc.sort) || attrTest(mDataSrc.type) || attrTest(mDataSrc.filter)
		);
		oCol._setter = null;
	
		oCol.fnGetData = function (rowData, type, meta) {
			var innerData = mData( rowData, type, undefined, meta );
	
			return mRender && type ?
				mRender( innerData, type, rowData, meta ) :
				innerData;
		};
		oCol.fnSetData = function ( rowData, val, meta ) {
			return _fnSetObjectDataFn( mDataSrc )( rowData, val, meta );
		};
	
		// Indicate if DataTables should read DOM data as an object or array
		// Used in _fnGetRowElements
		if ( typeof mDataSrc !== 'number' ) {
			oSettings._rowReadObject = true;
		}
	
		/* Feature sorting overrides column specific when off */
		if ( !oSettings.oFeatures.bSort )
		{
			oCol.bSortable = false;
			th.addClass( oClasses.sSortableNone ); // Have to add class here as order event isn't called
		}
	
		/* Check that the class assignment is correct for sorting */
		var bAsc = $.inArray('asc', oCol.asSorting) !== -1;
		var bDesc = $.inArray('desc', oCol.asSorting) !== -1;
		if ( !oCol.bSortable || (!bAsc && !bDesc) )
		{
			oCol.sSortingClass = oClasses.sSortableNone;
			oCol.sSortingClassJUI = "";
		}
		else if ( bAsc && !bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableAsc;
			oCol.sSortingClassJUI = oClasses.sSortJUIAscAllowed;
		}
		else if ( !bAsc && bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableDesc;
			oCol.sSortingClassJUI = oClasses.sSortJUIDescAllowed;
		}
		else
		{
			oCol.sSortingClass = oClasses.sSortable;
			oCol.sSortingClassJUI = oClasses.sSortJUI;
		}
	}
	
	
	/**
	 * Adjust the table column widths for new data. Note: you would probably want to
	 * do a redraw after calling this function!
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAdjustColumnSizing ( settings )
	{
		/* Not interested in doing column width calculation if auto-width is disabled */
		if ( settings.oFeatures.bAutoWidth !== false )
		{
			var columns = settings.aoColumns;
	
			_fnCalculateColumnWidths( settings );
			for ( var i=0 , iLen=columns.length ; i<iLen ; i++ )
			{
				columns[i].nTh.style.width = columns[i].sWidth;
			}
		}
	
		var scroll = settings.oScroll;
		if ( scroll.sY !== '' || scroll.sX !== '')
		{
			_fnScrollDraw( settings );
		}
	
		_fnCallbackFire( settings, null, 'column-sizing', [settings] );
	}
	
	
	/**
	 * Convert the index of a visible column to the index in the data array (take account
	 * of hidden columns)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iMatch Visible column index to lookup
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnVisibleToColumnIndex( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
	
		return typeof aiVis[iMatch] === 'number' ?
			aiVis[iMatch] :
			null;
	}
	
	
	/**
	 * Convert the index of an index in the data array and convert it to the visible
	 *   column index (take account of hidden columns)
	 *  @param {int} iMatch Column index to lookup
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnIndexToVisible( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
		var iPos = $.inArray( iMatch, aiVis );
	
		return iPos !== -1 ? iPos : null;
	}
	
	
	/**
	 * Get the number of visible columns
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the number of visible columns
	 *  @memberof DataTable#oApi
	 */
	function _fnVisbleColumns( oSettings )
	{
		var vis = 0;
	
		// No reduce in IE8, use a loop for now
		$.each( oSettings.aoColumns, function ( i, col ) {
			if ( col.bVisible && $(col.nTh).css('display') !== 'none' ) {
				vis++;
			}
		} );
	
		return vis;
	}
	
	
	/**
	 * Get an array of column indexes that match a given property
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sParam Parameter in aoColumns to look for - typically
	 *    bVisible or bSearchable
	 *  @returns {array} Array of indexes with matched properties
	 *  @memberof DataTable#oApi
	 */
	function _fnGetColumns( oSettings, sParam )
	{
		var a = [];
	
		$.map( oSettings.aoColumns, function(val, i) {
			if ( val[sParam] ) {
				a.push( i );
			}
		} );
	
		return a;
	}
	
	
	/**
	 * Calculate the 'type' of a column
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnTypes ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var types = DataTable.ext.type.detect;
		var i, ien, j, jen, k, ken;
		var col, cell, detectedType, cache;
	
		// For each column, spin over the 
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			col = columns[i];
			cache = [];
	
			if ( ! col.sType && col._sManualType ) {
				col.sType = col._sManualType;
			}
			else if ( ! col.sType ) {
				for ( j=0, jen=types.length ; j<jen ; j++ ) {
					for ( k=0, ken=data.length ; k<ken ; k++ ) {
						// Use a cache array so we only need to get the type data
						// from the formatter once (when using multiple detectors)
						if ( cache[k] === undefined ) {
							cache[k] = _fnGetCellData( settings, k, i, 'type' );
						}
	
						detectedType = types[j]( cache[k], settings );
	
						// If null, then this type can't apply to this column, so
						// rather than testing all cells, break out. There is an
						// exception for the last type which is `html`. We need to
						// scan all rows since it is possible to mix string and HTML
						// types
						if ( ! detectedType && j !== types.length-1 ) {
							break;
						}
	
						// Only a single match is needed for html type since it is
						// bottom of the pile and very similar to string - but it
						// must not be empty
						if ( detectedType === 'html' && ! _empty(cache[k]) ) {
							break;
						}
					}
	
					// Type is valid for all data points in the column - use this
					// type
					if ( detectedType ) {
						col.sType = detectedType;
						break;
					}
				}
	
				// Fall back - if no type was detected, always use string
				if ( ! col.sType ) {
					col.sType = 'string';
				}
			}
		}
	}
	
	
	/**
	 * Take the column definitions and static columns arrays and calculate how
	 * they relate to column indexes. The callback function will then apply the
	 * definition found for a column to a suitable configuration object.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aoColDefs The aoColumnDefs array that is to be applied
	 *  @param {array} aoCols The aoColumns array that defines columns individually
	 *  @param {function} fn Callback function - takes two parameters, the calculated
	 *    column index and the definition for that column.
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyColumnDefs( oSettings, aoColDefs, aoCols, fn )
	{
		var i, iLen, j, jLen, k, kLen, def;
		var columns = oSettings.aoColumns;
	
		// Column definitions with aTargets
		if ( aoColDefs )
		{
			/* Loop over the definitions array - loop in reverse so first instance has priority */
			for ( i=aoColDefs.length-1 ; i>=0 ; i-- )
			{
				def = aoColDefs[i];
	
				/* Each definition can target multiple columns, as it is an array */
				var aTargets = def.target !== undefined
					? def.target
					: def.targets !== undefined
						? def.targets
						: def.aTargets;
	
				if ( ! Array.isArray( aTargets ) )
				{
					aTargets = [ aTargets ];
				}
	
				for ( j=0, jLen=aTargets.length ; j<jLen ; j++ )
				{
					if ( typeof aTargets[j] === 'number' && aTargets[j] >= 0 )
					{
						/* Add columns that we don't yet know about */
						while( columns.length <= aTargets[j] )
						{
							_fnAddColumn( oSettings );
						}
	
						/* Integer, basic index */
						fn( aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'number' && aTargets[j] < 0 )
					{
						/* Negative integer, right to left column counting */
						fn( columns.length+aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'string' )
					{
						/* Class name matching on TH element */
						for ( k=0, kLen=columns.length ; k<kLen ; k++ )
						{
							if ( aTargets[j] == "_all" ||
							     $(columns[k].nTh).hasClass( aTargets[j] ) )
							{
								fn( k, def );
							}
						}
					}
				}
			}
		}
	
		// Statically defined columns array
		if ( aoCols )
		{
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ )
			{
				fn( i, aoCols[i] );
			}
		}
	}
	
	/**
	 * Add a data array to the table, creating DOM node etc. This is the parallel to
	 * _fnGatherData, but for adding rows from a Javascript source, rather than a
	 * DOM source.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aData data array to be added
	 *  @param {node} [nTr] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @returns {int} >=0 if successful (index of new aoData entry), -1 if failed
	 *  @memberof DataTable#oApi
	 */
	function _fnAddData ( oSettings, aDataIn, nTr, anTds )
	{
		/* Create the object for storing information about this new row */
		var iRow = oSettings.aoData.length;
		var oData = $.extend( true, {}, DataTable.models.oRow, {
			src: nTr ? 'dom' : 'data',
			idx: iRow
		} );
	
		oData._aData = aDataIn;
		oSettings.aoData.push( oData );
	
		/* Create the cells */
		var nTd, sThisType;
		var columns = oSettings.aoColumns;
	
		// Invalidate the column types as the new data needs to be revalidated
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			columns[i].sType = null;
		}
	
		/* Add to the display array */
		oSettings.aiDisplayMaster.push( iRow );
	
		var id = oSettings.rowIdFn( aDataIn );
		if ( id !== undefined ) {
			oSettings.aIds[ id ] = oData;
		}
	
		/* Create the DOM information, or register it if already present */
		if ( nTr || ! oSettings.oFeatures.bDeferRender )
		{
			_fnCreateTr( oSettings, iRow, nTr, anTds );
		}
	
		return iRow;
	}
	
	
	/**
	 * Add one or more TR elements to the table. Generally we'd expect to
	 * use this for reading data from a DOM sourced table, but it could be
	 * used for an TR element. Note that if a TR is given, it is used (i.e.
	 * it is not cloned).
	 *  @param {object} settings dataTables settings object
	 *  @param {array|node|jQuery} trs The TR element(s) to add to the table
	 *  @returns {array} Array of indexes for the added rows
	 *  @memberof DataTable#oApi
	 */
	function _fnAddTr( settings, trs )
	{
		var row;
	
		// Allow an individual node to be passed in
		if ( ! (trs instanceof $) ) {
			trs = $(trs);
		}
	
		return trs.map( function (i, el) {
			row = _fnGetRowElements( settings, el );
			return _fnAddData( settings, row.data, el, row.cells );
		} );
	}
	
	
	/**
	 * Take a TR element and convert it to an index in aoData
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} n the TR element to find
	 *  @returns {int} index if the node is found, null if not
	 *  @memberof DataTable#oApi
	 */
	function _fnNodeToDataIndex( oSettings, n )
	{
		return (n._DT_RowIndex!==undefined) ? n._DT_RowIndex : null;
	}
	
	
	/**
	 * Take a TD element and convert it into a column data index (not the visible index)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow The row number the TD/TH can be found in
	 *  @param {node} n The TD/TH element to find
	 *  @returns {int} index if the node is found, -1 if not
	 *  @memberof DataTable#oApi
	 */
	function _fnNodeToColumnIndex( oSettings, iRow, n )
	{
		return $.inArray( n, oSettings.aoData[ iRow ].anCells );
	}
	
	
	/**
	 * Get the data for a given cell from the internal cache, taking into account data mapping
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {string} type data get type ('display', 'type' 'filter|search' 'sort|order')
	 *  @returns {*} Cell data
	 *  @memberof DataTable#oApi
	 */
	function _fnGetCellData( settings, rowIdx, colIdx, type )
	{
		if (type === 'search') {
			type = 'filter';
		}
		else if (type === 'order') {
			type = 'sort';
		}
	
		var draw           = settings.iDraw;
		var col            = settings.aoColumns[colIdx];
		var rowData        = settings.aoData[rowIdx]._aData;
		var defaultContent = col.sDefaultContent;
		var cellData       = col.fnGetData( rowData, type, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		} );
	
		if ( cellData === undefined ) {
			if ( settings.iDrawError != draw && defaultContent === null ) {
				_fnLog( settings, 0, "Requested unknown parameter "+
					(typeof col.mData=='function' ? '{function}' : "'"+col.mData+"'")+
					" for row "+rowIdx+", column "+colIdx, 4 );
				settings.iDrawError = draw;
			}
			return defaultContent;
		}
	
		// When the data source is null and a specific data type is requested (i.e.
		// not the original data), we can use default column data
		if ( (cellData === rowData || cellData === null) && defaultContent !== null && type !== undefined ) {
			cellData = defaultContent;
		}
		else if ( typeof cellData === 'function' ) {
			// If the data source is a function, then we run it and use the return,
			// executing in the scope of the data object (for instances)
			return cellData.call( rowData );
		}
	
		if ( cellData === null && type === 'display' ) {
			return '';
		}
	
		if ( type === 'filter' ) {
			var fomatters = DataTable.ext.type.search;
	
			if ( fomatters[ col.sType ] ) {
				cellData = fomatters[ col.sType ]( cellData );
			}
		}
	
		return cellData;
	}
	
	
	/**
	 * Set the value for a specific cell, into the internal data cache
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {*} val Value to set
	 *  @memberof DataTable#oApi
	 */
	function _fnSetCellData( settings, rowIdx, colIdx, val )
	{
		var col     = settings.aoColumns[colIdx];
		var rowData = settings.aoData[rowIdx]._aData;
	
		col.fnSetData( rowData, val, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		}  );
	}
	
	
	// Private variable that is used to match action syntax in the data property object
	var __reArray = /\[.*?\]$/;
	var __reFn = /\(\)$/;
	
	/**
	 * Split string on periods, taking into account escaped periods
	 * @param  {string} str String to split
	 * @return {array} Split string
	 */
	function _fnSplitObjNotation( str )
	{
		return $.map( str.match(/(\\.|[^\.])+/g) || [''], function ( s ) {
			return s.replace(/\\\./g, '.');
		} );
	}
	
	
	/**
	 * Return a function that can be used to get data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data get function
	 *  @memberof DataTable#oApi
	 */
	var _fnGetObjectDataFn = DataTable.util.get;
	
	
	/**
	 * Return a function that can be used to set data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data set function
	 *  @memberof DataTable#oApi
	 */
	var _fnSetObjectDataFn = DataTable.util.set;
	
	
	/**
	 * Return an array with the full table data
	 *  @param {object} oSettings dataTables settings object
	 *  @returns array {array} aData Master data array
	 *  @memberof DataTable#oApi
	 */
	function _fnGetDataMaster ( settings )
	{
		return _pluck( settings.aoData, '_aData' );
	}
	
	
	/**
	 * Nuke the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnClearTable( settings )
	{
		settings.aoData.length = 0;
		settings.aiDisplayMaster.length = 0;
		settings.aiDisplay.length = 0;
		settings.aIds = {};
	}
	
	
	 /**
	 * Take an array of integers (index array) and remove a target integer (value - not
	 * the key!)
	 *  @param {array} a Index array to target
	 *  @param {int} iTarget value to find
	 *  @memberof DataTable#oApi
	 */
	function _fnDeleteIndex( a, iTarget, splice )
	{
		var iTargetIndex = -1;
	
		for ( var i=0, iLen=a.length ; i<iLen ; i++ )
		{
			if ( a[i] == iTarget )
			{
				iTargetIndex = i;
			}
			else if ( a[i] > iTarget )
			{
				a[i]--;
			}
		}
	
		if ( iTargetIndex != -1 && splice === undefined )
		{
			a.splice( iTargetIndex, 1 );
		}
	}
	
	
	/**
	 * Mark cached data as invalid such that a re-read of the data will occur when
	 * the cached data is next requested. Also update from the data source object.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {int}    rowIdx   Row index to invalidate
	 * @param {string} [src]    Source to invalidate from: undefined, 'auto', 'dom'
	 *     or 'data'
	 * @param {int}    [colIdx] Column index to invalidate. If undefined the whole
	 *     row will be invalidated
	 * @memberof DataTable#oApi
	 *
	 * @todo For the modularisation of v1.11 this will need to become a callback, so
	 *   the sort and filter methods can subscribe to it. That will required
	 *   initialisation options for sorting, which is why it is not already baked in
	 */
	function _fnInvalidate( settings, rowIdx, src, colIdx )
	{
		var row = settings.aoData[ rowIdx ];
		var i, ien;
		var cellWrite = function ( cell, col ) {
			// This is very frustrating, but in IE if you just write directly
			// to innerHTML, and elements that are overwritten are GC'ed,
			// even if there is a reference to them elsewhere
			while ( cell.childNodes.length ) {
				cell.removeChild( cell.firstChild );
			}
	
			cell.innerHTML = _fnGetCellData( settings, rowIdx, col, 'display' );
		};
	
		// Are we reading last data from DOM or the data object?
		if ( src === 'dom' || ((! src || src === 'auto') && row.src === 'dom') ) {
			// Read the data from the DOM
			row._aData = _fnGetRowElements(
					settings, row, colIdx, colIdx === undefined ? undefined : row._aData
				)
				.data;
		}
		else {
			// Reading from data object, update the DOM
			var cells = row.anCells;
	
			if ( cells ) {
				if ( colIdx !== undefined ) {
					cellWrite( cells[colIdx], colIdx );
				}
				else {
					for ( i=0, ien=cells.length ; i<ien ; i++ ) {
						cellWrite( cells[i], i );
					}
				}
			}
		}
	
		// For both row and cell invalidation, the cached data for sorting and
		// filtering is nulled out
		row._aSortData = null;
		row._aFilterData = null;
	
		// Invalidate the type for a specific column (if given) or all columns since
		// the data might have changed
		var cols = settings.aoColumns;
		if ( colIdx !== undefined ) {
			cols[ colIdx ].sType = null;
		}
		else {
			for ( i=0, ien=cols.length ; i<ien ; i++ ) {
				cols[i].sType = null;
			}
	
			// Update DataTables special `DT_*` attributes for the row
			_fnRowAttributes( settings, row );
		}
	}
	
	
	/**
	 * Build a data source object from an HTML row, reading the contents of the
	 * cells that are in the row.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {node|object} TR element from which to read data or existing row
	 *   object from which to re-read the data from the cells
	 * @param {int} [colIdx] Optional column index
	 * @param {array|object} [d] Data source object. If `colIdx` is given then this
	 *   parameter should also be given and will be used to write the data into.
	 *   Only the column in question will be written
	 * @returns {object} Object with two parameters: `data` the data read, in
	 *   document order, and `cells` and array of nodes (they can be useful to the
	 *   caller, so rather than needing a second traversal to get them, just return
	 *   them from here).
	 * @memberof DataTable#oApi
	 */
	function _fnGetRowElements( settings, row, colIdx, d )
	{
		var
			tds = [],
			td = row.firstChild,
			name, col, o, i=0, contents,
			columns = settings.aoColumns,
			objectRead = settings._rowReadObject;
	
		// Allow the data object to be passed in, or construct
		d = d !== undefined ?
			d :
			objectRead ?
				{} :
				[];
	
		var attr = function ( str, td  ) {
			if ( typeof str === 'string' ) {
				var idx = str.indexOf('@');
	
				if ( idx !== -1 ) {
					var attr = str.substring( idx+1 );
					var setter = _fnSetObjectDataFn( str );
					setter( d, td.getAttribute( attr ) );
				}
			}
		};
	
		// Read data from a cell and store into the data object
		var cellProcess = function ( cell ) {
			if ( colIdx === undefined || colIdx === i ) {
				col = columns[i];
				contents = (cell.innerHTML).trim();
	
				if ( col && col._bAttrSrc ) {
					var setter = _fnSetObjectDataFn( col.mData._ );
					setter( d, contents );
	
					attr( col.mData.sort, cell );
					attr( col.mData.type, cell );
					attr( col.mData.filter, cell );
				}
				else {
					// Depending on the `data` option for the columns the data can
					// be read to either an object or an array.
					if ( objectRead ) {
						if ( ! col._setter ) {
							// Cache the setter function
							col._setter = _fnSetObjectDataFn( col.mData );
						}
						col._setter( d, contents );
					}
					else {
						d[i] = contents;
					}
				}
			}
	
			i++;
		};
	
		if ( td ) {
			// `tr` element was passed in
			while ( td ) {
				name = td.nodeName.toUpperCase();
	
				if ( name == "TD" || name == "TH" ) {
					cellProcess( td );
					tds.push( td );
				}
	
				td = td.nextSibling;
			}
		}
		else {
			// Existing row object passed in
			tds = row.anCells;
	
			for ( var j=0, jen=tds.length ; j<jen ; j++ ) {
				cellProcess( tds[j] );
			}
		}
	
		// Read the ID from the DOM if present
		var rowNode = row.firstChild ? row : row.nTr;
	
		if ( rowNode ) {
			var id = rowNode.getAttribute( 'id' );
	
			if ( id ) {
				_fnSetObjectDataFn( settings.rowId )( d, id );
			}
		}
	
		return {
			data: d,
			cells: tds
		};
	}
	/**
	 * Create a new TR element (and it's TD children) for a row
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow Row to consider
	 *  @param {node} [nTrIn] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @memberof DataTable#oApi
	 */
	function _fnCreateTr ( oSettings, iRow, nTrIn, anTds )
	{
		var
			row = oSettings.aoData[iRow],
			rowData = row._aData,
			cells = [],
			nTr, nTd, oCol,
			i, iLen, create;
	
		if ( row.nTr === null )
		{
			nTr = nTrIn || document.createElement('tr');
	
			row.nTr = nTr;
			row.anCells = cells;
	
			/* Use a private property on the node to allow reserve mapping from the node
			 * to the aoData array for fast look up
			 */
			nTr._DT_RowIndex = iRow;
	
			/* Special parameters can be given by the data source to be used on the row */
			_fnRowAttributes( oSettings, row );
	
			/* Process each column */
			for ( i=0, iLen=oSettings.aoColumns.length ; i<iLen ; i++ )
			{
				oCol = oSettings.aoColumns[i];
				create = nTrIn ? false : true;
	
				nTd = create ? document.createElement( oCol.sCellType ) : anTds[i];
				nTd._DT_CellIndex = {
					row: iRow,
					column: i
				};
				
				cells.push( nTd );
	
				// Need to create the HTML if new, or if a rendering function is defined
				if ( create || ((oCol.mRender || oCol.mData !== i) &&
					 (!$.isPlainObject(oCol.mData) || oCol.mData._ !== i+'.display')
				)) {
					nTd.innerHTML = _fnGetCellData( oSettings, iRow, i, 'display' );
				}
	
				/* Add user defined class */
				if ( oCol.sClass )
				{
					nTd.className += ' '+oCol.sClass;
				}
	
				// Visibility - add or remove as required
				if ( oCol.bVisible && ! nTrIn )
				{
					nTr.appendChild( nTd );
				}
				else if ( ! oCol.bVisible && nTrIn )
				{
					nTd.parentNode.removeChild( nTd );
				}
	
				if ( oCol.fnCreatedCell )
				{
					oCol.fnCreatedCell.call( oSettings.oInstance,
						nTd, _fnGetCellData( oSettings, iRow, i ), rowData, iRow, i
					);
				}
			}
	
			_fnCallbackFire( oSettings, 'aoRowCreatedCallback', null, [nTr, rowData, iRow, cells] );
		}
	}
	
	
	/**
	 * Add attributes to a row based on the special `DT_*` parameters in a data
	 * source object.
	 *  @param {object} settings DataTables settings object
	 *  @param {object} DataTables row object for the row to be modified
	 *  @memberof DataTable#oApi
	 */
	function _fnRowAttributes( settings, row )
	{
		var tr = row.nTr;
		var data = row._aData;
	
		if ( tr ) {
			var id = settings.rowIdFn( data );
	
			if ( id ) {
				tr.id = id;
			}
	
			if ( data.DT_RowClass ) {
				// Remove any classes added by DT_RowClass before
				var a = data.DT_RowClass.split(' ');
				row.__rowc = row.__rowc ?
					_unique( row.__rowc.concat( a ) ) :
					a;
	
				$(tr)
					.removeClass( row.__rowc.join(' ') )
					.addClass( data.DT_RowClass );
			}
	
			if ( data.DT_RowAttr ) {
				$(tr).attr( data.DT_RowAttr );
			}
	
			if ( data.DT_RowData ) {
				$(tr).data( data.DT_RowData );
			}
		}
	}
	
	
	/**
	 * Create the HTML header for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBuildHead( oSettings )
	{
		var i, ien, cell, row, column;
		var thead = oSettings.nTHead;
		var tfoot = oSettings.nTFoot;
		var createHeader = $('th, td', thead).length === 0;
		var classes = oSettings.oClasses;
		var columns = oSettings.aoColumns;
	
		if ( createHeader ) {
			row = $('<tr/>').appendTo( thead );
		}
	
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			column = columns[i];
			cell = $( column.nTh ).addClass( column.sClass );
	
			if ( createHeader ) {
				cell.appendTo( row );
			}
	
			// 1.11 move into sorting
			if ( oSettings.oFeatures.bSort ) {
				cell.addClass( column.sSortingClass );
	
				if ( column.bSortable !== false ) {
					cell
						.attr( 'tabindex', oSettings.iTabIndex )
						.attr( 'aria-controls', oSettings.sTableId );
	
					_fnSortAttachListener( oSettings, column.nTh, i );
				}
			}
	
			if ( column.sTitle != cell[0].innerHTML ) {
				cell.html( column.sTitle );
			}
	
			_fnRenderer( oSettings, 'header' )(
				oSettings, cell, column, classes
			);
		}
	
		if ( createHeader ) {
			_fnDetectHeader( oSettings.aoHeader, thead );
		}
	
		/* Deal with the footer - add classes if required */
		$(thead).children('tr').children('th, td').addClass( classes.sHeaderTH );
		$(tfoot).children('tr').children('th, td').addClass( classes.sFooterTH );
	
		// Cache the footer cells. Note that we only take the cells from the first
		// row in the footer. If there is more than one row the user wants to
		// interact with, they need to use the table().foot() method. Note also this
		// allows cells to be used for multiple columns using colspan
		if ( tfoot !== null ) {
			var cells = oSettings.aoFooter[0];
	
			for ( i=0, ien=cells.length ; i<ien ; i++ ) {
				column = columns[i];
				column.nTf = cells[i].cell;
	
				if ( column.sClass ) {
					$(column.nTf).addClass( column.sClass );
				}
			}
		}
	}
	
	
	/**
	 * Draw the header (or footer) element based on the column visibility states. The
	 * methodology here is to use the layout array from _fnDetectHeader, modified for
	 * the instantaneous column visibility, to construct the new layout. The grid is
	 * traversed over cell at a time in a rows x columns grid fashion, although each
	 * cell insert can cover multiple elements in the grid - which is tracks using the
	 * aApplied array. Cell inserts in the grid will only occur where there isn't
	 * already a cell in that position.
	 *  @param {object} oSettings dataTables settings object
	 *  @param array {objects} aoSource Layout array from _fnDetectHeader
	 *  @param {boolean} [bIncludeHidden=false] If true then include the hidden columns in the calc,
	 *  @memberof DataTable#oApi
	 */
	function _fnDrawHead( oSettings, aoSource, bIncludeHidden )
	{
		var i, iLen, j, jLen, k, kLen, n, nLocalTr;
		var aoLocal = [];
		var aApplied = [];
		var iColumns = oSettings.aoColumns.length;
		var iRowspan, iColspan;
	
		if ( ! aoSource )
		{
			return;
		}
	
		if (  bIncludeHidden === undefined )
		{
			bIncludeHidden = false;
		}
	
		/* Make a copy of the master layout array, but without the visible columns in it */
		for ( i=0, iLen=aoSource.length ; i<iLen ; i++ )
		{
			aoLocal[i] = aoSource[i].slice();
			aoLocal[i].nTr = aoSource[i].nTr;
	
			/* Remove any columns which are currently hidden */
			for ( j=iColumns-1 ; j>=0 ; j-- )
			{
				if ( !oSettings.aoColumns[j].bVisible && !bIncludeHidden )
				{
					aoLocal[i].splice( j, 1 );
				}
			}
	
			/* Prep the applied array - it needs an element for each row */
			aApplied.push( [] );
		}
	
		for ( i=0, iLen=aoLocal.length ; i<iLen ; i++ )
		{
			nLocalTr = aoLocal[i].nTr;
	
			/* All cells are going to be replaced, so empty out the row */
			if ( nLocalTr )
			{
				while( (n = nLocalTr.firstChild) )
				{
					nLocalTr.removeChild( n );
				}
			}
	
			for ( j=0, jLen=aoLocal[i].length ; j<jLen ; j++ )
			{
				iRowspan = 1;
				iColspan = 1;
	
				/* Check to see if there is already a cell (row/colspan) covering our target
				 * insert point. If there is, then there is nothing to do.
				 */
				if ( aApplied[i][j] === undefined )
				{
					nLocalTr.appendChild( aoLocal[i][j].cell );
					aApplied[i][j] = 1;
	
					/* Expand the cell to cover as many rows as needed */
					while ( aoLocal[i+iRowspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i+iRowspan][j].cell )
					{
						aApplied[i+iRowspan][j] = 1;
						iRowspan++;
					}
	
					/* Expand the cell to cover as many columns as needed */
					while ( aoLocal[i][j+iColspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i][j+iColspan].cell )
					{
						/* Must update the applied array over the rows for the columns */
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aApplied[i+k][j+iColspan] = 1;
						}
						iColspan++;
					}
	
					/* Do the actual expansion in the DOM */
					$(aoLocal[i][j].cell)
						.attr('rowspan', iRowspan)
						.attr('colspan', iColspan);
				}
			}
		}
	}
	
	
	/**
	 * Insert the required TR nodes into the table for display
	 *  @param {object} oSettings dataTables settings object
	 *  @param ajaxComplete true after ajax call to complete rendering
	 *  @memberof DataTable#oApi
	 */
	function _fnDraw( oSettings, ajaxComplete )
	{
		// Allow for state saving and a custom start position
		_fnStart( oSettings );
	
		/* Provide a pre-callback function which can be used to cancel the draw is false is returned */
		var aPreDraw = _fnCallbackFire( oSettings, 'aoPreDrawCallback', 'preDraw', [oSettings] );
		if ( $.inArray( false, aPreDraw ) !== -1 )
		{
			_fnProcessingDisplay( oSettings, false );
			return;
		}
	
		var anRows = [];
		var iRowCount = 0;
		var asStripeClasses = oSettings.asStripeClasses;
		var iStripes = asStripeClasses.length;
		var oLang = oSettings.oLanguage;
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var aiDisplay = oSettings.aiDisplay;
		var iDisplayStart = oSettings._iDisplayStart;
		var iDisplayEnd = oSettings.fnDisplayEnd();
	
		oSettings.bDrawing = true;
	
		/* Server-side processing draw intercept */
		if ( oSettings.bDeferLoading )
		{
			oSettings.bDeferLoading = false;
			oSettings.iDraw++;
			_fnProcessingDisplay( oSettings, false );
		}
		else if ( !bServerSide )
		{
			oSettings.iDraw++;
		}
		else if ( !oSettings.bDestroying && !ajaxComplete)
		{
			_fnAjaxUpdate( oSettings );
			return;
		}
	
		if ( aiDisplay.length !== 0 )
		{
			var iStart = bServerSide ? 0 : iDisplayStart;
			var iEnd = bServerSide ? oSettings.aoData.length : iDisplayEnd;
	
			for ( var j=iStart ; j<iEnd ; j++ )
			{
				var iDataIndex = aiDisplay[j];
				var aoData = oSettings.aoData[ iDataIndex ];
				if ( aoData.nTr === null )
				{
					_fnCreateTr( oSettings, iDataIndex );
				}
	
				var nRow = aoData.nTr;
	
				/* Remove the old striping classes and then add the new one */
				if ( iStripes !== 0 )
				{
					var sStripe = asStripeClasses[ iRowCount % iStripes ];
					if ( aoData._sRowStripe != sStripe )
					{
						$(nRow).removeClass( aoData._sRowStripe ).addClass( sStripe );
						aoData._sRowStripe = sStripe;
					}
				}
	
				// Row callback functions - might want to manipulate the row
				// iRowCount and j are not currently documented. Are they at all
				// useful?
				_fnCallbackFire( oSettings, 'aoRowCallback', null,
					[nRow, aoData._aData, iRowCount, j, iDataIndex] );
	
				anRows.push( nRow );
				iRowCount++;
			}
		}
		else
		{
			/* Table is empty - create a row with an empty message in it */
			var sZero = oLang.sZeroRecords;
			if ( oSettings.iDraw == 1 &&  _fnDataSource( oSettings ) == 'ajax' )
			{
				sZero = oLang.sLoadingRecords;
			}
			else if ( oLang.sEmptyTable && oSettings.fnRecordsTotal() === 0 )
			{
				sZero = oLang.sEmptyTable;
			}
	
			anRows[ 0 ] = $( '<tr/>', { 'class': iStripes ? asStripeClasses[0] : '' } )
				.append( $('<td />', {
					'valign':  'top',
					'colSpan': _fnVisbleColumns( oSettings ),
					'class':   oSettings.oClasses.sRowEmpty
				} ).html( sZero ) )[0];
		}
	
		/* Header and footer callbacks */
		_fnCallbackFire( oSettings, 'aoHeaderCallback', 'header', [ $(oSettings.nTHead).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		_fnCallbackFire( oSettings, 'aoFooterCallback', 'footer', [ $(oSettings.nTFoot).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		var body = $(oSettings.nTBody);
	
		body.children().detach();
		body.append( $(anRows) );
	
		/* Call all required callback functions for the end of a draw */
		_fnCallbackFire( oSettings, 'aoDrawCallback', 'draw', [oSettings] );
	
		/* Draw is complete, sorting and filtering must be as well */
		oSettings.bSorted = false;
		oSettings.bFiltered = false;
		oSettings.bDrawing = false;
	}
	
	
	/**
	 * Redraw the table - taking account of the various features which are enabled
	 *  @param {object} oSettings dataTables settings object
	 *  @param {boolean} [holdPosition] Keep the current paging position. By default
	 *    the paging is reset to the first page
	 *  @memberof DataTable#oApi
	 */
	function _fnReDraw( settings, holdPosition )
	{
		var
			features = settings.oFeatures,
			sort     = features.bSort,
			filter   = features.bFilter;
	
		if ( sort ) {
			_fnSort( settings );
		}
	
		if ( filter ) {
			_fnFilterComplete( settings, settings.oPreviousSearch );
		}
		else {
			// No filtering, so we want to just use the display master
			settings.aiDisplay = settings.aiDisplayMaster.slice();
		}
	
		if ( holdPosition !== true ) {
			settings._iDisplayStart = 0;
		}
	
		// Let any modules know about the draw hold position state (used by
		// scrolling internally)
		settings._drawHold = holdPosition;
	
		_fnDraw( settings );
	
		settings._drawHold = false;
	}
	
	
	/**
	 * Add the options to the page HTML for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAddOptionsHtml ( oSettings )
	{
		var classes = oSettings.oClasses;
		var table = $(oSettings.nTable);
		var holding = $('<div/>').insertBefore( table ); // Holding element for speed
		var features = oSettings.oFeatures;
	
		// All DataTables are wrapped in a div
		var insert = $('<div/>', {
			id:      oSettings.sTableId+'_wrapper',
			'class': classes.sWrapper + (oSettings.nTFoot ? '' : ' '+classes.sNoFooter)
		} );
	
		oSettings.nHolding = holding[0];
		oSettings.nTableWrapper = insert[0];
		oSettings.nTableReinsertBefore = oSettings.nTable.nextSibling;
	
		/* Loop over the user set positioning and place the elements as needed */
		var aDom = oSettings.sDom.split('');
		var featureNode, cOption, nNewNode, cNext, sAttr, j;
		for ( var i=0 ; i<aDom.length ; i++ )
		{
			featureNode = null;
			cOption = aDom[i];
	
			if ( cOption == '<' )
			{
				/* New container div */
				nNewNode = $('<div/>')[0];
	
				/* Check to see if we should append an id and/or a class name to the container */
				cNext = aDom[i+1];
				if ( cNext == "'" || cNext == '"' )
				{
					sAttr = "";
					j = 2;
					while ( aDom[i+j] != cNext )
					{
						sAttr += aDom[i+j];
						j++;
					}
	
					/* Replace jQuery UI constants @todo depreciated */
					if ( sAttr == "H" )
					{
						sAttr = classes.sJUIHeader;
					}
					else if ( sAttr == "F" )
					{
						sAttr = classes.sJUIFooter;
					}
	
					/* The attribute can be in the format of "#id.class", "#id" or "class" This logic
					 * breaks the string into parts and applies them as needed
					 */
					if ( sAttr.indexOf('.') != -1 )
					{
						var aSplit = sAttr.split('.');
						nNewNode.id = aSplit[0].substr(1, aSplit[0].length-1);
						nNewNode.className = aSplit[1];
					}
					else if ( sAttr.charAt(0) == "#" )
					{
						nNewNode.id = sAttr.substr(1, sAttr.length-1);
					}
					else
					{
						nNewNode.className = sAttr;
					}
	
					i += j; /* Move along the position array */
				}
	
				insert.append( nNewNode );
				insert = $(nNewNode);
			}
			else if ( cOption == '>' )
			{
				/* End container div */
				insert = insert.parent();
			}
			// @todo Move options into their own plugins?
			else if ( cOption == 'l' && features.bPaginate && features.bLengthChange )
			{
				/* Length */
				featureNode = _fnFeatureHtmlLength( oSettings );
			}
			else if ( cOption == 'f' && features.bFilter )
			{
				/* Filter */
				featureNode = _fnFeatureHtmlFilter( oSettings );
			}
			else if ( cOption == 'r' && features.bProcessing )
			{
				/* pRocessing */
				featureNode = _fnFeatureHtmlProcessing( oSettings );
			}
			else if ( cOption == 't' )
			{
				/* Table */
				featureNode = _fnFeatureHtmlTable( oSettings );
			}
			else if ( cOption ==  'i' && features.bInfo )
			{
				/* Info */
				featureNode = _fnFeatureHtmlInfo( oSettings );
			}
			else if ( cOption == 'p' && features.bPaginate )
			{
				/* Pagination */
				featureNode = _fnFeatureHtmlPaginate( oSettings );
			}
			else if ( DataTable.ext.feature.length !== 0 )
			{
				/* Plug-in features */
				var aoFeatures = DataTable.ext.feature;
				for ( var k=0, kLen=aoFeatures.length ; k<kLen ; k++ )
				{
					if ( cOption == aoFeatures[k].cFeature )
					{
						featureNode = aoFeatures[k].fnInit( oSettings );
						break;
					}
				}
			}
	
			/* Add to the 2D features array */
			if ( featureNode )
			{
				var aanFeatures = oSettings.aanFeatures;
	
				if ( ! aanFeatures[cOption] )
				{
					aanFeatures[cOption] = [];
				}
	
				aanFeatures[cOption].push( featureNode );
				insert.append( featureNode );
			}
		}
	
		/* Built our DOM structure - replace the holding div with what we want */
		holding.replaceWith( insert );
		oSettings.nHolding = null;
	}
	
	
	/**
	 * Use the DOM source to create up an array of header cells. The idea here is to
	 * create a layout grid (array) of rows x columns, which contains a reference
	 * to the cell that that point in the grid (regardless of col/rowspan), such that
	 * any column / row could be removed and the new grid constructed
	 *  @param array {object} aLayout Array to store the calculated layout in
	 *  @param {node} nThead The header/footer element for the table
	 *  @memberof DataTable#oApi
	 */
	function _fnDetectHeader ( aLayout, nThead )
	{
		var nTrs = $(nThead).children('tr');
		var nTr, nCell;
		var i, k, l, iLen, jLen, iColShifted, iColumn, iColspan, iRowspan;
		var bUnique;
		var fnShiftCol = function ( a, i, j ) {
			var k = a[i];
	                while ( k[j] ) {
				j++;
			}
			return j;
		};
	
		aLayout.splice( 0, aLayout.length );
	
		/* We know how many rows there are in the layout - so prep it */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			aLayout.push( [] );
		}
	
		/* Calculate a layout array */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			nTr = nTrs[i];
			iColumn = 0;
	
			/* For every cell in the row... */
			nCell = nTr.firstChild;
			while ( nCell ) {
				if ( nCell.nodeName.toUpperCase() == "TD" ||
				     nCell.nodeName.toUpperCase() == "TH" )
				{
					/* Get the col and rowspan attributes from the DOM and sanitise them */
					iColspan = nCell.getAttribute('colspan') * 1;
					iRowspan = nCell.getAttribute('rowspan') * 1;
					iColspan = (!iColspan || iColspan===0 || iColspan===1) ? 1 : iColspan;
					iRowspan = (!iRowspan || iRowspan===0 || iRowspan===1) ? 1 : iRowspan;
	
					/* There might be colspan cells already in this row, so shift our target
					 * accordingly
					 */
					iColShifted = fnShiftCol( aLayout, i, iColumn );
	
					/* Cache calculation for unique columns */
					bUnique = iColspan === 1 ? true : false;
	
					/* If there is col / rowspan, copy the information into the layout grid */
					for ( l=0 ; l<iColspan ; l++ )
					{
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aLayout[i+k][iColShifted+l] = {
								"cell": nCell,
								"unique": bUnique
							};
							aLayout[i+k].nTr = nTr;
						}
					}
				}
				nCell = nCell.nextSibling;
			}
		}
	}
	
	
	/**
	 * Get an array of unique th elements, one for each column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} nHeader automatically detect the layout from this node - optional
	 *  @param {array} aLayout thead/tfoot layout from _fnDetectHeader - optional
	 *  @returns array {node} aReturn list of unique th's
	 *  @memberof DataTable#oApi
	 */
	function _fnGetUniqueThs ( oSettings, nHeader, aLayout )
	{
		var aReturn = [];
		if ( !aLayout )
		{
			aLayout = oSettings.aoHeader;
			if ( nHeader )
			{
				aLayout = [];
				_fnDetectHeader( aLayout, nHeader );
			}
		}
	
		for ( var i=0, iLen=aLayout.length ; i<iLen ; i++ )
		{
			for ( var j=0, jLen=aLayout[i].length ; j<jLen ; j++ )
			{
				if ( aLayout[i][j].unique &&
					 (!aReturn[j] || !oSettings.bSortCellsTop) )
				{
					aReturn[j] = aLayout[i][j].cell;
				}
			}
		}
	
		return aReturn;
	}
	
	/**
	 * Set the start position for draw
	 *  @param {object} oSettings dataTables settings object
	 */
	function _fnStart( oSettings )
	{
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var iInitDisplayStart = oSettings.iInitDisplayStart;
	
		// Check and see if we have an initial draw position from state saving
		if ( iInitDisplayStart !== undefined && iInitDisplayStart !== -1 )
		{
			oSettings._iDisplayStart = bServerSide ?
				iInitDisplayStart :
				iInitDisplayStart >= oSettings.fnRecordsDisplay() ?
					0 :
					iInitDisplayStart;
	
			oSettings.iInitDisplayStart = -1;
		}
	}
	
	/**
	 * Create an Ajax call based on the table's settings, taking into account that
	 * parameters can have multiple forms, and backwards compatibility.
	 *
	 * @param {object} oSettings dataTables settings object
	 * @param {array} data Data to send to the server, required by
	 *     DataTables - may be augmented by developer callbacks
	 * @param {function} fn Callback function to run when data is obtained
	 */
	function _fnBuildAjax( oSettings, data, fn )
	{
		// Compatibility with 1.9-, allow fnServerData and event to manipulate
		_fnCallbackFire( oSettings, 'aoServerParams', 'serverParams', [data] );
	
		// Convert to object based for 1.10+ if using the old array scheme which can
		// come from server-side processing or serverParams
		if ( data && Array.isArray(data) ) {
			var tmp = {};
			var rbracket = /(.*?)\[\]$/;
	
			$.each( data, function (key, val) {
				var match = val.name.match(rbracket);
	
				if ( match ) {
					// Support for arrays
					var name = match[0];
	
					if ( ! tmp[ name ] ) {
						tmp[ name ] = [];
					}
					tmp[ name ].push( val.value );
				}
				else {
					tmp[val.name] = val.value;
				}
			} );
			data = tmp;
		}
	
		var ajaxData;
		var ajax = oSettings.ajax;
		var instance = oSettings.oInstance;
		var callback = function ( json ) {
			var status = oSettings.jqXHR
				? oSettings.jqXHR.status
				: null;
	
			if ( json === null || (typeof status === 'number' && status == 204 ) ) {
				json = {};
				_fnAjaxDataSrc( oSettings, json, [] );
			}
	
			var error = json.error || json.sError;
			if ( error ) {
				_fnLog( oSettings, 0, error );
			}
	
			oSettings.json = json;
	
			_fnCallbackFire( oSettings, null, 'xhr', [oSettings, json, oSettings.jqXHR] );
			fn( json );
		};
	
		if ( $.isPlainObject( ajax ) && ajax.data )
		{
			ajaxData = ajax.data;
	
			var newData = typeof ajaxData === 'function' ?
				ajaxData( data, oSettings ) :  // fn can manipulate data or return
				ajaxData;                      // an object object or array to merge
	
			// If the function returned something, use that alone
			data = typeof ajaxData === 'function' && newData ?
				newData :
				$.extend( true, data, newData );
	
			// Remove the data property as we've resolved it already and don't want
			// jQuery to do it again (it is restored at the end of the function)
			delete ajax.data;
		}
	
		var baseAjax = {
			"data": data,
			"success": callback,
			"dataType": "json",
			"cache": false,
			"type": oSettings.sServerMethod,
			"error": function (xhr, error, thrown) {
				var ret = _fnCallbackFire( oSettings, null, 'xhr', [oSettings, null, oSettings.jqXHR] );
	
				if ( $.inArray( true, ret ) === -1 ) {
					if ( error == "parsererror" ) {
						_fnLog( oSettings, 0, 'Invalid JSON response', 1 );
					}
					else if ( xhr.readyState === 4 ) {
						_fnLog( oSettings, 0, 'Ajax error', 7 );
					}
				}
	
				_fnProcessingDisplay( oSettings, false );
			}
		};
	
		// Store the data submitted for the API
		oSettings.oAjaxData = data;
	
		// Allow plug-ins and external processes to modify the data
		_fnCallbackFire( oSettings, null, 'preXhr', [oSettings, data] );
	
		if ( oSettings.fnServerData )
		{
			// DataTables 1.9- compatibility
			oSettings.fnServerData.call( instance,
				oSettings.sAjaxSource,
				$.map( data, function (val, key) { // Need to convert back to 1.9 trad format
					return { name: key, value: val };
				} ),
				callback,
				oSettings
			);
		}
		else if ( oSettings.sAjaxSource || typeof ajax === 'string' )
		{
			// DataTables 1.9- compatibility
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, {
				url: ajax || oSettings.sAjaxSource
			} ) );
		}
		else if ( typeof ajax === 'function' )
		{
			// Is a function - let the caller define what needs to be done
			oSettings.jqXHR = ajax.call( instance, data, callback, oSettings );
		}
		else
		{
			// Object to extend the base settings
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, ajax ) );
	
			// Restore for next time around
			ajax.data = ajaxData;
		}
	}
	
	
	/**
	 * Update the table using an Ajax call
	 *  @param {object} settings dataTables settings object
	 *  @returns {boolean} Block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdate( settings )
	{
		settings.iDraw++;
		_fnProcessingDisplay( settings, true );
	
		_fnBuildAjax(
			settings,
			_fnAjaxParameters( settings ),
			function(json) {
				_fnAjaxUpdateDraw( settings, json );
			}
		);
	}
	
	
	/**
	 * Build up the parameters in an object needed for a server-side processing
	 * request. Note that this is basically done twice, is different ways - a modern
	 * method which is used by default in DataTables 1.10 which uses objects and
	 * arrays, or the 1.9- method with is name / value pairs. 1.9 method is used if
	 * the sAjaxSource option is used in the initialisation, or the legacyAjax
	 * option is set.
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {bool} block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxParameters( settings )
	{
		var
			columns = settings.aoColumns,
			columnCount = columns.length,
			features = settings.oFeatures,
			preSearch = settings.oPreviousSearch,
			preColSearch = settings.aoPreSearchCols,
			i, data = [], dataProp, column, columnSearch,
			sort = _fnSortFlatten( settings ),
			displayStart = settings._iDisplayStart,
			displayLength = features.bPaginate !== false ?
				settings._iDisplayLength :
				-1;
	
		var param = function ( name, value ) {
			data.push( { 'name': name, 'value': value } );
		};
	
		// DataTables 1.9- compatible method
		param( 'sEcho',          settings.iDraw );
		param( 'iColumns',       columnCount );
		param( 'sColumns',       _pluck( columns, 'sName' ).join(',') );
		param( 'iDisplayStart',  displayStart );
		param( 'iDisplayLength', displayLength );
	
		// DataTables 1.10+ method
		var d = {
			draw:    settings.iDraw,
			columns: [],
			order:   [],
			start:   displayStart,
			length:  displayLength,
			search:  {
				value: preSearch.sSearch,
				regex: preSearch.bRegex
			}
		};
	
		for ( i=0 ; i<columnCount ; i++ ) {
			column = columns[i];
			columnSearch = preColSearch[i];
			dataProp = typeof column.mData=="function" ? 'function' : column.mData ;
	
			d.columns.push( {
				data:       dataProp,
				name:       column.sName,
				searchable: column.bSearchable,
				orderable:  column.bSortable,
				search:     {
					value: columnSearch.sSearch,
					regex: columnSearch.bRegex
				}
			} );
	
			param( "mDataProp_"+i, dataProp );
	
			if ( features.bFilter ) {
				param( 'sSearch_'+i,     columnSearch.sSearch );
				param( 'bRegex_'+i,      columnSearch.bRegex );
				param( 'bSearchable_'+i, column.bSearchable );
			}
	
			if ( features.bSort ) {
				param( 'bSortable_'+i, column.bSortable );
			}
		}
	
		if ( features.bFilter ) {
			param( 'sSearch', preSearch.sSearch );
			param( 'bRegex', preSearch.bRegex );
		}
	
		if ( features.bSort ) {
			$.each( sort, function ( i, val ) {
				d.order.push( { column: val.col, dir: val.dir } );
	
				param( 'iSortCol_'+i, val.col );
				param( 'sSortDir_'+i, val.dir );
			} );
	
			param( 'iSortingCols', sort.length );
		}
	
		// If the legacy.ajax parameter is null, then we automatically decide which
		// form to use, based on sAjaxSource
		var legacy = DataTable.ext.legacy.ajax;
		if ( legacy === null ) {
			return settings.sAjaxSource ? data : d;
		}
	
		// Otherwise, if legacy has been specified then we use that to decide on the
		// form
		return legacy ? data : d;
	}
	
	
	/**
	 * Data the data from the server (nuking the old) and redraw the table
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} json json data return from the server.
	 *  @param {string} json.sEcho Tracking flag for DataTables to match requests
	 *  @param {int} json.iTotalRecords Number of records in the data set, not accounting for filtering
	 *  @param {int} json.iTotalDisplayRecords Number of records in the data set, accounting for filtering
	 *  @param {array} json.aaData The data to display on this page
	 *  @param {string} [json.sColumns] Column ordering (sName, comma separated)
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdateDraw ( settings, json )
	{
		// v1.10 uses camelCase variables, while 1.9 uses Hungarian notation.
		// Support both
		var compat = function ( old, modern ) {
			return json[old] !== undefined ? json[old] : json[modern];
		};
	
		var data = _fnAjaxDataSrc( settings, json );
		var draw            = compat( 'sEcho',                'draw' );
		var recordsTotal    = compat( 'iTotalRecords',        'recordsTotal' );
		var recordsFiltered = compat( 'iTotalDisplayRecords', 'recordsFiltered' );
	
		if ( draw !== undefined ) {
			// Protect against out of sequence returns
			if ( draw*1 < settings.iDraw ) {
				return;
			}
			settings.iDraw = draw * 1;
		}
	
		// No data in returned object, so rather than an array, we show an empty table
		if ( ! data ) {
			data = [];
		}
	
		_fnClearTable( settings );
		settings._iRecordsTotal   = parseInt(recordsTotal, 10);
		settings._iRecordsDisplay = parseInt(recordsFiltered, 10);
	
		for ( var i=0, ien=data.length ; i<ien ; i++ ) {
			_fnAddData( settings, data[i] );
		}
		settings.aiDisplay = settings.aiDisplayMaster.slice();
	
		_fnDraw( settings, true );
	
		if ( ! settings._bInitComplete ) {
			_fnInitComplete( settings, json );
		}
	
		_fnProcessingDisplay( settings, false );
	}
	
	
	/**
	 * Get the data from the JSON data source to use for drawing a table. Using
	 * `_fnGetObjectDataFn` allows the data to be sourced from a property of the
	 * source object, or from a processing function.
	 *  @param {object} oSettings dataTables settings object
	 *  @param  {object} json Data source object / array from the server
	 *  @return {array} Array of data to use
	 */
	 function _fnAjaxDataSrc ( oSettings, json, write )
	 {
		var dataSrc = $.isPlainObject( oSettings.ajax ) && oSettings.ajax.dataSrc !== undefined ?
			oSettings.ajax.dataSrc :
			oSettings.sAjaxDataProp; // Compatibility with 1.9-.
	
		if ( ! write ) {
			if ( dataSrc === 'data' ) {
				// If the default, then we still want to support the old style, and safely ignore
				// it if possible
				return json.aaData || json[dataSrc];
			}
	
			return dataSrc !== "" ?
				_fnGetObjectDataFn( dataSrc )( json ) :
				json;
		}
	
		// set
		_fnSetObjectDataFn( dataSrc )( json, write );
	}
	
	/**
	 * Generate the node required for filtering text
	 *  @returns {node} Filter control element
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlFilter ( settings )
	{
		var classes = settings.oClasses;
		var tableId = settings.sTableId;
		var language = settings.oLanguage;
		var previousSearch = settings.oPreviousSearch;
		var features = settings.aanFeatures;
		var input = '<input type="search" class="'+classes.sFilterInput+'"/>';
	
		var str = language.sSearch;
		str = str.match(/_INPUT_/) ?
			str.replace('_INPUT_', input) :
			str+input;
	
		var filter = $('<div/>', {
				'id': ! features.f ? tableId+'_filter' : null,
				'class': classes.sFilter
			} )
			.append( $('<label/>' ).append( str ) );
	
		var searchFn = function(event) {
			/* Update all other filter input elements for the new display */
			var n = features.f;
			var val = !this.value ? "" : this.value; // mental IE8 fix :-(
			if(previousSearch.return && event.key !== "Enter") {
				return;
			}
			/* Now do the filter */
			if ( val != previousSearch.sSearch ) {
				_fnFilterComplete( settings, {
					"sSearch": val,
					"bRegex": previousSearch.bRegex,
					"bSmart": previousSearch.bSmart ,
					"bCaseInsensitive": previousSearch.bCaseInsensitive,
					"return": previousSearch.return
				} );
	
				// Need to redraw, without resorting
				settings._iDisplayStart = 0;
				_fnDraw( settings );
			}
		};
	
		var searchDelay = settings.searchDelay !== null ?
			settings.searchDelay :
			_fnDataSource( settings ) === 'ssp' ?
				400 :
				0;
	
		var jqFilter = $('input', filter)
			.val( previousSearch.sSearch )
			.attr( 'placeholder', language.sSearchPlaceholder )
			.on(
				'keyup.DT search.DT input.DT paste.DT cut.DT',
				searchDelay ?
					_fnThrottle( searchFn, searchDelay ) :
					searchFn
			)
			.on( 'mouseup', function(e) {
				// Edge fix! Edge 17 does not trigger anything other than mouse events when clicking
				// on the clear icon (Edge bug 17584515). This is safe in other browsers as `searchFn`
				// checks the value to see if it has changed. In other browsers it won't have.
				setTimeout( function () {
					searchFn.call(jqFilter[0], e);
				}, 10);
			} )
			.on( 'keypress.DT', function(e) {
				/* Prevent form submission */
				if ( e.keyCode == 13 ) {
					return false;
				}
			} )
			.attr('aria-controls', tableId);
	
		// Update the input elements whenever the table is filtered
		$(settings.nTable).on( 'search.dt.DT', function ( ev, s ) {
			if ( settings === s ) {
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame...
				try {
					if ( jqFilter[0] !== document.activeElement ) {
						jqFilter.val( previousSearch.sSearch );
					}
				}
				catch ( e ) {}
			}
		} );
	
		return filter[0];
	}
	
	
	/**
	 * Filter the table using both the global filter and column based filtering
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oSearch search information
	 *  @param {int} [iForce] force a research of the master array (1) or not (undefined or 0)
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterComplete ( oSettings, oInput, iForce )
	{
		var oPrevSearch = oSettings.oPreviousSearch;
		var aoPrevSearch = oSettings.aoPreSearchCols;
		var fnSaveFilter = function ( oFilter ) {
			/* Save the filtering values */
			oPrevSearch.sSearch = oFilter.sSearch;
			oPrevSearch.bRegex = oFilter.bRegex;
			oPrevSearch.bSmart = oFilter.bSmart;
			oPrevSearch.bCaseInsensitive = oFilter.bCaseInsensitive;
			oPrevSearch.return = oFilter.return;
		};
		var fnRegex = function ( o ) {
			// Backwards compatibility with the bEscapeRegex option
			return o.bEscapeRegex !== undefined ? !o.bEscapeRegex : o.bRegex;
		};
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo As per sort - can this be moved into an event handler?
		_fnColumnTypes( oSettings );
	
		/* In server-side processing all filtering is done by the server, so no point hanging around here */
		if ( _fnDataSource( oSettings ) != 'ssp' )
		{
			/* Global filter */
			_fnFilter( oSettings, oInput.sSearch, iForce, fnRegex(oInput), oInput.bSmart, oInput.bCaseInsensitive, oInput.return );
			fnSaveFilter( oInput );
	
			/* Now do the individual column filter */
			for ( var i=0 ; i<aoPrevSearch.length ; i++ )
			{
				_fnFilterColumn( oSettings, aoPrevSearch[i].sSearch, i, fnRegex(aoPrevSearch[i]),
					aoPrevSearch[i].bSmart, aoPrevSearch[i].bCaseInsensitive );
			}
	
			/* Custom filtering */
			_fnFilterCustom( oSettings );
		}
		else
		{
			fnSaveFilter( oInput );
		}
	
		/* Tell the draw function we have been filtering */
		oSettings.bFiltered = true;
		_fnCallbackFire( oSettings, null, 'search', [oSettings] );
	}
	
	
	/**
	 * Apply custom filtering functions
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCustom( settings )
	{
		var filters = DataTable.ext.search;
		var displayRows = settings.aiDisplay;
		var row, rowIdx;
	
		for ( var i=0, ien=filters.length ; i<ien ; i++ ) {
			var rows = [];
	
			// Loop over each row and see if it should be included
			for ( var j=0, jen=displayRows.length ; j<jen ; j++ ) {
				rowIdx = displayRows[ j ];
				row = settings.aoData[ rowIdx ];
	
				if ( filters[i]( settings, row._aFilterData, rowIdx, row._aData, j ) ) {
					rows.push( rowIdx );
				}
			}
	
			// So the array reference doesn't break set the results into the
			// existing array
			displayRows.length = 0;
			$.merge( displayRows, rows );
		}
	}
	
	
	/**
	 * Filter the table on a per-column basis
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sInput string to filter on
	 *  @param {int} iColumn column to filter
	 *  @param {bool} bRegex treat search string as a regular expression or not
	 *  @param {bool} bSmart use smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insensitive matching or not
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterColumn ( settings, searchStr, colIdx, regex, smart, caseInsensitive )
	{
		if ( searchStr === '' ) {
			return;
		}
	
		var data;
		var out = [];
		var display = settings.aiDisplay;
		var rpSearch = _fnFilterCreateSearch( searchStr, regex, smart, caseInsensitive );
	
		for ( var i=0 ; i<display.length ; i++ ) {
			data = settings.aoData[ display[i] ]._aFilterData[ colIdx ];
	
			if ( rpSearch.test( data ) ) {
				out.push( display[i] );
			}
		}
	
		settings.aiDisplay = out;
	}
	
	
	/**
	 * Filter the data table based on user input and draw the table
	 *  @param {object} settings dataTables settings object
	 *  @param {string} input string to filter on
	 *  @param {int} force optional - force a research of the master array (1) or not (undefined or 0)
	 *  @param {bool} regex treat as a regular expression or not
	 *  @param {bool} smart perform smart filtering or not
	 *  @param {bool} caseInsensitive Do case insensitive matching or not
	 *  @memberof DataTable#oApi
	 */
	function _fnFilter( settings, input, force, regex, smart, caseInsensitive )
	{
		var rpSearch = _fnFilterCreateSearch( input, regex, smart, caseInsensitive );
		var prevSearch = settings.oPreviousSearch.sSearch;
		var displayMaster = settings.aiDisplayMaster;
		var display, invalidated, i;
		var filtered = [];
	
		// Need to take account of custom filtering functions - always filter
		if ( DataTable.ext.search.length !== 0 ) {
			force = true;
		}
	
		// Check if any of the rows were invalidated
		invalidated = _fnFilterData( settings );
	
		// If the input is blank - we just want the full data set
		if ( input.length <= 0 ) {
			settings.aiDisplay = displayMaster.slice();
		}
		else {
			// New search - start from the master array
			if ( invalidated ||
				 force ||
				 regex ||
				 prevSearch.length > input.length ||
				 input.indexOf(prevSearch) !== 0 ||
				 settings.bSorted // On resort, the display master needs to be
				                  // re-filtered since indexes will have changed
			) {
				settings.aiDisplay = displayMaster.slice();
			}
	
			// Search the display array
			display = settings.aiDisplay;
	
			for ( i=0 ; i<display.length ; i++ ) {
				if ( rpSearch.test( settings.aoData[ display[i] ]._sFilterRow ) ) {
					filtered.push( display[i] );
				}
			}
	
			settings.aiDisplay = filtered;
		}
	}
	
	
	/**
	 * Build a regular expression object suitable for searching a table
	 *  @param {string} sSearch string to search for
	 *  @param {bool} bRegex treat as a regular expression or not
	 *  @param {bool} bSmart perform smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insensitive matching or not
	 *  @returns {RegExp} constructed object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCreateSearch( search, regex, smart, caseInsensitive )
	{
		search = regex ?
			search :
			_fnEscapeRegex( search );
		
		if ( smart ) {
			/* For smart filtering we want to allow the search to work regardless of
			 * word order. We also want double quoted text to be preserved, so word
			 * order is important - a la google. So this is what we want to
			 * generate:
			 * 
			 * ^(?=.*?\bone\b)(?=.*?\btwo three\b)(?=.*?\bfour\b).*$
			 */
			var a = $.map( search.match( /"[^"]+"|[^ ]+/g ) || [''], function ( word ) {
				if ( word.charAt(0) === '"' ) {
					var m = word.match( /^"(.*)"$/ );
					word = m ? m[1] : word;
				}
	
				return word.replace('"', '');
			} );
	
			search = '^(?=.*?'+a.join( ')(?=.*?' )+').*$';
		}
	
		return new RegExp( search, caseInsensitive ? 'i' : '' );
	}
	
	
	/**
	 * Escape a string such that it can be used in a regular expression
	 *  @param {string} sVal string to escape
	 *  @returns {string} escaped string
	 *  @memberof DataTable#oApi
	 */
	var _fnEscapeRegex = DataTable.util.escapeRegex;
	
	var __filter_div = $('<div>')[0];
	var __filter_div_textContent = __filter_div.textContent !== undefined;
	
	// Update the filtering data for each row if needed (by invalidation or first run)
	function _fnFilterData ( settings )
	{
		var columns = settings.aoColumns;
		var column;
		var i, j, ien, jen, filterData, cellData, row;
		var wasInvalidated = false;
	
		for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aFilterData ) {
				filterData = [];
	
				for ( j=0, jen=columns.length ; j<jen ; j++ ) {
					column = columns[j];
	
					if ( column.bSearchable ) {
						cellData = _fnGetCellData( settings, i, j, 'filter' );
	
						// Search in DataTables 1.10 is string based. In 1.11 this
						// should be altered to also allow strict type checking.
						if ( cellData === null ) {
							cellData = '';
						}
	
						if ( typeof cellData !== 'string' && cellData.toString ) {
							cellData = cellData.toString();
						}
					}
					else {
						cellData = '';
					}
	
					// If it looks like there is an HTML entity in the string,
					// attempt to decode it so sorting works as expected. Note that
					// we could use a single line of jQuery to do this, but the DOM
					// method used here is much faster http://jsperf.com/html-decode
					if ( cellData.indexOf && cellData.indexOf('&') !== -1 ) {
						__filter_div.innerHTML = cellData;
						cellData = __filter_div_textContent ?
							__filter_div.textContent :
							__filter_div.innerText;
					}
	
					if ( cellData.replace ) {
						cellData = cellData.replace(/[\r\n\u2028]/g, '');
					}
	
					filterData.push( cellData );
				}
	
				row._aFilterData = filterData;
				row._sFilterRow = filterData.join('  ');
				wasInvalidated = true;
			}
		}
	
		return wasInvalidated;
	}
	
	
	/**
	 * Convert from the internal Hungarian notation to camelCase for external
	 * interaction
	 *  @param {object} obj Object to convert
	 *  @returns {object} Inverted object
	 *  @memberof DataTable#oApi
	 */
	function _fnSearchToCamel ( obj )
	{
		return {
			search:          obj.sSearch,
			smart:           obj.bSmart,
			regex:           obj.bRegex,
			caseInsensitive: obj.bCaseInsensitive
		};
	}
	
	
	
	/**
	 * Convert from camelCase notation to the internal Hungarian. We could use the
	 * Hungarian convert function here, but this is cleaner
	 *  @param {object} obj Object to convert
	 *  @returns {object} Inverted object
	 *  @memberof DataTable#oApi
	 */
	function _fnSearchToHung ( obj )
	{
		return {
			sSearch:          obj.search,
			bSmart:           obj.smart,
			bRegex:           obj.regex,
			bCaseInsensitive: obj.caseInsensitive
		};
	}
	
	/**
	 * Generate the node required for the info display
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {node} Information element
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlInfo ( settings )
	{
		var
			tid = settings.sTableId,
			nodes = settings.aanFeatures.i,
			n = $('<div/>', {
				'class': settings.oClasses.sInfo,
				'id': ! nodes ? tid+'_info' : null
			} );
	
		if ( ! nodes ) {
			// Update display on each draw
			settings.aoDrawCallback.push( {
				"fn": _fnUpdateInfo,
				"sName": "information"
			} );
	
			n
				.attr( 'role', 'status' )
				.attr( 'aria-live', 'polite' );
	
			// Table is described by our info div
			$(settings.nTable).attr( 'aria-describedby', tid+'_info' );
		}
	
		return n[0];
	}
	
	
	/**
	 * Update the information elements in the display
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnUpdateInfo ( settings )
	{
		/* Show information about the table */
		var nodes = settings.aanFeatures.i;
		if ( nodes.length === 0 ) {
			return;
		}
	
		var
			lang  = settings.oLanguage,
			start = settings._iDisplayStart+1,
			end   = settings.fnDisplayEnd(),
			max   = settings.fnRecordsTotal(),
			total = settings.fnRecordsDisplay(),
			out   = total ?
				lang.sInfo :
				lang.sInfoEmpty;
	
		if ( total !== max ) {
			/* Record set after filtering */
			out += ' ' + lang.sInfoFiltered;
		}
	
		// Convert the macros
		out += lang.sInfoPostFix;
		out = _fnInfoMacros( settings, out );
	
		var callback = lang.fnInfoCallback;
		if ( callback !== null ) {
			out = callback.call( settings.oInstance,
				settings, start, end, max, total, out
			);
		}
	
		$(nodes).html( out );
	}
	
	
	function _fnInfoMacros ( settings, str )
	{
		// When infinite scrolling, we are always starting at 1. _iDisplayStart is used only
		// internally
		var
			formatter  = settings.fnFormatNumber,
			start      = settings._iDisplayStart+1,
			len        = settings._iDisplayLength,
			vis        = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return str.
			replace(/_START_/g, formatter.call( settings, start ) ).
			replace(/_END_/g,   formatter.call( settings, settings.fnDisplayEnd() ) ).
			replace(/_MAX_/g,   formatter.call( settings, settings.fnRecordsTotal() ) ).
			replace(/_TOTAL_/g, formatter.call( settings, vis ) ).
			replace(/_PAGE_/g,  formatter.call( settings, all ? 1 : Math.ceil( start / len ) ) ).
			replace(/_PAGES_/g, formatter.call( settings, all ? 1 : Math.ceil( vis / len ) ) );
	}
	
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnInitialise ( settings )
	{
		var i, iLen, iAjaxStart=settings.iInitDisplayStart;
		var columns = settings.aoColumns, column;
		var features = settings.oFeatures;
		var deferLoading = settings.bDeferLoading; // value modified by the draw
	
		/* Ensure that the table data is fully initialised */
		if ( ! settings.bInitialised ) {
			setTimeout( function(){ _fnInitialise( settings ); }, 200 );
			return;
		}
	
		/* Show the display HTML options */
		_fnAddOptionsHtml( settings );
	
		/* Build and draw the header / footer for the table */
		_fnBuildHead( settings );
		_fnDrawHead( settings, settings.aoHeader );
		_fnDrawHead( settings, settings.aoFooter );
	
		/* Okay to show that something is going on now */
		_fnProcessingDisplay( settings, true );
	
		/* Calculate sizes for columns */
		if ( features.bAutoWidth ) {
			_fnCalculateColumnWidths( settings );
		}
	
		for ( i=0, iLen=columns.length ; i<iLen ; i++ ) {
			column = columns[i];
	
			if ( column.sWidth ) {
				column.nTh.style.width = _fnStringToCss( column.sWidth );
			}
		}
	
		_fnCallbackFire( settings, null, 'preInit', [settings] );
	
		// If there is default sorting required - let's do it. The sort function
		// will do the drawing for us. Otherwise we draw the table regardless of the
		// Ajax source - this allows the table to look initialised for Ajax sourcing
		// data (show 'loading' message possibly)
		_fnReDraw( settings );
	
		// Server-side processing init complete is done by _fnAjaxUpdateDraw
		var dataSrc = _fnDataSource( settings );
		if ( dataSrc != 'ssp' || deferLoading ) {
			// if there is an ajax source load the data
			if ( dataSrc == 'ajax' ) {
				_fnBuildAjax( settings, [], function(json) {
					var aData = _fnAjaxDataSrc( settings, json );
	
					// Got the data - add it to the table
					for ( i=0 ; i<aData.length ; i++ ) {
						_fnAddData( settings, aData[i] );
					}
	
					// Reset the init display for cookie saving. We've already done
					// a filter, and therefore cleared it before. So we need to make
					// it appear 'fresh'
					settings.iInitDisplayStart = iAjaxStart;
	
					_fnReDraw( settings );
	
					_fnProcessingDisplay( settings, false );
					_fnInitComplete( settings, json );
				}, settings );
			}
			else {
				_fnProcessingDisplay( settings, false );
				_fnInitComplete( settings );
			}
		}
	}
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} [json] JSON from the server that completed the table, if using Ajax source
	 *    with client-side processing (optional)
	 *  @memberof DataTable#oApi
	 */
	function _fnInitComplete ( settings, json )
	{
		settings._bInitComplete = true;
	
		// When data was added after the initialisation (data or Ajax) we need to
		// calculate the column sizing
		if ( json || settings.oInit.aaData ) {
			_fnAdjustColumnSizing( settings );
		}
	
		_fnCallbackFire( settings, null, 'plugin-init', [settings, json] );
		_fnCallbackFire( settings, 'aoInitComplete', 'init', [settings, json] );
	}
	
	
	function _fnLengthChange ( settings, val )
	{
		var len = parseInt( val, 10 );
		settings._iDisplayLength = len;
	
		_fnLengthOverflow( settings );
	
		// Fire length change event
		_fnCallbackFire( settings, null, 'length', [settings, len] );
	}
	
	
	/**
	 * Generate the node required for user display length changing
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Display length feature node
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlLength ( settings )
	{
		var
			classes  = settings.oClasses,
			tableId  = settings.sTableId,
			menu     = settings.aLengthMenu,
			d2       = Array.isArray( menu[0] ),
			lengths  = d2 ? menu[0] : menu,
			language = d2 ? menu[1] : menu;
	
		var select = $('<select/>', {
			'name':          tableId+'_length',
			'aria-controls': tableId,
			'class':         classes.sLengthSelect
		} );
	
		for ( var i=0, ien=lengths.length ; i<ien ; i++ ) {
			select[0][ i ] = new Option(
				typeof language[i] === 'number' ?
					settings.fnFormatNumber( language[i] ) :
					language[i],
				lengths[i]
			);
		}
	
		var div = $('<div><label/></div>').addClass( classes.sLength );
		if ( ! settings.aanFeatures.l ) {
			div[0].id = tableId+'_length';
		}
	
		div.children().append(
			settings.oLanguage.sLengthMenu.replace( '_MENU_', select[0].outerHTML )
		);
	
		// Can't use `select` variable as user might provide their own and the
		// reference is broken by the use of outerHTML
		$('select', div)
			.val( settings._iDisplayLength )
			.on( 'change.DT', function(e) {
				_fnLengthChange( settings, $(this).val() );
				_fnDraw( settings );
			} );
	
		// Update node value whenever anything changes the table's length
		$(settings.nTable).on( 'length.dt.DT', function (e, s, len) {
			if ( settings === s ) {
				$('select', div).val( len );
			}
		} );
	
		return div[0];
	}
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Note that most of the paging logic is done in
	 * DataTable.ext.pager
	 */
	
	/**
	 * Generate the node required for default pagination
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {node} Pagination feature node
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlPaginate ( settings )
	{
		var
			type   = settings.sPaginationType,
			plugin = DataTable.ext.pager[ type ],
			modern = typeof plugin === 'function',
			redraw = function( settings ) {
				_fnDraw( settings );
			},
			node = $('<div/>').addClass( settings.oClasses.sPaging + type )[0],
			features = settings.aanFeatures;
	
		if ( ! modern ) {
			plugin.fnInit( settings, node, redraw );
		}
	
		/* Add a draw callback for the pagination on first instance, to update the paging display */
		if ( ! features.p )
		{
			node.id = settings.sTableId+'_paginate';
	
			settings.aoDrawCallback.push( {
				"fn": function( settings ) {
					if ( modern ) {
						var
							start      = settings._iDisplayStart,
							len        = settings._iDisplayLength,
							visRecords = settings.fnRecordsDisplay(),
							all        = len === -1,
							page = all ? 0 : Math.ceil( start / len ),
							pages = all ? 1 : Math.ceil( visRecords / len ),
							buttons = plugin(page, pages),
							i, ien;
	
						for ( i=0, ien=features.p.length ; i<ien ; i++ ) {
							_fnRenderer( settings, 'pageButton' )(
								settings, features.p[i], i, buttons, page, pages
							);
						}
					}
					else {
						plugin.fnUpdate( settings, redraw );
					}
				},
				"sName": "pagination"
			} );
		}
	
		return node;
	}
	
	
	/**
	 * Alter the display settings to change the page
	 *  @param {object} settings DataTables settings object
	 *  @param {string|int} action Paging action to take: "first", "previous",
	 *    "next" or "last" or page number to jump to (integer)
	 *  @param [bool] redraw Automatically draw the update or not
	 *  @returns {bool} true page has changed, false - no change
	 *  @memberof DataTable#oApi
	 */
	function _fnPageChange ( settings, action, redraw )
	{
		var
			start     = settings._iDisplayStart,
			len       = settings._iDisplayLength,
			records   = settings.fnRecordsDisplay();
	
		if ( records === 0 || len === -1 )
		{
			start = 0;
		}
		else if ( typeof action === "number" )
		{
			start = action * len;
	
			if ( start > records )
			{
				start = 0;
			}
		}
		else if ( action == "first" )
		{
			start = 0;
		}
		else if ( action == "previous" )
		{
			start = len >= 0 ?
				start - len :
				0;
	
			if ( start < 0 )
			{
			  start = 0;
			}
		}
		else if ( action == "next" )
		{
			if ( start + len < records )
			{
				start += len;
			}
		}
		else if ( action == "last" )
		{
			start = Math.floor( (records-1) / len) * len;
		}
		else
		{
			_fnLog( settings, 0, "Unknown paging action: "+action, 5 );
		}
	
		var changed = settings._iDisplayStart !== start;
		settings._iDisplayStart = start;
	
		if ( changed ) {
			_fnCallbackFire( settings, null, 'page', [settings] );
	
			if ( redraw ) {
				_fnDraw( settings );
			}
		}
	
		return changed;
	}
	
	
	
	/**
	 * Generate the node required for the processing node
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Processing element
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlProcessing ( settings )
	{
		return $('<div/>', {
				'id': ! settings.aanFeatures.r ? settings.sTableId+'_processing' : null,
				'class': settings.oClasses.sProcessing
			} )
			.html( settings.oLanguage.sProcessing )
			.append('<div><div></div><div></div><div></div><div></div></div>')
			.insertBefore( settings.nTable )[0];
	}
	
	
	/**
	 * Display or hide the processing indicator
	 *  @param {object} settings dataTables settings object
	 *  @param {bool} show Show the processing indicator (true) or not (false)
	 *  @memberof DataTable#oApi
	 */
	function _fnProcessingDisplay ( settings, show )
	{
		if ( settings.oFeatures.bProcessing ) {
			$(settings.aanFeatures.r).css( 'display', show ? 'block' : 'none' );
		}
	
		_fnCallbackFire( settings, null, 'processing', [settings, show] );
	}
	
	/**
	 * Add any control elements for the table - specifically scrolling
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Node to add to the DOM
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlTable ( settings )
	{
		var table = $(settings.nTable);
	
		// Scrolling from here on in
		var scroll = settings.oScroll;
	
		if ( scroll.sX === '' && scroll.sY === '' ) {
			return settings.nTable;
		}
	
		var scrollX = scroll.sX;
		var scrollY = scroll.sY;
		var classes = settings.oClasses;
		var caption = table.children('caption');
		var captionSide = caption.length ? caption[0]._captionSide : null;
		var headerClone = $( table[0].cloneNode(false) );
		var footerClone = $( table[0].cloneNode(false) );
		var footer = table.children('tfoot');
		var _div = '<div/>';
		var size = function ( s ) {
			return !s ? null : _fnStringToCss( s );
		};
	
		if ( ! footer.length ) {
			footer = null;
		}
	
		/*
		 * The HTML structure that we want to generate in this function is:
		 *  div - scroller
		 *    div - scroll head
		 *      div - scroll head inner
		 *        table - scroll head table
		 *          thead - thead
		 *    div - scroll body
		 *      table - table (master table)
		 *        thead - thead clone for sizing
		 *        tbody - tbody
		 *    div - scroll foot
		 *      div - scroll foot inner
		 *        table - scroll foot table
		 *          tfoot - tfoot
		 */
		var scroller = $( _div, { 'class': classes.sScrollWrapper } )
			.append(
				$(_div, { 'class': classes.sScrollHead } )
					.css( {
						overflow: 'hidden',
						position: 'relative',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollHeadInner } )
							.css( {
								'box-sizing': 'content-box',
								width: scroll.sXInner || '100%'
							} )
							.append(
								headerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'top' ? caption : null )
									.append(
										table.children('thead')
									)
							)
					)
			)
			.append(
				$(_div, { 'class': classes.sScrollBody } )
					.css( {
						position: 'relative',
						overflow: 'auto',
						width: size( scrollX )
					} )
					.append( table )
			);
	
		if ( footer ) {
			scroller.append(
				$(_div, { 'class': classes.sScrollFoot } )
					.css( {
						overflow: 'hidden',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollFootInner } )
							.append(
								footerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'bottom' ? caption : null )
									.append(
										table.children('tfoot')
									)
							)
					)
			);
		}
	
		var children = scroller.children();
		var scrollHead = children[0];
		var scrollBody = children[1];
		var scrollFoot = footer ? children[2] : null;
	
		// When the body is scrolled, then we also want to scroll the headers
		if ( scrollX ) {
			$(scrollBody).on( 'scroll.DT', function (e) {
				var scrollLeft = this.scrollLeft;
	
				scrollHead.scrollLeft = scrollLeft;
	
				if ( footer ) {
					scrollFoot.scrollLeft = scrollLeft;
				}
			} );
		}
	
		$(scrollBody).css('max-height', scrollY);
		if (! scroll.bCollapse) {
			$(scrollBody).css('height', scrollY);
		}
	
		settings.nScrollHead = scrollHead;
		settings.nScrollBody = scrollBody;
		settings.nScrollFoot = scrollFoot;
	
		// On redraw - align columns
		settings.aoDrawCallback.push( {
			"fn": _fnScrollDraw,
			"sName": "scrolling"
		} );
	
		return scroller[0];
	}
	
	
	
	/**
	 * Update the header, footer and body tables for resizing - i.e. column
	 * alignment.
	 *
	 * Welcome to the most horrible function DataTables. The process that this
	 * function follows is basically:
	 *   1. Re-create the table inside the scrolling div
	 *   2. Take live measurements from the DOM
	 *   3. Apply the measurements to align the columns
	 *   4. Clean up
	 *
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnScrollDraw ( settings )
	{
		// Given that this is such a monster function, a lot of variables are use
		// to try and keep the minimised size as small as possible
		var
			scroll         = settings.oScroll,
			scrollX        = scroll.sX,
			scrollXInner   = scroll.sXInner,
			scrollY        = scroll.sY,
			barWidth       = scroll.iBarWidth,
			divHeader      = $(settings.nScrollHead),
			divHeaderStyle = divHeader[0].style,
			divHeaderInner = divHeader.children('div'),
			divHeaderInnerStyle = divHeaderInner[0].style,
			divHeaderTable = divHeaderInner.children('table'),
			divBodyEl      = settings.nScrollBody,
			divBody        = $(divBodyEl),
			divBodyStyle   = divBodyEl.style,
			divFooter      = $(settings.nScrollFoot),
			divFooterInner = divFooter.children('div'),
			divFooterTable = divFooterInner.children('table'),
			header         = $(settings.nTHead),
			table          = $(settings.nTable),
			tableEl        = table[0],
			tableStyle     = tableEl.style,
			footer         = settings.nTFoot ? $(settings.nTFoot) : null,
			browser        = settings.oBrowser,
			ie67           = browser.bScrollOversize,
			dtHeaderCells  = _pluck( settings.aoColumns, 'nTh' ),
			headerTrgEls, footerTrgEls,
			headerSrcEls, footerSrcEls,
			headerCopy, footerCopy,
			headerWidths=[], footerWidths=[],
			headerContent=[], footerContent=[],
			idx, correction, sanityWidth,
			zeroOut = function(nSizer) {
				var style = nSizer.style;
				style.paddingTop = "0";
				style.paddingBottom = "0";
				style.borderTopWidth = "0";
				style.borderBottomWidth = "0";
				style.height = 0;
			};
	
		// If the scrollbar visibility has changed from the last draw, we need to
		// adjust the column sizes as the table width will have changed to account
		// for the scrollbar
		var scrollBarVis = divBodyEl.scrollHeight > divBodyEl.clientHeight;
		
		if ( settings.scrollBarVis !== scrollBarVis && settings.scrollBarVis !== undefined ) {
			settings.scrollBarVis = scrollBarVis;
			_fnAdjustColumnSizing( settings );
			return; // adjust column sizing will call this function again
		}
		else {
			settings.scrollBarVis = scrollBarVis;
		}
	
		/*
		 * 1. Re-create the table inside the scrolling div
		 */
	
		// Remove the old minimised thead and tfoot elements in the inner table
		table.children('thead, tfoot').remove();
	
		if ( footer ) {
			footerCopy = footer.clone().prependTo( table );
			footerTrgEls = footer.find('tr'); // the original tfoot is in its own table and must be sized
			footerSrcEls = footerCopy.find('tr');
			footerCopy.find('[id]').removeAttr('id');
		}
	
		// Clone the current header and footer elements and then place it into the inner table
		headerCopy = header.clone().prependTo( table );
		headerTrgEls = header.find('tr'); // original header is in its own table
		headerSrcEls = headerCopy.find('tr');
		headerCopy.find('th, td').removeAttr('tabindex');
		headerCopy.find('[id]').removeAttr('id');
	
	
		/*
		 * 2. Take live measurements from the DOM - do not alter the DOM itself!
		 */
	
		// Remove old sizing and apply the calculated column widths
		// Get the unique column headers in the newly created (cloned) header. We want to apply the
		// calculated sizes to this header
		if ( ! scrollX )
		{
			divBodyStyle.width = '100%';
			divHeader[0].style.width = '100%';
		}
	
		$.each( _fnGetUniqueThs( settings, headerCopy ), function ( i, el ) {
			idx = _fnVisibleToColumnIndex( settings, i );
			el.style.width = settings.aoColumns[idx].sWidth;
		} );
	
		if ( footer ) {
			_fnApplyToChildren( function(n) {
				n.style.width = "";
			}, footerSrcEls );
		}
	
		// Size the table as a whole
		sanityWidth = table.outerWidth();
		if ( scrollX === "" ) {
			// No x scrolling
			tableStyle.width = "100%";
	
			// IE7 will make the width of the table when 100% include the scrollbar
			// - which is shouldn't. When there is a scrollbar we need to take this
			// into account.
			if ( ie67 && (table.find('tbody').height() > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( table.outerWidth() - barWidth);
			}
	
			// Recalculate the sanity width
			sanityWidth = table.outerWidth();
		}
		else if ( scrollXInner !== "" ) {
			// legacy x scroll inner has been given - use it
			tableStyle.width = _fnStringToCss(scrollXInner);
	
			// Recalculate the sanity width
			sanityWidth = table.outerWidth();
		}
	
		// Hidden header should have zero height, so remove padding and borders. Then
		// set the width based on the real headers
	
		// Apply all styles in one pass
		_fnApplyToChildren( zeroOut, headerSrcEls );
	
		// Read all widths in next pass
		_fnApplyToChildren( function(nSizer) {
			var style = window.getComputedStyle ?
				window.getComputedStyle(nSizer).width :
				_fnStringToCss( $(nSizer).width() );
	
			headerContent.push( nSizer.innerHTML );
			headerWidths.push( style );
		}, headerSrcEls );
	
		// Apply all widths in final pass
		_fnApplyToChildren( function(nToSize, i) {
			nToSize.style.width = headerWidths[i];
		}, headerTrgEls );
	
		$(headerSrcEls).css('height', 0);
	
		/* Same again with the footer if we have one */
		if ( footer )
		{
			_fnApplyToChildren( zeroOut, footerSrcEls );
	
			_fnApplyToChildren( function(nSizer) {
				footerContent.push( nSizer.innerHTML );
				footerWidths.push( _fnStringToCss( $(nSizer).css('width') ) );
			}, footerSrcEls );
	
			_fnApplyToChildren( function(nToSize, i) {
				nToSize.style.width = footerWidths[i];
			}, footerTrgEls );
	
			$(footerSrcEls).height(0);
		}
	
	
		/*
		 * 3. Apply the measurements
		 */
	
		// "Hide" the header and footer that we used for the sizing. We need to keep
		// the content of the cell so that the width applied to the header and body
		// both match, but we want to hide it completely. We want to also fix their
		// width to what they currently are
		_fnApplyToChildren( function(nSizer, i) {
			nSizer.innerHTML = '<div class="dataTables_sizing">'+headerContent[i]+'</div>';
			nSizer.childNodes[0].style.height = "0";
			nSizer.childNodes[0].style.overflow = "hidden";
			nSizer.style.width = headerWidths[i];
		}, headerSrcEls );
	
		if ( footer )
		{
			_fnApplyToChildren( function(nSizer, i) {
				nSizer.innerHTML = '<div class="dataTables_sizing">'+footerContent[i]+'</div>';
				nSizer.childNodes[0].style.height = "0";
				nSizer.childNodes[0].style.overflow = "hidden";
				nSizer.style.width = footerWidths[i];
			}, footerSrcEls );
		}
	
		// Sanity check that the table is of a sensible width. If not then we are going to get
		// misalignment - try to prevent this by not allowing the table to shrink below its min width
		if ( Math.round(table.outerWidth()) < Math.round(sanityWidth) )
		{
			// The min width depends upon if we have a vertical scrollbar visible or not */
			correction = ((divBodyEl.scrollHeight > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")) ?
					sanityWidth+barWidth :
					sanityWidth;
	
			// IE6/7 are a law unto themselves...
			if ( ie67 && (divBodyEl.scrollHeight >
				divBodyEl.offsetHeight || divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( correction-barWidth );
			}
	
			// And give the user a warning that we've stopped the table getting too small
			if ( scrollX === "" || scrollXInner !== "" ) {
				_fnLog( settings, 1, 'Possible column misalignment', 6 );
			}
		}
		else
		{
			correction = '100%';
		}
	
		// Apply to the container elements
		divBodyStyle.width = _fnStringToCss( correction );
		divHeaderStyle.width = _fnStringToCss( correction );
	
		if ( footer ) {
			settings.nScrollFoot.style.width = _fnStringToCss( correction );
		}
	
	
		/*
		 * 4. Clean up
		 */
		if ( ! scrollY ) {
			/* IE7< puts a vertical scrollbar in place (when it shouldn't be) due to subtracting
			 * the scrollbar height from the visible display, rather than adding it on. We need to
			 * set the height in order to sort this. Don't want to do it in any other browsers.
			 */
			if ( ie67 ) {
				divBodyStyle.height = _fnStringToCss( tableEl.offsetHeight+barWidth );
			}
		}
	
		/* Finally set the width's of the header and footer tables */
		var iOuterWidth = table.outerWidth();
		divHeaderTable[0].style.width = _fnStringToCss( iOuterWidth );
		divHeaderInnerStyle.width = _fnStringToCss( iOuterWidth );
	
		// Figure out if there are scrollbar present - if so then we need a the header and footer to
		// provide a bit more space to allow "overflow" scrolling (i.e. past the scrollbar)
		var bScrolling = table.height() > divBodyEl.clientHeight || divBody.css('overflow-y') == "scroll";
		var padding = 'padding' + (browser.bScrollbarLeft ? 'Left' : 'Right' );
		divHeaderInnerStyle[ padding ] = bScrolling ? barWidth+"px" : "0px";
	
		if ( footer ) {
			divFooterTable[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style[padding] = bScrolling ? barWidth+"px" : "0px";
		}
	
		// Correct DOM ordering for colgroup - comes before the thead
		table.children('colgroup').insertBefore( table.children('thead') );
	
		/* Adjust the position of the header in case we loose the y-scrollbar */
		divBody.trigger('scroll');
	
		// If sorting or filtering has occurred, jump the scrolling back to the top
		// only if we aren't holding the position
		if ( (settings.bSorted || settings.bFiltered) && ! settings._drawHold ) {
			divBodyEl.scrollTop = 0;
		}
	}
	
	
	
	/**
	 * Apply a given function to the display child nodes of an element array (typically
	 * TD children of TR rows
	 *  @param {function} fn Method to apply to the objects
	 *  @param array {nodes} an1 List of elements to look through for display children
	 *  @param array {nodes} an2 Another list (identical structure to the first) - optional
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyToChildren( fn, an1, an2 )
	{
		var index=0, i=0, iLen=an1.length;
		var nNode1, nNode2;
	
		while ( i < iLen ) {
			nNode1 = an1[i].firstChild;
			nNode2 = an2 ? an2[i].firstChild : null;
	
			while ( nNode1 ) {
				if ( nNode1.nodeType === 1 ) {
					if ( an2 ) {
						fn( nNode1, nNode2, index );
					}
					else {
						fn( nNode1, index );
					}
	
					index++;
				}
	
				nNode1 = nNode1.nextSibling;
				nNode2 = an2 ? nNode2.nextSibling : null;
			}
	
			i++;
		}
	}
	
	
	
	var __re_html_remove = /<.*?>/g;
	
	
	/**
	 * Calculate the width of columns for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnCalculateColumnWidths ( oSettings )
	{
		var
			table = oSettings.nTable,
			columns = oSettings.aoColumns,
			scroll = oSettings.oScroll,
			scrollY = scroll.sY,
			scrollX = scroll.sX,
			scrollXInner = scroll.sXInner,
			columnCount = columns.length,
			visibleColumns = _fnGetColumns( oSettings, 'bVisible' ),
			headerCells = $('th', oSettings.nTHead),
			tableWidthAttr = table.getAttribute('width'), // from DOM element
			tableContainer = table.parentNode,
			userInputs = false,
			i, column, columnIdx, width, outerWidth,
			browser = oSettings.oBrowser,
			ie67 = browser.bScrollOversize;
	
		var styleWidth = table.style.width;
		if ( styleWidth && styleWidth.indexOf('%') !== -1 ) {
			tableWidthAttr = styleWidth;
		}
	
		/* Convert any user input sizes into pixel sizes */
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			column = columns[ visibleColumns[i] ];
	
			if ( column.sWidth !== null ) {
				column.sWidth = _fnConvertToWidth( column.sWidthOrig, tableContainer );
	
				userInputs = true;
			}
		}
	
		/* If the number of columns in the DOM equals the number that we have to
		 * process in DataTables, then we can use the offsets that are created by
		 * the web- browser. No custom sizes can be set in order for this to happen,
		 * nor scrolling used
		 */
		if ( ie67 || ! userInputs && ! scrollX && ! scrollY &&
		     columnCount == _fnVisbleColumns( oSettings ) &&
		     columnCount == headerCells.length
		) {
			for ( i=0 ; i<columnCount ; i++ ) {
				var colIdx = _fnVisibleToColumnIndex( oSettings, i );
	
				if ( colIdx !== null ) {
					columns[ colIdx ].sWidth = _fnStringToCss( headerCells.eq(i).width() );
				}
			}
		}
		else
		{
			// Otherwise construct a single row, worst case, table with the widest
			// node in the data, assign any user defined widths, then insert it into
			// the DOM and allow the browser to do all the hard work of calculating
			// table widths
			var tmpTable = $(table).clone() // don't use cloneNode - IE8 will remove events on the main table
				.css( 'visibility', 'hidden' )
				.removeAttr( 'id' );
	
			// Clean up the table body
			tmpTable.find('tbody tr').remove();
			var tr = $('<tr/>').appendTo( tmpTable.find('tbody') );
	
			// Clone the table header and footer - we can't use the header / footer
			// from the cloned table, since if scrolling is active, the table's
			// real header and footer are contained in different table tags
			tmpTable.find('thead, tfoot').remove();
			tmpTable
				.append( $(oSettings.nTHead).clone() )
				.append( $(oSettings.nTFoot).clone() );
	
			// Remove any assigned widths from the footer (from scrolling)
			tmpTable.find('tfoot th, tfoot td').css('width', '');
	
			// Apply custom sizing to the cloned header
			headerCells = _fnGetUniqueThs( oSettings, tmpTable.find('thead')[0] );
	
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				column = columns[ visibleColumns[i] ];
	
				headerCells[i].style.width = column.sWidthOrig !== null && column.sWidthOrig !== '' ?
					_fnStringToCss( column.sWidthOrig ) :
					'';
	
				// For scrollX we need to force the column width otherwise the
				// browser will collapse it. If this width is smaller than the
				// width the column requires, then it will have no effect
				if ( column.sWidthOrig && scrollX ) {
					$( headerCells[i] ).append( $('<div/>').css( {
						width: column.sWidthOrig,
						margin: 0,
						padding: 0,
						border: 0,
						height: 1
					} ) );
				}
			}
	
			// Find the widest cell for each column and put it into the table
			if ( oSettings.aoData.length ) {
				for ( i=0 ; i<visibleColumns.length ; i++ ) {
					columnIdx = visibleColumns[i];
					column = columns[ columnIdx ];
	
					$( _fnGetWidestNode( oSettings, columnIdx ) )
						.clone( false )
						.append( column.sContentPadding )
						.appendTo( tr );
				}
			}
	
			// Tidy the temporary table - remove name attributes so there aren't
			// duplicated in the dom (radio elements for example)
			$('[name]', tmpTable).removeAttr('name');
	
			// Table has been built, attach to the document so we can work with it.
			// A holding element is used, positioned at the top of the container
			// with minimal height, so it has no effect on if the container scrolls
			// or not. Otherwise it might trigger scrolling when it actually isn't
			// needed
			var holder = $('<div/>').css( scrollX || scrollY ?
					{
						position: 'absolute',
						top: 0,
						left: 0,
						height: 1,
						right: 0,
						overflow: 'hidden'
					} :
					{}
				)
				.append( tmpTable )
				.appendTo( tableContainer );
	
			// When scrolling (X or Y) we want to set the width of the table as 
			// appropriate. However, when not scrolling leave the table width as it
			// is. This results in slightly different, but I think correct behaviour
			if ( scrollX && scrollXInner ) {
				tmpTable.width( scrollXInner );
			}
			else if ( scrollX ) {
				tmpTable.css( 'width', 'auto' );
				tmpTable.removeAttr('width');
	
				// If there is no width attribute or style, then allow the table to
				// collapse
				if ( tmpTable.width() < tableContainer.clientWidth && tableWidthAttr ) {
					tmpTable.width( tableContainer.clientWidth );
				}
			}
			else if ( scrollY ) {
				tmpTable.width( tableContainer.clientWidth );
			}
			else if ( tableWidthAttr ) {
				tmpTable.width( tableWidthAttr );
			}
	
			// Get the width of each column in the constructed table - we need to
			// know the inner width (so it can be assigned to the other table's
			// cells) and the outer width so we can calculate the full width of the
			// table. This is safe since DataTables requires a unique cell for each
			// column, but if ever a header can span multiple columns, this will
			// need to be modified.
			var total = 0;
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				var cell = $(headerCells[i]);
				var border = cell.outerWidth() - cell.width();
	
				// Use getBounding... where possible (not IE8-) because it can give
				// sub-pixel accuracy, which we then want to round up!
				var bounding = browser.bBounding ?
					Math.ceil( headerCells[i].getBoundingClientRect().width ) :
					cell.outerWidth();
	
				// Total is tracked to remove any sub-pixel errors as the outerWidth
				// of the table might not equal the total given here (IE!).
				total += bounding;
	
				// Width for each column to use
				columns[ visibleColumns[i] ].sWidth = _fnStringToCss( bounding - border );
			}
	
			table.style.width = _fnStringToCss( total );
	
			// Finished with the table - ditch it
			holder.remove();
		}
	
		// If there is a width attr, we want to attach an event listener which
		// allows the table sizing to automatically adjust when the window is
		// resized. Use the width attr rather than CSS, since we can't know if the
		// CSS is a relative value or absolute - DOM read is always px.
		if ( tableWidthAttr ) {
			table.style.width = _fnStringToCss( tableWidthAttr );
		}
	
		if ( (tableWidthAttr || scrollX) && ! oSettings._reszEvt ) {
			var bindResize = function () {
				$(window).on('resize.DT-'+oSettings.sInstance, _fnThrottle( function () {
					_fnAdjustColumnSizing( oSettings );
				} ) );
			};
	
			// IE6/7 will crash if we bind a resize event handler on page load.
			// To be removed in 1.11 which drops IE6/7 support
			if ( ie67 ) {
				setTimeout( bindResize, 1000 );
			}
			else {
				bindResize();
			}
	
			oSettings._reszEvt = true;
		}
	}
	
	
	/**
	 * Throttle the calls to a function. Arguments and context are maintained for
	 * the throttled function
	 *  @param {function} fn Function to be called
	 *  @param {int} [freq=200] call frequency in mS
	 *  @returns {function} wrapped function
	 *  @memberof DataTable#oApi
	 */
	var _fnThrottle = DataTable.util.throttle;
	
	
	/**
	 * Convert a CSS unit width to pixels (e.g. 2em)
	 *  @param {string} width width to be converted
	 *  @param {node} parent parent to get the with for (required for relative widths) - optional
	 *  @returns {int} width in pixels
	 *  @memberof DataTable#oApi
	 */
	function _fnConvertToWidth ( width, parent )
	{
		if ( ! width ) {
			return 0;
		}
	
		var n = $('<div/>')
			.css( 'width', _fnStringToCss( width ) )
			.appendTo( parent || document.body );
	
		var val = n[0].offsetWidth;
		n.remove();
	
		return val;
	}
	
	
	/**
	 * Get the widest node
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {node} widest table node
	 *  @memberof DataTable#oApi
	 */
	function _fnGetWidestNode( settings, colIdx )
	{
		var idx = _fnGetMaxLenString( settings, colIdx );
		if ( idx < 0 ) {
			return null;
		}
	
		var data = settings.aoData[ idx ];
		return ! data.nTr ? // Might not have been created when deferred rendering
			$('<td/>').html( _fnGetCellData( settings, idx, colIdx, 'display' ) )[0] :
			data.anCells[ colIdx ];
	}
	
	
	/**
	 * Get the maximum strlen for each data column
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {string} max string length for each column
	 *  @memberof DataTable#oApi
	 */
	function _fnGetMaxLenString( settings, colIdx )
	{
		var s, max=-1, maxIdx = -1;
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			s = _fnGetCellData( settings, i, colIdx, 'display' )+'';
			s = s.replace( __re_html_remove, '' );
			s = s.replace( /&nbsp;/g, ' ' );
	
			if ( s.length > max ) {
				max = s.length;
				maxIdx = i;
			}
		}
	
		return maxIdx;
	}
	
	
	/**
	 * Append a CSS unit (only if required) to a string
	 *  @param {string} value to css-ify
	 *  @returns {string} value with css unit
	 *  @memberof DataTable#oApi
	 */
	function _fnStringToCss( s )
	{
		if ( s === null ) {
			return '0px';
		}
	
		if ( typeof s == 'number' ) {
			return s < 0 ?
				'0px' :
				s+'px';
		}
	
		// Check it has a unit character already
		return s.match(/\d$/) ?
			s+'px' :
			s;
	}
	
	
	
	function _fnSortFlatten ( settings )
	{
		var
			i, iLen, k, kLen,
			aSort = [],
			aiOrig = [],
			aoColumns = settings.aoColumns,
			aDataSort, iCol, sType, srcCol,
			fixed = settings.aaSortingFixed,
			fixedObj = $.isPlainObject( fixed ),
			nestedSort = [],
			add = function ( a ) {
				if ( a.length && ! Array.isArray( a[0] ) ) {
					// 1D array
					nestedSort.push( a );
				}
				else {
					// 2D array
					$.merge( nestedSort, a );
				}
			};
	
		// Build the sort array, with pre-fix and post-fix options if they have been
		// specified
		if ( Array.isArray( fixed ) ) {
			add( fixed );
		}
	
		if ( fixedObj && fixed.pre ) {
			add( fixed.pre );
		}
	
		add( settings.aaSorting );
	
		if (fixedObj && fixed.post ) {
			add( fixed.post );
		}
	
		for ( i=0 ; i<nestedSort.length ; i++ )
		{
			srcCol = nestedSort[i][0];
			aDataSort = aoColumns[ srcCol ].aDataSort;
	
			for ( k=0, kLen=aDataSort.length ; k<kLen ; k++ )
			{
				iCol = aDataSort[k];
				sType = aoColumns[ iCol ].sType || 'string';
	
				if ( nestedSort[i]._idx === undefined ) {
					nestedSort[i]._idx = $.inArray( nestedSort[i][1], aoColumns[iCol].asSorting );
				}
	
				aSort.push( {
					src:       srcCol,
					col:       iCol,
					dir:       nestedSort[i][1],
					index:     nestedSort[i]._idx,
					type:      sType,
					formatter: DataTable.ext.type.order[ sType+"-pre" ]
				} );
			}
		}
	
		return aSort;
	}
	
	/**
	 * Change the order of the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 *  @todo This really needs split up!
	 */
	function _fnSort ( oSettings )
	{
		var
			i, ien, iLen, j, jLen, k, kLen,
			sDataType, nTh,
			aiOrig = [],
			oExtSort = DataTable.ext.type.order,
			aoData = oSettings.aoData,
			aoColumns = oSettings.aoColumns,
			aDataSort, data, iCol, sType, oSort,
			formatters = 0,
			sortCol,
			displayMaster = oSettings.aiDisplayMaster,
			aSort;
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo Can this be moved into a 'data-ready' handler which is called when
		//   data is going to be used in the table?
		_fnColumnTypes( oSettings );
	
		aSort = _fnSortFlatten( oSettings );
	
		for ( i=0, ien=aSort.length ; i<ien ; i++ ) {
			sortCol = aSort[i];
	
			// Track if we can use the fast sort algorithm
			if ( sortCol.formatter ) {
				formatters++;
			}
	
			// Load the data needed for the sort, for each cell
			_fnSortData( oSettings, sortCol.col );
		}
	
		/* No sorting required if server-side or no sorting array */
		if ( _fnDataSource( oSettings ) != 'ssp' && aSort.length !== 0 )
		{
			// Create a value - key array of the current row positions such that we can use their
			// current position during the sort, if values match, in order to perform stable sorting
			for ( i=0, iLen=displayMaster.length ; i<iLen ; i++ ) {
				aiOrig[ displayMaster[i] ] = i;
			}
	
			/* Do the sort - here we want multi-column sorting based on a given data source (column)
			 * and sorting function (from oSort) in a certain direction. It's reasonably complex to
			 * follow on it's own, but this is what we want (example two column sorting):
			 *  fnLocalSorting = function(a,b){
			 *    var iTest;
			 *    iTest = oSort['string-asc']('data11', 'data12');
			 *      if (iTest !== 0)
			 *        return iTest;
			 *    iTest = oSort['numeric-desc']('data21', 'data22');
			 *    if (iTest !== 0)
			 *      return iTest;
			 *    return oSort['numeric-asc']( aiOrig[a], aiOrig[b] );
			 *  }
			 * Basically we have a test for each sorting column, if the data in that column is equal,
			 * test the next column. If all columns match, then we use a numeric sort on the row
			 * positions in the original data array to provide a stable sort.
			 *
			 * Note - I know it seems excessive to have two sorting methods, but the first is around
			 * 15% faster, so the second is only maintained for backwards compatibility with sorting
			 * methods which do not have a pre-sort formatting function.
			 */
			if ( formatters === aSort.length ) {
				// All sort types have formatting functions
				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, test, sort,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						test = x<y ? -1 : x>y ? 1 : 0;
						if ( test !== 0 ) {
							return sort.dir === 'asc' ? test : -test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
			else {
				// Depreciated - remove in 1.11 (providing a plug-in option)
				// Not all sort types have formatting methods, so we have to call their sorting
				// methods.
				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, l, test, sort, fn,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						fn = oExtSort[ sort.type+"-"+sort.dir ] || oExtSort[ "string-"+sort.dir ];
						test = fn( x, y );
						if ( test !== 0 ) {
							return test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
		}
	
		/* Tell the draw function that we have sorted the data */
		oSettings.bSorted = true;
	}
	
	
	function _fnSortAria ( settings )
	{
		var label;
		var nextSort;
		var columns = settings.aoColumns;
		var aSort = _fnSortFlatten( settings );
		var oAria = settings.oLanguage.oAria;
	
		// ARIA attributes - need to loop all columns, to update all (removing old
		// attributes as needed)
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			var col = columns[i];
			var asSorting = col.asSorting;
			var sTitle = col.ariaTitle || col.sTitle.replace( /<.*?>/g, "" );
			var th = col.nTh;
	
			// IE7 is throwing an error when setting these properties with jQuery's
			// attr() and removeAttr() methods...
			th.removeAttribute('aria-sort');
	
			/* In ARIA only the first sorting column can be marked as sorting - no multi-sort option */
			if ( col.bSortable ) {
				if ( aSort.length > 0 && aSort[0].col == i ) {
					th.setAttribute('aria-sort', aSort[0].dir=="asc" ? "ascending" : "descending" );
					nextSort = asSorting[ aSort[0].index+1 ] || asSorting[0];
				}
				else {
					nextSort = asSorting[0];
				}
	
				label = sTitle + ( nextSort === "asc" ?
					oAria.sSortAscending :
					oAria.sSortDescending
				);
			}
			else {
				label = sTitle;
			}
	
			th.setAttribute('aria-label', label);
		}
	}
	
	
	/**
	 * Function to run on user sort request
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {boolean} [append=false] Append the requested sort to the existing
	 *    sort if true (i.e. multi-column sort)
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortListener ( settings, colIdx, append, callback )
	{
		var col = settings.aoColumns[ colIdx ];
		var sorting = settings.aaSorting;
		var asSorting = col.asSorting;
		var nextSortIdx;
		var next = function ( a, overflow ) {
			var idx = a._idx;
			if ( idx === undefined ) {
				idx = $.inArray( a[1], asSorting );
			}
	
			return idx+1 < asSorting.length ?
				idx+1 :
				overflow ?
					null :
					0;
		};
	
		// Convert to 2D array if needed
		if ( typeof sorting[0] === 'number' ) {
			sorting = settings.aaSorting = [ sorting ];
		}
	
		// If appending the sort then we are multi-column sorting
		if ( append && settings.oFeatures.bSortMulti ) {
			// Are we already doing some kind of sort on this column?
			var sortIdx = $.inArray( colIdx, _pluck(sorting, '0') );
	
			if ( sortIdx !== -1 ) {
				// Yes, modify the sort
				nextSortIdx = next( sorting[sortIdx], true );
	
				if ( nextSortIdx === null && sorting.length === 1 ) {
					nextSortIdx = 0; // can't remove sorting completely
				}
	
				if ( nextSortIdx === null ) {
					sorting.splice( sortIdx, 1 );
				}
				else {
					sorting[sortIdx][1] = asSorting[ nextSortIdx ];
					sorting[sortIdx]._idx = nextSortIdx;
				}
			}
			else {
				// No sort on this column yet
				sorting.push( [ colIdx, asSorting[0], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
		}
		else if ( sorting.length && sorting[0][0] == colIdx ) {
			// Single column - already sorting on this column, modify the sort
			nextSortIdx = next( sorting[0] );
	
			sorting.length = 1;
			sorting[0][1] = asSorting[ nextSortIdx ];
			sorting[0]._idx = nextSortIdx;
		}
		else {
			// Single column - sort only on this column
			sorting.length = 0;
			sorting.push( [ colIdx, asSorting[0] ] );
			sorting[0]._idx = 0;
		}
	
		// Run the sort by calling a full redraw
		_fnReDraw( settings );
	
		// callback used for async user interaction
		if ( typeof callback == 'function' ) {
			callback( settings );
		}
	}
	
	
	/**
	 * Attach a sort handler (click) to a node
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortAttachListener ( settings, attachTo, colIdx, callback )
	{
		var col = settings.aoColumns[ colIdx ];
	
		_fnBindAction( attachTo, {}, function (e) {
			/* If the column is not sortable - don't to anything */
			if ( col.bSortable === false ) {
				return;
			}
	
			// If processing is enabled use a timeout to allow the processing
			// display to be shown - otherwise to it synchronously
			if ( settings.oFeatures.bProcessing ) {
				_fnProcessingDisplay( settings, true );
	
				setTimeout( function() {
					_fnSortListener( settings, colIdx, e.shiftKey, callback );
	
					// In server-side processing, the draw callback will remove the
					// processing display
					if ( _fnDataSource( settings ) !== 'ssp' ) {
						_fnProcessingDisplay( settings, false );
					}
				}, 0 );
			}
			else {
				_fnSortListener( settings, colIdx, e.shiftKey, callback );
			}
		} );
	}
	
	
	/**
	 * Set the sorting classes on table's body, Note: it is safe to call this function
	 * when bSort and bSortClasses are false
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSortingClasses( settings )
	{
		var oldSort = settings.aLastSort;
		var sortClass = settings.oClasses.sSortColumn;
		var sort = _fnSortFlatten( settings );
		var features = settings.oFeatures;
		var i, ien, colIdx;
	
		if ( features.bSort && features.bSortClasses ) {
			// Remove old sorting classes
			for ( i=0, ien=oldSort.length ; i<ien ; i++ ) {
				colIdx = oldSort[i].src;
	
				// Remove column sorting
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.removeClass( sortClass + (i<2 ? i+1 : 3) );
			}
	
			// Add new column sorting
			for ( i=0, ien=sort.length ; i<ien ; i++ ) {
				colIdx = sort[i].src;
	
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.addClass( sortClass + (i<2 ? i+1 : 3) );
			}
		}
	
		settings.aLastSort = sort;
	}
	
	
	// Get the data to sort a column, be it from cache, fresh (populating the
	// cache), or from a sort formatter
	function _fnSortData( settings, idx )
	{
		// Custom sorting function - provided by the sort data type
		var column = settings.aoColumns[ idx ];
		var customSort = DataTable.ext.order[ column.sSortDataType ];
		var customData;
	
		if ( customSort ) {
			customData = customSort.call( settings.oInstance, settings, idx,
				_fnColumnIndexToVisible( settings, idx )
			);
		}
	
		// Use / populate cache
		var row, cellData;
		var formatter = DataTable.ext.type.order[ column.sType+"-pre" ];
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aSortData ) {
				row._aSortData = [];
			}
	
			if ( ! row._aSortData[idx] || customSort ) {
				cellData = customSort ?
					customData[i] : // If there was a custom sort function, use data from there
					_fnGetCellData( settings, i, idx, 'sort' );
	
				row._aSortData[ idx ] = formatter ?
					formatter( cellData ) :
					cellData;
			}
		}
	}
	
	
	
	/**
	 * Save the state of a table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSaveState ( settings )
	{
		if (settings._bLoadingState) {
			return;
		}
	
		/* Store the interesting variables */
		var state = {
			time:    +new Date(),
			start:   settings._iDisplayStart,
			length:  settings._iDisplayLength,
			order:   $.extend( true, [], settings.aaSorting ),
			search:  _fnSearchToCamel( settings.oPreviousSearch ),
			columns: $.map( settings.aoColumns, function ( col, i ) {
				return {
					visible: col.bVisible,
					search: _fnSearchToCamel( settings.aoPreSearchCols[i] )
				};
			} )
		};
	
		settings.oSavedState = state;
		_fnCallbackFire( settings, "aoStateSaveParams", 'stateSaveParams', [settings, state] );
		
		if ( settings.oFeatures.bStateSave && !settings.bDestroying )
		{
			settings.fnStateSaveCallback.call( settings.oInstance, settings, state );
		}	
	}
	
	
	/**
	 * Attempt to load a saved table state
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oInit DataTables init object so we can override settings
	 *  @param {function} callback Callback to execute when the state has been loaded
	 *  @memberof DataTable#oApi
	 */
	function _fnLoadState ( settings, oInit, callback )
	{
		if ( ! settings.oFeatures.bStateSave ) {
			callback();
			return;
		}
	
		var loaded = function(state) {
			_fnImplementState(settings, state, callback);
		}
	
		var state = settings.fnStateLoadCallback.call( settings.oInstance, settings, loaded );
	
		if ( state !== undefined ) {
			_fnImplementState( settings, state, callback );
		}
		// otherwise, wait for the loaded callback to be executed
	
		return true;
	}
	
	function _fnImplementState ( settings, s, callback) {
		var i, ien;
		var columns = settings.aoColumns;
		settings._bLoadingState = true;
	
		// When StateRestore was introduced the state could now be implemented at any time
		// Not just initialisation. To do this an api instance is required in some places
		var api = settings._bInitComplete ? new DataTable.Api(settings) : null;
	
		if ( ! s || ! s.time ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Allow custom and plug-in manipulation functions to alter the saved data set and
		// cancelling of loading by returning false
		var abStateLoad = _fnCallbackFire( settings, 'aoStateLoadParams', 'stateLoadParams', [settings, s] );
		if ( $.inArray( false, abStateLoad ) !== -1 ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Reject old data
		var duration = settings.iStateDuration;
		if ( duration > 0 && s.time < +new Date() - (duration*1000) ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Number of columns have changed - all bets are off, no restore of settings
		if ( s.columns && columns.length !== s.columns.length ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Store the saved state so it might be accessed at any time
		settings.oLoadedState = $.extend( true, {}, s );
	
		// Page Length
		if ( s.length !== undefined ) {
			// If already initialised just set the value directly so that the select element is also updated
			if (api) {
				api.page.len(s.length)
			}
			else {
				settings._iDisplayLength   = s.length;
			}
		}
	
		// Restore key features - todo - for 1.11 this needs to be done by
		// subscribed events
		if ( s.start !== undefined ) {
			if(api === null) {
				settings._iDisplayStart    = s.start;
				settings.iInitDisplayStart = s.start;
			}
			else {
				_fnPageChange(settings, s.start/settings._iDisplayLength);
			}
		}
	
		// Order
		if ( s.order !== undefined ) {
			settings.aaSorting = [];
			$.each( s.order, function ( i, col ) {
				settings.aaSorting.push( col[0] >= columns.length ?
					[ 0, col[1] ] :
					col
				);
			} );
		}
	
		// Search
		if ( s.search !== undefined ) {
			$.extend( settings.oPreviousSearch, _fnSearchToHung( s.search ) );
		}
	
		// Columns
		if ( s.columns ) {
			for ( i=0, ien=s.columns.length ; i<ien ; i++ ) {
				var col = s.columns[i];
	
				// Visibility
				if ( col.visible !== undefined ) {
					// If the api is defined, the table has been initialised so we need to use it rather than internal settings
					if (api) {
						// Don't redraw the columns on every iteration of this loop, we will do this at the end instead
						api.column(i).visible(col.visible, false);
					}
					else {
						columns[i].bVisible = col.visible;
					}
				}
	
				// Search
				if ( col.search !== undefined ) {
					$.extend( settings.aoPreSearchCols[i], _fnSearchToHung( col.search ) );
				}
			}
			
			// If the api is defined then we need to adjust the columns once the visibility has been changed
			if (api) {
				api.columns.adjust();
			}
		}
	
		settings._bLoadingState = false;
		_fnCallbackFire( settings, 'aoStateLoaded', 'stateLoaded', [settings, s] );
		callback();
	};
	
	
	/**
	 * Return the settings object for a particular table
	 *  @param {node} table table we are using as a dataTable
	 *  @returns {object} Settings object - or null if not found
	 *  @memberof DataTable#oApi
	 */
	function _fnSettingsFromNode ( table )
	{
		var settings = DataTable.settings;
		var idx = $.inArray( table, _pluck( settings, 'nTable' ) );
	
		return idx !== -1 ?
			settings[ idx ] :
			null;
	}
	
	
	/**
	 * Log an error message
	 *  @param {object} settings dataTables settings object
	 *  @param {int} level log error messages, or display them to the user
	 *  @param {string} msg error message
	 *  @param {int} tn Technical note id to get more information about the error.
	 *  @memberof DataTable#oApi
	 */
	function _fnLog( settings, level, msg, tn )
	{
		msg = 'DataTables warning: '+
			(settings ? 'table id='+settings.sTableId+' - ' : '')+msg;
	
		if ( tn ) {
			msg += '. For more information about this error, please see '+
			'http://datatables.net/tn/'+tn;
		}
	
		if ( ! level  ) {
			// Backwards compatibility pre 1.10
			var ext = DataTable.ext;
			var type = ext.sErrMode || ext.errMode;
	
			if ( settings ) {
				_fnCallbackFire( settings, null, 'error', [ settings, tn, msg ] );
			}
	
			if ( type == 'alert' ) {
				alert( msg );
			}
			else if ( type == 'throw' ) {
				throw new Error(msg);
			}
			else if ( typeof type == 'function' ) {
				type( settings, tn, msg );
			}
		}
		else if ( window.console && console.log ) {
			console.log( msg );
		}
	}
	
	
	/**
	 * See if a property is defined on one object, if so assign it to the other object
	 *  @param {object} ret target object
	 *  @param {object} src source object
	 *  @param {string} name property
	 *  @param {string} [mappedName] name to map too - optional, name used if not given
	 *  @memberof DataTable#oApi
	 */
	function _fnMap( ret, src, name, mappedName )
	{
		if ( Array.isArray( name ) ) {
			$.each( name, function (i, val) {
				if ( Array.isArray( val ) ) {
					_fnMap( ret, src, val[0], val[1] );
				}
				else {
					_fnMap( ret, src, val );
				}
			} );
	
			return;
		}
	
		if ( mappedName === undefined ) {
			mappedName = name;
		}
	
		if ( src[name] !== undefined ) {
			ret[mappedName] = src[name];
		}
	}
	
	
	/**
	 * Extend objects - very similar to jQuery.extend, but deep copy objects, and
	 * shallow copy arrays. The reason we need to do this, is that we don't want to
	 * deep copy array init values (such as aaSorting) since the dev wouldn't be
	 * able to override them, but we do want to deep copy arrays.
	 *  @param {object} out Object to extend
	 *  @param {object} extender Object from which the properties will be applied to
	 *      out
	 *  @param {boolean} breakRefs If true, then arrays will be sliced to take an
	 *      independent copy with the exception of the `data` or `aaData` parameters
	 *      if they are present. This is so you can pass in a collection to
	 *      DataTables and have that used as your data source without breaking the
	 *      references
	 *  @returns {object} out Reference, just for convenience - out === the return.
	 *  @memberof DataTable#oApi
	 *  @todo This doesn't take account of arrays inside the deep copied objects.
	 */
	function _fnExtend( out, extender, breakRefs )
	{
		var val;
	
		for ( var prop in extender ) {
			if ( extender.hasOwnProperty(prop) ) {
				val = extender[prop];
	
				if ( $.isPlainObject( val ) ) {
					if ( ! $.isPlainObject( out[prop] ) ) {
						out[prop] = {};
					}
					$.extend( true, out[prop], val );
				}
				else if ( breakRefs && prop !== 'data' && prop !== 'aaData' && Array.isArray(val) ) {
					out[prop] = val.slice();
				}
				else {
					out[prop] = val;
				}
			}
		}
	
		return out;
	}
	
	
	/**
	 * Bind an event handers to allow a click or return key to activate the callback.
	 * This is good for accessibility since a return on the keyboard will have the
	 * same effect as a click, if the element has focus.
	 *  @param {element} n Element to bind the action to
	 *  @param {object} oData Data object to pass to the triggered function
	 *  @param {function} fn Callback function for when the event is triggered
	 *  @memberof DataTable#oApi
	 */
	function _fnBindAction( n, oData, fn )
	{
		$(n)
			.on( 'click.DT', oData, function (e) {
					$(n).trigger('blur'); // Remove focus outline for mouse users
					fn(e);
				} )
			.on( 'keypress.DT', oData, function (e){
					if ( e.which === 13 ) {
						e.preventDefault();
						fn(e);
					}
				} )
			.on( 'selectstart.DT', function () {
					/* Take the brutal approach to cancelling text selection */
					return false;
				} );
	}
	
	
	/**
	 * Register a callback function. Easily allows a callback function to be added to
	 * an array store of callback functions that can then all be called together.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sStore Name of the array storage for the callbacks in oSettings
	 *  @param {function} fn Function to be called back
	 *  @param {string} sName Identifying name for the callback (i.e. a label)
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackReg( oSettings, sStore, fn, sName )
	{
		if ( fn )
		{
			oSettings[sStore].push( {
				"fn": fn,
				"sName": sName
			} );
		}
	}
	
	
	/**
	 * Fire callback functions and trigger events. Note that the loop over the
	 * callback array store is done backwards! Further note that you do not want to
	 * fire off triggers in time sensitive applications (for example cell creation)
	 * as its slow.
	 *  @param {object} settings dataTables settings object
	 *  @param {string} callbackArr Name of the array storage for the callbacks in
	 *      oSettings
	 *  @param {string} eventName Name of the jQuery custom event to trigger. If
	 *      null no trigger is fired
	 *  @param {array} args Array of arguments to pass to the callback function /
	 *      trigger
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackFire( settings, callbackArr, eventName, args )
	{
		var ret = [];
	
		if ( callbackArr ) {
			ret = $.map( settings[callbackArr].slice().reverse(), function (val, i) {
				return val.fn.apply( settings.oInstance, args );
			} );
		}
	
		if ( eventName !== null ) {
			var e = $.Event( eventName+'.dt' );
	
			$(settings.nTable).trigger( e, args );
	
			ret.push( e.result );
		}
	
		return ret;
	}
	
	
	function _fnLengthOverflow ( settings )
	{
		var
			start = settings._iDisplayStart,
			end = settings.fnDisplayEnd(),
			len = settings._iDisplayLength;
	
		/* If we have space to show extra rows (backing up from the end point - then do so */
		if ( start >= end )
		{
			start = end - len;
		}
	
		// Keep the start record on the current page
		start -= (start % len);
	
		if ( len === -1 || start < 0 )
		{
			start = 0;
		}
	
		settings._iDisplayStart = start;
	}
	
	
	function _fnRenderer( settings, type )
	{
		var renderer = settings.renderer;
		var host = DataTable.ext.renderer[type];
	
		if ( $.isPlainObject( renderer ) && renderer[type] ) {
			// Specific renderer for this type. If available use it, otherwise use
			// the default.
			return host[renderer[type]] || host._;
		}
		else if ( typeof renderer === 'string' ) {
			// Common renderer - if there is one available for this type use it,
			// otherwise use the default
			return host[renderer] || host._;
		}
	
		// Use the default
		return host._;
	}
	
	
	/**
	 * Detect the data source being used for the table. Used to simplify the code
	 * a little (ajax) and to make it compress a little smaller.
	 *
	 *  @param {object} settings dataTables settings object
	 *  @returns {string} Data source
	 *  @memberof DataTable#oApi
	 */
	function _fnDataSource ( settings )
	{
		if ( settings.oFeatures.bServerSide ) {
			return 'ssp';
		}
		else if ( settings.ajax || settings.sAjaxSource ) {
			return 'ajax';
		}
		return 'dom';
	}
	
	
	
	
	/**
	 * Computed structure of the DataTables API, defined by the options passed to
	 * `DataTable.Api.register()` when building the API.
	 *
	 * The structure is built in order to speed creation and extension of the Api
	 * objects since the extensions are effectively pre-parsed.
	 *
	 * The array is an array of objects with the following structure, where this
	 * base array represents the Api prototype base:
	 *
	 *     [
	 *       {
	 *         name:      'data'                -- string   - Property name
	 *         val:       function () {},       -- function - Api method (or undefined if just an object
	 *         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	 *         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	 *       },
	 *       {
	 *         name:     'row'
	 *         val:       {},
	 *         methodExt: [ ... ],
	 *         propExt:   [
	 *           {
	 *             name:      'data'
	 *             val:       function () {},
	 *             methodExt: [ ... ],
	 *             propExt:   [ ... ]
	 *           },
	 *           ...
	 *         ]
	 *       }
	 *     ]
	 *
	 * @type {Array}
	 * @ignore
	 */
	var __apiStruct = [];
	
	
	/**
	 * `Array.prototype` reference.
	 *
	 * @type object
	 * @ignore
	 */
	var __arrayProto = Array.prototype;
	
	
	/**
	 * Abstraction for `context` parameter of the `Api` constructor to allow it to
	 * take several different forms for ease of use.
	 *
	 * Each of the input parameter types will be converted to a DataTables settings
	 * object where possible.
	 *
	 * @param  {string|node|jQuery|object} mixed DataTable identifier. Can be one
	 *   of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 *   * `DataTables.Api` - API instance
	 * @return {array|null} Matching DataTables settings objects. `null` or
	 *   `undefined` is returned if no matching DataTable is found.
	 * @ignore
	 */
	var _toSettings = function ( mixed )
	{
		var idx, jq;
		var settings = DataTable.settings;
		var tables = $.map( settings, function (el, i) {
			return el.nTable;
		} );
	
		if ( ! mixed ) {
			return [];
		}
		else if ( mixed.nTable && mixed.oApi ) {
			// DataTables settings object
			return [ mixed ];
		}
		else if ( mixed.nodeName && mixed.nodeName.toLowerCase() === 'table' ) {
			// Table node
			idx = $.inArray( mixed, tables );
			return idx !== -1 ? [ settings[idx] ] : null;
		}
		else if ( mixed && typeof mixed.settings === 'function' ) {
			return mixed.settings().toArray();
		}
		else if ( typeof mixed === 'string' ) {
			// jQuery selector
			jq = $(mixed);
		}
		else if ( mixed instanceof $ ) {
			// jQuery object (also DataTables instance)
			jq = mixed;
		}
	
		if ( jq ) {
			return jq.map( function(i) {
				idx = $.inArray( this, tables );
				return idx !== -1 ? settings[idx] : null;
			} ).toArray();
		}
	};
	
	
	/**
	 * DataTables API class - used to control and interface with  one or more
	 * DataTables enhanced tables.
	 *
	 * The API class is heavily based on jQuery, presenting a chainable interface
	 * that you can use to interact with tables. Each instance of the API class has
	 * a "context" - i.e. the tables that it will operate on. This could be a single
	 * table, all tables on a page or a sub-set thereof.
	 *
	 * Additionally the API is designed to allow you to easily work with the data in
	 * the tables, retrieving and manipulating it as required. This is done by
	 * presenting the API class as an array like interface. The contents of the
	 * array depend upon the actions requested by each method (for example
	 * `rows().nodes()` will return an array of nodes, while `rows().data()` will
	 * return an array of objects or arrays depending upon your table's
	 * configuration). The API object has a number of array like methods (`push`,
	 * `pop`, `reverse` etc) as well as additional helper methods (`each`, `pluck`,
	 * `unique` etc) to assist your working with the data held in a table.
	 *
	 * Most methods (those which return an Api instance) are chainable, which means
	 * the return from a method call also has all of the methods available that the
	 * top level object had. For example, these two calls are equivalent:
	 *
	 *     // Not chained
	 *     api.row.add( {...} );
	 *     api.draw();
	 *
	 *     // Chained
	 *     api.row.add( {...} ).draw();
	 *
	 * @class DataTable.Api
	 * @param {array|object|string|jQuery} context DataTable identifier. This is
	 *   used to define which DataTables enhanced tables this API will operate on.
	 *   Can be one of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 * @param {array} [data] Data to initialise the Api instance with.
	 *
	 * @example
	 *   // Direct initialisation during DataTables construction
	 *   var api = $('#example').DataTable();
	 *
	 * @example
	 *   // Initialisation using a DataTables jQuery object
	 *   var api = $('#example').dataTable().api();
	 *
	 * @example
	 *   // Initialisation as a constructor
	 *   var api = new $.fn.DataTable.Api( 'table.dataTable' );
	 */
	_Api = function ( context, data )
	{
		if ( ! (this instanceof _Api) ) {
			return new _Api( context, data );
		}
	
		var settings = [];
		var ctxSettings = function ( o ) {
			var a = _toSettings( o );
			if ( a ) {
				settings.push.apply( settings, a );
			}
		};
	
		if ( Array.isArray( context ) ) {
			for ( var i=0, ien=context.length ; i<ien ; i++ ) {
				ctxSettings( context[i] );
			}
		}
		else {
			ctxSettings( context );
		}
	
		// Remove duplicates
		this.context = _unique( settings );
	
		// Initial data
		if ( data ) {
			$.merge( this, data );
		}
	
		// selector
		this.selector = {
			rows: null,
			cols: null,
			opts: null
		};
	
		_Api.extend( this, this, __apiStruct );
	};
	
	DataTable.Api = _Api;
	
	// Don't destroy the existing prototype, just extend it. Required for jQuery 2's
	// isPlainObject.
	$.extend( _Api.prototype, {
		any: function ()
		{
			return this.count() !== 0;
		},
	
	
		concat:  __arrayProto.concat,
	
	
		context: [], // array of table settings objects
	
	
		count: function ()
		{
			return this.flatten().length;
		},
	
	
		each: function ( fn )
		{
			for ( var i=0, ien=this.length ; i<ien; i++ ) {
				fn.call( this, this[i], i, this );
			}
	
			return this;
		},
	
	
		eq: function ( idx )
		{
			var ctx = this.context;
	
			return ctx.length > idx ?
				new _Api( ctx[idx], this[idx] ) :
				null;
		},
	
	
		filter: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.filter ) {
				a = __arrayProto.filter.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					if ( fn.call( this, this[i], i, this ) ) {
						a.push( this[i] );
					}
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		flatten: function ()
		{
			var a = [];
			return new _Api( this.context, a.concat.apply( a, this.toArray() ) );
		},
	
	
		join:    __arrayProto.join,
	
	
		indexOf: __arrayProto.indexOf || function (obj, start)
		{
			for ( var i=(start || 0), ien=this.length ; i<ien ; i++ ) {
				if ( this[i] === obj ) {
					return i;
				}
			}
			return -1;
		},
	
		iterator: function ( flatten, type, fn, alwaysNew ) {
			var
				a = [], ret,
				i, ien, j, jen,
				context = this.context,
				rows, items, item,
				selector = this.selector;
	
			// Argument shifting
			if ( typeof flatten === 'string' ) {
				alwaysNew = fn;
				fn = type;
				type = flatten;
				flatten = false;
			}
	
			for ( i=0, ien=context.length ; i<ien ; i++ ) {
				var apiInst = new _Api( context[i] );
	
				if ( type === 'table' ) {
					ret = fn.call( apiInst, context[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'columns' || type === 'rows' ) {
					// this has same length as context - one entry for each table
					ret = fn.call( apiInst, context[i], this[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'column' || type === 'column-rows' || type === 'row' || type === 'cell' ) {
					// columns and rows share the same structure.
					// 'this' is an array of column indexes for each context
					items = this[i];
	
					if ( type === 'column-rows' ) {
						rows = _selector_row_indexes( context[i], selector.opts );
					}
	
					for ( j=0, jen=items.length ; j<jen ; j++ ) {
						item = items[j];
	
						if ( type === 'cell' ) {
							ret = fn.call( apiInst, context[i], item.row, item.column, i, j );
						}
						else {
							ret = fn.call( apiInst, context[i], item, i, j, rows );
						}
	
						if ( ret !== undefined ) {
							a.push( ret );
						}
					}
				}
			}
	
			if ( a.length || alwaysNew ) {
				var api = new _Api( context, flatten ? a.concat.apply( [], a ) : a );
				var apiSelector = api.selector;
				apiSelector.rows = selector.rows;
				apiSelector.cols = selector.cols;
				apiSelector.opts = selector.opts;
				return api;
			}
			return this;
		},
	
	
		lastIndexOf: __arrayProto.lastIndexOf || function (obj, start)
		{
			// Bit cheeky...
			return this.indexOf.apply( this.toArray.reverse(), arguments );
		},
	
	
		length:  0,
	
	
		map: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.map ) {
				a = __arrayProto.map.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					a.push( fn.call( this, this[i], i ) );
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		pluck: function ( prop )
		{
			let fn = DataTable.util.get(prop);
	
			return this.map( function ( el ) {
				return fn(el);
			} );
		},
	
		pop:     __arrayProto.pop,
	
	
		push:    __arrayProto.push,
	
	
		// Does not return an API instance
		reduce: __arrayProto.reduce || function ( fn, init )
		{
			return _fnReduce( this, fn, init, 0, this.length, 1 );
		},
	
	
		reduceRight: __arrayProto.reduceRight || function ( fn, init )
		{
			return _fnReduce( this, fn, init, this.length-1, -1, -1 );
		},
	
	
		reverse: __arrayProto.reverse,
	
	
		// Object with rows, columns and opts
		selector: null,
	
	
		shift:   __arrayProto.shift,
	
	
		slice: function () {
			return new _Api( this.context, this );
		},
	
	
		sort:    __arrayProto.sort, // ? name - order?
	
	
		splice:  __arrayProto.splice,
	
	
		toArray: function ()
		{
			return __arrayProto.slice.call( this );
		},
	
	
		to$: function ()
		{
			return $( this );
		},
	
	
		toJQuery: function ()
		{
			return $( this );
		},
	
	
		unique: function ()
		{
			return new _Api( this.context, _unique(this) );
		},
	
	
		unshift: __arrayProto.unshift
	} );
	
	
	_Api.extend = function ( scope, obj, ext )
	{
		// Only extend API instances and static properties of the API
		if ( ! ext.length || ! obj || ( ! (obj instanceof _Api) && ! obj.__dt_wrapper ) ) {
			return;
		}
	
		var
			i, ien,
			struct,
			methodScoping = function ( scope, fn, struc ) {
				return function () {
					var ret = fn.apply( scope, arguments );
	
					// Method extension
					_Api.extend( ret, ret, struc.methodExt );
					return ret;
				};
			};
	
		for ( i=0, ien=ext.length ; i<ien ; i++ ) {
			struct = ext[i];
	
			// Value
			obj[ struct.name ] = struct.type === 'function' ?
				methodScoping( scope, struct.val, struct ) :
				struct.type === 'object' ?
					{} :
					struct.val;
	
			obj[ struct.name ].__dt_wrapper = true;
	
			// Property extension
			_Api.extend( scope, obj[ struct.name ], struct.propExt );
		}
	};
	
	
	// @todo - Is there need for an augment function?
	// _Api.augment = function ( inst, name )
	// {
	// 	// Find src object in the structure from the name
	// 	var parts = name.split('.');
	
	// 	_Api.extend( inst, obj );
	// };
	
	
	//     [
	//       {
	//         name:      'data'                -- string   - Property name
	//         val:       function () {},       -- function - Api method (or undefined if just an object
	//         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	//         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	//       },
	//       {
	//         name:     'row'
	//         val:       {},
	//         methodExt: [ ... ],
	//         propExt:   [
	//           {
	//             name:      'data'
	//             val:       function () {},
	//             methodExt: [ ... ],
	//             propExt:   [ ... ]
	//           },
	//           ...
	//         ]
	//       }
	//     ]
	
	_Api.register = _api_register = function ( name, val )
	{
		if ( Array.isArray( name ) ) {
			for ( var j=0, jen=name.length ; j<jen ; j++ ) {
				_Api.register( name[j], val );
			}
			return;
		}
	
		var
			i, ien,
			heir = name.split('.'),
			struct = __apiStruct,
			key, method;
	
		var find = function ( src, name ) {
			for ( var i=0, ien=src.length ; i<ien ; i++ ) {
				if ( src[i].name === name ) {
					return src[i];
				}
			}
			return null;
		};
	
		for ( i=0, ien=heir.length ; i<ien ; i++ ) {
			method = heir[i].indexOf('()') !== -1;
			key = method ?
				heir[i].replace('()', '') :
				heir[i];
	
			var src = find( struct, key );
			if ( ! src ) {
				src = {
					name:      key,
					val:       {},
					methodExt: [],
					propExt:   [],
					type:      'object'
				};
				struct.push( src );
			}
	
			if ( i === ien-1 ) {
				src.val = val;
				src.type = typeof val === 'function' ?
					'function' :
					$.isPlainObject( val ) ?
						'object' :
						'other';
			}
			else {
				struct = method ?
					src.methodExt :
					src.propExt;
			}
		}
	};
	
	_Api.registerPlural = _api_registerPlural = function ( pluralName, singularName, val ) {
		_Api.register( pluralName, val );
	
		_Api.register( singularName, function () {
			var ret = val.apply( this, arguments );
	
			if ( ret === this ) {
				// Returned item is the API instance that was passed in, return it
				return this;
			}
			else if ( ret instanceof _Api ) {
				// New API instance returned, want the value from the first item
				// in the returned array for the singular result.
				return ret.length ?
					Array.isArray( ret[0] ) ?
						new _Api( ret.context, ret[0] ) : // Array results are 'enhanced'
						ret[0] :
					undefined;
			}
	
			// Non-API return - just fire it back
			return ret;
		} );
	};
	
	
	/**
	 * Selector for HTML tables. Apply the given selector to the give array of
	 * DataTables settings objects.
	 *
	 * @param {string|integer} [selector] jQuery selector string or integer
	 * @param  {array} Array of DataTables settings objects to be filtered
	 * @return {array}
	 * @ignore
	 */
	var __table_selector = function ( selector, a )
	{
		if ( Array.isArray(selector) ) {
			return $.map( selector, function (item) {
				return __table_selector(item, a);
			} );
		}
	
		// Integer is used to pick out a table by index
		if ( typeof selector === 'number' ) {
			return [ a[ selector ] ];
		}
	
		// Perform a jQuery selector on the table nodes
		var nodes = $.map( a, function (el, i) {
			return el.nTable;
		} );
	
		return $(nodes)
			.filter( selector )
			.map( function (i) {
				// Need to translate back from the table node to the settings
				var idx = $.inArray( this, nodes );
				return a[ idx ];
			} )
			.toArray();
	};
	
	
	
	/**
	 * Context selector for the API's context (i.e. the tables the API instance
	 * refers to.
	 *
	 * @name    DataTable.Api#tables
	 * @param {string|integer} [selector] Selector to pick which tables the iterator
	 *   should operate on. If not given, all tables in the current context are
	 *   used. This can be given as a jQuery selector (for example `':gt(0)'`) to
	 *   select multiple tables or as an integer to select a single table.
	 * @returns {DataTable.Api} Returns a new API instance if a selector is given.
	 */
	_api_register( 'tables()', function ( selector ) {
		// A new instance is created if there was a selector specified
		return selector !== undefined && selector !== null ?
			new _Api( __table_selector( selector, this.context ) ) :
			this;
	} );
	
	
	_api_register( 'table()', function ( selector ) {
		var tables = this.tables( selector );
		var ctx = tables.context;
	
		// Truncate to the first matched table
		return ctx.length ?
			new _Api( ctx[0] ) :
			tables;
	} );
	
	
	_api_registerPlural( 'tables().nodes()', 'table().node()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTable;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().body()', 'table().body()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTBody;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().header()', 'table().header()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTHead;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().footer()', 'table().footer()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTFoot;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().containers()', 'table().container()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTableWrapper;
		}, 1 );
	} );
	
	
	
	/**
	 * Redraw the tables in the current context.
	 */
	_api_register( 'draw()', function ( paging ) {
		return this.iterator( 'table', function ( settings ) {
			if ( paging === 'page' ) {
				_fnDraw( settings );
			}
			else {
				if ( typeof paging === 'string' ) {
					paging = paging === 'full-hold' ?
						false :
						true;
				}
	
				_fnReDraw( settings, paging===false );
			}
		} );
	} );
	
	
	
	/**
	 * Get the current page index.
	 *
	 * @return {integer} Current page index (zero based)
	 *//**
	 * Set the current page.
	 *
	 * Note that if you attempt to show a page which does not exist, DataTables will
	 * not throw an error, but rather reset the paging.
	 *
	 * @param {integer|string} action The paging action to take. This can be one of:
	 *  * `integer` - The page index to jump to
	 *  * `string` - An action to take:
	 *    * `first` - Jump to first page.
	 *    * `next` - Jump to the next page
	 *    * `previous` - Jump to previous page
	 *    * `last` - Jump to the last page.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page()', function ( action ) {
		if ( action === undefined ) {
			return this.page.info().page; // not an expensive call
		}
	
		// else, have an action to take on all tables
		return this.iterator( 'table', function ( settings ) {
			_fnPageChange( settings, action );
		} );
	} );
	
	
	/**
	 * Paging information for the first table in the current context.
	 *
	 * If you require paging information for another table, use the `table()` method
	 * with a suitable selector.
	 *
	 * @return {object} Object with the following properties set:
	 *  * `page` - Current page index (zero based - i.e. the first page is `0`)
	 *  * `pages` - Total number of pages
	 *  * `start` - Display index for the first record shown on the current page
	 *  * `end` - Display index for the last record shown on the current page
	 *  * `length` - Display length (number of records). Note that generally `start
	 *    + length = end`, but this is not always true, for example if there are
	 *    only 2 records to show on the final page, with a length of 10.
	 *  * `recordsTotal` - Full data set length
	 *  * `recordsDisplay` - Data set length once the current filtering criterion
	 *    are applied.
	 */
	_api_register( 'page.info()', function ( action ) {
		if ( this.context.length === 0 ) {
			return undefined;
		}
	
		var
			settings   = this.context[0],
			start      = settings._iDisplayStart,
			len        = settings.oFeatures.bPaginate ? settings._iDisplayLength : -1,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return {
			"page":           all ? 0 : Math.floor( start / len ),
			"pages":          all ? 1 : Math.ceil( visRecords / len ),
			"start":          start,
			"end":            settings.fnDisplayEnd(),
			"length":         len,
			"recordsTotal":   settings.fnRecordsTotal(),
			"recordsDisplay": visRecords,
			"serverSide":     _fnDataSource( settings ) === 'ssp'
		};
	} );
	
	
	/**
	 * Get the current page length.
	 *
	 * @return {integer} Current page length. Note `-1` indicates that all records
	 *   are to be shown.
	 *//**
	 * Set the current page length.
	 *
	 * @param {integer} Page length to set. Use `-1` to show all records.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page.len()', function ( len ) {
		// Note that we can't call this function 'length()' because `length`
		// is a Javascript property of functions which defines how many arguments
		// the function expects.
		if ( len === undefined ) {
			return this.context.length !== 0 ?
				this.context[0]._iDisplayLength :
				undefined;
		}
	
		// else, set the page length
		return this.iterator( 'table', function ( settings ) {
			_fnLengthChange( settings, len );
		} );
	} );
	
	
	
	var __reload = function ( settings, holdPosition, callback ) {
		// Use the draw event to trigger a callback
		if ( callback ) {
			var api = new _Api( settings );
	
			api.one( 'draw', function () {
				callback( api.ajax.json() );
			} );
		}
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			_fnReDraw( settings, holdPosition );
		}
		else {
			_fnProcessingDisplay( settings, true );
	
			// Cancel an existing request
			var xhr = settings.jqXHR;
			if ( xhr && xhr.readyState !== 4 ) {
				xhr.abort();
			}
	
			// Trigger xhr
			_fnBuildAjax( settings, [], function( json ) {
				_fnClearTable( settings );
	
				var data = _fnAjaxDataSrc( settings, json );
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					_fnAddData( settings, data[i] );
				}
	
				_fnReDraw( settings, holdPosition );
				_fnProcessingDisplay( settings, false );
			} );
		}
	};
	
	
	/**
	 * Get the JSON response from the last Ajax request that DataTables made to the
	 * server. Note that this returns the JSON from the first table in the current
	 * context.
	 *
	 * @return {object} JSON received from the server.
	 */
	_api_register( 'ajax.json()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].json;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Get the data submitted in the last Ajax request
	 */
	_api_register( 'ajax.params()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].oAjaxData;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Reload tables from the Ajax data source. Note that this function will
	 * automatically re-draw the table when the remote data has been loaded.
	 *
	 * @param {boolean} [reset=true] Reset (default) or hold the current paging
	 *   position. A full re-sort and re-filter is performed when this method is
	 *   called, which is why the pagination reset is the default action.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.reload()', function ( callback, resetPaging ) {
		return this.iterator( 'table', function (settings) {
			__reload( settings, resetPaging===false, callback );
		} );
	} );
	
	
	/**
	 * Get the current Ajax URL. Note that this returns the URL from the first
	 * table in the current context.
	 *
	 * @return {string} Current Ajax source URL
	 *//**
	 * Set the Ajax URL. Note that this will set the URL for all tables in the
	 * current context.
	 *
	 * @param {string} url URL to set.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url()', function ( url ) {
		var ctx = this.context;
	
		if ( url === undefined ) {
			// get
			if ( ctx.length === 0 ) {
				return undefined;
			}
			ctx = ctx[0];
	
			return ctx.ajax ?
				$.isPlainObject( ctx.ajax ) ?
					ctx.ajax.url :
					ctx.ajax :
				ctx.sAjaxSource;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( $.isPlainObject( settings.ajax ) ) {
				settings.ajax.url = url;
			}
			else {
				settings.ajax = url;
			}
			// No need to consider sAjaxSource here since DataTables gives priority
			// to `ajax` over `sAjaxSource`. So setting `ajax` here, renders any
			// value of `sAjaxSource` redundant.
		} );
	} );
	
	
	/**
	 * Load data from the newly set Ajax URL. Note that this method is only
	 * available when `ajax.url()` is used to set a URL. Additionally, this method
	 * has the same effect as calling `ajax.reload()` but is provided for
	 * convenience when setting a new URL. Like `ajax.reload()` it will
	 * automatically redraw the table once the remote data has been loaded.
	 *
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url().load()', function ( callback, resetPaging ) {
		// Same as a reload, but makes sense to present it for easy access after a
		// url change
		return this.iterator( 'table', function ( ctx ) {
			__reload( ctx, resetPaging===false, callback );
		} );
	} );
	
	
	
	
	var _selector_run = function ( type, selector, selectFn, settings, opts )
	{
		var
			out = [], res,
			a, i, ien, j, jen,
			selectorType = typeof selector;
	
		// Can't just check for isArray here, as an API or jQuery instance might be
		// given with their array like look
		if ( ! selector || selectorType === 'string' || selectorType === 'function' || selector.length === undefined ) {
			selector = [ selector ];
		}
	
		for ( i=0, ien=selector.length ; i<ien ; i++ ) {
			// Only split on simple strings - complex expressions will be jQuery selectors
			a = selector[i] && selector[i].split && ! selector[i].match(/[\[\(:]/) ?
				selector[i].split(',') :
				[ selector[i] ];
	
			for ( j=0, jen=a.length ; j<jen ; j++ ) {
				res = selectFn( typeof a[j] === 'string' ? (a[j]).trim() : a[j] );
	
				if ( res && res.length ) {
					out = out.concat( res );
				}
			}
		}
	
		// selector extensions
		var ext = _ext.selector[ type ];
		if ( ext.length ) {
			for ( i=0, ien=ext.length ; i<ien ; i++ ) {
				out = ext[i]( settings, opts, out );
			}
		}
	
		return _unique( out );
	};
	
	
	var _selector_opts = function ( opts )
	{
		if ( ! opts ) {
			opts = {};
		}
	
		// Backwards compatibility for 1.9- which used the terminology filter rather
		// than search
		if ( opts.filter && opts.search === undefined ) {
			opts.search = opts.filter;
		}
	
		return $.extend( {
			search: 'none',
			order: 'current',
			page: 'all'
		}, opts );
	};
	
	
	var _selector_first = function ( inst )
	{
		// Reduce the API instance to the first item found
		for ( var i=0, ien=inst.length ; i<ien ; i++ ) {
			if ( inst[i].length > 0 ) {
				// Assign the first element to the first item in the instance
				// and truncate the instance and context
				inst[0] = inst[i];
				inst[0].length = 1;
				inst.length = 1;
				inst.context = [ inst.context[i] ];
	
				return inst;
			}
		}
	
		// Not found - return an empty instance
		inst.length = 0;
		return inst;
	};
	
	
	var _selector_row_indexes = function ( settings, opts )
	{
		var
			i, ien, tmp, a=[],
			displayFiltered = settings.aiDisplay,
			displayMaster = settings.aiDisplayMaster;
	
		var
			search = opts.search,  // none, applied, removed
			order  = opts.order,   // applied, current, index (original - compatibility with 1.9)
			page   = opts.page;    // all, current
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			// In server-side processing mode, most options are irrelevant since
			// rows not shown don't exist and the index order is the applied order
			// Removed is a special case - for consistency just return an empty
			// array
			return search === 'removed' ?
				[] :
				_range( 0, displayMaster.length );
		}
		else if ( page == 'current' ) {
			// Current page implies that order=current and filter=applied, since it is
			// fairly senseless otherwise, regardless of what order and search actually
			// are
			for ( i=settings._iDisplayStart, ien=settings.fnDisplayEnd() ; i<ien ; i++ ) {
				a.push( displayFiltered[i] );
			}
		}
		else if ( order == 'current' || order == 'applied' ) {
			if ( search == 'none') {
				a = displayMaster.slice();
			}
			else if ( search == 'applied' ) {
				a = displayFiltered.slice();
			}
			else if ( search == 'removed' ) {
				// O(n+m) solution by creating a hash map
				var displayFilteredMap = {};
	
				for ( var i=0, ien=displayFiltered.length ; i<ien ; i++ ) {
					displayFilteredMap[displayFiltered[i]] = null;
				}
	
				a = $.map( displayMaster, function (el) {
					return ! displayFilteredMap.hasOwnProperty(el) ?
						el :
						null;
				} );
			}
		}
		else if ( order == 'index' || order == 'original' ) {
			for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				if ( search == 'none' ) {
					a.push( i );
				}
				else { // applied | removed
					tmp = $.inArray( i, displayFiltered );
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( i );
					}
				}
			}
		}
	
		return a;
	};
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Rows
	 *
	 * {}          - no selector - use all available rows
	 * {integer}   - row aoData index
	 * {node}      - TR node
	 * {string}    - jQuery selector to apply to the TR elements
	 * {array}     - jQuery array of nodes, or simply an array of TR nodes
	 *
	 */
	var __row_selector = function ( settings, selector, opts )
	{
		var rows;
		var run = function ( sel ) {
			var selInt = _intVal( sel );
			var i, ien;
			var aoData = settings.aoData;
	
			// Short cut - selector is a number and no options provided (default is
			// all records, so no need to check if the index is in there, since it
			// must be - dev error if the index doesn't exist).
			if ( selInt !== null && ! opts ) {
				return [ selInt ];
			}
	
			if ( ! rows ) {
				rows = _selector_row_indexes( settings, opts );
			}
	
			if ( selInt !== null && $.inArray( selInt, rows ) !== -1 ) {
				// Selector - integer
				return [ selInt ];
			}
			else if ( sel === null || sel === undefined || sel === '' ) {
				// Selector - none
				return rows;
			}
	
			// Selector - function
			if ( typeof sel === 'function' ) {
				return $.map( rows, function (idx) {
					var row = aoData[ idx ];
					return sel( idx, row._aData, row.nTr ) ? idx : null;
				} );
			}
	
			// Selector - node
			if ( sel.nodeName ) {
				var rowIdx = sel._DT_RowIndex;  // Property added by DT for fast lookup
				var cellIdx = sel._DT_CellIndex;
	
				if ( rowIdx !== undefined ) {
					// Make sure that the row is actually still present in the table
					return aoData[ rowIdx ] && aoData[ rowIdx ].nTr === sel ?
						[ rowIdx ] :
						[];
				}
				else if ( cellIdx ) {
					return aoData[ cellIdx.row ] && aoData[ cellIdx.row ].nTr === sel.parentNode ?
						[ cellIdx.row ] :
						[];
				}
				else {
					var host = $(sel).closest('*[data-dt-row]');
					return host.length ?
						[ host.data('dt-row') ] :
						[];
				}
			}
	
			// ID selector. Want to always be able to select rows by id, regardless
			// of if the tr element has been created or not, so can't rely upon
			// jQuery here - hence a custom implementation. This does not match
			// Sizzle's fast selector or HTML4 - in HTML5 the ID can be anything,
			// but to select it using a CSS selector engine (like Sizzle or
			// querySelect) it would need to need to be escaped for some characters.
			// DataTables simplifies this for row selectors since you can select
			// only a row. A # indicates an id any anything that follows is the id -
			// unescaped.
			if ( typeof sel === 'string' && sel.charAt(0) === '#' ) {
				// get row index from id
				var rowObj = settings.aIds[ sel.replace( /^#/, '' ) ];
				if ( rowObj !== undefined ) {
					return [ rowObj.idx ];
				}
	
				// need to fall through to jQuery in case there is DOM id that
				// matches
			}
			
			// Get nodes in the order from the `rows` array with null values removed
			var nodes = _removeEmpty(
				_pluck_order( settings.aoData, rows, 'nTr' )
			);
	
			// Selector - jQuery selector string, array of nodes or jQuery object/
			// As jQuery's .filter() allows jQuery objects to be passed in filter,
			// it also allows arrays, so this will cope with all three options
			return $(nodes)
				.filter( sel )
				.map( function () {
					return this._DT_RowIndex;
				} )
				.toArray();
		};
	
		return _selector_run( 'row', selector, run, settings, opts );
	};
	
	
	_api_register( 'rows()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __row_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in __row_selector?
		inst.selector.rows = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_register( 'rows().nodes()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return settings.aoData[ row ].nTr || undefined;
		}, 1 );
	} );
	
	_api_register( 'rows().data()', function () {
		return this.iterator( true, 'rows', function ( settings, rows ) {
			return _pluck_order( settings.aoData, rows, '_aData' );
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().cache()', 'row().cache()', function ( type ) {
		return this.iterator( 'row', function ( settings, row ) {
			var r = settings.aoData[ row ];
			return type === 'search' ? r._aFilterData : r._aSortData;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().invalidate()', 'row().invalidate()', function ( src ) {
		return this.iterator( 'row', function ( settings, row ) {
			_fnInvalidate( settings, row, src );
		} );
	} );
	
	_api_registerPlural( 'rows().indexes()', 'row().index()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return row;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().ids()', 'row().id()', function ( hash ) {
		var a = [];
		var context = this.context;
	
		// `iterator` will drop undefined values, but in this case we want them
		for ( var i=0, ien=context.length ; i<ien ; i++ ) {
			for ( var j=0, jen=this[i].length ; j<jen ; j++ ) {
				var id = context[i].rowIdFn( context[i].aoData[ this[i][j] ]._aData );
				a.push( (hash === true ? '#' : '' )+ id );
			}
		}
	
		return new _Api( context, a );
	} );
	
	_api_registerPlural( 'rows().remove()', 'row().remove()', function () {
		var that = this;
	
		this.iterator( 'row', function ( settings, row, thatIdx ) {
			var data = settings.aoData;
			var rowData = data[ row ];
			var i, ien, j, jen;
			var loopRow, loopCells;
	
			data.splice( row, 1 );
	
			// Update the cached indexes
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				loopRow = data[i];
				loopCells = loopRow.anCells;
	
				// Rows
				if ( loopRow.nTr !== null ) {
					loopRow.nTr._DT_RowIndex = i;
				}
	
				// Cells
				if ( loopCells !== null ) {
					for ( j=0, jen=loopCells.length ; j<jen ; j++ ) {
						loopCells[j]._DT_CellIndex.row = i;
					}
				}
			}
	
			// Delete from the display arrays
			_fnDeleteIndex( settings.aiDisplayMaster, row );
			_fnDeleteIndex( settings.aiDisplay, row );
			_fnDeleteIndex( that[ thatIdx ], row, false ); // maintain local indexes
	
			// For server-side processing tables - subtract the deleted row from the count
			if ( settings._iRecordsDisplay > 0 ) {
				settings._iRecordsDisplay--;
			}
	
			// Check for an 'overflow' they case for displaying the table
			_fnLengthOverflow( settings );
	
			// Remove the row's ID reference if there is one
			var id = settings.rowIdFn( rowData._aData );
			if ( id !== undefined ) {
				delete settings.aIds[ id ];
			}
		} );
	
		this.iterator( 'table', function ( settings ) {
			for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				settings.aoData[i].idx = i;
			}
		} );
	
		return this;
	} );
	
	
	_api_register( 'rows.add()', function ( rows ) {
		var newRows = this.iterator( 'table', function ( settings ) {
				var row, i, ien;
				var out = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
						out.push( _fnAddTr( settings, row )[0] );
					}
					else {
						out.push( _fnAddData( settings, row ) );
					}
				}
	
				return out;
			}, 1 );
	
		// Return an Api.rows() extended instance, so rows().nodes() etc can be used
		var modRows = this.rows( -1 );
		modRows.pop();
		$.merge( modRows, newRows );
	
		return modRows;
	} );
	
	
	
	
	
	/**
	 *
	 */
	_api_register( 'row()', function ( selector, opts ) {
		return _selector_first( this.rows( selector, opts ) );
	} );
	
	
	_api_register( 'row().data()', function ( data ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// Get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._aData :
				undefined;
		}
	
		// Set
		var row = ctx[0].aoData[ this[0] ];
		row._aData = data;
	
		// If the DOM has an id, and the data source is an array
		if ( Array.isArray( data ) && row.nTr && row.nTr.id ) {
			_fnSetObjectDataFn( ctx[0].rowId )( data, row.nTr.id );
		}
	
		// Automatically invalidate
		_fnInvalidate( ctx[0], this[0], 'data' );
	
		return this;
	} );
	
	
	_api_register( 'row().node()', function () {
		var ctx = this.context;
	
		return ctx.length && this.length ?
			ctx[0].aoData[ this[0] ].nTr || null :
			null;
	} );
	
	
	_api_register( 'row.add()', function ( row ) {
		// Allow a jQuery object to be passed in - only a single row is added from
		// it though - the first element in the set
		if ( row instanceof $ && row.length ) {
			row = row[0];
		}
	
		var rows = this.iterator( 'table', function ( settings ) {
			if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
				return _fnAddTr( settings, row )[0];
			}
			return _fnAddData( settings, row );
		} );
	
		// Return an Api.rows() extended instance, with the newly added row selected
		return this.row( rows[0] );
	} );
	
	
	$(document).on('plugin-init.dt', function (e, context) {
		var api = new _Api( context );
	
		api.on( 'stateSaveParams', function ( e, settings, d ) {
			// This could be more compact with the API, but it is a lot faster as a simple
			// internal loop
			var idFn = settings.rowIdFn;
			var data = settings.aoData;
			var ids = [];
	
			for (var i=0 ; i<data.length ; i++) {
				if (data[i]._detailsShow) {
					ids.push( '#' + idFn(data[i]._aData) );
				}
			}
	
			d.childRows = ids;
		})
	
		var loaded = api.state.loaded();
	
		if ( loaded && loaded.childRows ) {
			api
				.rows( $.map(loaded.childRows, function (id){
					return id.replace(/:/g, '\\:')
				}) )
				.every( function () {
					_fnCallbackFire( context, null, 'requestChild', [ this ] )
				});
		}
	});
	
	var __details_add = function ( ctx, row, data, klass )
	{
		// Convert to array of TR elements
		var rows = [];
		var addRow = function ( r, k ) {
			// Recursion to allow for arrays of jQuery objects
			if ( Array.isArray( r ) || r instanceof $ ) {
				for ( var i=0, ien=r.length ; i<ien ; i++ ) {
					addRow( r[i], k );
				}
				return;
			}
	
			// If we get a TR element, then just add it directly - up to the dev
			// to add the correct number of columns etc
			if ( r.nodeName && r.nodeName.toLowerCase() === 'tr' ) {
				rows.push( r );
			}
			else {
				// Otherwise create a row with a wrapper
				var created = $('<tr><td></td></tr>').addClass( k );
				$('td', created)
					.addClass( k )
					.html( r )
					[0].colSpan = _fnVisbleColumns( ctx );
	
				rows.push( created[0] );
			}
		};
	
		addRow( data, klass );
	
		if ( row._details ) {
			row._details.detach();
		}
	
		row._details = $(rows);
	
		// If the children were already shown, that state should be retained
		if ( row._detailsShow ) {
			row._details.insertAfter( row.nTr );
		}
	};
	
	
	// Make state saving of child row details async to allow them to be batch processed
	var __details_state = DataTable.util.throttle(
		function (ctx) {
			_fnSaveState( ctx[0] )
		},
		500
	);
	
	
	var __details_remove = function ( api, idx )
	{
		var ctx = api.context;
	
		if ( ctx.length ) {
			var row = ctx[0].aoData[ idx !== undefined ? idx : api[0] ];
	
			if ( row && row._details ) {
				row._details.remove();
	
				row._detailsShow = undefined;
				row._details = undefined;
				$( row.nTr ).removeClass( 'dt-hasChild' );
				__details_state( ctx );
			}
		}
	};
	
	
	var __details_display = function ( api, show ) {
		var ctx = api.context;
	
		if ( ctx.length && api.length ) {
			var row = ctx[0].aoData[ api[0] ];
	
			if ( row._details ) {
				row._detailsShow = show;
	
				if ( show ) {
					row._details.insertAfter( row.nTr );
					$( row.nTr ).addClass( 'dt-hasChild' );
				}
				else {
					row._details.detach();
					$( row.nTr ).removeClass( 'dt-hasChild' );
				}
	
				_fnCallbackFire( ctx[0], null, 'childRow', [ show, api.row( api[0] ) ] )
	
				__details_events( ctx[0] );
				__details_state( ctx );
			}
		}
	};
	
	
	var __details_events = function ( settings )
	{
		var api = new _Api( settings );
		var namespace = '.dt.DT_details';
		var drawEvent = 'draw'+namespace;
		var colvisEvent = 'column-sizing'+namespace;
		var destroyEvent = 'destroy'+namespace;
		var data = settings.aoData;
	
		api.off( drawEvent +' '+ colvisEvent +' '+ destroyEvent );
	
		if ( _pluck( data, '_details' ).length > 0 ) {
			// On each draw, insert the required elements into the document
			api.on( drawEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				api.rows( {page:'current'} ).eq(0).each( function (idx) {
					// Internal data grab
					var row = data[ idx ];
	
					if ( row._detailsShow ) {
						row._details.insertAfter( row.nTr );
					}
				} );
			} );
	
			// Column visibility change - update the colspan
			api.on( colvisEvent, function ( e, ctx, idx, vis ) {
				if ( settings !== ctx ) {
					return;
				}
	
				// Update the colspan for the details rows (note, only if it already has
				// a colspan)
				var row, visible = _fnVisbleColumns( ctx );
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					row = data[i];
	
					if ( row._details ) {
						row._details.children('td[colspan]').attr('colspan', visible );
					}
				}
			} );
	
			// Table destroyed - nuke any child rows
			api.on( destroyEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					if ( data[i]._details ) {
						__details_remove( api, i );
					}
				}
			} );
		}
	};
	
	// Strings for the method names to help minification
	var _emp = '';
	var _child_obj = _emp+'row().child';
	var _child_mth = _child_obj+'()';
	
	// data can be:
	//  tr
	//  string
	//  jQuery or array of any of the above
	_api_register( _child_mth, function ( data, klass ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._details :
				undefined;
		}
		else if ( data === true ) {
			// show
			this.child.show();
		}
		else if ( data === false ) {
			// remove
			__details_remove( this );
		}
		else if ( ctx.length && this.length ) {
			// set
			__details_add( ctx[0], ctx[0].aoData[ this[0] ], data, klass );
		}
	
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.show()',
		_child_mth+'.show()' // only when `child()` was called with parameters (without
	], function ( show ) {   // it returns an object and this method is not executed)
		__details_display( this, true );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.hide()',
		_child_mth+'.hide()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, false );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.remove()',
		_child_mth+'.remove()' // only when `child()` was called with parameters (without
	], function () {           // it returns an object and this method is not executed)
		__details_remove( this );
		return this;
	} );
	
	
	_api_register( _child_obj+'.isShown()', function () {
		var ctx = this.context;
	
		if ( ctx.length && this.length ) {
			// _detailsShown as false or undefined will fall through to return false
			return ctx[0].aoData[ this[0] ]._detailsShow || false;
		}
		return false;
	} );
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Columns
	 *
	 * {integer}           - column index (>=0 count from left, <0 count from right)
	 * "{integer}:visIdx"  - visible column index (i.e. translate to column index)  (>=0 count from left, <0 count from right)
	 * "{integer}:visible" - alias for {integer}:visIdx  (>=0 count from left, <0 count from right)
	 * "{string}:name"     - column name
	 * "{string}"          - jQuery selector on column header nodes
	 *
	 */
	
	// can be an array of these items, comma separated list, or an array of comma
	// separated lists
	
	var __re_column_selector = /^([^:]+):(name|visIdx|visible)$/;
	
	
	// r1 and r2 are redundant - but it means that the parameters match for the
	// iterator callback in columns().data()
	var __columnData = function ( settings, column, r1, r2, rows ) {
		var a = [];
		for ( var row=0, ien=rows.length ; row<ien ; row++ ) {
			a.push( _fnGetCellData( settings, rows[row], column ) );
		}
		return a;
	};
	
	
	var __column_selector = function ( settings, selector, opts )
	{
		var
			columns = settings.aoColumns,
			names = _pluck( columns, 'sName' ),
			nodes = _pluck( columns, 'nTh' );
	
		var run = function ( s ) {
			var selInt = _intVal( s );
	
			// Selector - all
			if ( s === '' ) {
				return _range( columns.length );
			}
	
			// Selector - index
			if ( selInt !== null ) {
				return [ selInt >= 0 ?
					selInt : // Count from left
					columns.length + selInt // Count from right (+ because its a negative value)
				];
			}
	
			// Selector = function
			if ( typeof s === 'function' ) {
				var rows = _selector_row_indexes( settings, opts );
	
				return $.map( columns, function (col, idx) {
					return s(
							idx,
							__columnData( settings, idx, 0, 0, rows ),
							nodes[ idx ]
						) ? idx : null;
				} );
			}
	
			// jQuery or string selector
			var match = typeof s === 'string' ?
				s.match( __re_column_selector ) :
				'';
	
			if ( match ) {
				switch( match[2] ) {
					case 'visIdx':
					case 'visible':
						var idx = parseInt( match[1], 10 );
						// Visible index given, convert to column index
						if ( idx < 0 ) {
							// Counting from the right
							var visColumns = $.map( columns, function (col,i) {
								return col.bVisible ? i : null;
							} );
							return [ visColumns[ visColumns.length + idx ] ];
						}
						// Counting from the left
						return [ _fnVisibleToColumnIndex( settings, idx ) ];
	
					case 'name':
						// match by name. `names` is column index complete and in order
						return $.map( names, function (name, i) {
							return name === match[1] ? i : null;
						} );
	
					default:
						return [];
				}
			}
	
			// Cell in the table body
			if ( s.nodeName && s._DT_CellIndex ) {
				return [ s._DT_CellIndex.column ];
			}
	
			// jQuery selector on the TH elements for the columns
			var jqResult = $( nodes )
				.filter( s )
				.map( function () {
					return $.inArray( this, nodes ); // `nodes` is column index complete and in order
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise a node which might have a `dt-column` data attribute, or be
			// a child or such an element
			var host = $(s).closest('*[data-dt-column]');
			return host.length ?
				[ host.data('dt-column') ] :
				[];
		};
	
		return _selector_run( 'column', selector, run, settings, opts );
	};
	
	
	var __setColumnVis = function ( settings, column, vis ) {
		var
			cols = settings.aoColumns,
			col  = cols[ column ],
			data = settings.aoData,
			row, cells, i, ien, tr;
	
		// Get
		if ( vis === undefined ) {
			return col.bVisible;
		}
	
		// Set
		// No change
		if ( col.bVisible === vis ) {
			return;
		}
	
		if ( vis ) {
			// Insert column
			// Need to decide if we should use appendChild or insertBefore
			var insertBefore = $.inArray( true, _pluck(cols, 'bVisible'), column+1 );
	
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				tr = data[i].nTr;
				cells = data[i].anCells;
	
				if ( tr ) {
					// insertBefore can act like appendChild if 2nd arg is null
					tr.insertBefore( cells[ column ], cells[ insertBefore ] || null );
				}
			}
		}
		else {
			// Remove column
			$( _pluck( settings.aoData, 'anCells', column ) ).detach();
		}
	
		// Common actions
		col.bVisible = vis;
	};
	
	
	_api_register( 'columns()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __column_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in _row_selector?
		inst.selector.cols = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_registerPlural( 'columns().header()', 'column().header()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTh;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().footer()', 'column().footer()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTf;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().data()', 'column().data()', function () {
		return this.iterator( 'column-rows', __columnData, 1 );
	} );
	
	_api_registerPlural( 'columns().dataSrc()', 'column().dataSrc()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].mData;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().cache()', 'column().cache()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows,
				type === 'search' ? '_aFilterData' : '_aSortData', column
			);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().nodes()', 'column().nodes()', function () {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows, 'anCells', column ) ;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().visible()', 'column().visible()', function ( vis, calc ) {
		var that = this;
		var ret = this.iterator( 'column', function ( settings, column ) {
			if ( vis === undefined ) {
				return settings.aoColumns[ column ].bVisible;
			} // else
			__setColumnVis( settings, column, vis );
		} );
	
		// Group the column visibility changes
		if ( vis !== undefined ) {
			this.iterator( 'table', function ( settings ) {
				// Redraw the header after changes
				_fnDrawHead( settings, settings.aoHeader );
				_fnDrawHead( settings, settings.aoFooter );
		
				// Update colspan for no records display. Child rows and extensions will use their own
				// listeners to do this - only need to update the empty table item here
				if ( ! settings.aiDisplay.length ) {
					$(settings.nTBody).find('td[colspan]').attr('colspan', _fnVisbleColumns(settings));
				}
		
				_fnSaveState( settings );
	
				// Second loop once the first is done for events
				that.iterator( 'column', function ( settings, column ) {
					_fnCallbackFire( settings, null, 'column-visibility', [settings, column, vis, calc] );
				} );
	
				if ( calc === undefined || calc ) {
					that.columns.adjust();
				}
			});
		}
	
		return ret;
	} );
	
	_api_registerPlural( 'columns().indexes()', 'column().index()', function ( type ) {
		return this.iterator( 'column', function ( settings, column ) {
			return type === 'visible' ?
				_fnColumnIndexToVisible( settings, column ) :
				column;
		}, 1 );
	} );
	
	_api_register( 'columns.adjust()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnAdjustColumnSizing( settings );
		}, 1 );
	} );
	
	_api_register( 'column.index()', function ( type, idx ) {
		if ( this.context.length !== 0 ) {
			var ctx = this.context[0];
	
			if ( type === 'fromVisible' || type === 'toData' ) {
				return _fnVisibleToColumnIndex( ctx, idx );
			}
			else if ( type === 'fromData' || type === 'toVisible' ) {
				return _fnColumnIndexToVisible( ctx, idx );
			}
		}
	} );
	
	_api_register( 'column()', function ( selector, opts ) {
		return _selector_first( this.columns( selector, opts ) );
	} );
	
	var __cell_selector = function ( settings, selector, opts )
	{
		var data = settings.aoData;
		var rows = _selector_row_indexes( settings, opts );
		var cells = _removeEmpty( _pluck_order( data, rows, 'anCells' ) );
		var allCells = $(_flatten( [], cells ));
		var row;
		var columns = settings.aoColumns.length;
		var a, i, ien, j, o, host;
	
		var run = function ( s ) {
			var fnSelector = typeof s === 'function';
	
			if ( s === null || s === undefined || fnSelector ) {
				// All cells and function selectors
				a = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					for ( j=0 ; j<columns ; j++ ) {
						o = {
							row: row,
							column: j
						};
	
						if ( fnSelector ) {
							// Selector - function
							host = data[ row ];
	
							if ( s( o, _fnGetCellData(settings, row, j), host.anCells ? host.anCells[j] : null ) ) {
								a.push( o );
							}
						}
						else {
							// Selector - all
							a.push( o );
						}
					}
				}
	
				return a;
			}
			
			// Selector - index
			if ( $.isPlainObject( s ) ) {
				// Valid cell index and its in the array of selectable rows
				return s.column !== undefined && s.row !== undefined && $.inArray( s.row, rows ) !== -1 ?
					[s] :
					[];
			}
	
			// Selector - jQuery filtered cells
			var jqResult = allCells
				.filter( s )
				.map( function (i, el) {
					return { // use a new object, in case someone changes the values
						row:    el._DT_CellIndex.row,
						column: el._DT_CellIndex.column
	 				};
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise the selector is a node, and there is one last option - the
			// element might be a child of an element which has dt-row and dt-column
			// data attributes
			host = $(s).closest('*[data-dt-row]');
			return host.length ?
				[ {
					row: host.data('dt-row'),
					column: host.data('dt-column')
				} ] :
				[];
		};
	
		return _selector_run( 'cell', selector, run, settings, opts );
	};
	
	
	
	
	_api_register( 'cells()', function ( rowSelector, columnSelector, opts ) {
		// Argument shifting
		if ( $.isPlainObject( rowSelector ) ) {
			// Indexes
			if ( rowSelector.row === undefined ) {
				// Selector options in first parameter
				opts = rowSelector;
				rowSelector = null;
			}
			else {
				// Cell index objects in first parameter
				opts = columnSelector;
				columnSelector = null;
			}
		}
		if ( $.isPlainObject( columnSelector ) ) {
			opts = columnSelector;
			columnSelector = null;
		}
	
		// Cell selector
		if ( columnSelector === null || columnSelector === undefined ) {
			return this.iterator( 'table', function ( settings ) {
				return __cell_selector( settings, rowSelector, _selector_opts( opts ) );
			} );
		}
	
		// The default built in options need to apply to row and columns
		var internalOpts = opts ? {
			page: opts.page,
			order: opts.order,
			search: opts.search
		} : {};
	
		// Row + column selector
		var columns = this.columns( columnSelector, internalOpts );
		var rows = this.rows( rowSelector, internalOpts );
		var i, ien, j, jen;
	
		var cellsNoOpts = this.iterator( 'table', function ( settings, idx ) {
			var a = [];
	
			for ( i=0, ien=rows[idx].length ; i<ien ; i++ ) {
				for ( j=0, jen=columns[idx].length ; j<jen ; j++ ) {
					a.push( {
						row:    rows[idx][i],
						column: columns[idx][j]
					} );
				}
			}
	
			return a;
		}, 1 );
	
		// There is currently only one extension which uses a cell selector extension
		// It is a _major_ performance drag to run this if it isn't needed, so this is
		// an extension specific check at the moment
		var cells = opts && opts.selected ?
			this.cells( cellsNoOpts, opts ) :
			cellsNoOpts;
	
		$.extend( cells.selector, {
			cols: columnSelector,
			rows: rowSelector,
			opts: opts
		} );
	
		return cells;
	} );
	
	
	_api_registerPlural( 'cells().nodes()', 'cell().node()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			var data = settings.aoData[ row ];
	
			return data && data.anCells ?
				data.anCells[ column ] :
				undefined;
		}, 1 );
	} );
	
	
	_api_register( 'cells().data()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().cache()', 'cell().cache()', function ( type ) {
		type = type === 'search' ? '_aFilterData' : '_aSortData';
	
		return this.iterator( 'cell', function ( settings, row, column ) {
			return settings.aoData[ row ][ type ][ column ];
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().render()', 'cell().render()', function ( type ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column, type );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().indexes()', 'cell().index()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return {
				row: row,
				column: column,
				columnVisible: _fnColumnIndexToVisible( settings, column )
			};
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().invalidate()', 'cell().invalidate()', function ( src ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			_fnInvalidate( settings, row, src, column );
		} );
	} );
	
	
	
	_api_register( 'cell()', function ( rowSelector, columnSelector, opts ) {
		return _selector_first( this.cells( rowSelector, columnSelector, opts ) );
	} );
	
	
	_api_register( 'cell().data()', function ( data ) {
		var ctx = this.context;
		var cell = this[0];
	
		if ( data === undefined ) {
			// Get
			return ctx.length && cell.length ?
				_fnGetCellData( ctx[0], cell[0].row, cell[0].column ) :
				undefined;
		}
	
		// Set
		_fnSetCellData( ctx[0], cell[0].row, cell[0].column, data );
		_fnInvalidate( ctx[0], cell[0].row, 'data', cell[0].column );
	
		return this;
	} );
	
	
	
	/**
	 * Get current ordering (sorting) that has been applied to the table.
	 *
	 * @returns {array} 2D array containing the sorting information for the first
	 *   table in the current context. Each element in the parent array represents
	 *   a column being sorted upon (i.e. multi-sorting with two columns would have
	 *   2 inner arrays). The inner arrays may have 2 or 3 elements. The first is
	 *   the column index that the sorting condition applies to, the second is the
	 *   direction of the sort (`desc` or `asc`) and, optionally, the third is the
	 *   index of the sorting order from the `column.sorting` initialisation array.
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {integer} order Column index to sort upon.
	 * @param {string} direction Direction of the sort to be applied (`asc` or `desc`)
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 1D array of sorting information to be applied.
	 * @param {array} [...] Optional additional sorting conditions
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 2D array of sorting information to be applied.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order()', function ( order, dir ) {
		var ctx = this.context;
	
		if ( order === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].aaSorting :
				undefined;
		}
	
		// set
		if ( typeof order === 'number' ) {
			// Simple column / direction passed in
			order = [ [ order, dir ] ];
		}
		else if ( order.length && ! Array.isArray( order[0] ) ) {
			// Arguments passed in (list of 1D arrays)
			order = Array.prototype.slice.call( arguments );
		}
		// otherwise a 2D array was passed in
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSorting = order.slice();
		} );
	} );
	
	
	/**
	 * Attach a sort listener to an element for a given column
	 *
	 * @param {node|jQuery|string} node Identifier for the element(s) to attach the
	 *   listener to. This can take the form of a single DOM node, a jQuery
	 *   collection of nodes or a jQuery selector which will identify the node(s).
	 * @param {integer} column the column that a click on this node will sort on
	 * @param {function} [callback] callback function when sort is run
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order.listener()', function ( node, column, callback ) {
		return this.iterator( 'table', function ( settings ) {
			_fnSortAttachListener( settings, node, column, callback );
		} );
	} );
	
	
	_api_register( 'order.fixed()', function ( set ) {
		if ( ! set ) {
			var ctx = this.context;
			var fixed = ctx.length ?
				ctx[0].aaSortingFixed :
				undefined;
	
			return Array.isArray( fixed ) ?
				{ pre: fixed } :
				fixed;
		}
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSortingFixed = $.extend( true, {}, set );
		} );
	} );
	
	
	// Order by the selected column(s)
	_api_register( [
		'columns().order()',
		'column().order()'
	], function ( dir ) {
		var that = this;
	
		return this.iterator( 'table', function ( settings, i ) {
			var sort = [];
	
			$.each( that[i], function (j, col) {
				sort.push( [ col, dir ] );
			} );
	
			settings.aaSorting = sort;
		} );
	} );
	
	
	
	_api_register( 'search()', function ( input, regex, smart, caseInsen ) {
		var ctx = this.context;
	
		if ( input === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].oPreviousSearch.sSearch :
				undefined;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( ! settings.oFeatures.bFilter ) {
				return;
			}
	
			_fnFilterComplete( settings, $.extend( {}, settings.oPreviousSearch, {
				"sSearch": input+"",
				"bRegex":  regex === null ? false : regex,
				"bSmart":  smart === null ? true  : smart,
				"bCaseInsensitive": caseInsen === null ? true : caseInsen
			} ), 1 );
		} );
	} );
	
	
	_api_registerPlural(
		'columns().search()',
		'column().search()',
		function ( input, regex, smart, caseInsen ) {
			return this.iterator( 'column', function ( settings, column ) {
				var preSearch = settings.aoPreSearchCols;
	
				if ( input === undefined ) {
					// get
					return preSearch[ column ].sSearch;
				}
	
				// set
				if ( ! settings.oFeatures.bFilter ) {
					return;
				}
	
				$.extend( preSearch[ column ], {
					"sSearch": input+"",
					"bRegex":  regex === null ? false : regex,
					"bSmart":  smart === null ? true  : smart,
					"bCaseInsensitive": caseInsen === null ? true : caseInsen
				} );
	
				_fnFilterComplete( settings, settings.oPreviousSearch, 1 );
			} );
		}
	);
	
	/*
	 * State API methods
	 */
	
	_api_register( 'state()', function () {
		return this.context.length ?
			this.context[0].oSavedState :
			null;
	} );
	
	
	_api_register( 'state.clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			// Save an empty object
			settings.fnStateSaveCallback.call( settings.oInstance, settings, {} );
		} );
	} );
	
	
	_api_register( 'state.loaded()', function () {
		return this.context.length ?
			this.context[0].oLoadedState :
			null;
	} );
	
	
	_api_register( 'state.save()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnSaveState( settings );
		} );
	} );
	
	
	
	/**
	 * Provide a common method for plug-ins to check the version of DataTables being
	 * used, in order to ensure compatibility.
	 *
	 *  @param {string} version Version string to check for, in the format "X.Y.Z".
	 *    Note that the formats "X" and "X.Y" are also acceptable.
	 *  @returns {boolean} true if this version of DataTables is greater or equal to
	 *    the required version, or false if this version of DataTales is not
	 *    suitable
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    alert( $.fn.dataTable.versionCheck( '1.9.0' ) );
	 */
	DataTable.versionCheck = DataTable.fnVersionCheck = function( version )
	{
		var aThis = DataTable.version.split('.');
		var aThat = version.split('.');
		var iThis, iThat;
	
		for ( var i=0, iLen=aThat.length ; i<iLen ; i++ ) {
			iThis = parseInt( aThis[i], 10 ) || 0;
			iThat = parseInt( aThat[i], 10 ) || 0;
	
			// Parts are the same, keep comparing
			if (iThis === iThat) {
				continue;
			}
	
			// Parts are different, return immediately
			return iThis > iThat;
		}
	
		return true;
	};
	
	
	/**
	 * Check if a `<table>` node is a DataTable table already or not.
	 *
	 *  @param {node|jquery|string} table Table node, jQuery object or jQuery
	 *      selector for the table to test. Note that if more than more than one
	 *      table is passed on, only the first will be checked
	 *  @returns {boolean} true the table given is a DataTable, or false otherwise
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    if ( ! $.fn.DataTable.isDataTable( '#example' ) ) {
	 *      $('#example').dataTable();
	 *    }
	 */
	DataTable.isDataTable = DataTable.fnIsDataTable = function ( table )
	{
		var t = $(table).get(0);
		var is = false;
	
		if ( table instanceof DataTable.Api ) {
			return true;
		}
	
		$.each( DataTable.settings, function (i, o) {
			var head = o.nScrollHead ? $('table', o.nScrollHead)[0] : null;
			var foot = o.nScrollFoot ? $('table', o.nScrollFoot)[0] : null;
	
			if ( o.nTable === t || head === t || foot === t ) {
				is = true;
			}
		} );
	
		return is;
	};
	
	
	/**
	 * Get all DataTable tables that have been initialised - optionally you can
	 * select to get only currently visible tables.
	 *
	 *  @param {boolean} [visible=false] Flag to indicate if you want all (default)
	 *    or visible tables only.
	 *  @returns {array} Array of `table` nodes (not DataTable instances) which are
	 *    DataTables
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    $.each( $.fn.dataTable.tables(true), function () {
	 *      $(table).DataTable().columns.adjust();
	 *    } );
	 */
	DataTable.tables = DataTable.fnTables = function ( visible )
	{
		var api = false;
	
		if ( $.isPlainObject( visible ) ) {
			api = visible.api;
			visible = visible.visible;
		}
	
		var a = $.map( DataTable.settings, function (o) {
			if ( !visible || (visible && $(o.nTable).is(':visible')) ) {
				return o.nTable;
			}
		} );
	
		return api ?
			new _Api( a ) :
			a;
	};
	
	
	/**
	 * Convert from camel case parameters to Hungarian notation. This is made public
	 * for the extensions to provide the same ability as DataTables core to accept
	 * either the 1.9 style Hungarian notation, or the 1.10+ style camelCase
	 * parameters.
	 *
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 */
	DataTable.camelToHungarian = _fnCamelToHungarian;
	
	
	
	/**
	 *
	 */
	_api_register( '$()', function ( selector, opts ) {
		var
			rows   = this.rows( opts ).nodes(), // Get all rows
			jqRows = $(rows);
	
		return $( [].concat(
			jqRows.filter( selector ).toArray(),
			jqRows.find( selector ).toArray()
		) );
	} );
	
	
	// jQuery functions to operate on the tables
	$.each( [ 'on', 'one', 'off' ], function (i, key) {
		_api_register( key+'()', function ( /* event, handler */ ) {
			var args = Array.prototype.slice.call(arguments);
	
			// Add the `dt` namespace automatically if it isn't already present
			args[0] = $.map( args[0].split( /\s/ ), function ( e ) {
				return ! e.match(/\.dt\b/) ?
					e+'.dt' :
					e;
				} ).join( ' ' );
	
			var inst = $( this.tables().nodes() );
			inst[key].apply( inst, args );
			return this;
		} );
	} );
	
	
	_api_register( 'clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnClearTable( settings );
		} );
	} );
	
	
	_api_register( 'settings()', function () {
		return new _Api( this.context, this.context );
	} );
	
	
	_api_register( 'init()', function () {
		var ctx = this.context;
		return ctx.length ? ctx[0].oInit : null;
	} );
	
	
	_api_register( 'data()', function () {
		return this.iterator( 'table', function ( settings ) {
			return _pluck( settings.aoData, '_aData' );
		} ).flatten();
	} );
	
	
	_api_register( 'destroy()', function ( remove ) {
		remove = remove || false;
	
		return this.iterator( 'table', function ( settings ) {
			var classes   = settings.oClasses;
			var table     = settings.nTable;
			var tbody     = settings.nTBody;
			var thead     = settings.nTHead;
			var tfoot     = settings.nTFoot;
			var jqTable   = $(table);
			var jqTbody   = $(tbody);
			var jqWrapper = $(settings.nTableWrapper);
			var rows      = $.map( settings.aoData, function (r) { return r.nTr; } );
			var i, ien;
	
			// Flag to note that the table is currently being destroyed - no action
			// should be taken
			settings.bDestroying = true;
	
			// Fire off the destroy callbacks for plug-ins etc
			_fnCallbackFire( settings, "aoDestroyCallback", "destroy", [settings] );
	
			// If not being removed from the document, make all columns visible
			if ( ! remove ) {
				new _Api( settings ).columns().visible( true );
			}
	
			// Blitz all `DT` namespaced events (these are internal events, the
			// lowercase, `dt` events are user subscribed and they are responsible
			// for removing them
			jqWrapper.off('.DT').find(':not(tbody *)').off('.DT');
			$(window).off('.DT-'+settings.sInstance);
	
			// When scrolling we had to break the table up - restore it
			if ( table != thead.parentNode ) {
				jqTable.children('thead').detach();
				jqTable.append( thead );
			}
	
			if ( tfoot && table != tfoot.parentNode ) {
				jqTable.children('tfoot').detach();
				jqTable.append( tfoot );
			}
	
			settings.aaSorting = [];
			settings.aaSortingFixed = [];
			_fnSortingClasses( settings );
	
			$( rows ).removeClass( settings.asStripeClasses.join(' ') );
	
			$('th, td', thead).removeClass( classes.sSortable+' '+
				classes.sSortableAsc+' '+classes.sSortableDesc+' '+classes.sSortableNone
			);
	
			// Add the TR elements back into the table in their original order
			jqTbody.children().detach();
			jqTbody.append( rows );
	
			var orig = settings.nTableWrapper.parentNode;
	
			// Remove the DataTables generated nodes, events and classes
			var removedMethod = remove ? 'remove' : 'detach';
			jqTable[ removedMethod ]();
			jqWrapper[ removedMethod ]();
	
			// If we need to reattach the table to the document
			if ( ! remove && orig ) {
				// insertBefore acts like appendChild if !arg[1]
				orig.insertBefore( table, settings.nTableReinsertBefore );
	
				// Restore the width of the original table - was read from the style property,
				// so we can restore directly to that
				jqTable
					.css( 'width', settings.sDestroyWidth )
					.removeClass( classes.sTable );
	
				// If the were originally stripe classes - then we add them back here.
				// Note this is not fool proof (for example if not all rows had stripe
				// classes - but it's a good effort without getting carried away
				ien = settings.asDestroyStripes.length;
	
				if ( ien ) {
					jqTbody.children().each( function (i) {
						$(this).addClass( settings.asDestroyStripes[i % ien] );
					} );
				}
			}
	
			/* Remove the settings object from the settings array */
			var idx = $.inArray( settings, DataTable.settings );
			if ( idx !== -1 ) {
				DataTable.settings.splice( idx, 1 );
			}
		} );
	} );
	
	
	// Add the `every()` method for rows, columns and cells in a compact form
	$.each( [ 'column', 'row', 'cell' ], function ( i, type ) {
		_api_register( type+'s().every()', function ( fn ) {
			var opts = this.selector.opts;
			var api = this;
	
			return this.iterator( type, function ( settings, arg1, arg2, arg3, arg4 ) {
				// Rows and columns:
				//  arg1 - index
				//  arg2 - table counter
				//  arg3 - loop counter
				//  arg4 - undefined
				// Cells:
				//  arg1 - row index
				//  arg2 - column index
				//  arg3 - table counter
				//  arg4 - loop counter
				fn.call(
					api[ type ](
						arg1,
						type==='cell' ? arg2 : opts,
						type==='cell' ? opts : undefined
					),
					arg1, arg2, arg3, arg4
				);
			} );
		} );
	} );
	
	
	// i18n method for extensions to be able to use the language object from the
	// DataTable
	_api_register( 'i18n()', function ( token, def, plural ) {
		var ctx = this.context[0];
		var resolved = _fnGetObjectDataFn( token )( ctx.oLanguage );
	
		if ( resolved === undefined ) {
			resolved = def;
		}
	
		if ( plural !== undefined && $.isPlainObject( resolved ) ) {
			resolved = resolved[ plural ] !== undefined ?
				resolved[ plural ] :
				resolved._;
		}
	
		return resolved.replace( '%d', plural ); // nb: plural might be undefined,
	} );	
	/**
	 * Version string for plug-ins to check compatibility. Allowed format is
	 * `a.b.c-d` where: a:int, b:int, c:int, d:string(dev|beta|alpha). `d` is used
	 * only for non-release builds. See http://semver.org/ for more information.
	 *  @member
	 *  @type string
	 *  @default Version number
	 */
	DataTable.version = "1.12.1";
	
	/**
	 * Private data store, containing all of the settings objects that are
	 * created for the tables on a given page.
	 *
	 * Note that the `DataTable.settings` object is aliased to
	 * `jQuery.fn.dataTableExt` through which it may be accessed and
	 * manipulated, or `jQuery.fn.dataTable.settings`.
	 *  @member
	 *  @type array
	 *  @default []
	 *  @private
	 */
	DataTable.settings = [];
	
	/**
	 * Object models container, for the various models that DataTables has
	 * available to it. These models define the objects that are used to hold
	 * the active state and configuration of the table.
	 *  @namespace
	 */
	DataTable.models = {};
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * search information for the global filter and individual column filters.
	 *  @namespace
	 */
	DataTable.models.oSearch = {
		/**
		 * Flag to indicate if the filtering should be case insensitive or not
		 *  @type boolean
		 *  @default true
		 */
		"bCaseInsensitive": true,
	
		/**
		 * Applied search term
		 *  @type string
		 *  @default <i>Empty string</i>
		 */
		"sSearch": "",
	
		/**
		 * Flag to indicate if the search term should be interpreted as a
		 * regular expression (true) or not (false) and therefore and special
		 * regex characters escaped.
		 *  @type boolean
		 *  @default false
		 */
		"bRegex": false,
	
		/**
		 * Flag to indicate if DataTables is to use its smart filtering or not.
		 *  @type boolean
		 *  @default true
		 */
		"bSmart": true,
	
		/**
		 * Flag to indicate if DataTables should only trigger a search when
		 * the return key is pressed.
		 *  @type boolean
		 *  @default false
		 */
		"return": false
	};
	
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * each individual row. This is the object format used for the settings
	 * aoData array.
	 *  @namespace
	 */
	DataTable.models.oRow = {
		/**
		 * TR element for the row
		 *  @type node
		 *  @default null
		 */
		"nTr": null,
	
		/**
		 * Array of TD elements for each row. This is null until the row has been
		 * created.
		 *  @type array nodes
		 *  @default []
		 */
		"anCells": null,
	
		/**
		 * Data object from the original data source for the row. This is either
		 * an array if using the traditional form of DataTables, or an object if
		 * using mData options. The exact type will depend on the passed in
		 * data from the data source, or will be an array if using DOM a data
		 * source.
		 *  @type array|object
		 *  @default []
		 */
		"_aData": [],
	
		/**
		 * Sorting data cache - this array is ostensibly the same length as the
		 * number of columns (although each index is generated only as it is
		 * needed), and holds the data that is used for sorting each column in the
		 * row. We do this cache generation at the start of the sort in order that
		 * the formatting of the sort data need be done only once for each cell
		 * per sort. This array should not be read from or written to by anything
		 * other than the master sorting methods.
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_aSortData": null,
	
		/**
		 * Per cell filtering data cache. As per the sort data cache, used to
		 * increase the performance of the filtering in DataTables
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_aFilterData": null,
	
		/**
		 * Filtering data cache. This is the same as the cell filtering cache, but
		 * in this case a string rather than an array. This is easily computed with
		 * a join on `_aFilterData`, but is provided as a cache so the join isn't
		 * needed on every search (memory traded for performance)
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_sFilterRow": null,
	
		/**
		 * Cache of the class name that DataTables has applied to the row, so we
		 * can quickly look at this variable rather than needing to do a DOM check
		 * on className for the nTr property.
		 *  @type string
		 *  @default <i>Empty string</i>
		 *  @private
		 */
		"_sRowStripe": "",
	
		/**
		 * Denote if the original data source was from the DOM, or the data source
		 * object. This is used for invalidating data, so DataTables can
		 * automatically read data from the original source, unless uninstructed
		 * otherwise.
		 *  @type string
		 *  @default null
		 *  @private
		 */
		"src": null,
	
		/**
		 * Index in the aoData array. This saves an indexOf lookup when we have the
		 * object, but want to know the index
		 *  @type integer
		 *  @default -1
		 *  @private
		 */
		"idx": -1
	};
	
	
	/**
	 * Template object for the column information object in DataTables. This object
	 * is held in the settings aoColumns array and contains all the information that
	 * DataTables needs about each individual column.
	 *
	 * Note that this object is related to {@link DataTable.defaults.column}
	 * but this one is the internal data store for DataTables's cache of columns.
	 * It should NOT be manipulated outside of DataTables. Any configuration should
	 * be done through the initialisation options.
	 *  @namespace
	 */
	DataTable.models.oColumn = {
		/**
		 * Column index. This could be worked out on-the-fly with $.inArray, but it
		 * is faster to just hold it as a variable
		 *  @type integer
		 *  @default null
		 */
		"idx": null,
	
		/**
		 * A list of the columns that sorting should occur on when this column
		 * is sorted. That this property is an array allows multi-column sorting
		 * to be defined for a column (for example first name / last name columns
		 * would benefit from this). The values are integers pointing to the
		 * columns to be sorted on (typically it will be a single integer pointing
		 * at itself, but that doesn't need to be the case).
		 *  @type array
		 */
		"aDataSort": null,
	
		/**
		 * Define the sorting directions that are applied to the column, in sequence
		 * as the column is repeatedly sorted upon - i.e. the first value is used
		 * as the sorting direction when the column if first sorted (clicked on).
		 * Sort it again (click again) and it will move on to the next index.
		 * Repeat until loop.
		 *  @type array
		 */
		"asSorting": null,
	
		/**
		 * Flag to indicate if the column is searchable, and thus should be included
		 * in the filtering or not.
		 *  @type boolean
		 */
		"bSearchable": null,
	
		/**
		 * Flag to indicate if the column is sortable or not.
		 *  @type boolean
		 */
		"bSortable": null,
	
		/**
		 * Flag to indicate if the column is currently visible in the table or not
		 *  @type boolean
		 */
		"bVisible": null,
	
		/**
		 * Store for manual type assignment using the `column.type` option. This
		 * is held in store so we can manipulate the column's `sType` property.
		 *  @type string
		 *  @default null
		 *  @private
		 */
		"_sManualType": null,
	
		/**
		 * Flag to indicate if HTML5 data attributes should be used as the data
		 * source for filtering or sorting. True is either are.
		 *  @type boolean
		 *  @default false
		 *  @private
		 */
		"_bAttrSrc": false,
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 *  @type function
		 *  @param {element} nTd The TD node that has been created
		 *  @param {*} sData The Data for the cell
		 *  @param {array|object} oData The data for the whole row
		 *  @param {int} iRow The row index for the aoData data store
		 *  @default null
		 */
		"fnCreatedCell": null,
	
		/**
		 * Function to get data from a cell in a column. You should <b>never</b>
		 * access data directly through _aData internally in DataTables - always use
		 * the method attached to this property. It allows mData to function as
		 * required. This function is automatically assigned by the column
		 * initialisation method
		 *  @type function
		 *  @param {array|object} oData The data array/object for the array
		 *    (i.e. aoData[]._aData)
		 *  @param {string} sSpecific The specific data type you want to get -
		 *    'display', 'type' 'filter' 'sort'
		 *  @returns {*} The data for the cell from the given row's data
		 *  @default null
		 */
		"fnGetData": null,
	
		/**
		 * Function to set data for a cell in the column. You should <b>never</b>
		 * set the data directly to _aData internally in DataTables - always use
		 * this method. It allows mData to function as required. This function
		 * is automatically assigned by the column initialisation method
		 *  @type function
		 *  @param {array|object} oData The data array/object for the array
		 *    (i.e. aoData[]._aData)
		 *  @param {*} sValue Value to set
		 *  @default null
		 */
		"fnSetData": null,
	
		/**
		 * Property to read the value for the cells in the column from the data
		 * source array / object. If null, then the default content is used, if a
		 * function is given then the return from the function is used.
		 *  @type function|int|string|null
		 *  @default null
		 */
		"mData": null,
	
		/**
		 * Partner property to mData which is used (only when defined) to get
		 * the data - i.e. it is basically the same as mData, but without the
		 * 'set' option, and also the data fed to it is the result from mData.
		 * This is the rendering method to match the data method of mData.
		 *  @type function|int|string|null
		 *  @default null
		 */
		"mRender": null,
	
		/**
		 * Unique header TH/TD element for this column - this is what the sorting
		 * listener is attached to (if sorting is enabled.)
		 *  @type node
		 *  @default null
		 */
		"nTh": null,
	
		/**
		 * Unique footer TH/TD element for this column (if there is one). Not used
		 * in DataTables as such, but can be used for plug-ins to reference the
		 * footer for each column.
		 *  @type node
		 *  @default null
		 */
		"nTf": null,
	
		/**
		 * The class to apply to all TD elements in the table's TBODY for the column
		 *  @type string
		 *  @default null
		 */
		"sClass": null,
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 *  @type string
		 */
		"sContentPadding": null,
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because mData
		 * is set to null, or because the data source itself is null).
		 *  @type string
		 *  @default null
		 */
		"sDefaultContent": null,
	
		/**
		 * Name for the column, allowing reference to the column by name as well as
		 * by index (needs a lookup to work by name).
		 *  @type string
		 */
		"sName": null,
	
		/**
		 * Custom sorting data type - defines which of the available plug-ins in
		 * afnSortData the custom sorting will use - if any is defined.
		 *  @type string
		 *  @default std
		 */
		"sSortDataType": 'std',
	
		/**
		 * Class to be applied to the header element when sorting on this column
		 *  @type string
		 *  @default null
		 */
		"sSortingClass": null,
	
		/**
		 * Class to be applied to the header element when sorting on this column -
		 * when jQuery UI theming is used.
		 *  @type string
		 *  @default null
		 */
		"sSortingClassJUI": null,
	
		/**
		 * Title of the column - what is seen in the TH element (nTh).
		 *  @type string
		 */
		"sTitle": null,
	
		/**
		 * Column sorting and filtering type
		 *  @type string
		 *  @default null
		 */
		"sType": null,
	
		/**
		 * Width of the column
		 *  @type string
		 *  @default null
		 */
		"sWidth": null,
	
		/**
		 * Width of the column when it was first "encountered"
		 *  @type string
		 *  @default null
		 */
		"sWidthOrig": null
	};
	
	
	/*
	 * Developer note: The properties of the object below are given in Hungarian
	 * notation, that was used as the interface for DataTables prior to v1.10, however
	 * from v1.10 onwards the primary interface is camel case. In order to avoid
	 * breaking backwards compatibility utterly with this change, the Hungarian
	 * version is still, internally the primary interface, but is is not documented
	 * - hence the @name tags in each doc comment. This allows a Javascript function
	 * to create a map from Hungarian notation to camel case (going the other direction
	 * would require each property to be listed, which would add around 3K to the size
	 * of DataTables, while this method is about a 0.5K hit).
	 *
	 * Ultimately this does pave the way for Hungarian notation to be dropped
	 * completely, but that is a massive amount of work and will break current
	 * installs (therefore is on-hold until v2).
	 */
	
	/**
	 * Initialisation options that can be given to DataTables at initialisation
	 * time.
	 *  @namespace
	 */
	DataTable.defaults = {
		/**
		 * An array of data to use for the table, passed in at initialisation which
		 * will be used in preference to any data which is already in the DOM. This is
		 * particularly useful for constructing tables purely in Javascript, for
		 * example with a custom Ajax call.
		 *  @type array
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.data
		 *
		 *  @example
		 *    // Using a 2D array data source
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "data": [
		 *          ['Trident', 'Internet Explorer 4.0', 'Win 95+', 4, 'X'],
		 *          ['Trident', 'Internet Explorer 5.0', 'Win 95+', 5, 'C'],
		 *        ],
		 *        "columns": [
		 *          { "title": "Engine" },
		 *          { "title": "Browser" },
		 *          { "title": "Platform" },
		 *          { "title": "Version" },
		 *          { "title": "Grade" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using an array of objects as a data source (`data`)
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "data": [
		 *          {
		 *            "engine":   "Trident",
		 *            "browser":  "Internet Explorer 4.0",
		 *            "platform": "Win 95+",
		 *            "version":  4,
		 *            "grade":    "X"
		 *          },
		 *          {
		 *            "engine":   "Trident",
		 *            "browser":  "Internet Explorer 5.0",
		 *            "platform": "Win 95+",
		 *            "version":  5,
		 *            "grade":    "C"
		 *          }
		 *        ],
		 *        "columns": [
		 *          { "title": "Engine",   "data": "engine" },
		 *          { "title": "Browser",  "data": "browser" },
		 *          { "title": "Platform", "data": "platform" },
		 *          { "title": "Version",  "data": "version" },
		 *          { "title": "Grade",    "data": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"aaData": null,
	
	
		/**
		 * If ordering is enabled, then DataTables will perform a first pass sort on
		 * initialisation. You can define which column(s) the sort is performed
		 * upon, and the sorting direction, with this variable. The `sorting` array
		 * should contain an array for each column to be sorted initially containing
		 * the column's index and a direction string ('asc' or 'desc').
		 *  @type array
		 *  @default [[0,'asc']]
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.order
		 *
		 *  @example
		 *    // Sort by 3rd column first, and then 4th column
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "order": [[2,'asc'], [3,'desc']]
		 *      } );
		 *    } );
		 *
		 *    // No initial sorting
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "order": []
		 *      } );
		 *    } );
		 */
		"aaSorting": [[0,'asc']],
	
	
		/**
		 * This parameter is basically identical to the `sorting` parameter, but
		 * cannot be overridden by user interaction with the table. What this means
		 * is that you could have a column (visible or hidden) which the sorting
		 * will always be forced on first - any sorting after that (from the user)
		 * will then be performed as required. This can be useful for grouping rows
		 * together.
		 *  @type array
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.orderFixed
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "orderFixed": [[0,'asc']]
		 *      } );
		 *    } )
		 */
		"aaSortingFixed": [],
	
	
		/**
		 * DataTables can be instructed to load data to display in the table from a
		 * Ajax source. This option defines how that Ajax call is made and where to.
		 *
		 * The `ajax` property has three different modes of operation, depending on
		 * how it is defined. These are:
		 *
		 * * `string` - Set the URL from where the data should be loaded from.
		 * * `object` - Define properties for `jQuery.ajax`.
		 * * `function` - Custom data get function
		 *
		 * `string`
		 * --------
		 *
		 * As a string, the `ajax` property simply defines the URL from which
		 * DataTables will load data.
		 *
		 * `object`
		 * --------
		 *
		 * As an object, the parameters in the object are passed to
		 * [jQuery.ajax](http://api.jquery.com/jQuery.ajax/) allowing fine control
		 * of the Ajax request. DataTables has a number of default parameters which
		 * you can override using this option. Please refer to the jQuery
		 * documentation for a full description of the options available, although
		 * the following parameters provide additional options in DataTables or
		 * require special consideration:
		 *
		 * * `data` - As with jQuery, `data` can be provided as an object, but it
		 *   can also be used as a function to manipulate the data DataTables sends
		 *   to the server. The function takes a single parameter, an object of
		 *   parameters with the values that DataTables has readied for sending. An
		 *   object may be returned which will be merged into the DataTables
		 *   defaults, or you can add the items to the object that was passed in and
		 *   not return anything from the function. This supersedes `fnServerParams`
		 *   from DataTables 1.9-.
		 *
		 * * `dataSrc` - By default DataTables will look for the property `data` (or
		 *   `aaData` for compatibility with DataTables 1.9-) when obtaining data
		 *   from an Ajax source or for server-side processing - this parameter
		 *   allows that property to be changed. You can use Javascript dotted
		 *   object notation to get a data source for multiple levels of nesting, or
		 *   it my be used as a function. As a function it takes a single parameter,
		 *   the JSON returned from the server, which can be manipulated as
		 *   required, with the returned value being that used by DataTables as the
		 *   data source for the table. This supersedes `sAjaxDataProp` from
		 *   DataTables 1.9-.
		 *
		 * * `success` - Should not be overridden it is used internally in
		 *   DataTables. To manipulate / transform the data returned by the server
		 *   use `ajax.dataSrc`, or use `ajax` as a function (see below).
		 *
		 * `function`
		 * ----------
		 *
		 * As a function, making the Ajax call is left up to yourself allowing
		 * complete control of the Ajax request. Indeed, if desired, a method other
		 * than Ajax could be used to obtain the required data, such as Web storage
		 * or an AIR database.
		 *
		 * The function is given four parameters and no return is required. The
		 * parameters are:
		 *
		 * 1. _object_ - Data to send to the server
		 * 2. _function_ - Callback function that must be executed when the required
		 *    data has been obtained. That data should be passed into the callback
		 *    as the only parameter
		 * 3. _object_ - DataTables settings object for the table
		 *
		 * Note that this supersedes `fnServerData` from DataTables 1.9-.
		 *
		 *  @type string|object|function
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.ajax
		 *  @since 1.10.0
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax.
		 *   // Note DataTables expects data in the form `{ data: [ ...data... ] }` by default).
		 *   $('#example').dataTable( {
		 *     "ajax": "data.json"
		 *   } );
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax, using `dataSrc` to change
		 *   // `data` to `tableData` (i.e. `{ tableData: [ ...data... ] }`)
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": "tableData"
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax, using `dataSrc` to read data
		 *   // from a plain array rather than an array in an object
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": ""
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Manipulate the data returned from the server - add a link to data
		 *   // (note this can, should, be done using `render` for the column - this
		 *   // is just a simple example of how the data can be manipulated).
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": function ( json ) {
		 *         for ( var i=0, ien=json.length ; i<ien ; i++ ) {
		 *           json[i][0] = '<a href="/message/'+json[i][0]+'>View message</a>';
		 *         }
		 *         return json;
		 *       }
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Add data to the request
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "data": function ( d ) {
		 *         return {
		 *           "extra_search": $('#extra').val()
		 *         };
		 *       }
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Send request as POST
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "type": "POST"
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Get the data from localStorage (could interface with a form for
		 *   // adding, editing and removing rows).
		 *   $('#example').dataTable( {
		 *     "ajax": function (data, callback, settings) {
		 *       callback(
		 *         JSON.parse( localStorage.getItem('dataTablesData') )
		 *       );
		 *     }
		 *   } );
		 */
		"ajax": null,
	
	
		/**
		 * This parameter allows you to readily specify the entries in the length drop
		 * down menu that DataTables shows when pagination is enabled. It can be
		 * either a 1D array of options which will be used for both the displayed
		 * option and the value, or a 2D array which will use the array in the first
		 * position as the value, and the array in the second position as the
		 * displayed options (useful for language strings such as 'All').
		 *
		 * Note that the `pageLength` property will be automatically set to the
		 * first value given in this array, unless `pageLength` is also provided.
		 *  @type array
		 *  @default [ 10, 25, 50, 100 ]
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.lengthMenu
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "All"]]
		 *      } );
		 *    } );
		 */
		"aLengthMenu": [ 10, 25, 50, 100 ],
	
	
		/**
		 * The `columns` option in the initialisation parameter allows you to define
		 * details about the way individual columns behave. For a full list of
		 * column options that can be set, please see
		 * {@link DataTable.defaults.column}. Note that if you use `columns` to
		 * define your columns, you must have an entry in the array for every single
		 * column that you have in your table (these can be null if you don't which
		 * to specify any options).
		 *  @member
		 *
		 *  @name DataTable.defaults.column
		 */
		"aoColumns": null,
	
		/**
		 * Very similar to `columns`, `columnDefs` allows you to target a specific
		 * column, multiple columns, or all columns, using the `targets` property of
		 * each object in the array. This allows great flexibility when creating
		 * tables, as the `columnDefs` arrays can be of any length, targeting the
		 * columns you specifically want. `columnDefs` may use any of the column
		 * options available: {@link DataTable.defaults.column}, but it _must_
		 * have `targets` defined in each object in the array. Values in the `targets`
		 * array may be:
		 *   <ul>
		 *     <li>a string - class name will be matched on the TH for the column</li>
		 *     <li>0 or a positive integer - column index counting from the left</li>
		 *     <li>a negative integer - column index counting from the right</li>
		 *     <li>the string "_all" - all columns (i.e. assign a default)</li>
		 *   </ul>
		 *  @member
		 *
		 *  @name DataTable.defaults.columnDefs
		 */
		"aoColumnDefs": null,
	
	
		/**
		 * Basically the same as `search`, this parameter defines the individual column
		 * filtering state at initialisation time. The array must be of the same size
		 * as the number of columns, and each element be an object with the parameters
		 * `search` and `escapeRegex` (the latter is optional). 'null' is also
		 * accepted and the default will be used.
		 *  @type array
		 *  @default []
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.searchCols
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "searchCols": [
		 *          null,
		 *          { "search": "My filter" },
		 *          null,
		 *          { "search": "^[0-9]", "escapeRegex": false }
		 *        ]
		 *      } );
		 *    } )
		 */
		"aoSearchCols": [],
	
	
		/**
		 * An array of CSS classes that should be applied to displayed rows. This
		 * array may be of any length, and DataTables will apply each class
		 * sequentially, looping when required.
		 *  @type array
		 *  @default null <i>Will take the values determined by the `oClasses.stripe*`
		 *    options</i>
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.stripeClasses
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stripeClasses": [ 'strip1', 'strip2', 'strip3' ]
		 *      } );
		 *    } )
		 */
		"asStripeClasses": null,
	
	
		/**
		 * Enable or disable automatic column width calculation. This can be disabled
		 * as an optimisation (it takes some time to calculate the widths) if the
		 * tables widths are passed in using `columns`.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.autoWidth
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "autoWidth": false
		 *      } );
		 *    } );
		 */
		"bAutoWidth": true,
	
	
		/**
		 * Deferred rendering can provide DataTables with a huge speed boost when you
		 * are using an Ajax or JS data source for the table. This option, when set to
		 * true, will cause DataTables to defer the creation of the table elements for
		 * each row until they are needed for a draw - saving a significant amount of
		 * time.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.deferRender
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajax": "sources/arrays.txt",
		 *        "deferRender": true
		 *      } );
		 *    } );
		 */
		"bDeferRender": false,
	
	
		/**
		 * Replace a DataTable which matches the given selector and replace it with
		 * one which has the properties of the new initialisation object passed. If no
		 * table matches the selector, then the new DataTable will be constructed as
		 * per normal.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.destroy
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "srollY": "200px",
		 *        "paginate": false
		 *      } );
		 *
		 *      // Some time later....
		 *      $('#example').dataTable( {
		 *        "filter": false,
		 *        "destroy": true
		 *      } );
		 *    } );
		 */
		"bDestroy": false,
	
	
		/**
		 * Enable or disable filtering of data. Filtering in DataTables is "smart" in
		 * that it allows the end user to input multiple words (space separated) and
		 * will match a row containing those words, even if not in the order that was
		 * specified (this allow matching across multiple columns). Note that if you
		 * wish to use filtering in DataTables this must remain 'true' - to remove the
		 * default filtering input box and retain filtering abilities, please use
		 * {@link DataTable.defaults.dom}.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.searching
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "searching": false
		 *      } );
		 *    } );
		 */
		"bFilter": true,
	
	
		/**
		 * Enable or disable the table information display. This shows information
		 * about the data that is currently visible on the page, including information
		 * about filtered data if that action is being performed.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.info
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "info": false
		 *      } );
		 *    } );
		 */
		"bInfo": true,
	
	
		/**
		 * Allows the end user to select the size of a formatted page from a select
		 * menu (sizes are 10, 25, 50 and 100). Requires pagination (`paginate`).
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.lengthChange
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "lengthChange": false
		 *      } );
		 *    } );
		 */
		"bLengthChange": true,
	
	
		/**
		 * Enable or disable pagination.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.paging
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "paging": false
		 *      } );
		 *    } );
		 */
		"bPaginate": true,
	
	
		/**
		 * Enable or disable the display of a 'processing' indicator when the table is
		 * being processed (e.g. a sort). This is particularly useful for tables with
		 * large amounts of data where it can take a noticeable amount of time to sort
		 * the entries.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.processing
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "processing": true
		 *      } );
		 *    } );
		 */
		"bProcessing": false,
	
	
		/**
		 * Retrieve the DataTables object for the given selector. Note that if the
		 * table has already been initialised, this parameter will cause DataTables
		 * to simply return the object that has already been set up - it will not take
		 * account of any changes you might have made to the initialisation object
		 * passed to DataTables (setting this parameter to true is an acknowledgement
		 * that you understand this). `destroy` can be used to reinitialise a table if
		 * you need.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.retrieve
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      initTable();
		 *      tableActions();
		 *    } );
		 *
		 *    function initTable ()
		 *    {
		 *      return $('#example').dataTable( {
		 *        "scrollY": "200px",
		 *        "paginate": false,
		 *        "retrieve": true
		 *      } );
		 *    }
		 *
		 *    function tableActions ()
		 *    {
		 *      var table = initTable();
		 *      // perform API operations with oTable
		 *    }
		 */
		"bRetrieve": false,
	
	
		/**
		 * When vertical (y) scrolling is enabled, DataTables will force the height of
		 * the table's viewport to the given height at all times (useful for layout).
		 * However, this can look odd when filtering data down to a small data set,
		 * and the footer is left "floating" further down. This parameter (when
		 * enabled) will cause DataTables to collapse the table's viewport down when
		 * the result set will fit within the given Y height.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.scrollCollapse
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollY": "200",
		 *        "scrollCollapse": true
		 *      } );
		 *    } );
		 */
		"bScrollCollapse": false,
	
	
		/**
		 * Configure DataTables to use server-side processing. Note that the
		 * `ajax` parameter must also be given in order to give DataTables a
		 * source to obtain the required data for each draw.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverSide
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "xhr.php"
		 *      } );
		 *    } );
		 */
		"bServerSide": false,
	
	
		/**
		 * Enable or disable sorting of columns. Sorting of individual columns can be
		 * disabled by the `sortable` option for each column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.ordering
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "ordering": false
		 *      } );
		 *    } );
		 */
		"bSort": true,
	
	
		/**
		 * Enable or display DataTables' ability to sort multiple columns at the
		 * same time (activated by shift-click by the user).
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.orderMulti
		 *
		 *  @example
		 *    // Disable multiple column sorting ability
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "orderMulti": false
		 *      } );
		 *    } );
		 */
		"bSortMulti": true,
	
	
		/**
		 * Allows control over whether DataTables should use the top (true) unique
		 * cell that is found for a single column, or the bottom (false - default).
		 * This is useful when using complex headers.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.orderCellsTop
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "orderCellsTop": true
		 *      } );
		 *    } );
		 */
		"bSortCellsTop": false,
	
	
		/**
		 * Enable or disable the addition of the classes `sorting\_1`, `sorting\_2` and
		 * `sorting\_3` to the columns which are currently being sorted on. This is
		 * presented as a feature switch as it can increase processing time (while
		 * classes are removed and added) so for large data sets you might want to
		 * turn this off.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.orderClasses
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "orderClasses": false
		 *      } );
		 *    } );
		 */
		"bSortClasses": true,
	
	
		/**
		 * Enable or disable state saving. When enabled HTML5 `localStorage` will be
		 * used to save table display information such as pagination information,
		 * display length, filtering and sorting. As such when the end user reloads
		 * the page the display display will match what thy had previously set up.
		 *
		 * Due to the use of `localStorage` the default state saving is not supported
		 * in IE6 or 7. If state saving is required in those browsers, use
		 * `stateSaveCallback` to provide a storage solution such as cookies.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.stateSave
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "stateSave": true
		 *      } );
		 *    } );
		 */
		"bStateSave": false,
	
	
		/**
		 * This function is called when a TR element is created (and all TD child
		 * elements have been inserted), or registered if using a DOM source, allowing
		 * manipulation of the TR element (adding classes etc).
		 *  @type function
		 *  @param {node} row "TR" element for the current row
		 *  @param {array} data Raw data array for this row
		 *  @param {int} dataIndex The index of this row in the internal aoData array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.createdRow
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "createdRow": function( row, data, dataIndex ) {
		 *          // Bold the grade for all 'A' grade browsers
		 *          if ( data[4] == "A" )
		 *          {
		 *            $('td:eq(4)', row).html( '<b>A</b>' );
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnCreatedRow": null,
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify any aspect you want about the created DOM.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.drawCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "drawCallback": function( settings ) {
		 *          alert( 'DataTables has redrawn the table' );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnDrawCallback": null,
	
	
		/**
		 * Identical to fnHeaderCallback() but for the table footer this function
		 * allows you to modify the table footer on every 'draw' event.
		 *  @type function
		 *  @param {node} foot "TR" element for the footer
		 *  @param {array} data Full table data (as derived from the original HTML)
		 *  @param {int} start Index for the current display starting point in the
		 *    display array
		 *  @param {int} end Index for the current display ending point in the
		 *    display array
		 *  @param {array int} display Index array to translate the visual position
		 *    to the full data array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.footerCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "footerCallback": function( tfoot, data, start, end, display ) {
		 *          tfoot.getElementsByTagName('th')[0].innerHTML = "Starting index is "+start;
		 *        }
		 *      } );
		 *    } )
		 */
		"fnFooterCallback": null,
	
	
		/**
		 * When rendering large numbers in the information element for the table
		 * (i.e. "Showing 1 to 10 of 57 entries") DataTables will render large numbers
		 * to have a comma separator for the 'thousands' units (e.g. 1 million is
		 * rendered as "1,000,000") to help readability for the end user. This
		 * function will override the default method DataTables uses.
		 *  @type function
		 *  @member
		 *  @param {int} toFormat number to be formatted
		 *  @returns {string} formatted string for DataTables to show the number
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.formatNumber
		 *
		 *  @example
		 *    // Format a number using a single quote for the separator (note that
		 *    // this can also be done with the language.thousands option)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "formatNumber": function ( toFormat ) {
		 *          return toFormat.toString().replace(
		 *            /\B(?=(\d{3})+(?!\d))/g, "'"
		 *          );
		 *        };
		 *      } );
		 *    } );
		 */
		"fnFormatNumber": function ( toFormat ) {
			return toFormat.toString().replace(
				/\B(?=(\d{3})+(?!\d))/g,
				this.oLanguage.sThousands
			);
		},
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify the header row. This can be used to calculate and
		 * display useful information about the table.
		 *  @type function
		 *  @param {node} head "TR" element for the header
		 *  @param {array} data Full table data (as derived from the original HTML)
		 *  @param {int} start Index for the current display starting point in the
		 *    display array
		 *  @param {int} end Index for the current display ending point in the
		 *    display array
		 *  @param {array int} display Index array to translate the visual position
		 *    to the full data array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.headerCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "fheaderCallback": function( head, data, start, end, display ) {
		 *          head.getElementsByTagName('th')[0].innerHTML = "Displaying "+(end-start)+" records";
		 *        }
		 *      } );
		 *    } )
		 */
		"fnHeaderCallback": null,
	
	
		/**
		 * The information element can be used to convey information about the current
		 * state of the table. Although the internationalisation options presented by
		 * DataTables are quite capable of dealing with most customisations, there may
		 * be times where you wish to customise the string further. This callback
		 * allows you to do exactly that.
		 *  @type function
		 *  @param {object} oSettings DataTables settings object
		 *  @param {int} start Starting position in data for the draw
		 *  @param {int} end End position in data for the draw
		 *  @param {int} max Total number of rows in the table (regardless of
		 *    filtering)
		 *  @param {int} total Total number of rows in the data set, after filtering
		 *  @param {string} pre The string that DataTables has formatted using it's
		 *    own rules
		 *  @returns {string} The string to be displayed in the information element.
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.infoCallback
		 *
		 *  @example
		 *    $('#example').dataTable( {
		 *      "infoCallback": function( settings, start, end, max, total, pre ) {
		 *        return start +" to "+ end;
		 *      }
		 *    } );
		 */
		"fnInfoCallback": null,
	
	
		/**
		 * Called when the table has been initialised. Normally DataTables will
		 * initialise sequentially and there will be no need for this function,
		 * however, this does not hold true when using external language information
		 * since that is obtained using an async XHR call.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} json The JSON object request from the server - only
		 *    present if client-side Ajax sourced data is used
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.initComplete
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "initComplete": function(settings, json) {
		 *          alert( 'DataTables has finished its initialisation.' );
		 *        }
		 *      } );
		 *    } )
		 */
		"fnInitComplete": null,
	
	
		/**
		 * Called at the very start of each table draw and can be used to cancel the
		 * draw by returning false, any other return (including undefined) results in
		 * the full draw occurring).
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @returns {boolean} False will cancel the draw, anything else (including no
		 *    return) will allow it to complete.
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.preDrawCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "preDrawCallback": function( settings ) {
		 *          if ( $('#test').val() == 1 ) {
		 *            return false;
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnPreDrawCallback": null,
	
	
		/**
		 * This function allows you to 'post process' each row after it have been
		 * generated for each table draw, but before it is rendered on screen. This
		 * function might be used for setting the row class name etc.
		 *  @type function
		 *  @param {node} row "TR" element for the current row
		 *  @param {array} data Raw data array for this row
		 *  @param {int} displayIndex The display index for the current table draw
		 *  @param {int} displayIndexFull The index of the data in the full list of
		 *    rows (after filtering)
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.rowCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "rowCallback": function( row, data, displayIndex, displayIndexFull ) {
		 *          // Bold the grade for all 'A' grade browsers
		 *          if ( data[4] == "A" ) {
		 *            $('td:eq(4)', row).html( '<b>A</b>' );
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnRowCallback": null,
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * This parameter allows you to override the default function which obtains
		 * the data from the server so something more suitable for your application.
		 * For example you could use POST data, or pull information from a Gears or
		 * AIR database.
		 *  @type function
		 *  @member
		 *  @param {string} source HTTP source to obtain the data from (`ajax`)
		 *  @param {array} data A key/value pair object containing the data to send
		 *    to the server
		 *  @param {function} callback to be called on completion of the data get
		 *    process that will draw the data on the page.
		 *  @param {object} settings DataTables settings object
		 *
		 *  @dtopt Callbacks
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverData
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"fnServerData": null,
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 *  It is often useful to send extra data to the server when making an Ajax
		 * request - for example custom filtering information, and this callback
		 * function makes it trivial to send extra information to the server. The
		 * passed in parameter is the data set that has been constructed by
		 * DataTables, and you can add to this or modify it as you require.
		 *  @type function
		 *  @param {array} data Data array (array of objects which are name/value
		 *    pairs) that has been constructed by DataTables and will be sent to the
		 *    server. In the case of Ajax sourced data with server-side processing
		 *    this will be an empty array, for server-side processing there will be a
		 *    significant number of parameters!
		 *  @returns {undefined} Ensure that you modify the data array passed in,
		 *    as this is passed by reference.
		 *
		 *  @dtopt Callbacks
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverParams
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"fnServerParams": null,
	
	
		/**
		 * Load the table state. With this function you can define from where, and how, the
		 * state of a table is loaded. By default DataTables will load from `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 *  @type function
		 *  @member
		 *  @param {object} settings DataTables settings object
		 *  @param {object} callback Callback that can be executed when done. It
		 *    should be passed the loaded state object.
		 *  @return {object} The DataTables state object to be loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoadCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadCallback": function (settings, callback) {
		 *          $.ajax( {
		 *            "url": "/state_load",
		 *            "dataType": "json",
		 *            "success": function (json) {
		 *              callback( json );
		 *            }
		 *          } );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoadCallback": function ( settings ) {
			try {
				return JSON.parse(
					(settings.iStateDuration === -1 ? sessionStorage : localStorage).getItem(
						'DataTables_'+settings.sInstance+'_'+location.pathname
					)
				);
			} catch (e) {
				return {};
			}
		},
	
	
		/**
		 * Callback which allows modification of the saved state prior to loading that state.
		 * This callback is called when the table is loading state from the stored data, but
		 * prior to the settings object being modified by the saved state. Note that for
		 * plug-in authors, you should use the `stateLoadParams` event to load parameters for
		 * a plug-in.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object that is to be loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoadParams
		 *
		 *  @example
		 *    // Remove a saved filter, so filtering is never loaded
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadParams": function (settings, data) {
		 *          data.oSearch.sSearch = "";
		 *        }
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Disallow state loading by returning false
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadParams": function (settings, data) {
		 *          return false;
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoadParams": null,
	
	
		/**
		 * Callback that is called when the state has been loaded from the state saving method
		 * and the DataTables settings object has been modified as a result of the loaded state.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object that was loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoaded
		 *
		 *  @example
		 *    // Show an alert with the filtering value that was saved
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoaded": function (settings, data) {
		 *          alert( 'Saved filter was: '+data.oSearch.sSearch );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoaded": null,
	
	
		/**
		 * Save the table state. This function allows you to define where and how the state
		 * information for the table is stored By default DataTables will use `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 *  @type function
		 *  @member
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object to be saved
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateSaveCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateSaveCallback": function (settings, data) {
		 *          // Send an Ajax request to the server with the state object
		 *          $.ajax( {
		 *            "url": "/state_save",
		 *            "data": data,
		 *            "dataType": "json",
		 *            "method": "POST"
		 *            "success": function () {}
		 *          } );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateSaveCallback": function ( settings, data ) {
			try {
				(settings.iStateDuration === -1 ? sessionStorage : localStorage).setItem(
					'DataTables_'+settings.sInstance+'_'+location.pathname,
					JSON.stringify( data )
				);
			} catch (e) {}
		},
	
	
		/**
		 * Callback which allows modification of the state to be saved. Called when the table
		 * has changed state a new state save is required. This method allows modification of
		 * the state saving object prior to actually doing the save, including addition or
		 * other state properties or modification. Note that for plug-in authors, you should
		 * use the `stateSaveParams` event to save parameters for a plug-in.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object to be saved
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateSaveParams
		 *
		 *  @example
		 *    // Remove a saved filter, so filtering is never saved
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateSaveParams": function (settings, data) {
		 *          data.oSearch.sSearch = "";
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateSaveParams": null,
	
	
		/**
		 * Duration for which the saved state information is considered valid. After this period
		 * has elapsed the state will be returned to the default.
		 * Value is given in seconds.
		 *  @type int
		 *  @default 7200 <i>(2 hours)</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.stateDuration
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateDuration": 60*60*24; // 1 day
		 *      } );
		 *    } )
		 */
		"iStateDuration": 7200,
	
	
		/**
		 * When enabled DataTables will not make a request to the server for the first
		 * page draw - rather it will use the data already on the page (no sorting etc
		 * will be applied to it), thus saving on an XHR at load time. `deferLoading`
		 * is used to indicate that deferred loading is required, but it is also used
		 * to tell DataTables how many records there are in the full table (allowing
		 * the information element and pagination to be displayed correctly). In the case
		 * where a filtering is applied to the table on initial load, this can be
		 * indicated by giving the parameter as an array, where the first element is
		 * the number of records available after filtering and the second element is the
		 * number of records without filtering (allowing the table information element
		 * to be shown correctly).
		 *  @type int | array
		 *  @default null
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.deferLoading
		 *
		 *  @example
		 *    // 57 records available in the table, no filtering applied
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "scripts/server_processing.php",
		 *        "deferLoading": 57
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // 57 records after filtering, 100 without filtering (an initial filter applied)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "scripts/server_processing.php",
		 *        "deferLoading": [ 57, 100 ],
		 *        "search": {
		 *          "search": "my_filter"
		 *        }
		 *      } );
		 *    } );
		 */
		"iDeferLoading": null,
	
	
		/**
		 * Number of rows to display on a single page when using pagination. If
		 * feature enabled (`lengthChange`) then the end user will be able to override
		 * this to a custom setting using a pop-up menu.
		 *  @type int
		 *  @default 10
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.pageLength
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "pageLength": 50
		 *      } );
		 *    } )
		 */
		"iDisplayLength": 10,
	
	
		/**
		 * Define the starting point for data display when using DataTables with
		 * pagination. Note that this parameter is the number of records, rather than
		 * the page number, so if you have 10 records per page and want to start on
		 * the third page, it should be "20".
		 *  @type int
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.displayStart
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "displayStart": 20
		 *      } );
		 *    } )
		 */
		"iDisplayStart": 0,
	
	
		/**
		 * By default DataTables allows keyboard navigation of the table (sorting, paging,
		 * and filtering) by adding a `tabindex` attribute to the required elements. This
		 * allows you to tab through the controls and press the enter key to activate them.
		 * The tabindex is default 0, meaning that the tab follows the flow of the document.
		 * You can overrule this using this parameter if you wish. Use a value of -1 to
		 * disable built-in keyboard navigation.
		 *  @type int
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.tabIndex
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "tabIndex": 1
		 *      } );
		 *    } );
		 */
		"iTabIndex": 0,
	
	
		/**
		 * Classes that DataTables assigns to the various components and features
		 * that it adds to the HTML table. This allows classes to be configured
		 * during initialisation in addition to through the static
		 * {@link DataTable.ext.oStdClasses} object).
		 *  @namespace
		 *  @name DataTable.defaults.classes
		 */
		"oClasses": {},
	
	
		/**
		 * All strings that DataTables uses in the user interface that it creates
		 * are defined in this object, allowing you to modified them individually or
		 * completely replace them all as required.
		 *  @namespace
		 *  @name DataTable.defaults.language
		 */
		"oLanguage": {
			/**
			 * Strings that are used for WAI-ARIA labels and controls only (these are not
			 * actually visible on the page, but will be read by screenreaders, and thus
			 * must be internationalised as well).
			 *  @namespace
			 *  @name DataTable.defaults.language.aria
			 */
			"oAria": {
				/**
				 * ARIA label that is added to the table headers when the column may be
				 * sorted ascending by activing the column (click or return when focused).
				 * Note that the column header is prefixed to this string.
				 *  @type string
				 *  @default : activate to sort column ascending
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.aria.sortAscending
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "aria": {
				 *            "sortAscending": " - click/return to sort ascending"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sSortAscending": ": activate to sort column ascending",
	
				/**
				 * ARIA label that is added to the table headers when the column may be
				 * sorted descending by activing the column (click or return when focused).
				 * Note that the column header is prefixed to this string.
				 *  @type string
				 *  @default : activate to sort column ascending
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.aria.sortDescending
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "aria": {
				 *            "sortDescending": " - click/return to sort descending"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sSortDescending": ": activate to sort column descending"
			},
	
			/**
			 * Pagination string used by DataTables for the built-in pagination
			 * control types.
			 *  @namespace
			 *  @name DataTable.defaults.language.paginate
			 */
			"oPaginate": {
				/**
				 * Text to use when using the 'full_numbers' type of pagination for the
				 * button to take the user to the first page.
				 *  @type string
				 *  @default First
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.first
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "first": "First page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sFirst": "First",
	
	
				/**
				 * Text to use when using the 'full_numbers' type of pagination for the
				 * button to take the user to the last page.
				 *  @type string
				 *  @default Last
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.last
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "last": "Last page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sLast": "Last",
	
	
				/**
				 * Text to use for the 'next' pagination button (to take the user to the
				 * next page).
				 *  @type string
				 *  @default Next
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.next
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "next": "Next page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sNext": "Next",
	
	
				/**
				 * Text to use for the 'previous' pagination button (to take the user to
				 * the previous page).
				 *  @type string
				 *  @default Previous
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.previous
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "previous": "Previous page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sPrevious": "Previous"
			},
	
			/**
			 * This string is shown in preference to `zeroRecords` when the table is
			 * empty of data (regardless of filtering). Note that this is an optional
			 * parameter - if it is not given, the value of `zeroRecords` will be used
			 * instead (either the default or given value).
			 *  @type string
			 *  @default No data available in table
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.emptyTable
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "emptyTable": "No data available in table"
			 *        }
			 *      } );
			 *    } );
			 */
			"sEmptyTable": "No data available in table",
	
	
			/**
			 * This string gives information to the end user about the information
			 * that is current on display on the page. The following tokens can be
			 * used in the string and will be dynamically replaced as the table
			 * display updates. This tokens can be placed anywhere in the string, or
			 * removed as needed by the language requires:
			 *
			 * * `\_START\_` - Display index of the first record on the current page
			 * * `\_END\_` - Display index of the last record on the current page
			 * * `\_TOTAL\_` - Number of records in the table after filtering
			 * * `\_MAX\_` - Number of records in the table without filtering
			 * * `\_PAGE\_` - Current page number
			 * * `\_PAGES\_` - Total number of pages of data in the table
			 *
			 *  @type string
			 *  @default Showing _START_ to _END_ of _TOTAL_ entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.info
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "info": "Showing page _PAGE_ of _PAGES_"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfo": "Showing _START_ to _END_ of _TOTAL_ entries",
	
	
			/**
			 * Display information string for when the table is empty. Typically the
			 * format of this string should match `info`.
			 *  @type string
			 *  @default Showing 0 to 0 of 0 entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoEmpty
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoEmpty": "No entries to show"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoEmpty": "Showing 0 to 0 of 0 entries",
	
	
			/**
			 * When a user filters the information in a table, this string is appended
			 * to the information (`info`) to give an idea of how strong the filtering
			 * is. The variable _MAX_ is dynamically updated.
			 *  @type string
			 *  @default (filtered from _MAX_ total entries)
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoFiltered
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoFiltered": " - filtering from _MAX_ records"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoFiltered": "(filtered from _MAX_ total entries)",
	
	
			/**
			 * If can be useful to append extra information to the info string at times,
			 * and this variable does exactly that. This information will be appended to
			 * the `info` (`infoEmpty` and `infoFiltered` in whatever combination they are
			 * being used) at all times.
			 *  @type string
			 *  @default <i>Empty string</i>
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoPostFix
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoPostFix": "All records shown are derived from real information."
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoPostFix": "",
	
	
			/**
			 * This decimal place operator is a little different from the other
			 * language options since DataTables doesn't output floating point
			 * numbers, so it won't ever use this for display of a number. Rather,
			 * what this parameter does is modify the sort methods of the table so
			 * that numbers which are in a format which has a character other than
			 * a period (`.`) as a decimal place will be sorted numerically.
			 *
			 * Note that numbers with different decimal places cannot be shown in
			 * the same table and still be sortable, the table must be consistent.
			 * However, multiple different tables on the page can use different
			 * decimal place characters.
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.decimal
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "decimal": ","
			 *          "thousands": "."
			 *        }
			 *      } );
			 *    } );
			 */
			"sDecimal": "",
	
	
			/**
			 * DataTables has a build in number formatter (`formatNumber`) which is
			 * used to format large numbers that are used in the table information.
			 * By default a comma is used, but this can be trivially changed to any
			 * character you wish with this parameter.
			 *  @type string
			 *  @default ,
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.thousands
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "thousands": "'"
			 *        }
			 *      } );
			 *    } );
			 */
			"sThousands": ",",
	
	
			/**
			 * Detail the action that will be taken when the drop down menu for the
			 * pagination length option is changed. The '_MENU_' variable is replaced
			 * with a default select list of 10, 25, 50 and 100, and can be replaced
			 * with a custom select box if required.
			 *  @type string
			 *  @default Show _MENU_ entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.lengthMenu
			 *
			 *  @example
			 *    // Language change only
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "lengthMenu": "Display _MENU_ records"
			 *        }
			 *      } );
			 *    } );
			 *
			 *  @example
			 *    // Language and options change
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "lengthMenu": 'Display <select>'+
			 *            '<option value="10">10</option>'+
			 *            '<option value="20">20</option>'+
			 *            '<option value="30">30</option>'+
			 *            '<option value="40">40</option>'+
			 *            '<option value="50">50</option>'+
			 *            '<option value="-1">All</option>'+
			 *            '</select> records'
			 *        }
			 *      } );
			 *    } );
			 */
			"sLengthMenu": "Show _MENU_ entries",
	
	
			/**
			 * When using Ajax sourced data and during the first draw when DataTables is
			 * gathering the data, this message is shown in an empty row in the table to
			 * indicate to the end user the the data is being loaded. Note that this
			 * parameter is not used when loading data by server-side processing, just
			 * Ajax sourced data with client-side processing.
			 *  @type string
			 *  @default Loading...
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.loadingRecords
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "loadingRecords": "Please wait - loading..."
			 *        }
			 *      } );
			 *    } );
			 */
			"sLoadingRecords": "Loading...",
	
	
			/**
			 * Text which is displayed when the table is processing a user action
			 * (usually a sort command or similar).
			 *  @type string
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.processing
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "processing": "DataTables is currently busy"
			 *        }
			 *      } );
			 *    } );
			 */
			"sProcessing": "",
	
	
			/**
			 * Details the actions that will be taken when the user types into the
			 * filtering input text box. The variable "_INPUT_", if used in the string,
			 * is replaced with the HTML text box for the filtering input allowing
			 * control over where it appears in the string. If "_INPUT_" is not given
			 * then the input box is appended to the string automatically.
			 *  @type string
			 *  @default Search:
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.search
			 *
			 *  @example
			 *    // Input text box will be appended at the end automatically
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "search": "Filter records:"
			 *        }
			 *      } );
			 *    } );
			 *
			 *  @example
			 *    // Specify where the filter should appear
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "search": "Apply filter _INPUT_ to table"
			 *        }
			 *      } );
			 *    } );
			 */
			"sSearch": "Search:",
	
	
			/**
			 * Assign a `placeholder` attribute to the search `input` element
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.searchPlaceholder
			 */
			"sSearchPlaceholder": "",
	
	
			/**
			 * All of the language information can be stored in a file on the
			 * server-side, which DataTables will look up if this parameter is passed.
			 * It must store the URL of the language file, which is in a JSON format,
			 * and the object has the same properties as the oLanguage object in the
			 * initialiser object (i.e. the above parameters). Please refer to one of
			 * the example language files to see how this works in action.
			 *  @type string
			 *  @default <i>Empty string - i.e. disabled</i>
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.url
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "url": "http://www.sprymedia.co.uk/dataTables/lang.txt"
			 *        }
			 *      } );
			 *    } );
			 */
			"sUrl": "",
	
	
			/**
			 * Text shown inside the table records when the is no information to be
			 * displayed after filtering. `emptyTable` is shown when there is simply no
			 * information in the table at all (regardless of filtering).
			 *  @type string
			 *  @default No matching records found
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.zeroRecords
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "zeroRecords": "No records to display"
			 *        }
			 *      } );
			 *    } );
			 */
			"sZeroRecords": "No matching records found"
		},
	
	
		/**
		 * This parameter allows you to have define the global filtering state at
		 * initialisation time. As an object the `search` parameter must be
		 * defined, but all other parameters are optional. When `regex` is true,
		 * the search string will be treated as a regular expression, when false
		 * (default) it will be treated as a straight string. When `smart`
		 * DataTables will use it's smart filtering methods (to word match at
		 * any point in the data), when false this will not be done.
		 *  @namespace
		 *  @extends DataTable.models.oSearch
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.search
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "search": {"search": "Initial search"}
		 *      } );
		 *    } )
		 */
		"oSearch": $.extend( {}, DataTable.models.oSearch ),
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * By default DataTables will look for the property `data` (or `aaData` for
		 * compatibility with DataTables 1.9-) when obtaining data from an Ajax
		 * source or for server-side processing - this parameter allows that
		 * property to be changed. You can use Javascript dotted object notation to
		 * get a data source for multiple levels of nesting.
		 *  @type string
		 *  @default data
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.ajaxDataProp
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sAjaxDataProp": "data",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * You can instruct DataTables to load data from an external
		 * source using this parameter (use aData if you want to pass data in you
		 * already have). Simply provide a url a JSON object can be obtained from.
		 *  @type string
		 *  @default null
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.ajaxSource
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sAjaxSource": null,
	
	
		/**
		 * This initialisation variable allows you to specify exactly where in the
		 * DOM you want DataTables to inject the various controls it adds to the page
		 * (for example you might want the pagination controls at the top of the
		 * table). DIV elements (with or without a custom class) can also be added to
		 * aid styling. The follow syntax is used:
		 *   <ul>
		 *     <li>The following options are allowed:
		 *       <ul>
		 *         <li>'l' - Length changing</li>
		 *         <li>'f' - Filtering input</li>
		 *         <li>'t' - The table!</li>
		 *         <li>'i' - Information</li>
		 *         <li>'p' - Pagination</li>
		 *         <li>'r' - pRocessing</li>
		 *       </ul>
		 *     </li>
		 *     <li>The following constants are allowed:
		 *       <ul>
		 *         <li>'H' - jQueryUI theme "header" classes ('fg-toolbar ui-widget-header ui-corner-tl ui-corner-tr ui-helper-clearfix')</li>
		 *         <li>'F' - jQueryUI theme "footer" classes ('fg-toolbar ui-widget-header ui-corner-bl ui-corner-br ui-helper-clearfix')</li>
		 *       </ul>
		 *     </li>
		 *     <li>The following syntax is expected:
		 *       <ul>
		 *         <li>'&lt;' and '&gt;' - div elements</li>
		 *         <li>'&lt;"class" and '&gt;' - div with a class</li>
		 *         <li>'&lt;"#id" and '&gt;' - div with an ID</li>
		 *       </ul>
		 *     </li>
		 *     <li>Examples:
		 *       <ul>
		 *         <li>'&lt;"wrapper"flipt&gt;'</li>
		 *         <li>'&lt;lf&lt;t&gt;ip&gt;'</li>
		 *       </ul>
		 *     </li>
		 *   </ul>
		 *  @type string
		 *  @default lfrtip <i>(when `jQueryUI` is false)</i> <b>or</b>
		 *    <"H"lfr>t<"F"ip> <i>(when `jQueryUI` is true)</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.dom
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "dom": '&lt;"top"i&gt;rt&lt;"bottom"flp&gt;&lt;"clear"&gt;'
		 *      } );
		 *    } );
		 */
		"sDom": "lfrtip",
	
	
		/**
		 * Search delay option. This will throttle full table searches that use the
		 * DataTables provided search input element (it does not effect calls to
		 * `dt-api search()`, providing a delay before the search is made.
		 *  @type integer
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.searchDelay
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "searchDelay": 200
		 *      } );
		 *    } )
		 */
		"searchDelay": null,
	
	
		/**
		 * DataTables features six different built-in options for the buttons to
		 * display for pagination control:
		 *
		 * * `numbers` - Page number buttons only
		 * * `simple` - 'Previous' and 'Next' buttons only
		 * * 'simple_numbers` - 'Previous' and 'Next' buttons, plus page numbers
		 * * `full` - 'First', 'Previous', 'Next' and 'Last' buttons
		 * * `full_numbers` - 'First', 'Previous', 'Next' and 'Last' buttons, plus page numbers
		 * * `first_last_numbers` - 'First' and 'Last' buttons, plus page numbers
		 *  
		 * Further methods can be added using {@link DataTable.ext.oPagination}.
		 *  @type string
		 *  @default simple_numbers
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.pagingType
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "pagingType": "full_numbers"
		 *      } );
		 *    } )
		 */
		"sPaginationType": "simple_numbers",
	
	
		/**
		 * Enable horizontal scrolling. When a table is too wide to fit into a
		 * certain layout, or you have a large number of columns in the table, you
		 * can enable x-scrolling to show the table in a viewport, which can be
		 * scrolled. This property can be `true` which will allow the table to
		 * scroll horizontally when needed, or any CSS unit, or a number (in which
		 * case it will be treated as a pixel measurement). Setting as simply `true`
		 * is recommended.
		 *  @type boolean|string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.scrollX
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollX": true,
		 *        "scrollCollapse": true
		 *      } );
		 *    } );
		 */
		"sScrollX": "",
	
	
		/**
		 * This property can be used to force a DataTable to use more width than it
		 * might otherwise do when x-scrolling is enabled. For example if you have a
		 * table which requires to be well spaced, this parameter is useful for
		 * "over-sizing" the table, and thus forcing scrolling. This property can by
		 * any CSS unit, or a number (in which case it will be treated as a pixel
		 * measurement).
		 *  @type string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.scrollXInner
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollX": "100%",
		 *        "scrollXInner": "110%"
		 *      } );
		 *    } );
		 */
		"sScrollXInner": "",
	
	
		/**
		 * Enable vertical scrolling. Vertical scrolling will constrain the DataTable
		 * to the given height, and enable scrolling for any data which overflows the
		 * current viewport. This can be used as an alternative to paging to display
		 * a lot of data in a small area (although paging and scrolling can both be
		 * enabled at the same time). This property can be any CSS unit, or a number
		 * (in which case it will be treated as a pixel measurement).
		 *  @type string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.scrollY
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollY": "200px",
		 *        "paginate": false
		 *      } );
		 *    } );
		 */
		"sScrollY": "",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * Set the HTTP method that is used to make the Ajax call for server-side
		 * processing or Ajax sourced data.
		 *  @type string
		 *  @default GET
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverMethod
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sServerMethod": "GET",
	
	
		/**
		 * DataTables makes use of renderers when displaying HTML elements for
		 * a table. These renderers can be added or modified by plug-ins to
		 * generate suitable mark-up for a site. For example the Bootstrap
		 * integration plug-in for DataTables uses a paging button renderer to
		 * display pagination buttons in the mark-up required by Bootstrap.
		 *
		 * For further information about the renderers available see
		 * DataTable.ext.renderer
		 *  @type string|object
		 *  @default null
		 *
		 *  @name DataTable.defaults.renderer
		 *
		 */
		"renderer": null,
	
	
		/**
		 * Set the data property name that DataTables should use to get a row's id
		 * to set as the `id` property in the node.
		 *  @type string
		 *  @default DT_RowId
		 *
		 *  @name DataTable.defaults.rowId
		 */
		"rowId": "DT_RowId"
	};
	
	_fnHungarianMap( DataTable.defaults );
	
	
	
	/*
	 * Developer note - See note in model.defaults.js about the use of Hungarian
	 * notation and camel case.
	 */
	
	/**
	 * Column options that can be given to DataTables at initialisation time.
	 *  @namespace
	 */
	DataTable.defaults.column = {
		/**
		 * Define which column(s) an order will occur on for this column. This
		 * allows a column's ordering to take multiple columns into account when
		 * doing a sort or use the data from a different column. For example first
		 * name / last name columns make sense to do a multi-column sort over the
		 * two columns.
		 *  @type array|int
		 *  @default null <i>Takes the value of the column index automatically</i>
		 *
		 *  @name DataTable.defaults.column.orderData
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderData": [ 0, 1 ], "targets": [ 0 ] },
		 *          { "orderData": [ 1, 0 ], "targets": [ 1 ] },
		 *          { "orderData": 2, "targets": [ 2 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "orderData": [ 0, 1 ] },
		 *          { "orderData": [ 1, 0 ] },
		 *          { "orderData": 2 },
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"aDataSort": null,
		"iDataSort": -1,
	
	
		/**
		 * You can control the default ordering direction, and even alter the
		 * behaviour of the sort handler (i.e. only allow ascending ordering etc)
		 * using this parameter.
		 *  @type array
		 *  @default [ 'asc', 'desc' ]
		 *
		 *  @name DataTable.defaults.column.orderSequence
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderSequence": [ "asc" ], "targets": [ 1 ] },
		 *          { "orderSequence": [ "desc", "asc", "asc" ], "targets": [ 2 ] },
		 *          { "orderSequence": [ "desc" ], "targets": [ 3 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          { "orderSequence": [ "asc" ] },
		 *          { "orderSequence": [ "desc", "asc", "asc" ] },
		 *          { "orderSequence": [ "desc" ] },
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"asSorting": [ 'asc', 'desc' ],
	
	
		/**
		 * Enable or disable filtering on the data in this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.searchable
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "searchable": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "searchable": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bSearchable": true,
	
	
		/**
		 * Enable or disable ordering on this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.orderable
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderable": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "orderable": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bSortable": true,
	
	
		/**
		 * Enable or disable the display of this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.visible
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "visible": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "visible": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bVisible": true,
	
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 *  @type function
		 *  @param {element} td The TD node that has been created
		 *  @param {*} cellData The Data for the cell
		 *  @param {array|object} rowData The data for the whole row
		 *  @param {int} row The row index for the aoData data store
		 *  @param {int} col The column index for aoColumns
		 *
		 *  @name DataTable.defaults.column.createdCell
		 *  @dtopt Columns
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [3],
		 *          "createdCell": function (td, cellData, rowData, row, col) {
		 *            if ( cellData == "1.7" ) {
		 *              $(td).css('color', 'blue')
		 *            }
		 *          }
		 *        } ]
		 *      });
		 *    } );
		 */
		"fnCreatedCell": null,
	
	
		/**
		 * This parameter has been replaced by `data` in DataTables to ensure naming
		 * consistency. `dataProp` can still be used, as there is backwards
		 * compatibility in DataTables for this option, but it is strongly
		 * recommended that you use `data` in preference to `dataProp`.
		 *  @name DataTable.defaults.column.dataProp
		 */
	
	
		/**
		 * This property can be used to read data from any data source property,
		 * including deeply nested objects / properties. `data` can be given in a
		 * number of different ways which effect its behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object. Note that
		 *      function notation is recommended for use in `render` rather than
		 *      `data` as it is much simpler to use as a renderer.
		 * * `null` - use the original data source for the row rather than plucking
		 *   data directly from it. This action has effects on two other
		 *   initialisation options:
		 *    * `defaultContent` - When null is given as the `data` option and
		 *      `defaultContent` is specified for the column, the value defined by
		 *      `defaultContent` will be used for the cell.
		 *    * `render` - When null is used for the `data` option and the `render`
		 *      option is specified for the column, the whole data source for the
		 *      row is used for the renderer.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * `{array|object}` The data source for the row
		 *      * `{string}` The type call data requested - this will be 'set' when
		 *        setting data or 'filter', 'display', 'type', 'sort' or undefined
		 *        when gathering data. Note that when `undefined` is given for the
		 *        type DataTables expects to get the raw data for the object back<
		 *      * `{*}` Data to set when the second parameter is 'set'.
		 *    * Return:
		 *      * The return value from the function is not required when 'set' is
		 *        the type of call, but otherwise the return is what will be used
		 *        for the data requested.
		 *
		 * Note that `data` is a getter and setter option. If you just require
		 * formatting of data for output, you will likely want to use `render` which
		 * is simply a getter and thus simpler to use.
		 *
		 * Note that prior to DataTables 1.9.2 `data` was called `mDataProp`. The
		 * name change reflects the flexibility of this property and is consistent
		 * with the naming of mRender. If 'mDataProp' is given, then it will still
		 * be used by DataTables, as it automatically maps the old name to the new
		 * if required.
		 *
		 *  @type string|int|function|null
		 *  @default null <i>Use automatically calculated column index</i>
		 *
		 *  @name DataTable.defaults.column.data
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Read table data from objects
		 *    // JSON structure for each row:
		 *    //   {
		 *    //      "engine": {value},
		 *    //      "browser": {value},
		 *    //      "platform": {value},
		 *    //      "version": {value},
		 *    //      "grade": {value}
		 *    //   }
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/objects.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          { "data": "platform" },
		 *          { "data": "version" },
		 *          { "data": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Read information from deeply nested objects
		 *    // JSON structure for each row:
		 *    //   {
		 *    //      "engine": {value},
		 *    //      "browser": {value},
		 *    //      "platform": {
		 *    //         "inner": {value}
		 *    //      },
		 *    //      "details": [
		 *    //         {value}, {value}
		 *    //      ]
		 *    //   }
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/deep.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          { "data": "platform.inner" },
		 *          { "data": "details.0" },
		 *          { "data": "details.1" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `data` as a function to provide different information for
		 *    // sorting, filtering and display. In this case, currency (price)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": function ( source, type, val ) {
		 *            if (type === 'set') {
		 *              source.price = val;
		 *              // Store the computed display and filter values for efficiency
		 *              source.price_display = val=="" ? "" : "$"+numberFormat(val);
		 *              source.price_filter  = val=="" ? "" : "$"+numberFormat(val)+" "+val;
		 *              return;
		 *            }
		 *            else if (type === 'display') {
		 *              return source.price_display;
		 *            }
		 *            else if (type === 'filter') {
		 *              return source.price_filter;
		 *            }
		 *            // 'sort', 'type' and undefined all just use the integer
		 *            return source.price;
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using default content
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null,
		 *          "defaultContent": "Click to edit"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using array notation - outputting a list from an array
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": "name[, ]"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 */
		"mData": null,
	
	
		/**
		 * This property is the rendering partner to `data` and it is suggested that
		 * when you want to manipulate data for display (including filtering,
		 * sorting etc) without altering the underlying data for the table, use this
		 * property. `render` can be considered to be the the read only companion to
		 * `data` which is read / write (then as such more complex). Like `data`
		 * this option can be given in a number of different ways to effect its
		 * behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object.
		 * * `object` - use different data for the different data types requested by
		 *   DataTables ('filter', 'display', 'type' or 'sort'). The property names
		 *   of the object is the data type the property refers to and the value can
		 *   defined using an integer, string or function using the same rules as
		 *   `render` normally does. Note that an `_` option _must_ be specified.
		 *   This is the default value to use if you haven't specified a value for
		 *   the data type requested by DataTables.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * {array|object} The data source for the row (based on `data`)
		 *      * {string} The type call data requested - this will be 'filter',
		 *        'display', 'type' or 'sort'.
		 *      * {array|object} The full data source for the row (not based on
		 *        `data`)
		 *    * Return:
		 *      * The return value from the function is what will be used for the
		 *        data requested.
		 *
		 *  @type string|int|function|object|null
		 *  @default null Use the data source value.
		 *
		 *  @name DataTable.defaults.column.render
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Create a comma separated list from an array of objects
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/deep.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          {
		 *            "data": "platform",
		 *            "render": "[, ].name"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Execute a function to obtain data
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null, // Use the full data source object for the renderer's source
		 *          "render": "browserName()"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // As an object, extracting different data for the different types
		 *    // This would be used with a data source such as:
		 *    //   { "phone": 5552368, "phone_filter": "5552368 555-2368", "phone_display": "555-2368" }
		 *    // Here the `phone` integer is used for sorting and type detection, while `phone_filter`
		 *    // (which has both forms) is used for filtering for if a user inputs either format, while
		 *    // the formatted phone number is the one that is shown in the table.
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null, // Use the full data source object for the renderer's source
		 *          "render": {
		 *            "_": "phone",
		 *            "filter": "phone_filter",
		 *            "display": "phone_display"
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Use as a function to create a link from the data source
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": "download_link",
		 *          "render": function ( data, type, full ) {
		 *            return '<a href="'+data+'">Download</a>';
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 */
		"mRender": null,
	
	
		/**
		 * Change the cell type created for the column - either TD cells or TH cells. This
		 * can be useful as TH cells have semantic meaning in the table body, allowing them
		 * to act as a header for a row (you may wish to add scope='row' to the TH elements).
		 *  @type string
		 *  @default td
		 *
		 *  @name DataTable.defaults.column.cellType
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Make the first column use TH cells
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "cellType": "th"
		 *        } ]
		 *      } );
		 *    } );
		 */
		"sCellType": "td",
	
	
		/**
		 * Class to give to each cell in this column.
		 *  @type string
		 *  @default <i>Empty string</i>
		 *
		 *  @name DataTable.defaults.column.class
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "class": "my_class", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "class": "my_class" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sClass": "",
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 * Generally you shouldn't need this!
		 *  @type string
		 *  @default <i>Empty string<i>
		 *
		 *  @name DataTable.defaults.column.contentPadding
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          null,
		 *          {
		 *            "contentPadding": "mmm"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sContentPadding": "",
	
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because `data`
		 * is set to null, or because the data source itself is null).
		 *  @type string
		 *  @default null
		 *
		 *  @name DataTable.defaults.column.defaultContent
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          {
		 *            "data": null,
		 *            "defaultContent": "Edit",
		 *            "targets": [ -1 ]
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          null,
		 *          {
		 *            "data": null,
		 *            "defaultContent": "Edit"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sDefaultContent": null,
	
	
		/**
		 * This parameter is only used in DataTables' server-side processing. It can
		 * be exceptionally useful to know what columns are being displayed on the
		 * client side, and to map these to database fields. When defined, the names
		 * also allow DataTables to reorder information from the server if it comes
		 * back in an unexpected order (i.e. if you switch your columns around on the
		 * client-side, your server-side code does not also need updating).
		 *  @type string
		 *  @default <i>Empty string</i>
		 *
		 *  @name DataTable.defaults.column.name
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "name": "engine", "targets": [ 0 ] },
		 *          { "name": "browser", "targets": [ 1 ] },
		 *          { "name": "platform", "targets": [ 2 ] },
		 *          { "name": "version", "targets": [ 3 ] },
		 *          { "name": "grade", "targets": [ 4 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "name": "engine" },
		 *          { "name": "browser" },
		 *          { "name": "platform" },
		 *          { "name": "version" },
		 *          { "name": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sName": "",
	
	
		/**
		 * Defines a data source type for the ordering which can be used to read
		 * real-time information from the table (updating the internally cached
		 * version) prior to ordering. This allows ordering to occur on user
		 * editable elements such as form inputs.
		 *  @type string
		 *  @default std
		 *
		 *  @name DataTable.defaults.column.orderDataType
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderDataType": "dom-text", "targets": [ 2, 3 ] },
		 *          { "type": "numeric", "targets": [ 3 ] },
		 *          { "orderDataType": "dom-select", "targets": [ 4 ] },
		 *          { "orderDataType": "dom-checkbox", "targets": [ 5 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          { "orderDataType": "dom-text" },
		 *          { "orderDataType": "dom-text", "type": "numeric" },
		 *          { "orderDataType": "dom-select" },
		 *          { "orderDataType": "dom-checkbox" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sSortDataType": "std",
	
	
		/**
		 * The title of this column.
		 *  @type string
		 *  @default null <i>Derived from the 'TH' value for this column in the
		 *    original HTML table.</i>
		 *
		 *  @name DataTable.defaults.column.title
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "title": "My column title", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "title": "My column title" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sTitle": null,
	
	
		/**
		 * The type allows you to specify how the data for this column will be
		 * ordered. Four types (string, numeric, date and html (which will strip
		 * HTML tags before ordering)) are currently available. Note that only date
		 * formats understood by Javascript's Date() object will be accepted as type
		 * date. For example: "Mar 26, 2008 5:03 PM". May take the values: 'string',
		 * 'numeric', 'date' or 'html' (by default). Further types can be adding
		 * through plug-ins.
		 *  @type string
		 *  @default null <i>Auto-detected from raw data</i>
		 *
		 *  @name DataTable.defaults.column.type
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "type": "html", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "type": "html" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sType": null,
	
	
		/**
		 * Defining the width of the column, this parameter may take any CSS value
		 * (3em, 20px etc). DataTables applies 'smart' widths to columns which have not
		 * been given a specific width through this interface ensuring that the table
		 * remains readable.
		 *  @type string
		 *  @default null <i>Automatic</i>
		 *
		 *  @name DataTable.defaults.column.width
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "width": "20%", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "width": "20%" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sWidth": null
	};
	
	_fnHungarianMap( DataTable.defaults.column );
	
	
	
	/**
	 * DataTables settings object - this holds all the information needed for a
	 * given table, including configuration, data and current application of the
	 * table options. DataTables does not have a single instance for each DataTable
	 * with the settings attached to that instance, but rather instances of the
	 * DataTable "class" are created on-the-fly as needed (typically by a
	 * $().dataTable() call) and the settings object is then applied to that
	 * instance.
	 *
	 * Note that this object is related to {@link DataTable.defaults} but this
	 * one is the internal data store for DataTables's cache of columns. It should
	 * NOT be manipulated outside of DataTables. Any configuration should be done
	 * through the initialisation options.
	 *  @namespace
	 *  @todo Really should attach the settings object to individual instances so we
	 *    don't need to create new instances on each $().dataTable() call (if the
	 *    table already exists). It would also save passing oSettings around and
	 *    into every single function. However, this is a very significant
	 *    architecture change for DataTables and will almost certainly break
	 *    backwards compatibility with older installations. This is something that
	 *    will be done in 2.0.
	 */
	DataTable.models.oSettings = {
		/**
		 * Primary features of DataTables and their enablement state.
		 *  @namespace
		 */
		"oFeatures": {
	
			/**
			 * Flag to say if DataTables should automatically try to calculate the
			 * optimum table and columns widths (true) or not (false).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bAutoWidth": null,
	
			/**
			 * Delay the creation of TR and TD elements until they are actually
			 * needed by a driven page draw. This can give a significant speed
			 * increase for Ajax source and Javascript source data, but makes no
			 * difference at all for DOM and server-side processing tables.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bDeferRender": null,
	
			/**
			 * Enable filtering on the table or not. Note that if this is disabled
			 * then there is no filtering at all on the table, including fnFilter.
			 * To just remove the filtering input use sDom and remove the 'f' option.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bFilter": null,
	
			/**
			 * Table information element (the 'Showing x of y records' div) enable
			 * flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bInfo": null,
	
			/**
			 * Present a user control allowing the end user to change the page size
			 * when pagination is enabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bLengthChange": null,
	
			/**
			 * Pagination enabled or not. Note that if this is disabled then length
			 * changing must also be disabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bPaginate": null,
	
			/**
			 * Processing indicator enable flag whenever DataTables is enacting a
			 * user request - typically an Ajax request for server-side processing.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bProcessing": null,
	
			/**
			 * Server-side processing enabled flag - when enabled DataTables will
			 * get all data from the server for every draw - there is no filtering,
			 * sorting or paging done on the client-side.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bServerSide": null,
	
			/**
			 * Sorting enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSort": null,
	
			/**
			 * Multi-column sorting
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSortMulti": null,
	
			/**
			 * Apply a class to the columns which are being sorted to provide a
			 * visual highlight or not. This can slow things down when enabled since
			 * there is a lot of DOM interaction.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSortClasses": null,
	
			/**
			 * State saving enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bStateSave": null
		},
	
	
		/**
		 * Scrolling settings for a table.
		 *  @namespace
		 */
		"oScroll": {
			/**
			 * When the table is shorter in height than sScrollY, collapse the
			 * table container down to the height of the table (when true).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bCollapse": null,
	
			/**
			 * Width of the scrollbar for the web-browser's platform. Calculated
			 * during table initialisation.
			 *  @type int
			 *  @default 0
			 */
			"iBarWidth": 0,
	
			/**
			 * Viewport width for horizontal scrolling. Horizontal scrolling is
			 * disabled if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 */
			"sX": null,
	
			/**
			 * Width to expand the table to when using x-scrolling. Typically you
			 * should not need to use this.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 *  @deprecated
			 */
			"sXInner": null,
	
			/**
			 * Viewport height for vertical scrolling. Vertical scrolling is disabled
			 * if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 */
			"sY": null
		},
	
		/**
		 * Language information for the table.
		 *  @namespace
		 *  @extends DataTable.defaults.oLanguage
		 */
		"oLanguage": {
			/**
			 * Information callback function. See
			 * {@link DataTable.defaults.fnInfoCallback}
			 *  @type function
			 *  @default null
			 */
			"fnInfoCallback": null
		},
	
		/**
		 * Browser support parameters
		 *  @namespace
		 */
		"oBrowser": {
			/**
			 * Indicate if the browser incorrectly calculates width:100% inside a
			 * scrolling element (IE6/7)
			 *  @type boolean
			 *  @default false
			 */
			"bScrollOversize": false,
	
			/**
			 * Determine if the vertical scrollbar is on the right or left of the
			 * scrolling container - needed for rtl language layout, although not
			 * all browsers move the scrollbar (Safari).
			 *  @type boolean
			 *  @default false
			 */
			"bScrollbarLeft": false,
	
			/**
			 * Flag for if `getBoundingClientRect` is fully supported or not
			 *  @type boolean
			 *  @default false
			 */
			"bBounding": false,
	
			/**
			 * Browser scrollbar width
			 *  @type integer
			 *  @default 0
			 */
			"barWidth": 0
		},
	
	
		"ajax": null,
	
	
		/**
		 * Array referencing the nodes which are used for the features. The
		 * parameters of this object match what is allowed by sDom - i.e.
		 *   <ul>
		 *     <li>'l' - Length changing</li>
		 *     <li>'f' - Filtering input</li>
		 *     <li>'t' - The table!</li>
		 *     <li>'i' - Information</li>
		 *     <li>'p' - Pagination</li>
		 *     <li>'r' - pRocessing</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aanFeatures": [],
	
		/**
		 * Store data information - see {@link DataTable.models.oRow} for detailed
		 * information.
		 *  @type array
		 *  @default []
		 */
		"aoData": [],
	
		/**
		 * Array of indexes which are in the current display (after filtering etc)
		 *  @type array
		 *  @default []
		 */
		"aiDisplay": [],
	
		/**
		 * Array of indexes for display - no filtering
		 *  @type array
		 *  @default []
		 */
		"aiDisplayMaster": [],
	
		/**
		 * Map of row ids to data indexes
		 *  @type object
		 *  @default {}
		 */
		"aIds": {},
	
		/**
		 * Store information about each column that is in use
		 *  @type array
		 *  @default []
		 */
		"aoColumns": [],
	
		/**
		 * Store information about the table's header
		 *  @type array
		 *  @default []
		 */
		"aoHeader": [],
	
		/**
		 * Store information about the table's footer
		 *  @type array
		 *  @default []
		 */
		"aoFooter": [],
	
		/**
		 * Store the applied global search information in case we want to force a
		 * research or compare the old search to a new one.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @namespace
		 *  @extends DataTable.models.oSearch
		 */
		"oPreviousSearch": {},
	
		/**
		 * Store the applied search for each column - see
		 * {@link DataTable.models.oSearch} for the format that is used for the
		 * filtering information for each column.
		 *  @type array
		 *  @default []
		 */
		"aoPreSearchCols": [],
	
		/**
		 * Sorting that is applied to the table. Note that the inner arrays are
		 * used in the following manner:
		 * <ul>
		 *   <li>Index 0 - column number</li>
		 *   <li>Index 1 - current sorting direction</li>
		 * </ul>
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @todo These inner arrays should really be objects
		 */
		"aaSorting": null,
	
		/**
		 * Sorting that is always applied to the table (i.e. prefixed in front of
		 * aaSorting).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"aaSortingFixed": [],
	
		/**
		 * Classes to use for the striping of a table.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"asStripeClasses": null,
	
		/**
		 * If restoring a table - we should restore its striping classes as well
		 *  @type array
		 *  @default []
		 */
		"asDestroyStripes": [],
	
		/**
		 * If restoring a table - we should restore its width
		 *  @type int
		 *  @default 0
		 */
		"sDestroyWidth": 0,
	
		/**
		 * Callback functions array for every time a row is inserted (i.e. on a draw).
		 *  @type array
		 *  @default []
		 */
		"aoRowCallback": [],
	
		/**
		 * Callback functions for the header on each draw.
		 *  @type array
		 *  @default []
		 */
		"aoHeaderCallback": [],
	
		/**
		 * Callback function for the footer on each draw.
		 *  @type array
		 *  @default []
		 */
		"aoFooterCallback": [],
	
		/**
		 * Array of callback functions for draw callback functions
		 *  @type array
		 *  @default []
		 */
		"aoDrawCallback": [],
	
		/**
		 * Array of callback functions for row created function
		 *  @type array
		 *  @default []
		 */
		"aoRowCreatedCallback": [],
	
		/**
		 * Callback functions for just before the table is redrawn. A return of
		 * false will be used to cancel the draw.
		 *  @type array
		 *  @default []
		 */
		"aoPreDrawCallback": [],
	
		/**
		 * Callback functions for when the table has been initialised.
		 *  @type array
		 *  @default []
		 */
		"aoInitComplete": [],
	
	
		/**
		 * Callbacks for modifying the settings to be stored for state saving, prior to
		 * saving state.
		 *  @type array
		 *  @default []
		 */
		"aoStateSaveParams": [],
	
		/**
		 * Callbacks for modifying the settings that have been stored for state saving
		 * prior to using the stored values to restore the state.
		 *  @type array
		 *  @default []
		 */
		"aoStateLoadParams": [],
	
		/**
		 * Callbacks for operating on the settings object once the saved state has been
		 * loaded
		 *  @type array
		 *  @default []
		 */
		"aoStateLoaded": [],
	
		/**
		 * Cache the table ID for quick access
		 *  @type string
		 *  @default <i>Empty string</i>
		 */
		"sTableId": "",
	
		/**
		 * The TABLE node for the main table
		 *  @type node
		 *  @default null
		 */
		"nTable": null,
	
		/**
		 * Permanent ref to the thead element
		 *  @type node
		 *  @default null
		 */
		"nTHead": null,
	
		/**
		 * Permanent ref to the tfoot element - if it exists
		 *  @type node
		 *  @default null
		 */
		"nTFoot": null,
	
		/**
		 * Permanent ref to the tbody element
		 *  @type node
		 *  @default null
		 */
		"nTBody": null,
	
		/**
		 * Cache the wrapper node (contains all DataTables controlled elements)
		 *  @type node
		 *  @default null
		 */
		"nTableWrapper": null,
	
		/**
		 * Indicate if when using server-side processing the loading of data
		 * should be deferred until the second draw.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type boolean
		 *  @default false
		 */
		"bDeferLoading": false,
	
		/**
		 * Indicate if all required information has been read in
		 *  @type boolean
		 *  @default false
		 */
		"bInitialised": false,
	
		/**
		 * Information about open rows. Each object in the array has the parameters
		 * 'nTr' and 'nParent'
		 *  @type array
		 *  @default []
		 */
		"aoOpenRows": [],
	
		/**
		 * Dictate the positioning of DataTables' control elements - see
		 * {@link DataTable.model.oInit.sDom}.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default null
		 */
		"sDom": null,
	
		/**
		 * Search delay (in mS)
		 *  @type integer
		 *  @default null
		 */
		"searchDelay": null,
	
		/**
		 * Which type of pagination should be used.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default two_button
		 */
		"sPaginationType": "two_button",
	
		/**
		 * The state duration (for `stateSave`) in seconds.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type int
		 *  @default 0
		 */
		"iStateDuration": 0,
	
		/**
		 * Array of callback functions for state saving. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the JSON string to save that has been thus far created. Returns
		 *       a JSON string to be inserted into a json object
		 *       (i.e. '"param": [ 0, 1, 2]')</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aoStateSave": [],
	
		/**
		 * Array of callback functions for state loading. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the object stored. May return false to cancel state loading</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aoStateLoad": [],
	
		/**
		 * State that was saved. Useful for back reference
		 *  @type object
		 *  @default null
		 */
		"oSavedState": null,
	
		/**
		 * State that was loaded. Useful for back reference
		 *  @type object
		 *  @default null
		 */
		"oLoadedState": null,
	
		/**
		 * Source url for AJAX data for the table.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default null
		 */
		"sAjaxSource": null,
	
		/**
		 * Property from a given object from which to read the table data from. This
		 * can be an empty string (when not server-side processing), in which case
		 * it is  assumed an an array is given directly.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 */
		"sAjaxDataProp": null,
	
		/**
		 * The last jQuery XHR object that was used for server-side data gathering.
		 * This can be used for working with the XHR information in one of the
		 * callbacks
		 *  @type object
		 *  @default null
		 */
		"jqXHR": null,
	
		/**
		 * JSON returned from the server in the last Ajax request
		 *  @type object
		 *  @default undefined
		 */
		"json": undefined,
	
		/**
		 * Data submitted as part of the last Ajax request
		 *  @type object
		 *  @default undefined
		 */
		"oAjaxData": undefined,
	
		/**
		 * Function to get the server-side data.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type function
		 */
		"fnServerData": null,
	
		/**
		 * Functions which are called prior to sending an Ajax request so extra
		 * parameters can easily be sent to the server
		 *  @type array
		 *  @default []
		 */
		"aoServerParams": [],
	
		/**
		 * Send the XHR HTTP method - GET or POST (could be PUT or DELETE if
		 * required).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 */
		"sServerMethod": null,
	
		/**
		 * Format numbers for display.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type function
		 */
		"fnFormatNumber": null,
	
		/**
		 * List of options that can be used for the user selectable length menu.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"aLengthMenu": null,
	
		/**
		 * Counter for the draws that the table does. Also used as a tracker for
		 * server-side processing
		 *  @type int
		 *  @default 0
		 */
		"iDraw": 0,
	
		/**
		 * Indicate if a redraw is being done - useful for Ajax
		 *  @type boolean
		 *  @default false
		 */
		"bDrawing": false,
	
		/**
		 * Draw index (iDraw) of the last error when parsing the returned data
		 *  @type int
		 *  @default -1
		 */
		"iDrawError": -1,
	
		/**
		 * Paging display length
		 *  @type int
		 *  @default 10
		 */
		"_iDisplayLength": 10,
	
		/**
		 * Paging start point - aiDisplay index
		 *  @type int
		 *  @default 0
		 */
		"_iDisplayStart": 0,
	
		/**
		 * Server-side processing - number of records in the result set
		 * (i.e. before filtering), Use fnRecordsTotal rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 *  @type int
		 *  @default 0
		 *  @private
		 */
		"_iRecordsTotal": 0,
	
		/**
		 * Server-side processing - number of records in the current display set
		 * (i.e. after filtering). Use fnRecordsDisplay rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 *  @type boolean
		 *  @default 0
		 *  @private
		 */
		"_iRecordsDisplay": 0,
	
		/**
		 * The classes to use for the table
		 *  @type object
		 *  @default {}
		 */
		"oClasses": {},
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if filtering has been done in the draw. Deprecated in favour of
		 * events.
		 *  @type boolean
		 *  @default false
		 *  @deprecated
		 */
		"bFiltered": false,
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if sorting has been done in the draw. Deprecated in favour of
		 * events.
		 *  @type boolean
		 *  @default false
		 *  @deprecated
		 */
		"bSorted": false,
	
		/**
		 * Indicate that if multiple rows are in the header and there is more than
		 * one unique cell per column, if the top one (true) or bottom one (false)
		 * should be used for sorting / title by DataTables.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type boolean
		 */
		"bSortCellsTop": null,
	
		/**
		 * Initialisation object that is used for the table
		 *  @type object
		 *  @default null
		 */
		"oInit": null,
	
		/**
		 * Destroy callback functions - for plug-ins to attach themselves to the
		 * destroy so they can clean up markup and events.
		 *  @type array
		 *  @default []
		 */
		"aoDestroyCallback": [],
	
	
		/**
		 * Get the number of records in the current record set, before filtering
		 *  @type function
		 */
		"fnRecordsTotal": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsTotal * 1 :
				this.aiDisplayMaster.length;
		},
	
		/**
		 * Get the number of records in the current record set, after filtering
		 *  @type function
		 */
		"fnRecordsDisplay": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsDisplay * 1 :
				this.aiDisplay.length;
		},
	
		/**
		 * Get the display end point - aiDisplay index
		 *  @type function
		 */
		"fnDisplayEnd": function ()
		{
			var
				len      = this._iDisplayLength,
				start    = this._iDisplayStart,
				calc     = start + len,
				records  = this.aiDisplay.length,
				features = this.oFeatures,
				paginate = features.bPaginate;
	
			if ( features.bServerSide ) {
				return paginate === false || len === -1 ?
					start + records :
					Math.min( start+len, this._iRecordsDisplay );
			}
			else {
				return ! paginate || calc>records || len===-1 ?
					records :
					calc;
			}
		},
	
		/**
		 * The DataTables object for this table
		 *  @type object
		 *  @default null
		 */
		"oInstance": null,
	
		/**
		 * Unique identifier for each instance of the DataTables object. If there
		 * is an ID on the table node, then it takes that value, otherwise an
		 * incrementing internal counter is used.
		 *  @type string
		 *  @default null
		 */
		"sInstance": null,
	
		/**
		 * tabindex attribute value that is added to DataTables control elements, allowing
		 * keyboard navigation of the table and its controls.
		 */
		"iTabIndex": 0,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollHead": null,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollFoot": null,
	
		/**
		 * Last applied sort
		 *  @type array
		 *  @default []
		 */
		"aLastSort": [],
	
		/**
		 * Stored plug-in instances
		 *  @type object
		 *  @default {}
		 */
		"oPlugins": {},
	
		/**
		 * Function used to get a row's id from the row's data
		 *  @type function
		 *  @default null
		 */
		"rowIdFn": null,
	
		/**
		 * Data location where to store a row's id
		 *  @type string
		 *  @default null
		 */
		"rowId": null
	};
	
	/**
	 * Extension object for DataTables that is used to provide all extension
	 * options.
	 *
	 * Note that the `DataTable.ext` object is available through
	 * `jQuery.fn.dataTable.ext` where it may be accessed and manipulated. It is
	 * also aliased to `jQuery.fn.dataTableExt` for historic reasons.
	 *  @namespace
	 *  @extends DataTable.models.ext
	 */
	
	
	/**
	 * DataTables extensions
	 * 
	 * This namespace acts as a collection area for plug-ins that can be used to
	 * extend DataTables capabilities. Indeed many of the build in methods
	 * use this method to provide their own capabilities (sorting methods for
	 * example).
	 *
	 * Note that this namespace is aliased to `jQuery.fn.dataTableExt` for legacy
	 * reasons
	 *
	 *  @namespace
	 */
	DataTable.ext = _ext = {
		/**
		 * Buttons. For use with the Buttons extension for DataTables. This is
		 * defined here so other extensions can define buttons regardless of load
		 * order. It is _not_ used by DataTables core.
		 *
		 *  @type object
		 *  @default {}
		 */
		buttons: {},
	
	
		/**
		 * Element class names
		 *
		 *  @type object
		 *  @default {}
		 */
		classes: {},
	
	
		/**
		 * DataTables build type (expanded by the download builder)
		 *
		 *  @type string
		 */
		builder: "-source-",
	
	
		/**
		 * Error reporting.
		 * 
		 * How should DataTables report an error. Can take the value 'alert',
		 * 'throw', 'none' or a function.
		 *
		 *  @type string|function
		 *  @default alert
		 */
		errMode: "alert",
	
	
		/**
		 * Feature plug-ins.
		 * 
		 * This is an array of objects which describe the feature plug-ins that are
		 * available to DataTables. These feature plug-ins are then available for
		 * use through the `dom` initialisation option.
		 * 
		 * Each feature plug-in is described by an object which must have the
		 * following properties:
		 * 
		 * * `fnInit` - function that is used to initialise the plug-in,
		 * * `cFeature` - a character so the feature can be enabled by the `dom`
		 *   instillation option. This is case sensitive.
		 *
		 * The `fnInit` function has the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 *
		 * And the following return is expected:
		 * 
		 * * {node|null} The element which contains your feature. Note that the
		 *   return may also be void if your plug-in does not require to inject any
		 *   DOM elements into DataTables control (`dom`) - for example this might
		 *   be useful when developing a plug-in which allows table control via
		 *   keyboard entry
		 *
		 *  @type array
		 *
		 *  @example
		 *    $.fn.dataTable.ext.features.push( {
		 *      "fnInit": function( oSettings ) {
		 *        return new TableTools( { "oDTSettings": oSettings } );
		 *      },
		 *      "cFeature": "T"
		 *    } );
		 */
		feature: [],
	
	
		/**
		 * Row searching.
		 * 
		 * This method of searching is complimentary to the default type based
		 * searching, and a lot more comprehensive as it allows you complete control
		 * over the searching logic. Each element in this array is a function
		 * (parameters described below) that is called for every row in the table,
		 * and your logic decides if it should be included in the searching data set
		 * or not.
		 *
		 * Searching functions have the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{array|object}` Data for the row to be processed (same as the
		 *    original format that was passed in as the data source, or an array
		 *    from a DOM data source
		 * 3. `{int}` Row index ({@link DataTable.models.oSettings.aoData}), which
		 *    can be useful to retrieve the `TR` element if you need DOM interaction.
		 *
		 * And the following return is expected:
		 *
		 * * {boolean} Include the row in the searched result set (true) or not
		 *   (false)
		 *
		 * Note that as with the main search ability in DataTables, technically this
		 * is "filtering", since it is subtractive. However, for consistency in
		 * naming we call it searching here.
		 *
		 *  @type array
		 *  @default []
		 *
		 *  @example
		 *    // The following example shows custom search being applied to the
		 *    // fourth column (i.e. the data[3] index) based on two input values
		 *    // from the end-user, matching the data in a certain range.
		 *    $.fn.dataTable.ext.search.push(
		 *      function( settings, data, dataIndex ) {
		 *        var min = document.getElementById('min').value * 1;
		 *        var max = document.getElementById('max').value * 1;
		 *        var version = data[3] == "-" ? 0 : data[3]*1;
		 *
		 *        if ( min == "" && max == "" ) {
		 *          return true;
		 *        }
		 *        else if ( min == "" && version < max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && "" == max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && version < max ) {
		 *          return true;
		 *        }
		 *        return false;
		 *      }
		 *    );
		 */
		search: [],
	
	
		/**
		 * Selector extensions
		 *
		 * The `selector` option can be used to extend the options available for the
		 * selector modifier options (`selector-modifier` object data type) that
		 * each of the three built in selector types offer (row, column and cell +
		 * their plural counterparts). For example the Select extension uses this
		 * mechanism to provide an option to select only rows, columns and cells
		 * that have been marked as selected by the end user (`{selected: true}`),
		 * which can be used in conjunction with the existing built in selector
		 * options.
		 *
		 * Each property is an array to which functions can be pushed. The functions
		 * take three attributes:
		 *
		 * * Settings object for the host table
		 * * Options object (`selector-modifier` object type)
		 * * Array of selected item indexes
		 *
		 * The return is an array of the resulting item indexes after the custom
		 * selector has been applied.
		 *
		 *  @type object
		 */
		selector: {
			cell: [],
			column: [],
			row: []
		},
	
	
		/**
		 * Internal functions, exposed for used in plug-ins.
		 * 
		 * Please note that you should not need to use the internal methods for
		 * anything other than a plug-in (and even then, try to avoid if possible).
		 * The internal function may change between releases.
		 *
		 *  @type object
		 *  @default {}
		 */
		internal: {},
	
	
		/**
		 * Legacy configuration options. Enable and disable legacy options that
		 * are available in DataTables.
		 *
		 *  @type object
		 */
		legacy: {
			/**
			 * Enable / disable DataTables 1.9 compatible server-side processing
			 * requests
			 *
			 *  @type boolean
			 *  @default null
			 */
			ajax: null
		},
	
	
		/**
		 * Pagination plug-in methods.
		 * 
		 * Each entry in this object is a function and defines which buttons should
		 * be shown by the pagination rendering method that is used for the table:
		 * {@link DataTable.ext.renderer.pageButton}. The renderer addresses how the
		 * buttons are displayed in the document, while the functions here tell it
		 * what buttons to display. This is done by returning an array of button
		 * descriptions (what each button will do).
		 *
		 * Pagination types (the four built in options and any additional plug-in
		 * options defined here) can be used through the `paginationType`
		 * initialisation parameter.
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{int} page` The current page index
		 * 2. `{int} pages` The number of pages in the table
		 *
		 * Each function is expected to return an array where each element of the
		 * array can be one of:
		 *
		 * * `first` - Jump to first page when activated
		 * * `last` - Jump to last page when activated
		 * * `previous` - Show previous page when activated
		 * * `next` - Show next page when activated
		 * * `{int}` - Show page of the index given
		 * * `{array}` - A nested array containing the above elements to add a
		 *   containing 'DIV' element (might be useful for styling).
		 *
		 * Note that DataTables v1.9- used this object slightly differently whereby
		 * an object with two functions would be defined for each plug-in. That
		 * ability is still supported by DataTables 1.10+ to provide backwards
		 * compatibility, but this option of use is now decremented and no longer
		 * documented in DataTables 1.10+.
		 *
		 *  @type object
		 *  @default {}
		 *
		 *  @example
		 *    // Show previous, next and current page buttons only
		 *    $.fn.dataTableExt.oPagination.current = function ( page, pages ) {
		 *      return [ 'previous', page, 'next' ];
		 *    };
		 */
		pager: {},
	
	
		renderer: {
			pageButton: {},
			header: {}
		},
	
	
		/**
		 * Ordering plug-ins - custom data source
		 * 
		 * The extension options for ordering of data available here is complimentary
		 * to the default type based ordering that DataTables typically uses. It
		 * allows much greater control over the the data that is being used to
		 * order a column, but is necessarily therefore more complex.
		 * 
		 * This type of ordering is useful if you want to do ordering based on data
		 * live from the DOM (for example the contents of an 'input' element) rather
		 * than just the static string that DataTables knows of.
		 * 
		 * The way these plug-ins work is that you create an array of the values you
		 * wish to be ordering for the column in question and then return that
		 * array. The data in the array much be in the index order of the rows in
		 * the table (not the currently ordering order!). Which order data gathering
		 * function is run here depends on the `dt-init columns.orderDataType`
		 * parameter that is used for the column (if any).
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{int}` Target column index
		 *
		 * Each function is expected to return an array:
		 *
		 * * `{array}` Data for the column to be ordering upon
		 *
		 *  @type array
		 *
		 *  @example
		 *    // Ordering using `input` node values
		 *    $.fn.dataTable.ext.order['dom-text'] = function  ( settings, col )
		 *    {
		 *      return this.api().column( col, {order:'index'} ).nodes().map( function ( td, i ) {
		 *        return $('input', td).val();
		 *      } );
		 *    }
		 */
		order: {},
	
	
		/**
		 * Type based plug-ins.
		 *
		 * Each column in DataTables has a type assigned to it, either by automatic
		 * detection or by direct assignment using the `type` option for the column.
		 * The type of a column will effect how it is ordering and search (plug-ins
		 * can also make use of the column type if required).
		 *
		 * @namespace
		 */
		type: {
			/**
			 * Type detection functions.
			 *
			 * The functions defined in this object are used to automatically detect
			 * a column's type, making initialisation of DataTables super easy, even
			 * when complex data is in the table.
			 *
			 * The functions defined take two parameters:
			 *
		     *  1. `{*}` Data from the column cell to be analysed
		     *  2. `{settings}` DataTables settings object. This can be used to
		     *     perform context specific type detection - for example detection
		     *     based on language settings such as using a comma for a decimal
		     *     place. Generally speaking the options from the settings will not
		     *     be required
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Data type detected, or null if unknown (and thus
			 *   pass it on to the other type detection functions.
			 *
			 *  @type array
			 *
			 *  @example
			 *    // Currency type detection plug-in:
			 *    $.fn.dataTable.ext.type.detect.push(
			 *      function ( data, settings ) {
			 *        // Check the numeric part
			 *        if ( ! data.substring(1).match(/[0-9]/) ) {
			 *          return null;
			 *        }
			 *
			 *        // Check prefixed by currency
			 *        if ( data.charAt(0) == '$' || data.charAt(0) == '&pound;' ) {
			 *          return 'currency';
			 *        }
			 *        return null;
			 *      }
			 *    );
			 */
			detect: [],
	
	
			/**
			 * Type based search formatting.
			 *
			 * The type based searching functions can be used to pre-format the
			 * data to be search on. For example, it can be used to strip HTML
			 * tags or to de-format telephone numbers for numeric only searching.
			 *
			 * Note that is a search is not defined for a column of a given type,
			 * no search formatting will be performed.
			 * 
			 * Pre-processing of searching data plug-ins - When you assign the sType
			 * for a column (or have it automatically detected for you by DataTables
			 * or a type detection plug-in), you will typically be using this for
			 * custom sorting, but it can also be used to provide custom searching
			 * by allowing you to pre-processing the data and returning the data in
			 * the format that should be searched upon. This is done by adding
			 * functions this object with a parameter name which matches the sType
			 * for that target column. This is the corollary of <i>afnSortData</i>
			 * for searching data.
			 *
			 * The functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for searching
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Formatted string that will be used for the searching.
			 *
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    $.fn.dataTable.ext.type.search['title-numeric'] = function ( d ) {
			 *      return d.replace(/\n/g," ").replace( /<.*?>/g, "" );
			 *    }
			 */
			search: {},
	
	
			/**
			 * Type based ordering.
			 *
			 * The column type tells DataTables what ordering to apply to the table
			 * when a column is sorted upon. The order for each type that is defined,
			 * is defined by the functions available in this object.
			 *
			 * Each ordering option can be described by three properties added to
			 * this object:
			 *
			 * * `{type}-pre` - Pre-formatting function
			 * * `{type}-asc` - Ascending order function
			 * * `{type}-desc` - Descending order function
			 *
			 * All three can be used together, only `{type}-pre` or only
			 * `{type}-asc` and `{type}-desc` together. It is generally recommended
			 * that only `{type}-pre` is used, as this provides the optimal
			 * implementation in terms of speed, although the others are provided
			 * for compatibility with existing Javascript sort functions.
			 *
			 * `{type}-pre`: Functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for ordering
			 *
			 * And return:
			 *
			 * * `{*}` Data to be sorted upon
			 *
			 * `{type}-asc` and `{type}-desc`: Functions are typical Javascript sort
			 * functions, taking two parameters:
			 *
		     *  1. `{*}` Data to compare to the second parameter
		     *  2. `{*}` Data to compare to the first parameter
			 *
			 * And returning:
			 *
			 * * `{*}` Ordering match: <0 if first parameter should be sorted lower
			 *   than the second parameter, ===0 if the two parameters are equal and
			 *   >0 if the first parameter should be sorted height than the second
			 *   parameter.
			 * 
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    // Numeric ordering of formatted numbers with a pre-formatter
			 *    $.extend( $.fn.dataTable.ext.type.order, {
			 *      "string-pre": function(x) {
			 *        a = (a === "-" || a === "") ? 0 : a.replace( /[^\d\-\.]/g, "" );
			 *        return parseFloat( a );
			 *      }
			 *    } );
			 *
			 *  @example
			 *    // Case-sensitive string ordering, with no pre-formatting method
			 *    $.extend( $.fn.dataTable.ext.order, {
			 *      "string-case-asc": function(x,y) {
			 *        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
			 *      },
			 *      "string-case-desc": function(x,y) {
			 *        return ((x < y) ? 1 : ((x > y) ? -1 : 0));
			 *      }
			 *    } );
			 */
			order: {}
		},
	
		/**
		 * Unique DataTables instance counter
		 *
		 * @type int
		 * @private
		 */
		_unique: 0,
	
	
		//
		// Depreciated
		// The following properties are retained for backwards compatibility only.
		// The should not be used in new projects and will be removed in a future
		// version
		//
	
		/**
		 * Version check function.
		 *  @type function
		 *  @depreciated Since 1.10
		 */
		fnVersionCheck: DataTable.fnVersionCheck,
	
	
		/**
		 * Index for what 'this' index API functions should use
		 *  @type int
		 *  @deprecated Since v1.10
		 */
		iApiIndex: 0,
	
	
		/**
		 * jQuery UI class container
		 *  @type object
		 *  @deprecated Since v1.10
		 */
		oJUIClasses: {},
	
	
		/**
		 * Software version
		 *  @type string
		 *  @deprecated Since v1.10
		 */
		sVersion: DataTable.version
	};
	
	
	//
	// Backwards compatibility. Alias to pre 1.10 Hungarian notation counter parts
	//
	$.extend( _ext, {
		afnFiltering: _ext.search,
		aTypes:       _ext.type.detect,
		ofnSearch:    _ext.type.search,
		oSort:        _ext.type.order,
		afnSortData:  _ext.order,
		aoFeatures:   _ext.feature,
		oApi:         _ext.internal,
		oStdClasses:  _ext.classes,
		oPagination:  _ext.pager
	} );
	
	
	$.extend( DataTable.ext.classes, {
		"sTable": "dataTable",
		"sNoFooter": "no-footer",
	
		/* Paging buttons */
		"sPageButton": "paginate_button",
		"sPageButtonActive": "current",
		"sPageButtonDisabled": "disabled",
	
		/* Striping classes */
		"sStripeOdd": "odd",
		"sStripeEven": "even",
	
		/* Empty row */
		"sRowEmpty": "dataTables_empty",
	
		/* Features */
		"sWrapper": "dataTables_wrapper",
		"sFilter": "dataTables_filter",
		"sInfo": "dataTables_info",
		"sPaging": "dataTables_paginate paging_", /* Note that the type is postfixed */
		"sLength": "dataTables_length",
		"sProcessing": "dataTables_processing",
	
		/* Sorting */
		"sSortAsc": "sorting_asc",
		"sSortDesc": "sorting_desc",
		"sSortable": "sorting", /* Sortable in both directions */
		"sSortableAsc": "sorting_desc_disabled",
		"sSortableDesc": "sorting_asc_disabled",
		"sSortableNone": "sorting_disabled",
		"sSortColumn": "sorting_", /* Note that an int is postfixed for the sorting order */
	
		/* Filtering */
		"sFilterInput": "",
	
		/* Page length */
		"sLengthSelect": "",
	
		/* Scrolling */
		"sScrollWrapper": "dataTables_scroll",
		"sScrollHead": "dataTables_scrollHead",
		"sScrollHeadInner": "dataTables_scrollHeadInner",
		"sScrollBody": "dataTables_scrollBody",
		"sScrollFoot": "dataTables_scrollFoot",
		"sScrollFootInner": "dataTables_scrollFootInner",
	
		/* Misc */
		"sHeaderTH": "",
		"sFooterTH": "",
	
		// Deprecated
		"sSortJUIAsc": "",
		"sSortJUIDesc": "",
		"sSortJUI": "",
		"sSortJUIAscAllowed": "",
		"sSortJUIDescAllowed": "",
		"sSortJUIWrapper": "",
		"sSortIcon": "",
		"sJUIHeader": "",
		"sJUIFooter": ""
	} );
	
	
	var extPagination = DataTable.ext.pager;
	
	function _numbers ( page, pages ) {
		var
			numbers = [],
			buttons = extPagination.numbers_length,
			half = Math.floor( buttons / 2 ),
			i = 1;
	
		if ( pages <= buttons ) {
			numbers = _range( 0, pages );
		}
		else if ( page <= half ) {
			numbers = _range( 0, buttons-2 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
		}
		else if ( page >= pages - 1 - half ) {
			numbers = _range( pages-(buttons-2), pages );
			numbers.splice( 0, 0, 'ellipsis' ); // no unshift in ie6
			numbers.splice( 0, 0, 0 );
		}
		else {
			numbers = _range( page-half+2, page+half-1 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
			numbers.splice( 0, 0, 'ellipsis' );
			numbers.splice( 0, 0, 0 );
		}
	
		numbers.DT_el = 'span';
		return numbers;
	}
	
	
	$.extend( extPagination, {
		simple: function ( page, pages ) {
			return [ 'previous', 'next' ];
		},
	
		full: function ( page, pages ) {
			return [  'first', 'previous', 'next', 'last' ];
		},
	
		numbers: function ( page, pages ) {
			return [ _numbers(page, pages) ];
		},
	
		simple_numbers: function ( page, pages ) {
			return [ 'previous', _numbers(page, pages), 'next' ];
		},
	
		full_numbers: function ( page, pages ) {
			return [ 'first', 'previous', _numbers(page, pages), 'next', 'last' ];
		},
		
		first_last_numbers: function (page, pages) {
	 		return ['first', _numbers(page, pages), 'last'];
	 	},
	
		// For testing and plug-ins to use
		_numbers: _numbers,
	
		// Number of number buttons (including ellipsis) to show. _Must be odd!_
		numbers_length: 7
	} );
	
	
	$.extend( true, DataTable.ext.renderer, {
		pageButton: {
			_: function ( settings, host, idx, buttons, page, pages ) {
				var classes = settings.oClasses;
				var lang = settings.oLanguage.oPaginate;
				var aria = settings.oLanguage.oAria.paginate || {};
				var btnDisplay, btnClass, counter=0;
	
				var attach = function( container, buttons ) {
					var i, ien, node, button, tabIndex;
					var disabledClass = classes.sPageButtonDisabled;
					var clickHandler = function ( e ) {
						_fnPageChange( settings, e.data.action, true );
					};
	
					for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
						button = buttons[i];
	
						if ( Array.isArray( button ) ) {
							var inner = $( '<'+(button.DT_el || 'div')+'/>' )
								.appendTo( container );
							attach( inner, button );
						}
						else {
							btnDisplay = null;
							btnClass = button;
							tabIndex = settings.iTabIndex;
	
							switch ( button ) {
								case 'ellipsis':
									container.append('<span class="ellipsis">&#x2026;</span>');
									break;
	
								case 'first':
									btnDisplay = lang.sFirst;
	
									if ( page === 0 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'previous':
									btnDisplay = lang.sPrevious;
	
									if ( page === 0 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'next':
									btnDisplay = lang.sNext;
	
									if ( pages === 0 || page === pages-1 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'last':
									btnDisplay = lang.sLast;
	
									if ( pages === 0 || page === pages-1 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								default:
									btnDisplay = settings.fnFormatNumber( button + 1 );
									btnClass = page === button ?
										classes.sPageButtonActive : '';
									break;
							}
	
							if ( btnDisplay !== null ) {
								node = $('<a>', {
										'class': classes.sPageButton+' '+btnClass,
										'aria-controls': settings.sTableId,
										'aria-label': aria[ button ],
										'data-dt-idx': counter,
										'tabindex': tabIndex,
										'id': idx === 0 && typeof button === 'string' ?
											settings.sTableId +'_'+ button :
											null
									} )
									.html( btnDisplay )
									.appendTo( container );
	
								_fnBindAction(
									node, {action: button}, clickHandler
								);
	
								counter++;
							}
						}
					}
				};
	
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame. Try / catch the error. Not good for
				// accessibility, but neither are frames.
				var activeEl;
	
				try {
					// Because this approach is destroying and recreating the paging
					// elements, focus is lost on the select button which is bad for
					// accessibility. So we want to restore focus once the draw has
					// completed
					activeEl = $(host).find(document.activeElement).data('dt-idx');
				}
				catch (e) {}
	
				attach( $(host).empty(), buttons );
	
				if ( activeEl !== undefined ) {
					$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
				}
			}
		}
	} );
	
	
	
	// Built in type detection. See model.ext.aTypes for information about
	// what is required from this methods.
	$.extend( DataTable.ext.type.detect, [
		// Plain numbers - first since V8 detects some plain numbers as dates
		// e.g. Date.parse('55') (but not all, e.g. Date.parse('22')...).
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal ) ? 'num'+decimal : null;
		},
	
		// Dates (only those recognised by the browser's Date.parse)
		function ( d, settings )
		{
			// V8 tries _very_ hard to make a string passed into `Date.parse()`
			// valid, so we need to use a regex to restrict date formats. Use a
			// plug-in for anything other than ISO8601 style strings
			if ( d && !(d instanceof Date) && ! _re_date.test(d) ) {
				return null;
			}
			var parsed = Date.parse(d);
			return (parsed !== null && !isNaN(parsed)) || _empty(d) ? 'date' : null;
		},
	
		// Formatted numbers
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal, true ) ? 'num-fmt'+decimal : null;
		},
	
		// HTML numeric
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal ) ? 'html-num'+decimal : null;
		},
	
		// HTML numeric, formatted
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal, true ) ? 'html-num-fmt'+decimal : null;
		},
	
		// HTML (this is strict checking - there must be html)
		function ( d, settings )
		{
			return _empty( d ) || (typeof d === 'string' && d.indexOf('<') !== -1) ?
				'html' : null;
		}
	] );
	
	
	
	// Filter formatting functions. See model.ext.ofnSearch for information about
	// what is required from these methods.
	// 
	// Note that additional search methods are added for the html numbers and
	// html formatted numbers by `_addNumericSort()` when we know what the decimal
	// place is
	
	
	$.extend( DataTable.ext.type.search, {
		html: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data
						.replace( _re_new_lines, " " )
						.replace( _re_html, "" ) :
					'';
		},
	
		string: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data.replace( _re_new_lines, " " ) :
					data;
		}
	} );
	
	
	
	var __numericReplace = function ( d, decimalPlace, re1, re2 ) {
		if ( d !== 0 && (!d || d === '-') ) {
			return -Infinity;
		}
	
		// If a decimal place other than `.` is used, it needs to be given to the
		// function so we can detect it and replace with a `.` which is the only
		// decimal place Javascript recognises - it is not locale aware.
		if ( decimalPlace ) {
			d = _numToDecimal( d, decimalPlace );
		}
	
		if ( d.replace ) {
			if ( re1 ) {
				d = d.replace( re1, '' );
			}
	
			if ( re2 ) {
				d = d.replace( re2, '' );
			}
		}
	
		return d * 1;
	};
	
	
	// Add the numeric 'deformatting' functions for sorting and search. This is done
	// in a function to provide an easy ability for the language options to add
	// additional methods if a non-period decimal place is used.
	function _addNumericSort ( decimalPlace ) {
		$.each(
			{
				// Plain numbers
				"num": function ( d ) {
					return __numericReplace( d, decimalPlace );
				},
	
				// Formatted numbers
				"num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_formatted_numeric );
				},
	
				// HTML numeric
				"html-num": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html );
				},
	
				// HTML numeric, formatted
				"html-num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html, _re_formatted_numeric );
				}
			},
			function ( key, fn ) {
				// Add the ordering method
				_ext.type.order[ key+decimalPlace+'-pre' ] = fn;
	
				// For HTML types add a search formatter that will strip the HTML
				if ( key.match(/^html\-/) ) {
					_ext.type.search[ key+decimalPlace ] = _ext.type.search.html;
				}
			}
		);
	}
	
	
	// Default sort methods
	$.extend( _ext.type.order, {
		// Dates
		"date-pre": function ( d ) {
			var ts = Date.parse( d );
			return isNaN(ts) ? -Infinity : ts;
		},
	
		// html
		"html-pre": function ( a ) {
			return _empty(a) ?
				'' :
				a.replace ?
					a.replace( /<.*?>/g, "" ).toLowerCase() :
					a+'';
		},
	
		// string
		"string-pre": function ( a ) {
			// This is a little complex, but faster than always calling toString,
			// http://jsperf.com/tostring-v-check
			return _empty(a) ?
				'' :
				typeof a === 'string' ?
					a.toLowerCase() :
					! a.toString ?
						'' :
						a.toString();
		},
	
		// string-asc and -desc are retained only for compatibility with the old
		// sort methods
		"string-asc": function ( x, y ) {
			return ((x < y) ? -1 : ((x > y) ? 1 : 0));
		},
	
		"string-desc": function ( x, y ) {
			return ((x < y) ? 1 : ((x > y) ? -1 : 0));
		}
	} );
	
	
	// Numeric sorting types - order doesn't matter here
	_addNumericSort( '' );
	
	
	$.extend( true, DataTable.ext.renderer, {
		header: {
			_: function ( settings, cell, column, classes ) {
				// No additional mark-up required
				// Attach a sort listener to update on sort - note that using the
				// `DT` namespace will allow the event to be removed automatically
				// on destroy, while the `dt` namespaced event is the one we are
				// listening for
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) { // need to check this this is the host
						return;               // table, not a nested one
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass(
							classes.sSortAsc +' '+
							classes.sSortDesc
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
				} );
			},
	
			jqueryui: function ( settings, cell, column, classes ) {
				$('<div/>')
					.addClass( classes.sSortJUIWrapper )
					.append( cell.contents() )
					.append( $('<span/>')
						.addClass( classes.sSortIcon+' '+column.sSortingClassJUI )
					)
					.appendTo( cell );
	
				// Attach a sort listener to update on sort
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) {
						return;
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass( classes.sSortAsc +" "+classes.sSortDesc )
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
	
					cell
						.find( 'span.'+classes.sSortIcon )
						.removeClass(
							classes.sSortJUIAsc +" "+
							classes.sSortJUIDesc +" "+
							classes.sSortJUI +" "+
							classes.sSortJUIAscAllowed +" "+
							classes.sSortJUIDescAllowed
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortJUIAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortJUIDesc :
								column.sSortingClassJUI
						);
				} );
			}
		}
	} );
	
	/*
	 * Public helper functions. These aren't used internally by DataTables, or
	 * called by any of the options passed into DataTables, but they can be used
	 * externally by developers working with DataTables. They are helper functions
	 * to make working with DataTables a little bit easier.
	 */
	
	var __htmlEscapeEntities = function ( d ) {
		if (Array.isArray(d)) {
			d = d.join(',');
		}
	
		return typeof d === 'string' ?
			d
				.replace(/&/g, '&amp;')
				.replace(/</g, '&lt;')
				.replace(/>/g, '&gt;')
				.replace(/"/g, '&quot;') :
			d;
	};
	
	// Common logic for moment, luxon or a date action
	function __mld( dt, momentFn, luxonFn, dateFn, arg1 ) {
		if (window.moment) {
			return dt[momentFn]( arg1 );
		}
		else if (window.luxon) {
			return dt[luxonFn]( arg1 );
		}
		
		return dateFn ? dt[dateFn]( arg1 ) : dt;
	}
	
	
	var __mlWarning = false;
	function __mldObj (d, format, locale) {
		var dt;
	
		if (window.moment) {
			dt = window.moment.utc( d, format, locale, true );
	
			if (! dt.isValid()) {
				return null;
			}
		}
		else if (window.luxon) {
			dt = format
				? window.luxon.DateTime.fromFormat( d, format )
				: window.luxon.DateTime.fromISO( d );
	
			if (! dt.isValid) {
				return null;
			}
	
			dt.setLocale(locale);
		}
		else if (! format) {
			// No format given, must be ISO
			dt = new Date(d);
		}
		else {
			if (! __mlWarning) {
				alert('DataTables warning: Formatted date without Moment.js or Luxon - https://datatables.net/tn/17');
			}
	
			__mlWarning = true;
		}
	
		return dt;
	}
	
	// Wrapper for date, datetime and time which all operate the same way with the exception of
	// the output string for auto locale support
	function __mlHelper (localeString) {
		return function ( from, to, locale, def ) {
			// Luxon and Moment support
			// Argument shifting
			if ( arguments.length === 0 ) {
				locale = 'en';
				to = null; // means toLocaleString
				from = null; // means iso8601
			}
			else if ( arguments.length === 1 ) {
				locale = 'en';
				to = from;
				from = null;
			}
			else if ( arguments.length === 2 ) {
				locale = to;
				to = from;
				from = null;
			}
	
			var typeName = 'datetime-' + to;
	
			// Add type detection and sorting specific to this date format - we need to be able to identify
			// date type columns as such, rather than as numbers in extensions. Hence the need for this.
			if (! DataTable.ext.type.order[typeName]) {
				// The renderer will give the value to type detect as the type!
				DataTable.ext.type.detect.unshift(function (d) {
					return d === typeName ? typeName : false;
				});
	
				// The renderer gives us Moment, Luxon or Date obects for the sorting, all of which have a
				// `valueOf` which gives milliseconds epoch
				DataTable.ext.type.order[typeName + '-asc'] = function (a, b) {
					var x = a.valueOf();
					var y = b.valueOf();
	
					return x === y
						? 0
						: x < y
							? -1
							: 1;
				}
	
				DataTable.ext.type.order[typeName + '-desc'] = function (a, b) {
					var x = a.valueOf();
					var y = b.valueOf();
	
					return x === y
						? 0
						: x > y
							? -1
							: 1;
				}
			}
		
			return function ( d, type ) {
				// Allow for a default value
				if (d === null || d === undefined) {
					if (def === '--now') {
						// We treat everything as UTC further down, so no changes are
						// made, as such need to get the local date / time as if it were
						// UTC
						var local = new Date();
						d = new Date( Date.UTC(
							local.getFullYear(), local.getMonth(), local.getDate(),
							local.getHours(), local.getMinutes(), local.getSeconds()
						) );
					}
					else {
						d = '';
					}
				}
	
				if (type === 'type') {
					// Typing uses the type name for fast matching
					return typeName;
				}
	
				if (d === '') {
					return type !== 'sort'
						? ''
						: __mldObj('0000-01-01 00:00:00', null, locale);
				}
	
				// Shortcut. If `from` and `to` are the same, we are using the renderer to
				// format for ordering, not display - its already in the display format.
				if ( to !== null && from === to && type !== 'sort' && type !== 'type' && ! (d instanceof Date) ) {
					return d;
				}
	
				var dt = __mldObj(d, from, locale);
	
				if (dt === null) {
					return d;
				}
	
				if (type === 'sort') {
					return dt;
				}
				
				var formatted = to === null
					? __mld(dt, 'toDate', 'toJSDate', '')[localeString]()
					: __mld(dt, 'format', 'toFormat', 'toISOString', to);
	
				// XSS protection
				return type === 'display' ?
					__htmlEscapeEntities( formatted ) :
					formatted;
			};
		}
	}
	
	// Based on locale, determine standard number formatting
	// Fallback for legacy browsers is US English
	var __thousands = ',';
	var __decimal = '.';
	
	if (Intl) {
		try {
			var num = new Intl.NumberFormat().formatToParts(100000.1);
		
			for (var i=0 ; i<num.length ; i++) {
				if (num[i].type === 'group') {
					__thousands = num[i].value;
				}
				else if (num[i].type === 'decimal') {
					__decimal = num[i].value;
				}
			}
		}
		catch (e) {
			// noop
		}
	}
	
	// Formatted date time detection - use by declaring the formats you are going to use
	DataTable.datetime = function ( format, locale ) {
		var typeName = 'datetime-detect-' + format;
	
		if (! locale) {
			locale = 'en';
		}
	
		if (! DataTable.ext.type.order[typeName]) {
			DataTable.ext.type.detect.unshift(function (d) {
				var dt = __mldObj(d, format, locale);
				return d === '' || dt ? typeName : false;
			});
	
			DataTable.ext.type.order[typeName + '-pre'] = function (d) {
				return __mldObj(d, format, locale) || 0;
			}
		}
	}
	
	/**
	 * Helpers for `columns.render`.
	 *
	 * The options defined here can be used with the `columns.render` initialisation
	 * option to provide a display renderer. The following functions are defined:
	 *
	 * * `number` - Will format numeric data (defined by `columns.data`) for
	 *   display, retaining the original unformatted data for sorting and filtering.
	 *   It takes 5 parameters:
	 *   * `string` - Thousands grouping separator
	 *   * `string` - Decimal point indicator
	 *   * `integer` - Number of decimal points to show
	 *   * `string` (optional) - Prefix.
	 *   * `string` (optional) - Postfix (/suffix).
	 * * `text` - Escape HTML to help prevent XSS attacks. It has no optional
	 *   parameters.
	 *
	 * @example
	 *   // Column definition using the number renderer
	 *   {
	 *     data: "salary",
	 *     render: $.fn.dataTable.render.number( '\'', '.', 0, '$' )
	 *   }
	 *
	 * @namespace
	 */
	DataTable.render = {
		date: __mlHelper('toLocaleDateString'),
		datetime: __mlHelper('toLocaleString'),
		time: __mlHelper('toLocaleTimeString'),
		number: function ( thousands, decimal, precision, prefix, postfix ) {
			// Auto locale detection
			if (thousands === null || thousands === undefined) {
				thousands = __thousands;
			}
	
			if (decimal === null || decimal === undefined) {
				decimal = __decimal;
			}
	
			return {
				display: function ( d ) {
					if ( typeof d !== 'number' && typeof d !== 'string' ) {
						return d;
					}
	
					if (d === '' || d === null) {
						return d;
					}
	
					var negative = d < 0 ? '-' : '';
					var flo = parseFloat( d );
	
					// If NaN then there isn't much formatting that we can do - just
					// return immediately, escaping any HTML (this was supposed to
					// be a number after all)
					if ( isNaN( flo ) ) {
						return __htmlEscapeEntities( d );
					}
	
					flo = flo.toFixed( precision );
					d = Math.abs( flo );
	
					var intPart = parseInt( d, 10 );
					var floatPart = precision ?
						decimal+(d - intPart).toFixed( precision ).substring( 2 ):
						'';
	
					// If zero, then can't have a negative prefix
					if (intPart === 0 && parseFloat(floatPart) === 0) {
						negative = '';
					}
	
					return negative + (prefix||'') +
						intPart.toString().replace(
							/\B(?=(\d{3})+(?!\d))/g, thousands
						) +
						floatPart +
						(postfix||'');
				}
			};
		},
	
		text: function () {
			return {
				display: __htmlEscapeEntities,
				filter: __htmlEscapeEntities
			};
		}
	};
	
	
	/*
	 * This is really a good bit rubbish this method of exposing the internal methods
	 * publicly... - To be fixed in 2.0 using methods on the prototype
	 */
	
	
	/**
	 * Create a wrapper function for exporting an internal functions to an external API.
	 *  @param {string} fn API function name
	 *  @returns {function} wrapped function
	 *  @memberof DataTable#internal
	 */
	function _fnExternApiFunc (fn)
	{
		return function() {
			var args = [_fnSettingsFromNode( this[DataTable.ext.iApiIndex] )].concat(
				Array.prototype.slice.call(arguments)
			);
			return DataTable.ext.internal[fn].apply( this, args );
		};
	}
	
	
	/**
	 * Reference to internal functions for use by plug-in developers. Note that
	 * these methods are references to internal functions and are considered to be
	 * private. If you use these methods, be aware that they are liable to change
	 * between versions.
	 *  @namespace
	 */
	$.extend( DataTable.ext.internal, {
		_fnExternApiFunc: _fnExternApiFunc,
		_fnBuildAjax: _fnBuildAjax,
		_fnAjaxUpdate: _fnAjaxUpdate,
		_fnAjaxParameters: _fnAjaxParameters,
		_fnAjaxUpdateDraw: _fnAjaxUpdateDraw,
		_fnAjaxDataSrc: _fnAjaxDataSrc,
		_fnAddColumn: _fnAddColumn,
		_fnColumnOptions: _fnColumnOptions,
		_fnAdjustColumnSizing: _fnAdjustColumnSizing,
		_fnVisibleToColumnIndex: _fnVisibleToColumnIndex,
		_fnColumnIndexToVisible: _fnColumnIndexToVisible,
		_fnVisbleColumns: _fnVisbleColumns,
		_fnGetColumns: _fnGetColumns,
		_fnColumnTypes: _fnColumnTypes,
		_fnApplyColumnDefs: _fnApplyColumnDefs,
		_fnHungarianMap: _fnHungarianMap,
		_fnCamelToHungarian: _fnCamelToHungarian,
		_fnLanguageCompat: _fnLanguageCompat,
		_fnBrowserDetect: _fnBrowserDetect,
		_fnAddData: _fnAddData,
		_fnAddTr: _fnAddTr,
		_fnNodeToDataIndex: _fnNodeToDataIndex,
		_fnNodeToColumnIndex: _fnNodeToColumnIndex,
		_fnGetCellData: _fnGetCellData,
		_fnSetCellData: _fnSetCellData,
		_fnSplitObjNotation: _fnSplitObjNotation,
		_fnGetObjectDataFn: _fnGetObjectDataFn,
		_fnSetObjectDataFn: _fnSetObjectDataFn,
		_fnGetDataMaster: _fnGetDataMaster,
		_fnClearTable: _fnClearTable,
		_fnDeleteIndex: _fnDeleteIndex,
		_fnInvalidate: _fnInvalidate,
		_fnGetRowElements: _fnGetRowElements,
		_fnCreateTr: _fnCreateTr,
		_fnBuildHead: _fnBuildHead,
		_fnDrawHead: _fnDrawHead,
		_fnDraw: _fnDraw,
		_fnReDraw: _fnReDraw,
		_fnAddOptionsHtml: _fnAddOptionsHtml,
		_fnDetectHeader: _fnDetectHeader,
		_fnGetUniqueThs: _fnGetUniqueThs,
		_fnFeatureHtmlFilter: _fnFeatureHtmlFilter,
		_fnFilterComplete: _fnFilterComplete,
		_fnFilterCustom: _fnFilterCustom,
		_fnFilterColumn: _fnFilterColumn,
		_fnFilter: _fnFilter,
		_fnFilterCreateSearch: _fnFilterCreateSearch,
		_fnEscapeRegex: _fnEscapeRegex,
		_fnFilterData: _fnFilterData,
		_fnFeatureHtmlInfo: _fnFeatureHtmlInfo,
		_fnUpdateInfo: _fnUpdateInfo,
		_fnInfoMacros: _fnInfoMacros,
		_fnInitialise: _fnInitialise,
		_fnInitComplete: _fnInitComplete,
		_fnLengthChange: _fnLengthChange,
		_fnFeatureHtmlLength: _fnFeatureHtmlLength,
		_fnFeatureHtmlPaginate: _fnFeatureHtmlPaginate,
		_fnPageChange: _fnPageChange,
		_fnFeatureHtmlProcessing: _fnFeatureHtmlProcessing,
		_fnProcessingDisplay: _fnProcessingDisplay,
		_fnFeatureHtmlTable: _fnFeatureHtmlTable,
		_fnScrollDraw: _fnScrollDraw,
		_fnApplyToChildren: _fnApplyToChildren,
		_fnCalculateColumnWidths: _fnCalculateColumnWidths,
		_fnThrottle: _fnThrottle,
		_fnConvertToWidth: _fnConvertToWidth,
		_fnGetWidestNode: _fnGetWidestNode,
		_fnGetMaxLenString: _fnGetMaxLenString,
		_fnStringToCss: _fnStringToCss,
		_fnSortFlatten: _fnSortFlatten,
		_fnSort: _fnSort,
		_fnSortAria: _fnSortAria,
		_fnSortListener: _fnSortListener,
		_fnSortAttachListener: _fnSortAttachListener,
		_fnSortingClasses: _fnSortingClasses,
		_fnSortData: _fnSortData,
		_fnSaveState: _fnSaveState,
		_fnLoadState: _fnLoadState,
		_fnImplementState: _fnImplementState,
		_fnSettingsFromNode: _fnSettingsFromNode,
		_fnLog: _fnLog,
		_fnMap: _fnMap,
		_fnBindAction: _fnBindAction,
		_fnCallbackReg: _fnCallbackReg,
		_fnCallbackFire: _fnCallbackFire,
		_fnLengthOverflow: _fnLengthOverflow,
		_fnRenderer: _fnRenderer,
		_fnDataSource: _fnDataSource,
		_fnRowAttributes: _fnRowAttributes,
		_fnExtend: _fnExtend,
		_fnCalculateEnd: function () {} // Used by a lot of plug-ins, but redundant
		                                // in 1.10, so this dead-end function is
		                                // added to prevent errors
	} );
	
	
	// jQuery access
	$.fn.dataTable = DataTable;
	
	// Provide access to the host jQuery object (circular reference)
	DataTable.$ = $;
	
	// Legacy aliases
	$.fn.dataTableSettings = DataTable.settings;
	$.fn.dataTableExt = DataTable.ext;
	
	// With a capital `D` we return a DataTables API instance rather than a
	// jQuery object
	$.fn.DataTable = function ( opts ) {
		return $(this).dataTable( opts ).api();
	};
	
	// All properties that are available to $.fn.dataTable should also be
	// available on $.fn.DataTable
	$.each( DataTable, function ( prop, val ) {
		$.fn.DataTable[ prop ] = val;
	} );
	
	return DataTable;
}));

/*! DataTables Bootstrap 5 integration
 * 2020 SpryMedia Ltd - datatables.net/license
 */

/**
 * DataTables integration for Bootstrap 4. This requires Bootstrap 5 and
 * DataTables 1.10 or newer.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See http://datatables.net/manual/styling/bootstrap
 * for further information.
 */
(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				// Require DataTables, which attaches to jQuery, including
				// jQuery if needed and have a $ property so we can access the
				// jQuery object that is used
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	dom:
		"<'row'<'col-sm-12 col-md-6'l><'col-sm-12 col-md-6'f>>" +
		"<'row'<'col-sm-12'tr>>" +
		"<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7'p>>",
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( DataTable.ext.classes, {
	sWrapper:      "dataTables_wrapper dt-bootstrap5",
	sFilterInput:  "form-control form-control-sm",
	sLengthSelect: "form-select form-select-sm",
	sProcessing:   "dataTables_processing card",
	sPageButton:   "paginate_button page-item"
} );


/* Bootstrap paging button renderer */
DataTable.ext.renderer.pageButton.bootstrap = function ( settings, host, idx, buttons, page, pages ) {
	var api     = new DataTable.Api( settings );
	var classes = settings.oClasses;
	var lang    = settings.oLanguage.oPaginate;
	var aria = settings.oLanguage.oAria.paginate || {};
	var btnDisplay, btnClass, counter=0;

	var attach = function( container, buttons ) {
		var i, ien, node, button;
		var clickHandler = function ( e ) {
			e.preventDefault();
			if ( !$(e.currentTarget).hasClass('disabled') && api.page() != e.data.action ) {
				api.page( e.data.action ).draw( 'page' );
			}
		};

		for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
			button = buttons[i];

			if ( Array.isArray( button ) ) {
				attach( container, button );
			}
			else {
				btnDisplay = '';
				btnClass = '';

				switch ( button ) {
					case 'ellipsis':
						btnDisplay = '&#x2026;';
						btnClass = 'disabled';
						break;

					case 'first':
						btnDisplay = lang.sFirst;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'previous':
						btnDisplay = lang.sPrevious;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'next':
						btnDisplay = lang.sNext;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					case 'last':
						btnDisplay = lang.sLast;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					default:
						btnDisplay = button + 1;
						btnClass = page === button ?
							'active' : '';
						break;
				}

				if ( btnDisplay ) {
					node = $('<li>', {
							'class': classes.sPageButton+' '+btnClass,
							'id': idx === 0 && typeof button === 'string' ?
								settings.sTableId +'_'+ button :
								null
						} )
						.append( $('<a>', {
								'href': '#',
								'aria-controls': settings.sTableId,
								'aria-label': aria[ button ],
								'data-dt-idx': counter,
								'tabindex': settings.iTabIndex,
								'class': 'page-link'
							} )
							.html( btnDisplay )
						)
						.appendTo( container );

					settings.oApi._fnBindAction(
						node, {action: button}, clickHandler
					);

					counter++;
				}
			}
		}
	};

	// IE9 throws an 'unknown error' if document.activeElement is used
	// inside an iframe or frame. 
	var activeEl;

	try {
		// Because this approach is destroying and recreating the paging
		// elements, focus is lost on the select button which is bad for
		// accessibility. So we want to restore focus once the draw has
		// completed
		activeEl = $(host).find(document.activeElement).data('dt-idx');
	}
	catch (e) {}

	attach(
		$(host).empty().html('<ul class="pagination"/>').children('ul'),
		buttons
	);

	if ( activeEl !== undefined ) {
		$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
	}
};


return DataTable;
}));

"use strict";

//
// Datatables.net Initialization
//

// Set Defaults

var defaults = {
	"language": {		
		"info": "Showing _START_ to _END_ of _TOTAL_ records",
    	"infoEmpty": "Showing no records",
		"lengthMenu": "_MENU_",
		"processing": '<span class="spinner-border w-15px h-15px text-muted align-middle me-2"></span> <span class="text-gray-600">Loading...</span>',
		"paginate": {
			"first": '<i class="first"></i>',
			"last": '<i class="last"></i>',
			"next": '<i class="next"></i>',
			"previous": '<i class="previous"></i>'
		}
	}
};

$.extend(true, $.fn.dataTable.defaults, defaults);

/*! DataTables Bootstrap 4 integration
 * ©2011-2017 SpryMedia Ltd - datatables.net/license
 */

/**
 * DataTables integration for Bootstrap 4. This requires Bootstrap 4 and
 * DataTables 1.10 or newer.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See http://datatables.net/manual/styling/bootstrap
 * for further information.
 */
(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				// Require DataTables, which attaches to jQuery, including
				// jQuery if needed and have a $ property so we can access the
				// jQuery object that is used
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	dom: 
		"<'table-responsive'tr>" +
		
		"<'row'" + 
		"<'col-sm-12 col-md-5 d-flex align-items-center justify-content-center justify-content-md-start'li>" + 
		"<'col-sm-12 col-md-7 d-flex align-items-center justify-content-center justify-content-md-end'p>" +
		">",
	
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( DataTable.ext.classes, {
	sWrapper:      "dataTables_wrapper dt-bootstrap4",
	sFilterInput:  "form-control form-control-sm form-control-solid",
	sLengthSelect: "form-select form-select-sm form-select-solid",
	sProcessing:   "dataTables_processing",
	sPageButton:   "paginate_button page-item"
} );


/* Bootstrap paging button renderer */
DataTable.ext.renderer.pageButton.bootstrap = function ( settings, host, idx, buttons, page, pages ) {
	var api     = new DataTable.Api( settings );
	var classes = settings.oClasses;
	var lang    = settings.oLanguage.oPaginate;
	var aria = settings.oLanguage.oAria.paginate || {};
	var btnDisplay, btnClass, counter=0;

	var attach = function( container, buttons ) {
		var i, ien, node, button;
		var clickHandler = function ( e ) {
			e.preventDefault();
			if ( !$(e.currentTarget).hasClass('disabled') && api.page() != e.data.action ) {
				api.page( e.data.action ).draw( 'page' );
			}
		};

		for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
			button = buttons[i];

			if ( Array.isArray( button ) ) {
				attach( container, button );
			}
			else {
				btnDisplay = '';
				btnClass = '';

				switch ( button ) {
					case 'ellipsis':
						btnDisplay = '&#x2026;';
						btnClass = 'disabled';
						break;

					case 'first':
						btnDisplay = lang.sFirst;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'previous':
						btnDisplay = lang.sPrevious;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'next':
						btnDisplay = lang.sNext;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					case 'last':
						btnDisplay = lang.sLast;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					default:
						btnDisplay = button + 1;
						btnClass = page === button ?
							'active' : '';
						break;
				}

				if ( btnDisplay ) {
					node = $('<li>', {
							'class': classes.sPageButton+' '+btnClass,
							'id': idx === 0 && typeof button === 'string' ?
								settings.sTableId +'_'+ button :
								null
						} )
						.append( $('<a>', {
								'href': '#',
								'aria-controls': settings.sTableId,
								'aria-label': aria[ button ],
								'data-dt-idx': counter,
								'tabindex': settings.iTabIndex,
								'class': 'page-link'
							} )
							.html( btnDisplay )
						)
						.appendTo( container );

					settings.oApi._fnBindAction(
						node, {action: button}, clickHandler
					);

					counter++;
				}
			}
		}
	};

	// IE9 throws an 'unknown error' if document.activeElement is used
	// inside an iframe or frame.
	var activeEl;

	try {
		// Because this approach is destroying and recreating the paging
		// elements, focus is lost on the select button which is bad for
		// accessibility. So we want to restore focus once the draw has
		// completed
		activeEl = $(host).find(document.activeElement).data('dt-idx');
	}
	catch (e) {}

	attach(
		$(host).empty().html('<ul class="pagination"/>').children('ul'),
		buttons
	);

	if ( activeEl !== undefined ) {
		$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
	}
};


return DataTable;
}));

/*!

JSZip v3.10.0 - A JavaScript class for generating and reading zip files
<http://stuartk.com/jszip>

(c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/main/LICENSE.markdown.

JSZip uses the library pako released under the MIT license :
https://github.com/nodeca/pako/blob/main/LICENSE
*/

!function(e){if("object"==typeof exports&&"undefined"!=typeof module)module.exports=e();else if("function"==typeof define&&define.amd)define([],e);else{("undefined"!=typeof window?window:"undefined"!=typeof global?global:"undefined"!=typeof self?self:this).JSZip=e()}}(function(){return function s(a,o,h){function u(r,e){if(!o[r]){if(!a[r]){var t="function"==typeof require&&require;if(!e&&t)return t(r,!0);if(l)return l(r,!0);var n=new Error("Cannot find module '"+r+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[r]={exports:{}};a[r][0].call(i.exports,function(e){var t=a[r][1][e];return u(t||e)},i,i.exports,s,a,o,h)}return o[r].exports}for(var l="function"==typeof require&&require,e=0;e<h.length;e++)u(h[e]);return u}({1:[function(e,t,r){"use strict";var d=e("./utils"),c=e("./support"),p="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";r.encode=function(e){for(var t,r,n,i,s,a,o,h=[],u=0,l=e.length,f=l,c="string"!==d.getTypeOf(e);u<e.length;)f=l-u,n=c?(t=e[u++],r=u<l?e[u++]:0,u<l?e[u++]:0):(t=e.charCodeAt(u++),r=u<l?e.charCodeAt(u++):0,u<l?e.charCodeAt(u++):0),i=t>>2,s=(3&t)<<4|r>>4,a=1<f?(15&r)<<2|n>>6:64,o=2<f?63&n:64,h.push(p.charAt(i)+p.charAt(s)+p.charAt(a)+p.charAt(o));return h.join("")},r.decode=function(e){var t,r,n,i,s,a,o=0,h=0,u="data:";if(e.substr(0,u.length)===u)throw new Error("Invalid base64 input, it looks like a data url.");var l,f=3*(e=e.replace(/[^A-Za-z0-9\+\/\=]/g,"")).length/4;if(e.charAt(e.length-1)===p.charAt(64)&&f--,e.charAt(e.length-2)===p.charAt(64)&&f--,f%1!=0)throw new Error("Invalid base64 input, bad content length.");for(l=c.uint8array?new Uint8Array(0|f):new Array(0|f);o<e.length;)t=p.indexOf(e.charAt(o++))<<2|(i=p.indexOf(e.charAt(o++)))>>4,r=(15&i)<<4|(s=p.indexOf(e.charAt(o++)))>>2,n=(3&s)<<6|(a=p.indexOf(e.charAt(o++))),l[h++]=t,64!==s&&(l[h++]=r),64!==a&&(l[h++]=n);return l}},{"./support":30,"./utils":32}],2:[function(e,t,r){"use strict";var n=e("./external"),i=e("./stream/DataWorker"),s=e("./stream/Crc32Probe"),a=e("./stream/DataLengthProbe");function o(e,t,r,n,i){this.compressedSize=e,this.uncompressedSize=t,this.crc32=r,this.compression=n,this.compressedContent=i}o.prototype={getContentWorker:function(){var e=new i(n.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),t=this;return e.on("end",function(){if(this.streamInfo.data_length!==t.uncompressedSize)throw new Error("Bug : uncompressed data size mismatch")}),e},getCompressedWorker:function(){return new i(n.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize",this.compressedSize).withStreamInfo("uncompressedSize",this.uncompressedSize).withStreamInfo("crc32",this.crc32).withStreamInfo("compression",this.compression)}},o.createWorkerFrom=function(e,t,r){return e.pipe(new s).pipe(new a("uncompressedSize")).pipe(t.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression",t)},t.exports=o},{"./external":6,"./stream/Crc32Probe":25,"./stream/DataLengthProbe":26,"./stream/DataWorker":27}],3:[function(e,t,r){"use strict";var n=e("./stream/GenericWorker");r.STORE={magic:"\0\0",compressWorker:function(e){return new n("STORE compression")},uncompressWorker:function(){return new n("STORE decompression")}},r.DEFLATE=e("./flate")},{"./flate":7,"./stream/GenericWorker":28}],4:[function(e,t,r){"use strict";var n=e("./utils");var o=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t){return void 0!==e&&e.length?"string"!==n.getTypeOf(e)?function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t[a])];return-1^e}(0|t,e,e.length,0):function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t.charCodeAt(a))];return-1^e}(0|t,e,e.length,0):0}},{"./utils":32}],5:[function(e,t,r){"use strict";r.base64=!1,r.binary=!1,r.dir=!1,r.createFolders=!0,r.date=null,r.compression=null,r.compressionOptions=null,r.comment=null,r.unixPermissions=null,r.dosPermissions=null},{}],6:[function(e,t,r){"use strict";var n=null;n="undefined"!=typeof Promise?Promise:e("lie"),t.exports={Promise:n}},{lie:37}],7:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Uint32Array,i=e("pako"),s=e("./utils"),a=e("./stream/GenericWorker"),o=n?"uint8array":"array";function h(e,t){a.call(this,"FlateWorker/"+e),this._pako=null,this._pakoAction=e,this._pakoOptions=t,this.meta={}}r.magic="\b\0",s.inherits(h,a),h.prototype.processChunk=function(e){this.meta=e.meta,null===this._pako&&this._createPako(),this._pako.push(s.transformTo(o,e.data),!1)},h.prototype.flush=function(){a.prototype.flush.call(this),null===this._pako&&this._createPako(),this._pako.push([],!0)},h.prototype.cleanUp=function(){a.prototype.cleanUp.call(this),this._pako=null},h.prototype._createPako=function(){this._pako=new i[this._pakoAction]({raw:!0,level:this._pakoOptions.level||-1});var t=this;this._pako.onData=function(e){t.push({data:e,meta:t.meta})}},r.compressWorker=function(e){return new h("Deflate",e)},r.uncompressWorker=function(){return new h("Inflate",{})}},{"./stream/GenericWorker":28,"./utils":32,pako:38}],8:[function(e,t,r){"use strict";function A(e,t){var r,n="";for(r=0;r<t;r++)n+=String.fromCharCode(255&e),e>>>=8;return n}function n(e,t,r,n,i,s){var a,o,h=e.file,u=e.compression,l=s!==O.utf8encode,f=I.transformTo("string",s(h.name)),c=I.transformTo("string",O.utf8encode(h.name)),d=h.comment,p=I.transformTo("string",s(d)),m=I.transformTo("string",O.utf8encode(d)),_=c.length!==h.name.length,g=m.length!==d.length,b="",v="",y="",w=h.dir,k=h.date,x={crc32:0,compressedSize:0,uncompressedSize:0};t&&!r||(x.crc32=e.crc32,x.compressedSize=e.compressedSize,x.uncompressedSize=e.uncompressedSize);var S=0;t&&(S|=8),l||!_&&!g||(S|=2048);var z=0,C=0;w&&(z|=16),"UNIX"===i?(C=798,z|=function(e,t){var r=e;return e||(r=t?16893:33204),(65535&r)<<16}(h.unixPermissions,w)):(C=20,z|=function(e){return 63&(e||0)}(h.dosPermissions)),a=k.getUTCHours(),a<<=6,a|=k.getUTCMinutes(),a<<=5,a|=k.getUTCSeconds()/2,o=k.getUTCFullYear()-1980,o<<=4,o|=k.getUTCMonth()+1,o<<=5,o|=k.getUTCDate(),_&&(v=A(1,1)+A(B(f),4)+c,b+="up"+A(v.length,2)+v),g&&(y=A(1,1)+A(B(p),4)+m,b+="uc"+A(y.length,2)+y);var E="";return E+="\n\0",E+=A(S,2),E+=u.magic,E+=A(a,2),E+=A(o,2),E+=A(x.crc32,4),E+=A(x.compressedSize,4),E+=A(x.uncompressedSize,4),E+=A(f.length,2),E+=A(b.length,2),{fileRecord:R.LOCAL_FILE_HEADER+E+f+b,dirRecord:R.CENTRAL_FILE_HEADER+A(C,2)+E+A(p.length,2)+"\0\0\0\0"+A(z,4)+A(n,4)+f+b+p}}var I=e("../utils"),i=e("../stream/GenericWorker"),O=e("../utf8"),B=e("../crc32"),R=e("../signature");function s(e,t,r,n){i.call(this,"ZipFileWorker"),this.bytesWritten=0,this.zipComment=t,this.zipPlatform=r,this.encodeFileName=n,this.streamFiles=e,this.accumulate=!1,this.contentBuffer=[],this.dirRecords=[],this.currentSourceOffset=0,this.entriesCount=0,this.currentFile=null,this._sources=[]}I.inherits(s,i),s.prototype.push=function(e){var t=e.meta.percent||0,r=this.entriesCount,n=this._sources.length;this.accumulate?this.contentBuffer.push(e):(this.bytesWritten+=e.data.length,i.prototype.push.call(this,{data:e.data,meta:{currentFile:this.currentFile,percent:r?(t+100*(r-n-1))/r:100}}))},s.prototype.openedSource=function(e){this.currentSourceOffset=this.bytesWritten,this.currentFile=e.file.name;var t=this.streamFiles&&!e.file.dir;if(t){var r=n(e,t,!1,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);this.push({data:r.fileRecord,meta:{percent:0}})}else this.accumulate=!0},s.prototype.closedSource=function(e){this.accumulate=!1;var t=this.streamFiles&&!e.file.dir,r=n(e,t,!0,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);if(this.dirRecords.push(r.dirRecord),t)this.push({data:function(e){return R.DATA_DESCRIPTOR+A(e.crc32,4)+A(e.compressedSize,4)+A(e.uncompressedSize,4)}(e),meta:{percent:100}});else for(this.push({data:r.fileRecord,meta:{percent:0}});this.contentBuffer.length;)this.push(this.contentBuffer.shift());this.currentFile=null},s.prototype.flush=function(){for(var e=this.bytesWritten,t=0;t<this.dirRecords.length;t++)this.push({data:this.dirRecords[t],meta:{percent:100}});var r=this.bytesWritten-e,n=function(e,t,r,n,i){var s=I.transformTo("string",i(n));return R.CENTRAL_DIRECTORY_END+"\0\0\0\0"+A(e,2)+A(e,2)+A(t,4)+A(r,4)+A(s.length,2)+s}(this.dirRecords.length,r,e,this.zipComment,this.encodeFileName);this.push({data:n,meta:{percent:100}})},s.prototype.prepareNextSource=function(){this.previous=this._sources.shift(),this.openedSource(this.previous.streamInfo),this.isPaused?this.previous.pause():this.previous.resume()},s.prototype.registerPrevious=function(e){this._sources.push(e);var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.closedSource(t.previous.streamInfo),t._sources.length?t.prepareNextSource():t.end()}),e.on("error",function(e){t.error(e)}),this},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(!this.previous&&this._sources.length?(this.prepareNextSource(),!0):this.previous||this._sources.length||this.generatedError?void 0:(this.end(),!0))},s.prototype.error=function(e){var t=this._sources;if(!i.prototype.error.call(this,e))return!1;for(var r=0;r<t.length;r++)try{t[r].error(e)}catch(e){}return!0},s.prototype.lock=function(){i.prototype.lock.call(this);for(var e=this._sources,t=0;t<e.length;t++)e[t].lock()},t.exports=s},{"../crc32":4,"../signature":23,"../stream/GenericWorker":28,"../utf8":31,"../utils":32}],9:[function(e,t,r){"use strict";var u=e("../compressions"),n=e("./ZipFileWorker");r.generateWorker=function(e,a,t){var o=new n(a.streamFiles,t,a.platform,a.encodeFileName),h=0;try{e.forEach(function(e,t){h++;var r=function(e,t){var r=e||t,n=u[r];if(!n)throw new Error(r+" is not a valid compression method !");return n}(t.options.compression,a.compression),n=t.options.compressionOptions||a.compressionOptions||{},i=t.dir,s=t.date;t._compressWorker(r,n).withStreamInfo("file",{name:e,dir:i,date:s,comment:t.comment||"",unixPermissions:t.unixPermissions,dosPermissions:t.dosPermissions}).pipe(o)}),o.entriesCount=h}catch(e){o.error(e)}return o}},{"../compressions":3,"./ZipFileWorker":8}],10:[function(e,t,r){"use strict";function n(){if(!(this instanceof n))return new n;if(arguments.length)throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");this.files=Object.create(null),this.comment=null,this.root="",this.clone=function(){var e=new n;for(var t in this)"function"!=typeof this[t]&&(e[t]=this[t]);return e}}(n.prototype=e("./object")).loadAsync=e("./load"),n.support=e("./support"),n.defaults=e("./defaults"),n.version="3.10.0",n.loadAsync=function(e,t){return(new n).loadAsync(e,t)},n.external=e("./external"),t.exports=n},{"./defaults":5,"./external":6,"./load":11,"./object":15,"./support":30}],11:[function(e,t,r){"use strict";var u=e("./utils"),i=e("./external"),n=e("./utf8"),s=e("./zipEntries"),a=e("./stream/Crc32Probe"),l=e("./nodejsUtils");function f(n){return new i.Promise(function(e,t){var r=n.decompressed.getContentWorker().pipe(new a);r.on("error",function(e){t(e)}).on("end",function(){r.streamInfo.crc32!==n.decompressed.crc32?t(new Error("Corrupted zip : CRC32 mismatch")):e()}).resume()})}t.exports=function(e,o){var h=this;return o=u.extend(o||{},{base64:!1,checkCRC32:!1,optimizedBinaryString:!1,createFolders:!1,decodeFileName:n.utf8decode}),l.isNode&&l.isStream(e)?i.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")):u.prepareContent("the loaded zip file",e,!0,o.optimizedBinaryString,o.base64).then(function(e){var t=new s(o);return t.load(e),t}).then(function(e){var t=[i.Promise.resolve(e)],r=e.files;if(o.checkCRC32)for(var n=0;n<r.length;n++)t.push(f(r[n]));return i.Promise.all(t)}).then(function(e){for(var t=e.shift(),r=t.files,n=0;n<r.length;n++){var i=r[n],s=i.fileNameStr,a=u.resolve(i.fileNameStr);h.file(a,i.decompressed,{binary:!0,optimizedBinaryString:!0,date:i.date,dir:i.dir,comment:i.fileCommentStr.length?i.fileCommentStr:null,unixPermissions:i.unixPermissions,dosPermissions:i.dosPermissions,createFolders:o.createFolders}),i.dir||(h.file(a).unsafeOriginalName=s)}return t.zipComment.length&&(h.comment=t.zipComment),h})}},{"./external":6,"./nodejsUtils":14,"./stream/Crc32Probe":25,"./utf8":31,"./utils":32,"./zipEntries":33}],12:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../stream/GenericWorker");function s(e,t){i.call(this,"Nodejs stream input adapter for "+e),this._upstreamEnded=!1,this._bindStream(t)}n.inherits(s,i),s.prototype._bindStream=function(e){var t=this;(this._stream=e).pause(),e.on("data",function(e){t.push({data:e,meta:{percent:0}})}).on("error",function(e){t.isPaused?this.generatedError=e:t.error(e)}).on("end",function(){t.isPaused?t._upstreamEnded=!0:t.end()})},s.prototype.pause=function(){return!!i.prototype.pause.call(this)&&(this._stream.pause(),!0)},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(this._upstreamEnded?this.end():this._stream.resume(),!0)},t.exports=s},{"../stream/GenericWorker":28,"../utils":32}],13:[function(e,t,r){"use strict";var i=e("readable-stream").Readable;function n(e,t,r){i.call(this,t),this._helper=e;var n=this;e.on("data",function(e,t){n.push(e)||n._helper.pause(),r&&r(t)}).on("error",function(e){n.emit("error",e)}).on("end",function(){n.push(null)})}e("../utils").inherits(n,i),n.prototype._read=function(){this._helper.resume()},t.exports=n},{"../utils":32,"readable-stream":16}],14:[function(e,t,r){"use strict";t.exports={isNode:"undefined"!=typeof Buffer,newBufferFrom:function(e,t){if(Buffer.from&&Buffer.from!==Uint8Array.from)return Buffer.from(e,t);if("number"==typeof e)throw new Error('The "data" argument must not be a number');return new Buffer(e,t)},allocBuffer:function(e){if(Buffer.alloc)return Buffer.alloc(e);var t=new Buffer(e);return t.fill(0),t},isBuffer:function(e){return Buffer.isBuffer(e)},isStream:function(e){return e&&"function"==typeof e.on&&"function"==typeof e.pause&&"function"==typeof e.resume}}},{}],15:[function(e,t,r){"use strict";function s(e,t,r){var n,i=u.getTypeOf(t),s=u.extend(r||{},f);s.date=s.date||new Date,null!==s.compression&&(s.compression=s.compression.toUpperCase()),"string"==typeof s.unixPermissions&&(s.unixPermissions=parseInt(s.unixPermissions,8)),s.unixPermissions&&16384&s.unixPermissions&&(s.dir=!0),s.dosPermissions&&16&s.dosPermissions&&(s.dir=!0),s.dir&&(e=g(e)),s.createFolders&&(n=_(e))&&b.call(this,n,!0);var a="string"===i&&!1===s.binary&&!1===s.base64;r&&void 0!==r.binary||(s.binary=!a),(t instanceof c&&0===t.uncompressedSize||s.dir||!t||0===t.length)&&(s.base64=!1,s.binary=!0,t="",s.compression="STORE",i="string");var o=null;o=t instanceof c||t instanceof l?t:p.isNode&&p.isStream(t)?new m(e,t):u.prepareContent(e,t,s.binary,s.optimizedBinaryString,s.base64);var h=new d(e,o,s);this.files[e]=h}var i=e("./utf8"),u=e("./utils"),l=e("./stream/GenericWorker"),a=e("./stream/StreamHelper"),f=e("./defaults"),c=e("./compressedObject"),d=e("./zipObject"),o=e("./generate"),p=e("./nodejsUtils"),m=e("./nodejs/NodejsStreamInputAdapter"),_=function(e){"/"===e.slice(-1)&&(e=e.substring(0,e.length-1));var t=e.lastIndexOf("/");return 0<t?e.substring(0,t):""},g=function(e){return"/"!==e.slice(-1)&&(e+="/"),e},b=function(e,t){return t=void 0!==t?t:f.createFolders,e=g(e),this.files[e]||s.call(this,e,null,{dir:!0,createFolders:t}),this.files[e]};function h(e){return"[object RegExp]"===Object.prototype.toString.call(e)}var n={load:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},forEach:function(e){var t,r,n;for(t in this.files)n=this.files[t],(r=t.slice(this.root.length,t.length))&&t.slice(0,this.root.length)===this.root&&e(r,n)},filter:function(r){var n=[];return this.forEach(function(e,t){r(e,t)&&n.push(t)}),n},file:function(e,t,r){if(1!==arguments.length)return e=this.root+e,s.call(this,e,t,r),this;if(h(e)){var n=e;return this.filter(function(e,t){return!t.dir&&n.test(e)})}var i=this.files[this.root+e];return i&&!i.dir?i:null},folder:function(r){if(!r)return this;if(h(r))return this.filter(function(e,t){return t.dir&&r.test(e)});var e=this.root+r,t=b.call(this,e),n=this.clone();return n.root=t.name,n},remove:function(r){r=this.root+r;var e=this.files[r];if(e||("/"!==r.slice(-1)&&(r+="/"),e=this.files[r]),e&&!e.dir)delete this.files[r];else for(var t=this.filter(function(e,t){return t.name.slice(0,r.length)===r}),n=0;n<t.length;n++)delete this.files[t[n].name];return this},generate:function(e){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},generateInternalStream:function(e){var t,r={};try{if((r=u.extend(e||{},{streamFiles:!1,compression:"STORE",compressionOptions:null,type:"",platform:"DOS",comment:null,mimeType:"application/zip",encodeFileName:i.utf8encode})).type=r.type.toLowerCase(),r.compression=r.compression.toUpperCase(),"binarystring"===r.type&&(r.type="string"),!r.type)throw new Error("No output type specified.");u.checkSupport(r.type),"darwin"!==r.platform&&"freebsd"!==r.platform&&"linux"!==r.platform&&"sunos"!==r.platform||(r.platform="UNIX"),"win32"===r.platform&&(r.platform="DOS");var n=r.comment||this.comment||"";t=o.generateWorker(this,r,n)}catch(e){(t=new l("error")).error(e)}return new a(t,r.type||"string",r.mimeType)},generateAsync:function(e,t){return this.generateInternalStream(e).accumulate(t)},generateNodeStream:function(e,t){return(e=e||{}).type||(e.type="nodebuffer"),this.generateInternalStream(e).toNodejsStream(t)}};t.exports=n},{"./compressedObject":2,"./defaults":5,"./generate":9,"./nodejs/NodejsStreamInputAdapter":12,"./nodejsUtils":14,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31,"./utils":32,"./zipObject":35}],16:[function(e,t,r){t.exports=e("stream")},{stream:void 0}],17:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e);for(var t=0;t<this.data.length;t++)e[t]=255&e[t]}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data[this.zero+e]},i.prototype.lastIndexOfSignature=function(e){for(var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.length-4;0<=s;--s)if(this.data[s]===t&&this.data[s+1]===r&&this.data[s+2]===n&&this.data[s+3]===i)return s-this.zero;return-1},i.prototype.readAndCheckSignature=function(e){var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.readData(4);return t===s[0]&&r===s[1]&&n===s[2]&&i===s[3]},i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return[];var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],18:[function(e,t,r){"use strict";var n=e("../utils");function i(e){this.data=e,this.length=e.length,this.index=0,this.zero=0}i.prototype={checkOffset:function(e){this.checkIndex(this.index+e)},checkIndex:function(e){if(this.length<this.zero+e||e<0)throw new Error("End of data reached (data length = "+this.length+", asked index = "+e+"). Corrupted zip ?")},setIndex:function(e){this.checkIndex(e),this.index=e},skip:function(e){this.setIndex(this.index+e)},byteAt:function(e){},readInt:function(e){var t,r=0;for(this.checkOffset(e),t=this.index+e-1;t>=this.index;t--)r=(r<<8)+this.byteAt(t);return this.index+=e,r},readString:function(e){return n.transformTo("string",this.readData(e))},readData:function(e){},lastIndexOfSignature:function(e){},readAndCheckSignature:function(e){},readDate:function(){var e=this.readInt(4);return new Date(Date.UTC(1980+(e>>25&127),(e>>21&15)-1,e>>16&31,e>>11&31,e>>5&63,(31&e)<<1))}},t.exports=i},{"../utils":32}],19:[function(e,t,r){"use strict";var n=e("./Uint8ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./Uint8ArrayReader":21}],20:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data.charCodeAt(this.zero+e)},i.prototype.lastIndexOfSignature=function(e){return this.data.lastIndexOf(e)-this.zero},i.prototype.readAndCheckSignature=function(e){return e===this.readData(4)},i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],21:[function(e,t,r){"use strict";var n=e("./ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return new Uint8Array(0);var t=this.data.subarray(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./ArrayReader":17}],22:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../support"),s=e("./ArrayReader"),a=e("./StringReader"),o=e("./NodeBufferReader"),h=e("./Uint8ArrayReader");t.exports=function(e){var t=n.getTypeOf(e);return n.checkSupport(t),"string"!==t||i.uint8array?"nodebuffer"===t?new o(e):i.uint8array?new h(n.transformTo("uint8array",e)):new s(n.transformTo("array",e)):new a(e)}},{"../support":30,"../utils":32,"./ArrayReader":17,"./NodeBufferReader":19,"./StringReader":20,"./Uint8ArrayReader":21}],23:[function(e,t,r){"use strict";r.LOCAL_FILE_HEADER="PK",r.CENTRAL_FILE_HEADER="PK",r.CENTRAL_DIRECTORY_END="PK",r.ZIP64_CENTRAL_DIRECTORY_LOCATOR="PK",r.ZIP64_CENTRAL_DIRECTORY_END="PK",r.DATA_DESCRIPTOR="PK\b"},{}],24:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../utils");function s(e){n.call(this,"ConvertWorker to "+e),this.destType=e}i.inherits(s,n),s.prototype.processChunk=function(e){this.push({data:i.transformTo(this.destType,e.data),meta:e.meta})},t.exports=s},{"../utils":32,"./GenericWorker":28}],25:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../crc32");function s(){n.call(this,"Crc32Probe"),this.withStreamInfo("crc32",0)}e("../utils").inherits(s,n),s.prototype.processChunk=function(e){this.streamInfo.crc32=i(e.data,this.streamInfo.crc32||0),this.push(e)},t.exports=s},{"../crc32":4,"../utils":32,"./GenericWorker":28}],26:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataLengthProbe for "+e),this.propName=e,this.withStreamInfo(e,0)}n.inherits(s,i),s.prototype.processChunk=function(e){if(e){var t=this.streamInfo[this.propName]||0;this.streamInfo[this.propName]=t+e.data.length}i.prototype.processChunk.call(this,e)},t.exports=s},{"../utils":32,"./GenericWorker":28}],27:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataWorker");var t=this;this.dataIsReady=!1,this.index=0,this.max=0,this.data=null,this.type="",this._tickScheduled=!1,e.then(function(e){t.dataIsReady=!0,t.data=e,t.max=e&&e.length||0,t.type=n.getTypeOf(e),t.isPaused||t._tickAndRepeat()},function(e){t.error(e)})}n.inherits(s,i),s.prototype.cleanUp=function(){i.prototype.cleanUp.call(this),this.data=null},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(!this._tickScheduled&&this.dataIsReady&&(this._tickScheduled=!0,n.delay(this._tickAndRepeat,[],this)),!0)},s.prototype._tickAndRepeat=function(){this._tickScheduled=!1,this.isPaused||this.isFinished||(this._tick(),this.isFinished||(n.delay(this._tickAndRepeat,[],this),this._tickScheduled=!0))},s.prototype._tick=function(){if(this.isPaused||this.isFinished)return!1;var e=null,t=Math.min(this.max,this.index+16384);if(this.index>=this.max)return this.end();switch(this.type){case"string":e=this.data.substring(this.index,t);break;case"uint8array":e=this.data.subarray(this.index,t);break;case"array":case"nodebuffer":e=this.data.slice(this.index,t)}return this.index=t,this.push({data:e,meta:{percent:this.max?this.index/this.max*100:0}})},t.exports=s},{"../utils":32,"./GenericWorker":28}],28:[function(e,t,r){"use strict";function n(e){this.name=e||"default",this.streamInfo={},this.generatedError=null,this.extraStreamInfo={},this.isPaused=!0,this.isFinished=!1,this.isLocked=!1,this._listeners={data:[],end:[],error:[]},this.previous=null}n.prototype={push:function(e){this.emit("data",e)},end:function(){if(this.isFinished)return!1;this.flush();try{this.emit("end"),this.cleanUp(),this.isFinished=!0}catch(e){this.emit("error",e)}return!0},error:function(e){return!this.isFinished&&(this.isPaused?this.generatedError=e:(this.isFinished=!0,this.emit("error",e),this.previous&&this.previous.error(e),this.cleanUp()),!0)},on:function(e,t){return this._listeners[e].push(t),this},cleanUp:function(){this.streamInfo=this.generatedError=this.extraStreamInfo=null,this._listeners=[]},emit:function(e,t){if(this._listeners[e])for(var r=0;r<this._listeners[e].length;r++)this._listeners[e][r].call(this,t)},pipe:function(e){return e.registerPrevious(this)},registerPrevious:function(e){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.streamInfo=e.streamInfo,this.mergeStreamInfo(),this.previous=e;var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.end()}),e.on("error",function(e){t.error(e)}),this},pause:function(){return!this.isPaused&&!this.isFinished&&(this.isPaused=!0,this.previous&&this.previous.pause(),!0)},resume:function(){if(!this.isPaused||this.isFinished)return!1;var e=this.isPaused=!1;return this.generatedError&&(this.error(this.generatedError),e=!0),this.previous&&this.previous.resume(),!e},flush:function(){},processChunk:function(e){this.push(e)},withStreamInfo:function(e,t){return this.extraStreamInfo[e]=t,this.mergeStreamInfo(),this},mergeStreamInfo:function(){for(var e in this.extraStreamInfo)this.extraStreamInfo.hasOwnProperty(e)&&(this.streamInfo[e]=this.extraStreamInfo[e])},lock:function(){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.isLocked=!0,this.previous&&this.previous.lock()},toString:function(){var e="Worker "+this.name;return this.previous?this.previous+" -> "+e:e}},t.exports=n},{}],29:[function(e,t,r){"use strict";var h=e("../utils"),i=e("./ConvertWorker"),s=e("./GenericWorker"),u=e("../base64"),n=e("../support"),a=e("../external"),o=null;if(n.nodestream)try{o=e("../nodejs/NodejsStreamOutputAdapter")}catch(e){}function l(e,o){return new a.Promise(function(t,r){var n=[],i=e._internalType,s=e._outputType,a=e._mimeType;e.on("data",function(e,t){n.push(e),o&&o(t)}).on("error",function(e){n=[],r(e)}).on("end",function(){try{var e=function(e,t,r){switch(e){case"blob":return h.newBlob(h.transformTo("arraybuffer",t),r);case"base64":return u.encode(t);default:return h.transformTo(e,t)}}(s,function(e,t){var r,n=0,i=null,s=0;for(r=0;r<t.length;r++)s+=t[r].length;switch(e){case"string":return t.join("");case"array":return Array.prototype.concat.apply([],t);case"uint8array":for(i=new Uint8Array(s),r=0;r<t.length;r++)i.set(t[r],n),n+=t[r].length;return i;case"nodebuffer":return Buffer.concat(t);default:throw new Error("concat : unsupported type '"+e+"'")}}(i,n),a);t(e)}catch(e){r(e)}n=[]}).resume()})}function f(e,t,r){var n=t;switch(t){case"blob":case"arraybuffer":n="uint8array";break;case"base64":n="string"}try{this._internalType=n,this._outputType=t,this._mimeType=r,h.checkSupport(n),this._worker=e.pipe(new i(n)),e.lock()}catch(e){this._worker=new s("error"),this._worker.error(e)}}f.prototype={accumulate:function(e){return l(this,e)},on:function(e,t){var r=this;return"data"===e?this._worker.on(e,function(e){t.call(r,e.data,e.meta)}):this._worker.on(e,function(){h.delay(t,arguments,r)}),this},resume:function(){return h.delay(this._worker.resume,[],this._worker),this},pause:function(){return this._worker.pause(),this},toNodejsStream:function(e){if(h.checkSupport("nodestream"),"nodebuffer"!==this._outputType)throw new Error(this._outputType+" is not supported by this method");return new o(this,{objectMode:"nodebuffer"!==this._outputType},e)}},t.exports=f},{"../base64":1,"../external":6,"../nodejs/NodejsStreamOutputAdapter":13,"../support":30,"../utils":32,"./ConvertWorker":24,"./GenericWorker":28}],30:[function(e,t,r){"use strict";if(r.base64=!0,r.array=!0,r.string=!0,r.arraybuffer="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof Uint8Array,r.nodebuffer="undefined"!=typeof Buffer,r.uint8array="undefined"!=typeof Uint8Array,"undefined"==typeof ArrayBuffer)r.blob=!1;else{var n=new ArrayBuffer(0);try{r.blob=0===new Blob([n],{type:"application/zip"}).size}catch(e){try{var i=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);i.append(n),r.blob=0===i.getBlob("application/zip").size}catch(e){r.blob=!1}}}try{r.nodestream=!!e("readable-stream").Readable}catch(e){r.nodestream=!1}},{"readable-stream":16}],31:[function(e,t,s){"use strict";for(var o=e("./utils"),h=e("./support"),r=e("./nodejsUtils"),n=e("./stream/GenericWorker"),u=new Array(256),i=0;i<256;i++)u[i]=252<=i?6:248<=i?5:240<=i?4:224<=i?3:192<=i?2:1;u[254]=u[254]=1;function a(){n.call(this,"utf-8 decode"),this.leftOver=null}function l(){n.call(this,"utf-8 encode")}s.utf8encode=function(e){return h.nodebuffer?r.newBufferFrom(e,"utf-8"):function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=h.uint8array?new Uint8Array(o):new Array(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t}(e)},s.utf8decode=function(e){return h.nodebuffer?o.transformTo("nodebuffer",e).toString("utf-8"):function(e){var t,r,n,i,s=e.length,a=new Array(2*s);for(t=r=0;t<s;)if((n=e[t++])<128)a[r++]=n;else if(4<(i=u[n]))a[r++]=65533,t+=i-1;else{for(n&=2===i?31:3===i?15:7;1<i&&t<s;)n=n<<6|63&e[t++],i--;1<i?a[r++]=65533:n<65536?a[r++]=n:(n-=65536,a[r++]=55296|n>>10&1023,a[r++]=56320|1023&n)}return a.length!==r&&(a.subarray?a=a.subarray(0,r):a.length=r),o.applyFromCharCode(a)}(e=o.transformTo(h.uint8array?"uint8array":"array",e))},o.inherits(a,n),a.prototype.processChunk=function(e){var t=o.transformTo(h.uint8array?"uint8array":"array",e.data);if(this.leftOver&&this.leftOver.length){if(h.uint8array){var r=t;(t=new Uint8Array(r.length+this.leftOver.length)).set(this.leftOver,0),t.set(r,this.leftOver.length)}else t=this.leftOver.concat(t);this.leftOver=null}var n=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+u[e[r]]>t?r:t}(t),i=t;n!==t.length&&(h.uint8array?(i=t.subarray(0,n),this.leftOver=t.subarray(n,t.length)):(i=t.slice(0,n),this.leftOver=t.slice(n,t.length))),this.push({data:s.utf8decode(i),meta:e.meta})},a.prototype.flush=function(){this.leftOver&&this.leftOver.length&&(this.push({data:s.utf8decode(this.leftOver),meta:{}}),this.leftOver=null)},s.Utf8DecodeWorker=a,o.inherits(l,n),l.prototype.processChunk=function(e){this.push({data:s.utf8encode(e.data),meta:e.meta})},s.Utf8EncodeWorker=l},{"./nodejsUtils":14,"./stream/GenericWorker":28,"./support":30,"./utils":32}],32:[function(e,t,a){"use strict";var o=e("./support"),h=e("./base64"),r=e("./nodejsUtils"),u=e("./external");function n(e){return e}function l(e,t){for(var r=0;r<e.length;++r)t[r]=255&e.charCodeAt(r);return t}e("setimmediate"),a.newBlob=function(t,r){a.checkSupport("blob");try{return new Blob([t],{type:r})}catch(e){try{var n=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);return n.append(t),n.getBlob(r)}catch(e){throw new Error("Bug : can't construct the Blob.")}}};var i={stringifyByChunk:function(e,t,r){var n=[],i=0,s=e.length;if(s<=r)return String.fromCharCode.apply(null,e);for(;i<s;)"array"===t||"nodebuffer"===t?n.push(String.fromCharCode.apply(null,e.slice(i,Math.min(i+r,s)))):n.push(String.fromCharCode.apply(null,e.subarray(i,Math.min(i+r,s)))),i+=r;return n.join("")},stringifyByChar:function(e){for(var t="",r=0;r<e.length;r++)t+=String.fromCharCode(e[r]);return t},applyCanBeUsed:{uint8array:function(){try{return o.uint8array&&1===String.fromCharCode.apply(null,new Uint8Array(1)).length}catch(e){return!1}}(),nodebuffer:function(){try{return o.nodebuffer&&1===String.fromCharCode.apply(null,r.allocBuffer(1)).length}catch(e){return!1}}()}};function s(e){var t=65536,r=a.getTypeOf(e),n=!0;if("uint8array"===r?n=i.applyCanBeUsed.uint8array:"nodebuffer"===r&&(n=i.applyCanBeUsed.nodebuffer),n)for(;1<t;)try{return i.stringifyByChunk(e,r,t)}catch(e){t=Math.floor(t/2)}return i.stringifyByChar(e)}function f(e,t){for(var r=0;r<e.length;r++)t[r]=e[r];return t}a.applyFromCharCode=s;var c={};c.string={string:n,array:function(e){return l(e,new Array(e.length))},arraybuffer:function(e){return c.string.uint8array(e).buffer},uint8array:function(e){return l(e,new Uint8Array(e.length))},nodebuffer:function(e){return l(e,r.allocBuffer(e.length))}},c.array={string:s,array:n,arraybuffer:function(e){return new Uint8Array(e).buffer},uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(e)}},c.arraybuffer={string:function(e){return s(new Uint8Array(e))},array:function(e){return f(new Uint8Array(e),new Array(e.byteLength))},arraybuffer:n,uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(new Uint8Array(e))}},c.uint8array={string:s,array:function(e){return f(e,new Array(e.length))},arraybuffer:function(e){return e.buffer},uint8array:n,nodebuffer:function(e){return r.newBufferFrom(e)}},c.nodebuffer={string:s,array:function(e){return f(e,new Array(e.length))},arraybuffer:function(e){return c.nodebuffer.uint8array(e).buffer},uint8array:function(e){return f(e,new Uint8Array(e.length))},nodebuffer:n},a.transformTo=function(e,t){if(t=t||"",!e)return t;a.checkSupport(e);var r=a.getTypeOf(t);return c[r][e](t)},a.resolve=function(e){for(var t=e.split("/"),r=[],n=0;n<t.length;n++){var i=t[n];"."===i||""===i&&0!==n&&n!==t.length-1||(".."===i?r.pop():r.push(i))}return r.join("/")},a.getTypeOf=function(e){return"string"==typeof e?"string":"[object Array]"===Object.prototype.toString.call(e)?"array":o.nodebuffer&&r.isBuffer(e)?"nodebuffer":o.uint8array&&e instanceof Uint8Array?"uint8array":o.arraybuffer&&e instanceof ArrayBuffer?"arraybuffer":void 0},a.checkSupport=function(e){if(!o[e.toLowerCase()])throw new Error(e+" is not supported by this platform")},a.MAX_VALUE_16BITS=65535,a.MAX_VALUE_32BITS=-1,a.pretty=function(e){var t,r,n="";for(r=0;r<(e||"").length;r++)n+="\\x"+((t=e.charCodeAt(r))<16?"0":"")+t.toString(16).toUpperCase();return n},a.delay=function(e,t,r){setImmediate(function(){e.apply(r||null,t||[])})},a.inherits=function(e,t){function r(){}r.prototype=t.prototype,e.prototype=new r},a.extend=function(){var e,t,r={};for(e=0;e<arguments.length;e++)for(t in arguments[e])arguments[e].hasOwnProperty(t)&&void 0===r[t]&&(r[t]=arguments[e][t]);return r},a.prepareContent=function(r,e,n,i,s){return u.Promise.resolve(e).then(function(n){return o.blob&&(n instanceof Blob||-1!==["[object File]","[object Blob]"].indexOf(Object.prototype.toString.call(n)))&&"undefined"!=typeof FileReader?new u.Promise(function(t,r){var e=new FileReader;e.onload=function(e){t(e.target.result)},e.onerror=function(e){r(e.target.error)},e.readAsArrayBuffer(n)}):n}).then(function(e){var t=a.getTypeOf(e);return t?("arraybuffer"===t?e=a.transformTo("uint8array",e):"string"===t&&(s?e=h.decode(e):n&&!0!==i&&(e=function(e){return l(e,o.uint8array?new Uint8Array(e.length):new Array(e.length))}(e))),e):u.Promise.reject(new Error("Can't read the data of '"+r+"'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"))})}},{"./base64":1,"./external":6,"./nodejsUtils":14,"./support":30,setimmediate:54}],33:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),i=e("./utils"),s=e("./signature"),a=e("./zipEntry"),o=(e("./utf8"),e("./support"));function h(e){this.files=[],this.loadOptions=e}h.prototype={checkSignature:function(e){if(!this.reader.readAndCheckSignature(e)){this.reader.index-=4;var t=this.reader.readString(4);throw new Error("Corrupted zip or bug: unexpected signature ("+i.pretty(t)+", expected "+i.pretty(e)+")")}},isSignature:function(e,t){var r=this.reader.index;this.reader.setIndex(e);var n=this.reader.readString(4)===t;return this.reader.setIndex(r),n},readBlockEndOfCentral:function(){this.diskNumber=this.reader.readInt(2),this.diskWithCentralDirStart=this.reader.readInt(2),this.centralDirRecordsOnThisDisk=this.reader.readInt(2),this.centralDirRecords=this.reader.readInt(2),this.centralDirSize=this.reader.readInt(4),this.centralDirOffset=this.reader.readInt(4),this.zipCommentLength=this.reader.readInt(2);var e=this.reader.readData(this.zipCommentLength),t=o.uint8array?"uint8array":"array",r=i.transformTo(t,e);this.zipComment=this.loadOptions.decodeFileName(r)},readBlockZip64EndOfCentral:function(){this.zip64EndOfCentralSize=this.reader.readInt(8),this.reader.skip(4),this.diskNumber=this.reader.readInt(4),this.diskWithCentralDirStart=this.reader.readInt(4),this.centralDirRecordsOnThisDisk=this.reader.readInt(8),this.centralDirRecords=this.reader.readInt(8),this.centralDirSize=this.reader.readInt(8),this.centralDirOffset=this.reader.readInt(8),this.zip64ExtensibleData={};for(var e,t,r,n=this.zip64EndOfCentralSize-44;0<n;)e=this.reader.readInt(2),t=this.reader.readInt(4),r=this.reader.readData(t),this.zip64ExtensibleData[e]={id:e,length:t,value:r}},readBlockZip64EndOfCentralLocator:function(){if(this.diskWithZip64CentralDirStart=this.reader.readInt(4),this.relativeOffsetEndOfZip64CentralDir=this.reader.readInt(8),this.disksCount=this.reader.readInt(4),1<this.disksCount)throw new Error("Multi-volumes zip are not supported")},readLocalFiles:function(){var e,t;for(e=0;e<this.files.length;e++)t=this.files[e],this.reader.setIndex(t.localHeaderOffset),this.checkSignature(s.LOCAL_FILE_HEADER),t.readLocalPart(this.reader),t.handleUTF8(),t.processAttributes()},readCentralDir:function(){var e;for(this.reader.setIndex(this.centralDirOffset);this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);)(e=new a({zip64:this.zip64},this.loadOptions)).readCentralPart(this.reader),this.files.push(e);if(this.centralDirRecords!==this.files.length&&0!==this.centralDirRecords&&0===this.files.length)throw new Error("Corrupted zip or bug: expected "+this.centralDirRecords+" records in central dir, got "+this.files.length)},readEndOfCentral:function(){var e=this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);if(e<0)throw!this.isSignature(0,s.LOCAL_FILE_HEADER)?new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html"):new Error("Corrupted zip: can't find end of central directory");this.reader.setIndex(e);var t=e;if(this.checkSignature(s.CENTRAL_DIRECTORY_END),this.readBlockEndOfCentral(),this.diskNumber===i.MAX_VALUE_16BITS||this.diskWithCentralDirStart===i.MAX_VALUE_16BITS||this.centralDirRecordsOnThisDisk===i.MAX_VALUE_16BITS||this.centralDirRecords===i.MAX_VALUE_16BITS||this.centralDirSize===i.MAX_VALUE_32BITS||this.centralDirOffset===i.MAX_VALUE_32BITS){if(this.zip64=!0,(e=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR))<0)throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");if(this.reader.setIndex(e),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR),this.readBlockZip64EndOfCentralLocator(),!this.isSignature(this.relativeOffsetEndOfZip64CentralDir,s.ZIP64_CENTRAL_DIRECTORY_END)&&(this.relativeOffsetEndOfZip64CentralDir=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.relativeOffsetEndOfZip64CentralDir<0))throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.readBlockZip64EndOfCentral()}var r=this.centralDirOffset+this.centralDirSize;this.zip64&&(r+=20,r+=12+this.zip64EndOfCentralSize);var n=t-r;if(0<n)this.isSignature(t,s.CENTRAL_FILE_HEADER)||(this.reader.zero=n);else if(n<0)throw new Error("Corrupted zip: missing "+Math.abs(n)+" bytes.")},prepareReader:function(e){this.reader=n(e)},load:function(e){this.prepareReader(e),this.readEndOfCentral(),this.readCentralDir(),this.readLocalFiles()}},t.exports=h},{"./reader/readerFor":22,"./signature":23,"./support":30,"./utf8":31,"./utils":32,"./zipEntry":34}],34:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),s=e("./utils"),i=e("./compressedObject"),a=e("./crc32"),o=e("./utf8"),h=e("./compressions"),u=e("./support");function l(e,t){this.options=e,this.loadOptions=t}l.prototype={isEncrypted:function(){return 1==(1&this.bitFlag)},useUTF8:function(){return 2048==(2048&this.bitFlag)},readLocalPart:function(e){var t,r;if(e.skip(22),this.fileNameLength=e.readInt(2),r=e.readInt(2),this.fileName=e.readData(this.fileNameLength),e.skip(r),-1===this.compressedSize||-1===this.uncompressedSize)throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");if(null===(t=function(e){for(var t in h)if(h.hasOwnProperty(t)&&h[t].magic===e)return h[t];return null}(this.compressionMethod)))throw new Error("Corrupted zip : compression "+s.pretty(this.compressionMethod)+" unknown (inner file : "+s.transformTo("string",this.fileName)+")");this.decompressed=new i(this.compressedSize,this.uncompressedSize,this.crc32,t,e.readData(this.compressedSize))},readCentralPart:function(e){this.versionMadeBy=e.readInt(2),e.skip(2),this.bitFlag=e.readInt(2),this.compressionMethod=e.readString(2),this.date=e.readDate(),this.crc32=e.readInt(4),this.compressedSize=e.readInt(4),this.uncompressedSize=e.readInt(4);var t=e.readInt(2);if(this.extraFieldsLength=e.readInt(2),this.fileCommentLength=e.readInt(2),this.diskNumberStart=e.readInt(2),this.internalFileAttributes=e.readInt(2),this.externalFileAttributes=e.readInt(4),this.localHeaderOffset=e.readInt(4),this.isEncrypted())throw new Error("Encrypted zip are not supported");e.skip(t),this.readExtraFields(e),this.parseZIP64ExtraField(e),this.fileComment=e.readData(this.fileCommentLength)},processAttributes:function(){this.unixPermissions=null,this.dosPermissions=null;var e=this.versionMadeBy>>8;this.dir=!!(16&this.externalFileAttributes),0==e&&(this.dosPermissions=63&this.externalFileAttributes),3==e&&(this.unixPermissions=this.externalFileAttributes>>16&65535),this.dir||"/"!==this.fileNameStr.slice(-1)||(this.dir=!0)},parseZIP64ExtraField:function(e){if(this.extraFields[1]){var t=n(this.extraFields[1].value);this.uncompressedSize===s.MAX_VALUE_32BITS&&(this.uncompressedSize=t.readInt(8)),this.compressedSize===s.MAX_VALUE_32BITS&&(this.compressedSize=t.readInt(8)),this.localHeaderOffset===s.MAX_VALUE_32BITS&&(this.localHeaderOffset=t.readInt(8)),this.diskNumberStart===s.MAX_VALUE_32BITS&&(this.diskNumberStart=t.readInt(4))}},readExtraFields:function(e){var t,r,n,i=e.index+this.extraFieldsLength;for(this.extraFields||(this.extraFields={});e.index+4<i;)t=e.readInt(2),r=e.readInt(2),n=e.readData(r),this.extraFields[t]={id:t,length:r,value:n};e.setIndex(i)},handleUTF8:function(){var e=u.uint8array?"uint8array":"array";if(this.useUTF8())this.fileNameStr=o.utf8decode(this.fileName),this.fileCommentStr=o.utf8decode(this.fileComment);else{var t=this.findExtraFieldUnicodePath();if(null!==t)this.fileNameStr=t;else{var r=s.transformTo(e,this.fileName);this.fileNameStr=this.loadOptions.decodeFileName(r)}var n=this.findExtraFieldUnicodeComment();if(null!==n)this.fileCommentStr=n;else{var i=s.transformTo(e,this.fileComment);this.fileCommentStr=this.loadOptions.decodeFileName(i)}}},findExtraFieldUnicodePath:function(){var e=this.extraFields[28789];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileName)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null},findExtraFieldUnicodeComment:function(){var e=this.extraFields[25461];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileComment)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null}},t.exports=l},{"./compressedObject":2,"./compressions":3,"./crc32":4,"./reader/readerFor":22,"./support":30,"./utf8":31,"./utils":32}],35:[function(e,t,r){"use strict";function n(e,t,r){this.name=e,this.dir=r.dir,this.date=r.date,this.comment=r.comment,this.unixPermissions=r.unixPermissions,this.dosPermissions=r.dosPermissions,this._data=t,this._dataBinary=r.binary,this.options={compression:r.compression,compressionOptions:r.compressionOptions}}var s=e("./stream/StreamHelper"),i=e("./stream/DataWorker"),a=e("./utf8"),o=e("./compressedObject"),h=e("./stream/GenericWorker");n.prototype={internalStream:function(e){var t=null,r="string";try{if(!e)throw new Error("No output type specified.");var n="string"===(r=e.toLowerCase())||"text"===r;"binarystring"!==r&&"text"!==r||(r="string"),t=this._decompressWorker();var i=!this._dataBinary;i&&!n&&(t=t.pipe(new a.Utf8EncodeWorker)),!i&&n&&(t=t.pipe(new a.Utf8DecodeWorker))}catch(e){(t=new h("error")).error(e)}return new s(t,r,"")},async:function(e,t){return this.internalStream(e).accumulate(t)},nodeStream:function(e,t){return this.internalStream(e||"nodebuffer").toNodejsStream(t)},_compressWorker:function(e,t){if(this._data instanceof o&&this._data.compression.magic===e.magic)return this._data.getCompressedWorker();var r=this._decompressWorker();return this._dataBinary||(r=r.pipe(new a.Utf8EncodeWorker)),o.createWorkerFrom(r,e,t)},_decompressWorker:function(){return this._data instanceof o?this._data.getContentWorker():this._data instanceof h?this._data:new i(this._data)}};for(var u=["asText","asBinary","asNodeBuffer","asUint8Array","asArrayBuffer"],l=function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},f=0;f<u.length;f++)n.prototype[u[f]]=l;t.exports=n},{"./compressedObject":2,"./stream/DataWorker":27,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31}],36:[function(e,l,t){(function(t){"use strict";var r,n,e=t.MutationObserver||t.WebKitMutationObserver;if(e){var i=0,s=new e(u),a=t.document.createTextNode("");s.observe(a,{characterData:!0}),r=function(){a.data=i=++i%2}}else if(t.setImmediate||void 0===t.MessageChannel)r="document"in t&&"onreadystatechange"in t.document.createElement("script")?function(){var e=t.document.createElement("script");e.onreadystatechange=function(){u(),e.onreadystatechange=null,e.parentNode.removeChild(e),e=null},t.document.documentElement.appendChild(e)}:function(){setTimeout(u,0)};else{var o=new t.MessageChannel;o.port1.onmessage=u,r=function(){o.port2.postMessage(0)}}var h=[];function u(){var e,t;n=!0;for(var r=h.length;r;){for(t=h,h=[],e=-1;++e<r;)t[e]();r=h.length}n=!1}l.exports=function(e){1!==h.push(e)||n||r()}}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}],37:[function(e,t,r){"use strict";var i=e("immediate");function u(){}var l={},s=["REJECTED"],a=["FULFILLED"],n=["PENDING"];function o(e){if("function"!=typeof e)throw new TypeError("resolver must be a function");this.state=n,this.queue=[],this.outcome=void 0,e!==u&&d(this,e)}function h(e,t,r){this.promise=e,"function"==typeof t&&(this.onFulfilled=t,this.callFulfilled=this.otherCallFulfilled),"function"==typeof r&&(this.onRejected=r,this.callRejected=this.otherCallRejected)}function f(t,r,n){i(function(){var e;try{e=r(n)}catch(e){return l.reject(t,e)}e===t?l.reject(t,new TypeError("Cannot resolve promise with itself")):l.resolve(t,e)})}function c(e){var t=e&&e.then;if(e&&("object"==typeof e||"function"==typeof e)&&"function"==typeof t)return function(){t.apply(e,arguments)}}function d(t,e){var r=!1;function n(e){r||(r=!0,l.reject(t,e))}function i(e){r||(r=!0,l.resolve(t,e))}var s=p(function(){e(i,n)});"error"===s.status&&n(s.value)}function p(e,t){var r={};try{r.value=e(t),r.status="success"}catch(e){r.status="error",r.value=e}return r}(t.exports=o).prototype.finally=function(t){if("function"!=typeof t)return this;var r=this.constructor;return this.then(function(e){return r.resolve(t()).then(function(){return e})},function(e){return r.resolve(t()).then(function(){throw e})})},o.prototype.catch=function(e){return this.then(null,e)},o.prototype.then=function(e,t){if("function"!=typeof e&&this.state===a||"function"!=typeof t&&this.state===s)return this;var r=new this.constructor(u);this.state!==n?f(r,this.state===a?e:t,this.outcome):this.queue.push(new h(r,e,t));return r},h.prototype.callFulfilled=function(e){l.resolve(this.promise,e)},h.prototype.otherCallFulfilled=function(e){f(this.promise,this.onFulfilled,e)},h.prototype.callRejected=function(e){l.reject(this.promise,e)},h.prototype.otherCallRejected=function(e){f(this.promise,this.onRejected,e)},l.resolve=function(e,t){var r=p(c,t);if("error"===r.status)return l.reject(e,r.value);var n=r.value;if(n)d(e,n);else{e.state=a,e.outcome=t;for(var i=-1,s=e.queue.length;++i<s;)e.queue[i].callFulfilled(t)}return e},l.reject=function(e,t){e.state=s,e.outcome=t;for(var r=-1,n=e.queue.length;++r<n;)e.queue[r].callRejected(t);return e},o.resolve=function(e){if(e instanceof this)return e;return l.resolve(new this(u),e)},o.reject=function(e){var t=new this(u);return l.reject(t,e)},o.all=function(e){var r=this;if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var n=e.length,i=!1;if(!n)return this.resolve([]);var s=new Array(n),a=0,t=-1,o=new this(u);for(;++t<n;)h(e[t],t);return o;function h(e,t){r.resolve(e).then(function(e){s[t]=e,++a!==n||i||(i=!0,l.resolve(o,s))},function(e){i||(i=!0,l.reject(o,e))})}},o.race=function(e){var t=this;if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var r=e.length,n=!1;if(!r)return this.resolve([]);var i=-1,s=new this(u);for(;++i<r;)a=e[i],t.resolve(a).then(function(e){n||(n=!0,l.resolve(s,e))},function(e){n||(n=!0,l.reject(s,e))});var a;return s}},{immediate:36}],38:[function(e,t,r){"use strict";var n={};(0,e("./lib/utils/common").assign)(n,e("./lib/deflate"),e("./lib/inflate"),e("./lib/zlib/constants")),t.exports=n},{"./lib/deflate":39,"./lib/inflate":40,"./lib/utils/common":41,"./lib/zlib/constants":44}],39:[function(e,t,r){"use strict";var a=e("./zlib/deflate"),o=e("./utils/common"),h=e("./utils/strings"),i=e("./zlib/messages"),s=e("./zlib/zstream"),u=Object.prototype.toString,l=0,f=-1,c=0,d=8;function p(e){if(!(this instanceof p))return new p(e);this.options=o.assign({level:f,method:d,chunkSize:16384,windowBits:15,memLevel:8,strategy:c,to:""},e||{});var t=this.options;t.raw&&0<t.windowBits?t.windowBits=-t.windowBits:t.gzip&&0<t.windowBits&&t.windowBits<16&&(t.windowBits+=16),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new s,this.strm.avail_out=0;var r=a.deflateInit2(this.strm,t.level,t.method,t.windowBits,t.memLevel,t.strategy);if(r!==l)throw new Error(i[r]);if(t.header&&a.deflateSetHeader(this.strm,t.header),t.dictionary){var n;if(n="string"==typeof t.dictionary?h.string2buf(t.dictionary):"[object ArrayBuffer]"===u.call(t.dictionary)?new Uint8Array(t.dictionary):t.dictionary,(r=a.deflateSetDictionary(this.strm,n))!==l)throw new Error(i[r]);this._dict_set=!0}}function n(e,t){var r=new p(t);if(r.push(e,!0),r.err)throw r.msg||i[r.err];return r.result}p.prototype.push=function(e,t){var r,n,i=this.strm,s=this.options.chunkSize;if(this.ended)return!1;n=t===~~t?t:!0===t?4:0,"string"==typeof e?i.input=h.string2buf(e):"[object ArrayBuffer]"===u.call(e)?i.input=new Uint8Array(e):i.input=e,i.next_in=0,i.avail_in=i.input.length;do{if(0===i.avail_out&&(i.output=new o.Buf8(s),i.next_out=0,i.avail_out=s),1!==(r=a.deflate(i,n))&&r!==l)return this.onEnd(r),!(this.ended=!0);0!==i.avail_out&&(0!==i.avail_in||4!==n&&2!==n)||("string"===this.options.to?this.onData(h.buf2binstring(o.shrinkBuf(i.output,i.next_out))):this.onData(o.shrinkBuf(i.output,i.next_out)))}while((0<i.avail_in||0===i.avail_out)&&1!==r);return 4===n?(r=a.deflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===l):2!==n||(this.onEnd(l),!(i.avail_out=0))},p.prototype.onData=function(e){this.chunks.push(e)},p.prototype.onEnd=function(e){e===l&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=o.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Deflate=p,r.deflate=n,r.deflateRaw=function(e,t){return(t=t||{}).raw=!0,n(e,t)},r.gzip=function(e,t){return(t=t||{}).gzip=!0,n(e,t)}},{"./utils/common":41,"./utils/strings":42,"./zlib/deflate":46,"./zlib/messages":51,"./zlib/zstream":53}],40:[function(e,t,r){"use strict";var c=e("./zlib/inflate"),d=e("./utils/common"),p=e("./utils/strings"),m=e("./zlib/constants"),n=e("./zlib/messages"),i=e("./zlib/zstream"),s=e("./zlib/gzheader"),_=Object.prototype.toString;function a(e){if(!(this instanceof a))return new a(e);this.options=d.assign({chunkSize:16384,windowBits:0,to:""},e||{});var t=this.options;t.raw&&0<=t.windowBits&&t.windowBits<16&&(t.windowBits=-t.windowBits,0===t.windowBits&&(t.windowBits=-15)),!(0<=t.windowBits&&t.windowBits<16)||e&&e.windowBits||(t.windowBits+=32),15<t.windowBits&&t.windowBits<48&&0==(15&t.windowBits)&&(t.windowBits|=15),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new i,this.strm.avail_out=0;var r=c.inflateInit2(this.strm,t.windowBits);if(r!==m.Z_OK)throw new Error(n[r]);this.header=new s,c.inflateGetHeader(this.strm,this.header)}function o(e,t){var r=new a(t);if(r.push(e,!0),r.err)throw r.msg||n[r.err];return r.result}a.prototype.push=function(e,t){var r,n,i,s,a,o,h=this.strm,u=this.options.chunkSize,l=this.options.dictionary,f=!1;if(this.ended)return!1;n=t===~~t?t:!0===t?m.Z_FINISH:m.Z_NO_FLUSH,"string"==typeof e?h.input=p.binstring2buf(e):"[object ArrayBuffer]"===_.call(e)?h.input=new Uint8Array(e):h.input=e,h.next_in=0,h.avail_in=h.input.length;do{if(0===h.avail_out&&(h.output=new d.Buf8(u),h.next_out=0,h.avail_out=u),(r=c.inflate(h,m.Z_NO_FLUSH))===m.Z_NEED_DICT&&l&&(o="string"==typeof l?p.string2buf(l):"[object ArrayBuffer]"===_.call(l)?new Uint8Array(l):l,r=c.inflateSetDictionary(this.strm,o)),r===m.Z_BUF_ERROR&&!0===f&&(r=m.Z_OK,f=!1),r!==m.Z_STREAM_END&&r!==m.Z_OK)return this.onEnd(r),!(this.ended=!0);h.next_out&&(0!==h.avail_out&&r!==m.Z_STREAM_END&&(0!==h.avail_in||n!==m.Z_FINISH&&n!==m.Z_SYNC_FLUSH)||("string"===this.options.to?(i=p.utf8border(h.output,h.next_out),s=h.next_out-i,a=p.buf2string(h.output,i),h.next_out=s,h.avail_out=u-s,s&&d.arraySet(h.output,h.output,i,s,0),this.onData(a)):this.onData(d.shrinkBuf(h.output,h.next_out)))),0===h.avail_in&&0===h.avail_out&&(f=!0)}while((0<h.avail_in||0===h.avail_out)&&r!==m.Z_STREAM_END);return r===m.Z_STREAM_END&&(n=m.Z_FINISH),n===m.Z_FINISH?(r=c.inflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===m.Z_OK):n!==m.Z_SYNC_FLUSH||(this.onEnd(m.Z_OK),!(h.avail_out=0))},a.prototype.onData=function(e){this.chunks.push(e)},a.prototype.onEnd=function(e){e===m.Z_OK&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=d.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Inflate=a,r.inflate=o,r.inflateRaw=function(e,t){return(t=t||{}).raw=!0,o(e,t)},r.ungzip=o},{"./utils/common":41,"./utils/strings":42,"./zlib/constants":44,"./zlib/gzheader":47,"./zlib/inflate":49,"./zlib/messages":51,"./zlib/zstream":53}],41:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;r.assign=function(e){for(var t=Array.prototype.slice.call(arguments,1);t.length;){var r=t.shift();if(r){if("object"!=typeof r)throw new TypeError(r+"must be non-object");for(var n in r)r.hasOwnProperty(n)&&(e[n]=r[n])}}return e},r.shrinkBuf=function(e,t){return e.length===t?e:e.subarray?e.subarray(0,t):(e.length=t,e)};var i={arraySet:function(e,t,r,n,i){if(t.subarray&&e.subarray)e.set(t.subarray(r,r+n),i);else for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){var t,r,n,i,s,a;for(t=n=0,r=e.length;t<r;t++)n+=e[t].length;for(a=new Uint8Array(n),t=i=0,r=e.length;t<r;t++)s=e[t],a.set(s,i),i+=s.length;return a}},s={arraySet:function(e,t,r,n,i){for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){return[].concat.apply([],e)}};r.setTyped=function(e){e?(r.Buf8=Uint8Array,r.Buf16=Uint16Array,r.Buf32=Int32Array,r.assign(r,i)):(r.Buf8=Array,r.Buf16=Array,r.Buf32=Array,r.assign(r,s))},r.setTyped(n)},{}],42:[function(e,t,r){"use strict";var h=e("./common"),i=!0,s=!0;try{String.fromCharCode.apply(null,[0])}catch(e){i=!1}try{String.fromCharCode.apply(null,new Uint8Array(1))}catch(e){s=!1}for(var u=new h.Buf8(256),n=0;n<256;n++)u[n]=252<=n?6:248<=n?5:240<=n?4:224<=n?3:192<=n?2:1;function l(e,t){if(t<65537&&(e.subarray&&s||!e.subarray&&i))return String.fromCharCode.apply(null,h.shrinkBuf(e,t));for(var r="",n=0;n<t;n++)r+=String.fromCharCode(e[n]);return r}u[254]=u[254]=1,r.string2buf=function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=new h.Buf8(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t},r.buf2binstring=function(e){return l(e,e.length)},r.binstring2buf=function(e){for(var t=new h.Buf8(e.length),r=0,n=t.length;r<n;r++)t[r]=e.charCodeAt(r);return t},r.buf2string=function(e,t){var r,n,i,s,a=t||e.length,o=new Array(2*a);for(r=n=0;r<a;)if((i=e[r++])<128)o[n++]=i;else if(4<(s=u[i]))o[n++]=65533,r+=s-1;else{for(i&=2===s?31:3===s?15:7;1<s&&r<a;)i=i<<6|63&e[r++],s--;1<s?o[n++]=65533:i<65536?o[n++]=i:(i-=65536,o[n++]=55296|i>>10&1023,o[n++]=56320|1023&i)}return l(o,n)},r.utf8border=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+u[e[r]]>t?r:t}},{"./common":41}],43:[function(e,t,r){"use strict";t.exports=function(e,t,r,n){for(var i=65535&e|0,s=e>>>16&65535|0,a=0;0!==r;){for(r-=a=2e3<r?2e3:r;s=s+(i=i+t[n++]|0)|0,--a;);i%=65521,s%=65521}return i|s<<16|0}},{}],44:[function(e,t,r){"use strict";t.exports={Z_NO_FLUSH:0,Z_PARTIAL_FLUSH:1,Z_SYNC_FLUSH:2,Z_FULL_FLUSH:3,Z_FINISH:4,Z_BLOCK:5,Z_TREES:6,Z_OK:0,Z_STREAM_END:1,Z_NEED_DICT:2,Z_ERRNO:-1,Z_STREAM_ERROR:-2,Z_DATA_ERROR:-3,Z_BUF_ERROR:-5,Z_NO_COMPRESSION:0,Z_BEST_SPEED:1,Z_BEST_COMPRESSION:9,Z_DEFAULT_COMPRESSION:-1,Z_FILTERED:1,Z_HUFFMAN_ONLY:2,Z_RLE:3,Z_FIXED:4,Z_DEFAULT_STRATEGY:0,Z_BINARY:0,Z_TEXT:1,Z_UNKNOWN:2,Z_DEFLATED:8}},{}],45:[function(e,t,r){"use strict";var o=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t[a])];return-1^e}},{}],46:[function(e,t,r){"use strict";var h,c=e("../utils/common"),u=e("./trees"),d=e("./adler32"),p=e("./crc32"),n=e("./messages"),l=0,f=4,m=0,_=-2,g=-1,b=4,i=2,v=8,y=9,s=286,a=30,o=19,w=2*s+1,k=15,x=3,S=258,z=S+x+1,C=42,E=113,A=1,I=2,O=3,B=4;function R(e,t){return e.msg=n[t],t}function T(e){return(e<<1)-(4<e?9:0)}function D(e){for(var t=e.length;0<=--t;)e[t]=0}function F(e){var t=e.state,r=t.pending;r>e.avail_out&&(r=e.avail_out),0!==r&&(c.arraySet(e.output,t.pending_buf,t.pending_out,r,e.next_out),e.next_out+=r,t.pending_out+=r,e.total_out+=r,e.avail_out-=r,t.pending-=r,0===t.pending&&(t.pending_out=0))}function N(e,t){u._tr_flush_block(e,0<=e.block_start?e.block_start:-1,e.strstart-e.block_start,t),e.block_start=e.strstart,F(e.strm)}function U(e,t){e.pending_buf[e.pending++]=t}function P(e,t){e.pending_buf[e.pending++]=t>>>8&255,e.pending_buf[e.pending++]=255&t}function L(e,t){var r,n,i=e.max_chain_length,s=e.strstart,a=e.prev_length,o=e.nice_match,h=e.strstart>e.w_size-z?e.strstart-(e.w_size-z):0,u=e.window,l=e.w_mask,f=e.prev,c=e.strstart+S,d=u[s+a-1],p=u[s+a];e.prev_length>=e.good_match&&(i>>=2),o>e.lookahead&&(o=e.lookahead);do{if(u[(r=t)+a]===p&&u[r+a-1]===d&&u[r]===u[s]&&u[++r]===u[s+1]){s+=2,r++;do{}while(u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&s<c);if(n=S-(c-s),s=c-S,a<n){if(e.match_start=t,o<=(a=n))break;d=u[s+a-1],p=u[s+a]}}}while((t=f[t&l])>h&&0!=--i);return a<=e.lookahead?a:e.lookahead}function j(e){var t,r,n,i,s,a,o,h,u,l,f=e.w_size;do{if(i=e.window_size-e.lookahead-e.strstart,e.strstart>=f+(f-z)){for(c.arraySet(e.window,e.window,f,f,0),e.match_start-=f,e.strstart-=f,e.block_start-=f,t=r=e.hash_size;n=e.head[--t],e.head[t]=f<=n?n-f:0,--r;);for(t=r=f;n=e.prev[--t],e.prev[t]=f<=n?n-f:0,--r;);i+=f}if(0===e.strm.avail_in)break;if(a=e.strm,o=e.window,h=e.strstart+e.lookahead,u=i,l=void 0,l=a.avail_in,u<l&&(l=u),r=0===l?0:(a.avail_in-=l,c.arraySet(o,a.input,a.next_in,l,h),1===a.state.wrap?a.adler=d(a.adler,o,l,h):2===a.state.wrap&&(a.adler=p(a.adler,o,l,h)),a.next_in+=l,a.total_in+=l,l),e.lookahead+=r,e.lookahead+e.insert>=x)for(s=e.strstart-e.insert,e.ins_h=e.window[s],e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+1])&e.hash_mask;e.insert&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+x-1])&e.hash_mask,e.prev[s&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=s,s++,e.insert--,!(e.lookahead+e.insert<x)););}while(e.lookahead<z&&0!==e.strm.avail_in)}function Z(e,t){for(var r,n;;){if(e.lookahead<z){if(j(e),e.lookahead<z&&t===l)return A;if(0===e.lookahead)break}if(r=0,e.lookahead>=x&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!==r&&e.strstart-r<=e.w_size-z&&(e.match_length=L(e,r)),e.match_length>=x)if(n=u._tr_tally(e,e.strstart-e.match_start,e.match_length-x),e.lookahead-=e.match_length,e.match_length<=e.max_lazy_match&&e.lookahead>=x){for(e.match_length--;e.strstart++,e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart,0!=--e.match_length;);e.strstart++}else e.strstart+=e.match_length,e.match_length=0,e.ins_h=e.window[e.strstart],e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+1])&e.hash_mask;else n=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++;if(n&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=e.strstart<x-1?e.strstart:x-1,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}function W(e,t){for(var r,n,i;;){if(e.lookahead<z){if(j(e),e.lookahead<z&&t===l)return A;if(0===e.lookahead)break}if(r=0,e.lookahead>=x&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),e.prev_length=e.match_length,e.prev_match=e.match_start,e.match_length=x-1,0!==r&&e.prev_length<e.max_lazy_match&&e.strstart-r<=e.w_size-z&&(e.match_length=L(e,r),e.match_length<=5&&(1===e.strategy||e.match_length===x&&4096<e.strstart-e.match_start)&&(e.match_length=x-1)),e.prev_length>=x&&e.match_length<=e.prev_length){for(i=e.strstart+e.lookahead-x,n=u._tr_tally(e,e.strstart-1-e.prev_match,e.prev_length-x),e.lookahead-=e.prev_length-1,e.prev_length-=2;++e.strstart<=i&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!=--e.prev_length;);if(e.match_available=0,e.match_length=x-1,e.strstart++,n&&(N(e,!1),0===e.strm.avail_out))return A}else if(e.match_available){if((n=u._tr_tally(e,0,e.window[e.strstart-1]))&&N(e,!1),e.strstart++,e.lookahead--,0===e.strm.avail_out)return A}else e.match_available=1,e.strstart++,e.lookahead--}return e.match_available&&(n=u._tr_tally(e,0,e.window[e.strstart-1]),e.match_available=0),e.insert=e.strstart<x-1?e.strstart:x-1,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}function M(e,t,r,n,i){this.good_length=e,this.max_lazy=t,this.nice_length=r,this.max_chain=n,this.func=i}function H(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=v,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new c.Buf16(2*w),this.dyn_dtree=new c.Buf16(2*(2*a+1)),this.bl_tree=new c.Buf16(2*(2*o+1)),D(this.dyn_ltree),D(this.dyn_dtree),D(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new c.Buf16(k+1),this.heap=new c.Buf16(2*s+1),D(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new c.Buf16(2*s+1),D(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function G(e){var t;return e&&e.state?(e.total_in=e.total_out=0,e.data_type=i,(t=e.state).pending=0,t.pending_out=0,t.wrap<0&&(t.wrap=-t.wrap),t.status=t.wrap?C:E,e.adler=2===t.wrap?0:1,t.last_flush=l,u._tr_init(t),m):R(e,_)}function K(e){var t=G(e);return t===m&&function(e){e.window_size=2*e.w_size,D(e.head),e.max_lazy_match=h[e.level].max_lazy,e.good_match=h[e.level].good_length,e.nice_match=h[e.level].nice_length,e.max_chain_length=h[e.level].max_chain,e.strstart=0,e.block_start=0,e.lookahead=0,e.insert=0,e.match_length=e.prev_length=x-1,e.match_available=0,e.ins_h=0}(e.state),t}function Y(e,t,r,n,i,s){if(!e)return _;var a=1;if(t===g&&(t=6),n<0?(a=0,n=-n):15<n&&(a=2,n-=16),i<1||y<i||r!==v||n<8||15<n||t<0||9<t||s<0||b<s)return R(e,_);8===n&&(n=9);var o=new H;return(e.state=o).strm=e,o.wrap=a,o.gzhead=null,o.w_bits=n,o.w_size=1<<o.w_bits,o.w_mask=o.w_size-1,o.hash_bits=i+7,o.hash_size=1<<o.hash_bits,o.hash_mask=o.hash_size-1,o.hash_shift=~~((o.hash_bits+x-1)/x),o.window=new c.Buf8(2*o.w_size),o.head=new c.Buf16(o.hash_size),o.prev=new c.Buf16(o.w_size),o.lit_bufsize=1<<i+6,o.pending_buf_size=4*o.lit_bufsize,o.pending_buf=new c.Buf8(o.pending_buf_size),o.d_buf=1*o.lit_bufsize,o.l_buf=3*o.lit_bufsize,o.level=t,o.strategy=s,o.method=r,K(e)}h=[new M(0,0,0,0,function(e,t){var r=65535;for(r>e.pending_buf_size-5&&(r=e.pending_buf_size-5);;){if(e.lookahead<=1){if(j(e),0===e.lookahead&&t===l)return A;if(0===e.lookahead)break}e.strstart+=e.lookahead,e.lookahead=0;var n=e.block_start+r;if((0===e.strstart||e.strstart>=n)&&(e.lookahead=e.strstart-n,e.strstart=n,N(e,!1),0===e.strm.avail_out))return A;if(e.strstart-e.block_start>=e.w_size-z&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):(e.strstart>e.block_start&&(N(e,!1),e.strm.avail_out),A)}),new M(4,4,8,4,Z),new M(4,5,16,8,Z),new M(4,6,32,32,Z),new M(4,4,16,16,W),new M(8,16,32,32,W),new M(8,16,128,128,W),new M(8,32,128,256,W),new M(32,128,258,1024,W),new M(32,258,258,4096,W)],r.deflateInit=function(e,t){return Y(e,t,v,15,8,0)},r.deflateInit2=Y,r.deflateReset=K,r.deflateResetKeep=G,r.deflateSetHeader=function(e,t){return e&&e.state?2!==e.state.wrap?_:(e.state.gzhead=t,m):_},r.deflate=function(e,t){var r,n,i,s;if(!e||!e.state||5<t||t<0)return e?R(e,_):_;if(n=e.state,!e.output||!e.input&&0!==e.avail_in||666===n.status&&t!==f)return R(e,0===e.avail_out?-5:_);if(n.strm=e,r=n.last_flush,n.last_flush=t,n.status===C)if(2===n.wrap)e.adler=0,U(n,31),U(n,139),U(n,8),n.gzhead?(U(n,(n.gzhead.text?1:0)+(n.gzhead.hcrc?2:0)+(n.gzhead.extra?4:0)+(n.gzhead.name?8:0)+(n.gzhead.comment?16:0)),U(n,255&n.gzhead.time),U(n,n.gzhead.time>>8&255),U(n,n.gzhead.time>>16&255),U(n,n.gzhead.time>>24&255),U(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),U(n,255&n.gzhead.os),n.gzhead.extra&&n.gzhead.extra.length&&(U(n,255&n.gzhead.extra.length),U(n,n.gzhead.extra.length>>8&255)),n.gzhead.hcrc&&(e.adler=p(e.adler,n.pending_buf,n.pending,0)),n.gzindex=0,n.status=69):(U(n,0),U(n,0),U(n,0),U(n,0),U(n,0),U(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),U(n,3),n.status=E);else{var a=v+(n.w_bits-8<<4)<<8;a|=(2<=n.strategy||n.level<2?0:n.level<6?1:6===n.level?2:3)<<6,0!==n.strstart&&(a|=32),a+=31-a%31,n.status=E,P(n,a),0!==n.strstart&&(P(n,e.adler>>>16),P(n,65535&e.adler)),e.adler=1}if(69===n.status)if(n.gzhead.extra){for(i=n.pending;n.gzindex<(65535&n.gzhead.extra.length)&&(n.pending!==n.pending_buf_size||(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending!==n.pending_buf_size));)U(n,255&n.gzhead.extra[n.gzindex]),n.gzindex++;n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),n.gzindex===n.gzhead.extra.length&&(n.gzindex=0,n.status=73)}else n.status=73;if(73===n.status)if(n.gzhead.name){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.name.length?255&n.gzhead.name.charCodeAt(n.gzindex++):0,U(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.gzindex=0,n.status=91)}else n.status=91;if(91===n.status)if(n.gzhead.comment){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.comment.length?255&n.gzhead.comment.charCodeAt(n.gzindex++):0,U(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.status=103)}else n.status=103;if(103===n.status&&(n.gzhead.hcrc?(n.pending+2>n.pending_buf_size&&F(e),n.pending+2<=n.pending_buf_size&&(U(n,255&e.adler),U(n,e.adler>>8&255),e.adler=0,n.status=E)):n.status=E),0!==n.pending){if(F(e),0===e.avail_out)return n.last_flush=-1,m}else if(0===e.avail_in&&T(t)<=T(r)&&t!==f)return R(e,-5);if(666===n.status&&0!==e.avail_in)return R(e,-5);if(0!==e.avail_in||0!==n.lookahead||t!==l&&666!==n.status){var o=2===n.strategy?function(e,t){for(var r;;){if(0===e.lookahead&&(j(e),0===e.lookahead)){if(t===l)return A;break}if(e.match_length=0,r=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++,r&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}(n,t):3===n.strategy?function(e,t){for(var r,n,i,s,a=e.window;;){if(e.lookahead<=S){if(j(e),e.lookahead<=S&&t===l)return A;if(0===e.lookahead)break}if(e.match_length=0,e.lookahead>=x&&0<e.strstart&&(n=a[i=e.strstart-1])===a[++i]&&n===a[++i]&&n===a[++i]){s=e.strstart+S;do{}while(n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&i<s);e.match_length=S-(s-i),e.match_length>e.lookahead&&(e.match_length=e.lookahead)}if(e.match_length>=x?(r=u._tr_tally(e,1,e.match_length-x),e.lookahead-=e.match_length,e.strstart+=e.match_length,e.match_length=0):(r=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++),r&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}(n,t):h[n.level].func(n,t);if(o!==O&&o!==B||(n.status=666),o===A||o===O)return 0===e.avail_out&&(n.last_flush=-1),m;if(o===I&&(1===t?u._tr_align(n):5!==t&&(u._tr_stored_block(n,0,0,!1),3===t&&(D(n.head),0===n.lookahead&&(n.strstart=0,n.block_start=0,n.insert=0))),F(e),0===e.avail_out))return n.last_flush=-1,m}return t!==f?m:n.wrap<=0?1:(2===n.wrap?(U(n,255&e.adler),U(n,e.adler>>8&255),U(n,e.adler>>16&255),U(n,e.adler>>24&255),U(n,255&e.total_in),U(n,e.total_in>>8&255),U(n,e.total_in>>16&255),U(n,e.total_in>>24&255)):(P(n,e.adler>>>16),P(n,65535&e.adler)),F(e),0<n.wrap&&(n.wrap=-n.wrap),0!==n.pending?m:1)},r.deflateEnd=function(e){var t;return e&&e.state?(t=e.state.status)!==C&&69!==t&&73!==t&&91!==t&&103!==t&&t!==E&&666!==t?R(e,_):(e.state=null,t===E?R(e,-3):m):_},r.deflateSetDictionary=function(e,t){var r,n,i,s,a,o,h,u,l=t.length;if(!e||!e.state)return _;if(2===(s=(r=e.state).wrap)||1===s&&r.status!==C||r.lookahead)return _;for(1===s&&(e.adler=d(e.adler,t,l,0)),r.wrap=0,l>=r.w_size&&(0===s&&(D(r.head),r.strstart=0,r.block_start=0,r.insert=0),u=new c.Buf8(r.w_size),c.arraySet(u,t,l-r.w_size,r.w_size,0),t=u,l=r.w_size),a=e.avail_in,o=e.next_in,h=e.input,e.avail_in=l,e.next_in=0,e.input=t,j(r);r.lookahead>=x;){for(n=r.strstart,i=r.lookahead-(x-1);r.ins_h=(r.ins_h<<r.hash_shift^r.window[n+x-1])&r.hash_mask,r.prev[n&r.w_mask]=r.head[r.ins_h],r.head[r.ins_h]=n,n++,--i;);r.strstart=n,r.lookahead=x-1,j(r)}return r.strstart+=r.lookahead,r.block_start=r.strstart,r.insert=r.lookahead,r.lookahead=0,r.match_length=r.prev_length=x-1,r.match_available=0,e.next_in=o,e.input=h,e.avail_in=a,r.wrap=s,m},r.deflateInfo="pako deflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./messages":51,"./trees":52}],47:[function(e,t,r){"use strict";t.exports=function(){this.text=0,this.time=0,this.xflags=0,this.os=0,this.extra=null,this.extra_len=0,this.name="",this.comment="",this.hcrc=0,this.done=!1}},{}],48:[function(e,t,r){"use strict";t.exports=function(e,t){var r,n,i,s,a,o,h,u,l,f,c,d,p,m,_,g,b,v,y,w,k,x,S,z,C;r=e.state,n=e.next_in,z=e.input,i=n+(e.avail_in-5),s=e.next_out,C=e.output,a=s-(t-e.avail_out),o=s+(e.avail_out-257),h=r.dmax,u=r.wsize,l=r.whave,f=r.wnext,c=r.window,d=r.hold,p=r.bits,m=r.lencode,_=r.distcode,g=(1<<r.lenbits)-1,b=(1<<r.distbits)-1;e:do{p<15&&(d+=z[n++]<<p,p+=8,d+=z[n++]<<p,p+=8),v=m[d&g];t:for(;;){if(d>>>=y=v>>>24,p-=y,0===(y=v>>>16&255))C[s++]=65535&v;else{if(!(16&y)){if(0==(64&y)){v=m[(65535&v)+(d&(1<<y)-1)];continue t}if(32&y){r.mode=12;break e}e.msg="invalid literal/length code",r.mode=30;break e}w=65535&v,(y&=15)&&(p<y&&(d+=z[n++]<<p,p+=8),w+=d&(1<<y)-1,d>>>=y,p-=y),p<15&&(d+=z[n++]<<p,p+=8,d+=z[n++]<<p,p+=8),v=_[d&b];r:for(;;){if(d>>>=y=v>>>24,p-=y,!(16&(y=v>>>16&255))){if(0==(64&y)){v=_[(65535&v)+(d&(1<<y)-1)];continue r}e.msg="invalid distance code",r.mode=30;break e}if(k=65535&v,p<(y&=15)&&(d+=z[n++]<<p,(p+=8)<y&&(d+=z[n++]<<p,p+=8)),h<(k+=d&(1<<y)-1)){e.msg="invalid distance too far back",r.mode=30;break e}if(d>>>=y,p-=y,(y=s-a)<k){if(l<(y=k-y)&&r.sane){e.msg="invalid distance too far back",r.mode=30;break e}if(S=c,(x=0)===f){if(x+=u-y,y<w){for(w-=y;C[s++]=c[x++],--y;);x=s-k,S=C}}else if(f<y){if(x+=u+f-y,(y-=f)<w){for(w-=y;C[s++]=c[x++],--y;);if(x=0,f<w){for(w-=y=f;C[s++]=c[x++],--y;);x=s-k,S=C}}}else if(x+=f-y,y<w){for(w-=y;C[s++]=c[x++],--y;);x=s-k,S=C}for(;2<w;)C[s++]=S[x++],C[s++]=S[x++],C[s++]=S[x++],w-=3;w&&(C[s++]=S[x++],1<w&&(C[s++]=S[x++]))}else{for(x=s-k;C[s++]=C[x++],C[s++]=C[x++],C[s++]=C[x++],2<(w-=3););w&&(C[s++]=C[x++],1<w&&(C[s++]=C[x++]))}break}}break}}while(n<i&&s<o);n-=w=p>>3,d&=(1<<(p-=w<<3))-1,e.next_in=n,e.next_out=s,e.avail_in=n<i?i-n+5:5-(n-i),e.avail_out=s<o?o-s+257:257-(s-o),r.hold=d,r.bits=p}},{}],49:[function(e,t,r){"use strict";var I=e("../utils/common"),O=e("./adler32"),B=e("./crc32"),R=e("./inffast"),T=e("./inftrees"),D=1,F=2,N=0,U=-2,P=1,n=852,i=592;function L(e){return(e>>>24&255)+(e>>>8&65280)+((65280&e)<<8)+((255&e)<<24)}function s(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new I.Buf16(320),this.work=new I.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function a(e){var t;return e&&e.state?(t=e.state,e.total_in=e.total_out=t.total=0,e.msg="",t.wrap&&(e.adler=1&t.wrap),t.mode=P,t.last=0,t.havedict=0,t.dmax=32768,t.head=null,t.hold=0,t.bits=0,t.lencode=t.lendyn=new I.Buf32(n),t.distcode=t.distdyn=new I.Buf32(i),t.sane=1,t.back=-1,N):U}function o(e){var t;return e&&e.state?((t=e.state).wsize=0,t.whave=0,t.wnext=0,a(e)):U}function h(e,t){var r,n;return e&&e.state?(n=e.state,t<0?(r=0,t=-t):(r=1+(t>>4),t<48&&(t&=15)),t&&(t<8||15<t)?U:(null!==n.window&&n.wbits!==t&&(n.window=null),n.wrap=r,n.wbits=t,o(e))):U}function u(e,t){var r,n;return e?(n=new s,(e.state=n).window=null,(r=h(e,t))!==N&&(e.state=null),r):U}var l,f,c=!0;function j(e){if(c){var t;for(l=new I.Buf32(512),f=new I.Buf32(32),t=0;t<144;)e.lens[t++]=8;for(;t<256;)e.lens[t++]=9;for(;t<280;)e.lens[t++]=7;for(;t<288;)e.lens[t++]=8;for(T(D,e.lens,0,288,l,0,e.work,{bits:9}),t=0;t<32;)e.lens[t++]=5;T(F,e.lens,0,32,f,0,e.work,{bits:5}),c=!1}e.lencode=l,e.lenbits=9,e.distcode=f,e.distbits=5}function Z(e,t,r,n){var i,s=e.state;return null===s.window&&(s.wsize=1<<s.wbits,s.wnext=0,s.whave=0,s.window=new I.Buf8(s.wsize)),n>=s.wsize?(I.arraySet(s.window,t,r-s.wsize,s.wsize,0),s.wnext=0,s.whave=s.wsize):(n<(i=s.wsize-s.wnext)&&(i=n),I.arraySet(s.window,t,r-n,i,s.wnext),(n-=i)?(I.arraySet(s.window,t,r-n,n,0),s.wnext=n,s.whave=s.wsize):(s.wnext+=i,s.wnext===s.wsize&&(s.wnext=0),s.whave<s.wsize&&(s.whave+=i))),0}r.inflateReset=o,r.inflateReset2=h,r.inflateResetKeep=a,r.inflateInit=function(e){return u(e,15)},r.inflateInit2=u,r.inflate=function(e,t){var r,n,i,s,a,o,h,u,l,f,c,d,p,m,_,g,b,v,y,w,k,x,S,z,C=0,E=new I.Buf8(4),A=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!e||!e.state||!e.output||!e.input&&0!==e.avail_in)return U;12===(r=e.state).mode&&(r.mode=13),a=e.next_out,i=e.output,h=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,u=r.hold,l=r.bits,f=o,c=h,x=N;e:for(;;)switch(r.mode){case P:if(0===r.wrap){r.mode=13;break}for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(2&r.wrap&&35615===u){E[r.check=0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0),l=u=0,r.mode=2;break}if(r.flags=0,r.head&&(r.head.done=!1),!(1&r.wrap)||(((255&u)<<8)+(u>>8))%31){e.msg="incorrect header check",r.mode=30;break}if(8!=(15&u)){e.msg="unknown compression method",r.mode=30;break}if(l-=4,k=8+(15&(u>>>=4)),0===r.wbits)r.wbits=k;else if(k>r.wbits){e.msg="invalid window size",r.mode=30;break}r.dmax=1<<k,e.adler=r.check=1,r.mode=512&u?10:12,l=u=0;break;case 2:for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(r.flags=u,8!=(255&r.flags)){e.msg="unknown compression method",r.mode=30;break}if(57344&r.flags){e.msg="unknown header flags set",r.mode=30;break}r.head&&(r.head.text=u>>8&1),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=3;case 3:for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.head&&(r.head.time=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,E[2]=u>>>16&255,E[3]=u>>>24&255,r.check=B(r.check,E,4,0)),l=u=0,r.mode=4;case 4:for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.head&&(r.head.xflags=255&u,r.head.os=u>>8),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=5;case 5:if(1024&r.flags){for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.length=u,r.head&&(r.head.extra_len=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0}else r.head&&(r.head.extra=null);r.mode=6;case 6:if(1024&r.flags&&(o<(d=r.length)&&(d=o),d&&(r.head&&(k=r.head.extra_len-r.length,r.head.extra||(r.head.extra=new Array(r.head.extra_len)),I.arraySet(r.head.extra,n,s,d,k)),512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,r.length-=d),r.length))break e;r.length=0,r.mode=7;case 7:if(2048&r.flags){if(0===o)break e;for(d=0;k=n[s+d++],r.head&&k&&r.length<65536&&(r.head.name+=String.fromCharCode(k)),k&&d<o;);if(512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,k)break e}else r.head&&(r.head.name=null);r.length=0,r.mode=8;case 8:if(4096&r.flags){if(0===o)break e;for(d=0;k=n[s+d++],r.head&&k&&r.length<65536&&(r.head.comment+=String.fromCharCode(k)),k&&d<o;);if(512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,k)break e}else r.head&&(r.head.comment=null);r.mode=9;case 9:if(512&r.flags){for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u!==(65535&r.check)){e.msg="header crc mismatch",r.mode=30;break}l=u=0}r.head&&(r.head.hcrc=r.flags>>9&1,r.head.done=!0),e.adler=r.check=0,r.mode=12;break;case 10:for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}e.adler=r.check=L(u),l=u=0,r.mode=11;case 11:if(0===r.havedict)return e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,2;e.adler=r.check=1,r.mode=12;case 12:if(5===t||6===t)break e;case 13:if(r.last){u>>>=7&l,l-=7&l,r.mode=27;break}for(;l<3;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}switch(r.last=1&u,l-=1,3&(u>>>=1)){case 0:r.mode=14;break;case 1:if(j(r),r.mode=20,6!==t)break;u>>>=2,l-=2;break e;case 2:r.mode=17;break;case 3:e.msg="invalid block type",r.mode=30}u>>>=2,l-=2;break;case 14:for(u>>>=7&l,l-=7&l;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if((65535&u)!=(u>>>16^65535)){e.msg="invalid stored block lengths",r.mode=30;break}if(r.length=65535&u,l=u=0,r.mode=15,6===t)break e;case 15:r.mode=16;case 16:if(d=r.length){if(o<d&&(d=o),h<d&&(d=h),0===d)break e;I.arraySet(i,n,s,d,a),o-=d,s+=d,h-=d,a+=d,r.length-=d;break}r.mode=12;break;case 17:for(;l<14;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(r.nlen=257+(31&u),u>>>=5,l-=5,r.ndist=1+(31&u),u>>>=5,l-=5,r.ncode=4+(15&u),u>>>=4,l-=4,286<r.nlen||30<r.ndist){e.msg="too many length or distance symbols",r.mode=30;break}r.have=0,r.mode=18;case 18:for(;r.have<r.ncode;){for(;l<3;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.lens[A[r.have++]]=7&u,u>>>=3,l-=3}for(;r.have<19;)r.lens[A[r.have++]]=0;if(r.lencode=r.lendyn,r.lenbits=7,S={bits:r.lenbits},x=T(0,r.lens,0,19,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid code lengths set",r.mode=30;break}r.have=0,r.mode=19;case 19:for(;r.have<r.nlen+r.ndist;){for(;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(b<16)u>>>=_,l-=_,r.lens[r.have++]=b;else{if(16===b){for(z=_+2;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u>>>=_,l-=_,0===r.have){e.msg="invalid bit length repeat",r.mode=30;break}k=r.lens[r.have-1],d=3+(3&u),u>>>=2,l-=2}else if(17===b){for(z=_+3;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}l-=_,k=0,d=3+(7&(u>>>=_)),u>>>=3,l-=3}else{for(z=_+7;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}l-=_,k=0,d=11+(127&(u>>>=_)),u>>>=7,l-=7}if(r.have+d>r.nlen+r.ndist){e.msg="invalid bit length repeat",r.mode=30;break}for(;d--;)r.lens[r.have++]=k}}if(30===r.mode)break;if(0===r.lens[256]){e.msg="invalid code -- missing end-of-block",r.mode=30;break}if(r.lenbits=9,S={bits:r.lenbits},x=T(D,r.lens,0,r.nlen,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid literal/lengths set",r.mode=30;break}if(r.distbits=6,r.distcode=r.distdyn,S={bits:r.distbits},x=T(F,r.lens,r.nlen,r.ndist,r.distcode,0,r.work,S),r.distbits=S.bits,x){e.msg="invalid distances set",r.mode=30;break}if(r.mode=20,6===t)break e;case 20:r.mode=21;case 21:if(6<=o&&258<=h){e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,R(e,c),a=e.next_out,i=e.output,h=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,u=r.hold,l=r.bits,12===r.mode&&(r.back=-1);break}for(r.back=0;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(g&&0==(240&g)){for(v=_,y=g,w=b;g=(C=r.lencode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,r.length=b,0===g){r.mode=26;break}if(32&g){r.back=-1,r.mode=12;break}if(64&g){e.msg="invalid literal/length code",r.mode=30;break}r.extra=15&g,r.mode=22;case 22:if(r.extra){for(z=r.extra;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.length+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}r.was=r.length,r.mode=23;case 23:for(;g=(C=r.distcode[u&(1<<r.distbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(0==(240&g)){for(v=_,y=g,w=b;g=(C=r.distcode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,64&g){e.msg="invalid distance code",r.mode=30;break}r.offset=b,r.extra=15&g,r.mode=24;case 24:if(r.extra){for(z=r.extra;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.offset+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}if(r.offset>r.dmax){e.msg="invalid distance too far back",r.mode=30;break}r.mode=25;case 25:if(0===h)break e;if(d=c-h,r.offset>d){if((d=r.offset-d)>r.whave&&r.sane){e.msg="invalid distance too far back",r.mode=30;break}p=d>r.wnext?(d-=r.wnext,r.wsize-d):r.wnext-d,d>r.length&&(d=r.length),m=r.window}else m=i,p=a-r.offset,d=r.length;for(h<d&&(d=h),h-=d,r.length-=d;i[a++]=m[p++],--d;);0===r.length&&(r.mode=21);break;case 26:if(0===h)break e;i[a++]=r.length,h--,r.mode=21;break;case 27:if(r.wrap){for(;l<32;){if(0===o)break e;o--,u|=n[s++]<<l,l+=8}if(c-=h,e.total_out+=c,r.total+=c,c&&(e.adler=r.check=r.flags?B(r.check,i,c,a-c):O(r.check,i,c,a-c)),c=h,(r.flags?u:L(u))!==r.check){e.msg="incorrect data check",r.mode=30;break}l=u=0}r.mode=28;case 28:if(r.wrap&&r.flags){for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u!==(4294967295&r.total)){e.msg="incorrect length check",r.mode=30;break}l=u=0}r.mode=29;case 29:x=1;break e;case 30:x=-3;break e;case 31:return-4;case 32:default:return U}return e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,(r.wsize||c!==e.avail_out&&r.mode<30&&(r.mode<27||4!==t))&&Z(e,e.output,e.next_out,c-e.avail_out)?(r.mode=31,-4):(f-=e.avail_in,c-=e.avail_out,e.total_in+=f,e.total_out+=c,r.total+=c,r.wrap&&c&&(e.adler=r.check=r.flags?B(r.check,i,c,e.next_out-c):O(r.check,i,c,e.next_out-c)),e.data_type=r.bits+(r.last?64:0)+(12===r.mode?128:0)+(20===r.mode||15===r.mode?256:0),(0==f&&0===c||4===t)&&x===N&&(x=-5),x)},r.inflateEnd=function(e){if(!e||!e.state)return U;var t=e.state;return t.window&&(t.window=null),e.state=null,N},r.inflateGetHeader=function(e,t){var r;return e&&e.state?0==(2&(r=e.state).wrap)?U:((r.head=t).done=!1,N):U},r.inflateSetDictionary=function(e,t){var r,n=t.length;return e&&e.state?0!==(r=e.state).wrap&&11!==r.mode?U:11===r.mode&&O(1,t,n,0)!==r.check?-3:Z(e,t,n,n)?(r.mode=31,-4):(r.havedict=1,N):U},r.inflateInfo="pako inflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./inffast":48,"./inftrees":50}],50:[function(e,t,r){"use strict";var D=e("../utils/common"),F=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],N=[16,16,16,16,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,16,72,78],U=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0],P=[16,16,16,16,17,17,18,18,19,19,20,20,21,21,22,22,23,23,24,24,25,25,26,26,27,27,28,28,29,29,64,64];t.exports=function(e,t,r,n,i,s,a,o){var h,u,l,f,c,d,p,m,_,g=o.bits,b=0,v=0,y=0,w=0,k=0,x=0,S=0,z=0,C=0,E=0,A=null,I=0,O=new D.Buf16(16),B=new D.Buf16(16),R=null,T=0;for(b=0;b<=15;b++)O[b]=0;for(v=0;v<n;v++)O[t[r+v]]++;for(k=g,w=15;1<=w&&0===O[w];w--);if(w<k&&(k=w),0===w)return i[s++]=20971520,i[s++]=20971520,o.bits=1,0;for(y=1;y<w&&0===O[y];y++);for(k<y&&(k=y),b=z=1;b<=15;b++)if(z<<=1,(z-=O[b])<0)return-1;if(0<z&&(0===e||1!==w))return-1;for(B[1]=0,b=1;b<15;b++)B[b+1]=B[b]+O[b];for(v=0;v<n;v++)0!==t[r+v]&&(a[B[t[r+v]]++]=v);if(d=0===e?(A=R=a,19):1===e?(A=F,I-=257,R=N,T-=257,256):(A=U,R=P,-1),b=y,c=s,S=v=E=0,l=-1,f=(C=1<<(x=k))-1,1===e&&852<C||2===e&&592<C)return 1;for(;;){for(p=b-S,_=a[v]<d?(m=0,a[v]):a[v]>d?(m=R[T+a[v]],A[I+a[v]]):(m=96,0),h=1<<b-S,y=u=1<<x;i[c+(E>>S)+(u-=h)]=p<<24|m<<16|_|0,0!==u;);for(h=1<<b-1;E&h;)h>>=1;if(0!==h?(E&=h-1,E+=h):E=0,v++,0==--O[b]){if(b===w)break;b=t[r+a[v]]}if(k<b&&(E&f)!==l){for(0===S&&(S=k),c+=y,z=1<<(x=b-S);x+S<w&&!((z-=O[x+S])<=0);)x++,z<<=1;if(C+=1<<x,1===e&&852<C||2===e&&592<C)return 1;i[l=E&f]=k<<24|x<<16|c-s|0}}return 0!==E&&(i[c+E]=b-S<<24|64<<16|0),o.bits=k,0}},{"../utils/common":41}],51:[function(e,t,r){"use strict";t.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},{}],52:[function(e,t,r){"use strict";var i=e("../utils/common"),o=0,h=1;function n(e){for(var t=e.length;0<=--t;)e[t]=0}var s=0,a=29,u=256,l=u+1+a,f=30,c=19,_=2*l+1,g=15,d=16,p=7,m=256,b=16,v=17,y=18,w=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],k=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],x=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],S=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],z=new Array(2*(l+2));n(z);var C=new Array(2*f);n(C);var E=new Array(512);n(E);var A=new Array(256);n(A);var I=new Array(a);n(I);var O,B,R,T=new Array(f);function D(e,t,r,n,i){this.static_tree=e,this.extra_bits=t,this.extra_base=r,this.elems=n,this.max_length=i,this.has_stree=e&&e.length}function F(e,t){this.dyn_tree=e,this.max_code=0,this.stat_desc=t}function N(e){return e<256?E[e]:E[256+(e>>>7)]}function U(e,t){e.pending_buf[e.pending++]=255&t,e.pending_buf[e.pending++]=t>>>8&255}function P(e,t,r){e.bi_valid>d-r?(e.bi_buf|=t<<e.bi_valid&65535,U(e,e.bi_buf),e.bi_buf=t>>d-e.bi_valid,e.bi_valid+=r-d):(e.bi_buf|=t<<e.bi_valid&65535,e.bi_valid+=r)}function L(e,t,r){P(e,r[2*t],r[2*t+1])}function j(e,t){for(var r=0;r|=1&e,e>>>=1,r<<=1,0<--t;);return r>>>1}function Z(e,t,r){var n,i,s=new Array(g+1),a=0;for(n=1;n<=g;n++)s[n]=a=a+r[n-1]<<1;for(i=0;i<=t;i++){var o=e[2*i+1];0!==o&&(e[2*i]=j(s[o]++,o))}}function W(e){var t;for(t=0;t<l;t++)e.dyn_ltree[2*t]=0;for(t=0;t<f;t++)e.dyn_dtree[2*t]=0;for(t=0;t<c;t++)e.bl_tree[2*t]=0;e.dyn_ltree[2*m]=1,e.opt_len=e.static_len=0,e.last_lit=e.matches=0}function M(e){8<e.bi_valid?U(e,e.bi_buf):0<e.bi_valid&&(e.pending_buf[e.pending++]=e.bi_buf),e.bi_buf=0,e.bi_valid=0}function H(e,t,r,n){var i=2*t,s=2*r;return e[i]<e[s]||e[i]===e[s]&&n[t]<=n[r]}function G(e,t,r){for(var n=e.heap[r],i=r<<1;i<=e.heap_len&&(i<e.heap_len&&H(t,e.heap[i+1],e.heap[i],e.depth)&&i++,!H(t,n,e.heap[i],e.depth));)e.heap[r]=e.heap[i],r=i,i<<=1;e.heap[r]=n}function K(e,t,r){var n,i,s,a,o=0;if(0!==e.last_lit)for(;n=e.pending_buf[e.d_buf+2*o]<<8|e.pending_buf[e.d_buf+2*o+1],i=e.pending_buf[e.l_buf+o],o++,0===n?L(e,i,t):(L(e,(s=A[i])+u+1,t),0!==(a=w[s])&&P(e,i-=I[s],a),L(e,s=N(--n),r),0!==(a=k[s])&&P(e,n-=T[s],a)),o<e.last_lit;);L(e,m,t)}function Y(e,t){var r,n,i,s=t.dyn_tree,a=t.stat_desc.static_tree,o=t.stat_desc.has_stree,h=t.stat_desc.elems,u=-1;for(e.heap_len=0,e.heap_max=_,r=0;r<h;r++)0!==s[2*r]?(e.heap[++e.heap_len]=u=r,e.depth[r]=0):s[2*r+1]=0;for(;e.heap_len<2;)s[2*(i=e.heap[++e.heap_len]=u<2?++u:0)]=1,e.depth[i]=0,e.opt_len--,o&&(e.static_len-=a[2*i+1]);for(t.max_code=u,r=e.heap_len>>1;1<=r;r--)G(e,s,r);for(i=h;r=e.heap[1],e.heap[1]=e.heap[e.heap_len--],G(e,s,1),n=e.heap[1],e.heap[--e.heap_max]=r,e.heap[--e.heap_max]=n,s[2*i]=s[2*r]+s[2*n],e.depth[i]=(e.depth[r]>=e.depth[n]?e.depth[r]:e.depth[n])+1,s[2*r+1]=s[2*n+1]=i,e.heap[1]=i++,G(e,s,1),2<=e.heap_len;);e.heap[--e.heap_max]=e.heap[1],function(e,t){var r,n,i,s,a,o,h=t.dyn_tree,u=t.max_code,l=t.stat_desc.static_tree,f=t.stat_desc.has_stree,c=t.stat_desc.extra_bits,d=t.stat_desc.extra_base,p=t.stat_desc.max_length,m=0;for(s=0;s<=g;s++)e.bl_count[s]=0;for(h[2*e.heap[e.heap_max]+1]=0,r=e.heap_max+1;r<_;r++)p<(s=h[2*h[2*(n=e.heap[r])+1]+1]+1)&&(s=p,m++),h[2*n+1]=s,u<n||(e.bl_count[s]++,a=0,d<=n&&(a=c[n-d]),o=h[2*n],e.opt_len+=o*(s+a),f&&(e.static_len+=o*(l[2*n+1]+a)));if(0!==m){do{for(s=p-1;0===e.bl_count[s];)s--;e.bl_count[s]--,e.bl_count[s+1]+=2,e.bl_count[p]--,m-=2}while(0<m);for(s=p;0!==s;s--)for(n=e.bl_count[s];0!==n;)u<(i=e.heap[--r])||(h[2*i+1]!==s&&(e.opt_len+=(s-h[2*i+1])*h[2*i],h[2*i+1]=s),n--)}}(e,t),Z(s,u,e.bl_count)}function X(e,t,r){var n,i,s=-1,a=t[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),t[2*(r+1)+1]=65535,n=0;n<=r;n++)i=a,a=t[2*(n+1)+1],++o<h&&i===a||(o<u?e.bl_tree[2*i]+=o:0!==i?(i!==s&&e.bl_tree[2*i]++,e.bl_tree[2*b]++):o<=10?e.bl_tree[2*v]++:e.bl_tree[2*y]++,s=i,u=(o=0)===a?(h=138,3):i===a?(h=6,3):(h=7,4))}function V(e,t,r){var n,i,s=-1,a=t[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),n=0;n<=r;n++)if(i=a,a=t[2*(n+1)+1],!(++o<h&&i===a)){if(o<u)for(;L(e,i,e.bl_tree),0!=--o;);else 0!==i?(i!==s&&(L(e,i,e.bl_tree),o--),L(e,b,e.bl_tree),P(e,o-3,2)):o<=10?(L(e,v,e.bl_tree),P(e,o-3,3)):(L(e,y,e.bl_tree),P(e,o-11,7));s=i,u=(o=0)===a?(h=138,3):i===a?(h=6,3):(h=7,4)}}n(T);var q=!1;function J(e,t,r,n){P(e,(s<<1)+(n?1:0),3),function(e,t,r,n){M(e),n&&(U(e,r),U(e,~r)),i.arraySet(e.pending_buf,e.window,t,r,e.pending),e.pending+=r}(e,t,r,!0)}r._tr_init=function(e){q||(function(){var e,t,r,n,i,s=new Array(g+1);for(n=r=0;n<a-1;n++)for(I[n]=r,e=0;e<1<<w[n];e++)A[r++]=n;for(A[r-1]=n,n=i=0;n<16;n++)for(T[n]=i,e=0;e<1<<k[n];e++)E[i++]=n;for(i>>=7;n<f;n++)for(T[n]=i<<7,e=0;e<1<<k[n]-7;e++)E[256+i++]=n;for(t=0;t<=g;t++)s[t]=0;for(e=0;e<=143;)z[2*e+1]=8,e++,s[8]++;for(;e<=255;)z[2*e+1]=9,e++,s[9]++;for(;e<=279;)z[2*e+1]=7,e++,s[7]++;for(;e<=287;)z[2*e+1]=8,e++,s[8]++;for(Z(z,l+1,s),e=0;e<f;e++)C[2*e+1]=5,C[2*e]=j(e,5);O=new D(z,w,u+1,l,g),B=new D(C,k,0,f,g),R=new D(new Array(0),x,0,c,p)}(),q=!0),e.l_desc=new F(e.dyn_ltree,O),e.d_desc=new F(e.dyn_dtree,B),e.bl_desc=new F(e.bl_tree,R),e.bi_buf=0,e.bi_valid=0,W(e)},r._tr_stored_block=J,r._tr_flush_block=function(e,t,r,n){var i,s,a=0;0<e.level?(2===e.strm.data_type&&(e.strm.data_type=function(e){var t,r=4093624447;for(t=0;t<=31;t++,r>>>=1)if(1&r&&0!==e.dyn_ltree[2*t])return o;if(0!==e.dyn_ltree[18]||0!==e.dyn_ltree[20]||0!==e.dyn_ltree[26])return h;for(t=32;t<u;t++)if(0!==e.dyn_ltree[2*t])return h;return o}(e)),Y(e,e.l_desc),Y(e,e.d_desc),a=function(e){var t;for(X(e,e.dyn_ltree,e.l_desc.max_code),X(e,e.dyn_dtree,e.d_desc.max_code),Y(e,e.bl_desc),t=c-1;3<=t&&0===e.bl_tree[2*S[t]+1];t--);return e.opt_len+=3*(t+1)+5+5+4,t}(e),i=e.opt_len+3+7>>>3,(s=e.static_len+3+7>>>3)<=i&&(i=s)):i=s=r+5,r+4<=i&&-1!==t?J(e,t,r,n):4===e.strategy||s===i?(P(e,2+(n?1:0),3),K(e,z,C)):(P(e,4+(n?1:0),3),function(e,t,r,n){var i;for(P(e,t-257,5),P(e,r-1,5),P(e,n-4,4),i=0;i<n;i++)P(e,e.bl_tree[2*S[i]+1],3);V(e,e.dyn_ltree,t-1),V(e,e.dyn_dtree,r-1)}(e,e.l_desc.max_code+1,e.d_desc.max_code+1,a+1),K(e,e.dyn_ltree,e.dyn_dtree)),W(e),n&&M(e)},r._tr_tally=function(e,t,r){return e.pending_buf[e.d_buf+2*e.last_lit]=t>>>8&255,e.pending_buf[e.d_buf+2*e.last_lit+1]=255&t,e.pending_buf[e.l_buf+e.last_lit]=255&r,e.last_lit++,0===t?e.dyn_ltree[2*r]++:(e.matches++,t--,e.dyn_ltree[2*(A[r]+u+1)]++,e.dyn_dtree[2*N(t)]++),e.last_lit===e.lit_bufsize-1},r._tr_align=function(e){P(e,2,3),L(e,m,z),function(e){16===e.bi_valid?(U(e,e.bi_buf),e.bi_buf=0,e.bi_valid=0):8<=e.bi_valid&&(e.pending_buf[e.pending++]=255&e.bi_buf,e.bi_buf>>=8,e.bi_valid-=8)}(e)}},{"../utils/common":41}],53:[function(e,t,r){"use strict";t.exports=function(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},{}],54:[function(e,t,r){(function(e){!function(r,n){"use strict";if(!r.setImmediate){var i,s,t,a,o=1,h={},u=!1,l=r.document,e=Object.getPrototypeOf&&Object.getPrototypeOf(r);e=e&&e.setTimeout?e:r,i="[object process]"==={}.toString.call(r.process)?function(e){process.nextTick(function(){c(e)})}:function(){if(r.postMessage&&!r.importScripts){var e=!0,t=r.onmessage;return r.onmessage=function(){e=!1},r.postMessage("","*"),r.onmessage=t,e}}()?(a="setImmediate$"+Math.random()+"$",r.addEventListener?r.addEventListener("message",d,!1):r.attachEvent("onmessage",d),function(e){r.postMessage(a+e,"*")}):r.MessageChannel?((t=new MessageChannel).port1.onmessage=function(e){c(e.data)},function(e){t.port2.postMessage(e)}):l&&"onreadystatechange"in l.createElement("script")?(s=l.documentElement,function(e){var t=l.createElement("script");t.onreadystatechange=function(){c(e),t.onreadystatechange=null,s.removeChild(t),t=null},s.appendChild(t)}):function(e){setTimeout(c,0,e)},e.setImmediate=function(e){"function"!=typeof e&&(e=new Function(""+e));for(var t=new Array(arguments.length-1),r=0;r<t.length;r++)t[r]=arguments[r+1];var n={callback:e,args:t};return h[o]=n,i(o),o++},e.clearImmediate=f}function f(e){delete h[e]}function c(e){if(u)setTimeout(c,0,e);else{var t=h[e];if(t){u=!0;try{!function(e){var t=e.callback,r=e.args;switch(r.length){case 0:t();break;case 1:t(r[0]);break;case 2:t(r[0],r[1]);break;case 3:t(r[0],r[1],r[2]);break;default:t.apply(n,r)}}(t)}finally{f(e),u=!1}}}}function d(e){e.source===r&&"string"==typeof e.data&&0===e.data.indexOf(a)&&c(+e.data.slice(a.length))}}("undefined"==typeof self?void 0===e?this:e:self)}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[10])(10)});
/*! pdfmake v0.2.5, @license MIT, @link http://pdfmake.org */
!function webpackUniversalModuleDefinition(r,i){if("object"==typeof exports&&"object"==typeof module)module.exports=i();else if("function"==typeof define&&define.amd)define([],i);else{var o=i();for(var u in o)("object"==typeof exports?exports:r)[u]=o[u]}}("undefined"!=typeof self?self:this,(function(){return function(){var r={9282:function(r,i,o){"use strict";var u=o(4155);function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function _typeof(r){return typeof r}:function _typeof(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}var p,g,y=o(2136).codes,w=y.ERR_AMBIGUOUS_ARGUMENT,_=y.ERR_INVALID_ARG_TYPE,x=y.ERR_INVALID_ARG_VALUE,k=y.ERR_INVALID_RETURN_VALUE,P=y.ERR_MISSING_ARGS,O=o(5961),I=o(9539).inspect,B=o(9539).types,R=B.isPromise,D=B.isRegExp,N=Object.assign?Object.assign:o(8091).assign,W=Object.is?Object.is:o(609);new Map;function lazyLoadComparison(){var r=o(9158);p=r.isDeepEqual,g=r.isDeepStrictEqual}var U=!1,j=r.exports=ok,G={};function innerFail(r){if(r.message instanceof Error)throw r.message;throw new O(r)}function innerOk(r,i,o,u){if(!o){var p=!1;if(0===i)p=!0,u="No value argument passed to `assert.ok()`";else if(u instanceof Error)throw u;var g=new O({actual:o,expected:!0,message:u,operator:"==",stackStartFn:r});throw g.generatedMessage=p,g}}function ok(){for(var r=arguments.length,i=new Array(r),o=0;o<r;o++)i[o]=arguments[o];innerOk.apply(void 0,[ok,i.length].concat(i))}j.fail=function fail(r,i,o,p,g){var y,w=arguments.length;if(0===w)y="Failed";else if(1===w)o=r,r=void 0;else{if(!1===U){U=!0;var _=u.emitWarning?u.emitWarning:void 0;_("assert.fail() with more than one argument is deprecated. Please use assert.strictEqual() instead or only pass a message.","DeprecationWarning","DEP0094")}2===w&&(p="!=")}if(o instanceof Error)throw o;var x={actual:r,expected:i,operator:void 0===p?"fail":p,stackStartFn:g||fail};void 0!==o&&(x.message=o);var k=new O(x);throw y&&(k.message=y,k.generatedMessage=!0),k},j.AssertionError=O,j.ok=ok,j.equal=function equal(r,i,o){if(arguments.length<2)throw new P("actual","expected");r!=i&&innerFail({actual:r,expected:i,message:o,operator:"==",stackStartFn:equal})},j.notEqual=function notEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");r==i&&innerFail({actual:r,expected:i,message:o,operator:"!=",stackStartFn:notEqual})},j.deepEqual=function deepEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison(),p(r,i)||innerFail({actual:r,expected:i,message:o,operator:"deepEqual",stackStartFn:deepEqual})},j.notDeepEqual=function notDeepEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison(),p(r,i)&&innerFail({actual:r,expected:i,message:o,operator:"notDeepEqual",stackStartFn:notDeepEqual})},j.deepStrictEqual=function deepStrictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison(),g(r,i)||innerFail({actual:r,expected:i,message:o,operator:"deepStrictEqual",stackStartFn:deepStrictEqual})},j.notDeepStrictEqual=function notDeepStrictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison();g(r,i)&&innerFail({actual:r,expected:i,message:o,operator:"notDeepStrictEqual",stackStartFn:notDeepStrictEqual})},j.strictEqual=function strictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");W(r,i)||innerFail({actual:r,expected:i,message:o,operator:"strictEqual",stackStartFn:strictEqual})},j.notStrictEqual=function notStrictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");W(r,i)&&innerFail({actual:r,expected:i,message:o,operator:"notStrictEqual",stackStartFn:notStrictEqual})};var X=function Comparison(r,i,o){var u=this;!function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,Comparison),i.forEach((function(i){i in r&&(void 0!==o&&"string"==typeof o[i]&&D(r[i])&&r[i].test(o[i])?u[i]=o[i]:u[i]=r[i])}))};function compareExceptionKey(r,i,o,u,p,y){if(!(o in r)||!g(r[o],i[o])){if(!u){var w=new X(r,p),_=new X(i,p,r),x=new O({actual:w,expected:_,operator:"deepStrictEqual",stackStartFn:y});throw x.actual=r,x.expected=i,x.operator=y.name,x}innerFail({actual:r,expected:i,message:u,operator:y.name,stackStartFn:y})}}function expectedException(r,i,o,u){if("function"!=typeof i){if(D(i))return i.test(r);if(2===arguments.length)throw new _("expected",["Function","RegExp"],i);if("object"!==_typeof(r)||null===r){var g=new O({actual:r,expected:i,message:o,operator:"deepStrictEqual",stackStartFn:u});throw g.operator=u.name,g}var y=Object.keys(i);if(i instanceof Error)y.push("name","message");else if(0===y.length)throw new x("error",i,"may not be an empty object");return void 0===p&&lazyLoadComparison(),y.forEach((function(p){"string"==typeof r[p]&&D(i[p])&&i[p].test(r[p])||compareExceptionKey(r,i,p,o,y,u)})),!0}return void 0!==i.prototype&&r instanceof i||!Error.isPrototypeOf(i)&&!0===i.call({},r)}function getActual(r){if("function"!=typeof r)throw new _("fn","Function",r);try{r()}catch(r){return r}return G}function checkIsPromise(r){return R(r)||null!==r&&"object"===_typeof(r)&&"function"==typeof r.then&&"function"==typeof r.catch}function waitForActual(r){return Promise.resolve().then((function(){var i;if("function"==typeof r){if(!checkIsPromise(i=r()))throw new k("instance of Promise","promiseFn",i)}else{if(!checkIsPromise(r))throw new _("promiseFn",["Function","Promise"],r);i=r}return Promise.resolve().then((function(){return i})).then((function(){return G})).catch((function(r){return r}))}))}function expectsError(r,i,o,u){if("string"==typeof o){if(4===arguments.length)throw new _("error",["Object","Error","Function","RegExp"],o);if("object"===_typeof(i)&&null!==i){if(i.message===o)throw new w("error/message",'The error message "'.concat(i.message,'" is identical to the message.'))}else if(i===o)throw new w("error/message",'The error "'.concat(i,'" is identical to the message.'));u=o,o=void 0}else if(null!=o&&"object"!==_typeof(o)&&"function"!=typeof o)throw new _("error",["Object","Error","Function","RegExp"],o);if(i===G){var p="";o&&o.name&&(p+=" (".concat(o.name,")")),p+=u?": ".concat(u):".";var g="rejects"===r.name?"rejection":"exception";innerFail({actual:void 0,expected:o,operator:r.name,message:"Missing expected ".concat(g).concat(p),stackStartFn:r})}if(o&&!expectedException(i,o,u,r))throw i}function expectsNoError(r,i,o,u){if(i!==G){if("string"==typeof o&&(u=o,o=void 0),!o||expectedException(i,o)){var p=u?": ".concat(u):".",g="doesNotReject"===r.name?"rejection":"exception";innerFail({actual:i,expected:o,operator:r.name,message:"Got unwanted ".concat(g).concat(p,"\n")+'Actual message: "'.concat(i&&i.message,'"'),stackStartFn:r})}throw i}}function strict(){for(var r=arguments.length,i=new Array(r),o=0;o<r;o++)i[o]=arguments[o];innerOk.apply(void 0,[strict,i.length].concat(i))}j.throws=function throws(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];expectsError.apply(void 0,[throws,getActual(r)].concat(o))},j.rejects=function rejects(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];return waitForActual(r).then((function(r){return expectsError.apply(void 0,[rejects,r].concat(o))}))},j.doesNotThrow=function doesNotThrow(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];expectsNoError.apply(void 0,[doesNotThrow,getActual(r)].concat(o))},j.doesNotReject=function doesNotReject(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];return waitForActual(r).then((function(r){return expectsNoError.apply(void 0,[doesNotReject,r].concat(o))}))},j.ifError=function ifError(r){if(null!=r){var i="ifError got unwanted exception: ";"object"===_typeof(r)&&"string"==typeof r.message?0===r.message.length&&r.constructor?i+=r.constructor.name:i+=r.message:i+=I(r);var o=new O({actual:r,expected:null,operator:"ifError",message:i,stackStartFn:ifError}),u=r.stack;if("string"==typeof u){var p=u.split("\n");p.shift();for(var g=o.stack.split("\n"),y=0;y<p.length;y++){var w=g.indexOf(p[y]);if(-1!==w){g=g.slice(0,w);break}}o.stack="".concat(g.join("\n"),"\n").concat(p.join("\n"))}throw o}},j.strict=N(strict,j,{equal:j.strictEqual,deepEqual:j.deepStrictEqual,notEqual:j.notStrictEqual,notDeepEqual:j.notDeepStrictEqual}),j.strict.strict=j.strict},5961:function(r,i,o){"use strict";var u=o(4155);function _defineProperty(r,i,o){return i in r?Object.defineProperty(r,i,{value:o,enumerable:!0,configurable:!0,writable:!0}):r[i]=o,r}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,u.key,u)}}function _possibleConstructorReturn(r,i){return!i||"object"!==_typeof(i)&&"function"!=typeof i?_assertThisInitialized(r):i}function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}function _wrapNativeSuper(r){var i="function"==typeof Map?new Map:void 0;return _wrapNativeSuper=function _wrapNativeSuper(r){if(null===r||!function _isNativeFunction(r){return-1!==Function.toString.call(r).indexOf("[native code]")}(r))return r;if("function"!=typeof r)throw new TypeError("Super expression must either be null or a function");if(void 0!==i){if(i.has(r))return i.get(r);i.set(r,Wrapper)}function Wrapper(){return _construct(r,arguments,_getPrototypeOf(this).constructor)}return Wrapper.prototype=Object.create(r.prototype,{constructor:{value:Wrapper,enumerable:!1,writable:!0,configurable:!0}}),_setPrototypeOf(Wrapper,r)},_wrapNativeSuper(r)}function isNativeReflectConstruct(){if("undefined"==typeof Reflect||!Reflect.construct)return!1;if(Reflect.construct.sham)return!1;if("function"==typeof Proxy)return!0;try{return Date.prototype.toString.call(Reflect.construct(Date,[],(function(){}))),!0}catch(r){return!1}}function _construct(r,i,o){return _construct=isNativeReflectConstruct()?Reflect.construct:function _construct(r,i,o){var u=[null];u.push.apply(u,i);var p=new(Function.bind.apply(r,u));return o&&_setPrototypeOf(p,o.prototype),p},_construct.apply(null,arguments)}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}function _getPrototypeOf(r){return _getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf:function _getPrototypeOf(r){return r.__proto__||Object.getPrototypeOf(r)},_getPrototypeOf(r)}function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function _typeof(r){return typeof r}:function _typeof(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}var p=o(9539).inspect,g=o(2136).codes.ERR_INVALID_ARG_TYPE;function endsWith(r,i,o){return(void 0===o||o>r.length)&&(o=r.length),r.substring(o-i.length,o)===i}var y="",w="",_="",x="",k={deepStrictEqual:"Expected values to be strictly deep-equal:",strictEqual:"Expected values to be strictly equal:",strictEqualObject:'Expected "actual" to be reference-equal to "expected":',deepEqual:"Expected values to be loosely deep-equal:",equal:"Expected values to be loosely equal:",notDeepStrictEqual:'Expected "actual" not to be strictly deep-equal to:',notStrictEqual:'Expected "actual" to be strictly unequal to:',notStrictEqualObject:'Expected "actual" not to be reference-equal to "expected":',notDeepEqual:'Expected "actual" not to be loosely deep-equal to:',notEqual:'Expected "actual" to be loosely unequal to:',notIdentical:"Values identical but not reference-equal:"};function copyError(r){var i=Object.keys(r),o=Object.create(Object.getPrototypeOf(r));return i.forEach((function(i){o[i]=r[i]})),Object.defineProperty(o,"message",{value:r.message}),o}function inspectValue(r){return p(r,{compact:!1,customInspect:!1,depth:1e3,maxArrayLength:1/0,showHidden:!1,breakLength:1/0,showProxy:!1,sorted:!0,getters:!0})}function createErrDiff(r,i,o){var p="",g="",P=0,O="",I=!1,B=inspectValue(r),R=B.split("\n"),D=inspectValue(i).split("\n"),N=0,W="";if("strictEqual"===o&&"object"===_typeof(r)&&"object"===_typeof(i)&&null!==r&&null!==i&&(o="strictEqualObject"),1===R.length&&1===D.length&&R[0]!==D[0]){var U=R[0].length+D[0].length;if(U<=10){if(!("object"===_typeof(r)&&null!==r||"object"===_typeof(i)&&null!==i||0===r&&0===i))return"".concat(k[o],"\n\n")+"".concat(R[0]," !== ").concat(D[0],"\n")}else if("strictEqualObject"!==o){if(U<(u.stderr&&u.stderr.isTTY?u.stderr.columns:80)){for(;R[0][N]===D[0][N];)N++;N>2&&(W="\n  ".concat(function repeat(r,i){if(i=Math.floor(i),0==r.length||0==i)return"";var o=r.length*i;for(i=Math.floor(Math.log(i)/Math.log(2));i;)r+=r,i--;return r+r.substring(0,o-r.length)}(" ",N),"^"),N=0)}}}for(var j=R[R.length-1],G=D[D.length-1];j===G&&(N++<2?O="\n  ".concat(j).concat(O):p=j,R.pop(),D.pop(),0!==R.length&&0!==D.length);)j=R[R.length-1],G=D[D.length-1];var X=Math.max(R.length,D.length);if(0===X){var K=B.split("\n");if(K.length>30)for(K[26]="".concat(y,"...").concat(x);K.length>27;)K.pop();return"".concat(k.notIdentical,"\n\n").concat(K.join("\n"),"\n")}N>3&&(O="\n".concat(y,"...").concat(x).concat(O),I=!0),""!==p&&(O="\n  ".concat(p).concat(O),p="");var Y=0,J=k[o]+"\n".concat(w,"+ actual").concat(x," ").concat(_,"- expected").concat(x),$=" ".concat(y,"...").concat(x," Lines skipped");for(N=0;N<X;N++){var ee=N-P;if(R.length<N+1)ee>1&&N>2&&(ee>4?(g+="\n".concat(y,"...").concat(x),I=!0):ee>3&&(g+="\n  ".concat(D[N-2]),Y++),g+="\n  ".concat(D[N-1]),Y++),P=N,p+="\n".concat(_,"-").concat(x," ").concat(D[N]),Y++;else if(D.length<N+1)ee>1&&N>2&&(ee>4?(g+="\n".concat(y,"...").concat(x),I=!0):ee>3&&(g+="\n  ".concat(R[N-2]),Y++),g+="\n  ".concat(R[N-1]),Y++),P=N,g+="\n".concat(w,"+").concat(x," ").concat(R[N]),Y++;else{var te=D[N],re=R[N],ne=re!==te&&(!endsWith(re,",")||re.slice(0,-1)!==te);ne&&endsWith(te,",")&&te.slice(0,-1)===re&&(ne=!1,re+=","),ne?(ee>1&&N>2&&(ee>4?(g+="\n".concat(y,"...").concat(x),I=!0):ee>3&&(g+="\n  ".concat(R[N-2]),Y++),g+="\n  ".concat(R[N-1]),Y++),P=N,g+="\n".concat(w,"+").concat(x," ").concat(re),p+="\n".concat(_,"-").concat(x," ").concat(te),Y+=2):(g+=p,p="",1!==ee&&0!==N||(g+="\n  ".concat(re),Y++))}if(Y>20&&N<X-2)return"".concat(J).concat($,"\n").concat(g,"\n").concat(y,"...").concat(x).concat(p,"\n")+"".concat(y,"...").concat(x)}return"".concat(J).concat(I?$:"","\n").concat(g).concat(p).concat(O).concat(W)}var P=function(r){function AssertionError(r){var i;if(function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,AssertionError),"object"!==_typeof(r)||null===r)throw new g("options","Object",r);var o=r.message,p=r.operator,P=r.stackStartFn,O=r.actual,I=r.expected,B=Error.stackTraceLimit;if(Error.stackTraceLimit=0,null!=o)i=_possibleConstructorReturn(this,_getPrototypeOf(AssertionError).call(this,String(o)));else if(u.stderr&&u.stderr.isTTY&&(u.stderr&&u.stderr.getColorDepth&&1!==u.stderr.getColorDepth()?(y="[34m",w="[32m",x="[39m",_="[31m"):(y="",w="",x="",_="")),"object"===_typeof(O)&&null!==O&&"object"===_typeof(I)&&null!==I&&"stack"in O&&O instanceof Error&&"stack"in I&&I instanceof Error&&(O=copyError(O),I=copyError(I)),"deepStrictEqual"===p||"strictEqual"===p)i=_possibleConstructorReturn(this,_getPrototypeOf(AssertionError).call(this,createErrDiff(O,I,p)));else if("notDeepStrictEqual"===p||"notStrictEqual"===p){var R=k[p],D=inspectValue(O).split("\n");if("notStrictEqual"===p&&"object"===_typeof(O)&&null!==O&&(R=k.notStrictEqualObject),D.length>30)for(D[26]="".concat(y,"...").concat(x);D.length>27;)D.pop();i=1===D.length?_possibleConstructorReturn(this,_getPrototypeOf(AssertionError).call(this,"".concat(R," ").concat(D[0]))):_possibleConstructorReturn(this,_getPrototypeOf(AssertionError).call(this,"".concat(R,"\n\n").concat(D.join("\n"),"\n")))}else{var N=inspectValue(O),W="",U=k[p];"notDeepEqual"===p||"notEqual"===p?(N="".concat(k[p],"\n\n").concat(N)).length>1024&&(N="".concat(N.slice(0,1021),"...")):(W="".concat(inspectValue(I)),N.length>512&&(N="".concat(N.slice(0,509),"...")),W.length>512&&(W="".concat(W.slice(0,509),"...")),"deepEqual"===p||"equal"===p?N="".concat(U,"\n\n").concat(N,"\n\nshould equal\n\n"):W=" ".concat(p," ").concat(W)),i=_possibleConstructorReturn(this,_getPrototypeOf(AssertionError).call(this,"".concat(N).concat(W)))}return Error.stackTraceLimit=B,i.generatedMessage=!o,Object.defineProperty(_assertThisInitialized(i),"name",{value:"AssertionError [ERR_ASSERTION]",enumerable:!1,writable:!0,configurable:!0}),i.code="ERR_ASSERTION",i.actual=O,i.expected=I,i.operator=p,Error.captureStackTrace&&Error.captureStackTrace(_assertThisInitialized(i),P),i.stack,i.name="AssertionError",_possibleConstructorReturn(i)}return function _inherits(r,i){if("function"!=typeof i&&null!==i)throw new TypeError("Super expression must either be null or a function");r.prototype=Object.create(i&&i.prototype,{constructor:{value:r,writable:!0,configurable:!0}}),i&&_setPrototypeOf(r,i)}(AssertionError,r),function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),r}(AssertionError,[{key:"toString",value:function toString(){return"".concat(this.name," [").concat(this.code,"]: ").concat(this.message)}},{key:p.custom,value:function value(r,i){return p(this,function _objectSpread(r){for(var i=1;i<arguments.length;i++){var o=null!=arguments[i]?arguments[i]:{},u=Object.keys(o);"function"==typeof Object.getOwnPropertySymbols&&(u=u.concat(Object.getOwnPropertySymbols(o).filter((function(r){return Object.getOwnPropertyDescriptor(o,r).enumerable})))),u.forEach((function(i){_defineProperty(r,i,o[i])}))}return r}({},i,{customInspect:!1,depth:0}))}}]),AssertionError}(_wrapNativeSuper(Error));r.exports=P},2136:function(r,i,o){"use strict";function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function _typeof(r){return typeof r}:function _typeof(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}function _possibleConstructorReturn(r,i){return!i||"object"!==_typeof(i)&&"function"!=typeof i?function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}(r):i}function _getPrototypeOf(r){return _getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf:function _getPrototypeOf(r){return r.__proto__||Object.getPrototypeOf(r)},_getPrototypeOf(r)}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}var u,p,g={};function createErrorType(r,i,o){o||(o=Error);var u=function(o){function NodeError(o,u,p){var g;return function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,NodeError),g=_possibleConstructorReturn(this,_getPrototypeOf(NodeError).call(this,function getMessage(r,o,u){return"string"==typeof i?i:i(r,o,u)}(o,u,p))),g.code=r,g}return function _inherits(r,i){if("function"!=typeof i&&null!==i)throw new TypeError("Super expression must either be null or a function");r.prototype=Object.create(i&&i.prototype,{constructor:{value:r,writable:!0,configurable:!0}}),i&&_setPrototypeOf(r,i)}(NodeError,o),NodeError}(o);g[r]=u}function oneOf(r,i){if(Array.isArray(r)){var o=r.length;return r=r.map((function(r){return String(r)})),o>2?"one of ".concat(i," ").concat(r.slice(0,o-1).join(", "),", or ")+r[o-1]:2===o?"one of ".concat(i," ").concat(r[0]," or ").concat(r[1]):"of ".concat(i," ").concat(r[0])}return"of ".concat(i," ").concat(String(r))}createErrorType("ERR_AMBIGUOUS_ARGUMENT",'The "%s" argument is ambiguous. %s',TypeError),createErrorType("ERR_INVALID_ARG_TYPE",(function(r,i,p){var g,y;if(void 0===u&&(u=o(9282)),u("string"==typeof r,"'name' must be a string"),"string"==typeof i&&function startsWith(r,i,o){return r.substr(!o||o<0?0:+o,i.length)===i}(i,"not ")?(g="must not be",i=i.replace(/^not /,"")):g="must be",function endsWith(r,i,o){return(void 0===o||o>r.length)&&(o=r.length),r.substring(o-i.length,o)===i}(r," argument"))y="The ".concat(r," ").concat(g," ").concat(oneOf(i,"type"));else{var w=function includes(r,i,o){return"number"!=typeof o&&(o=0),!(o+i.length>r.length)&&-1!==r.indexOf(i,o)}(r,".")?"property":"argument";y='The "'.concat(r,'" ').concat(w," ").concat(g," ").concat(oneOf(i,"type"))}return y+=". Received type ".concat(_typeof(p))}),TypeError),createErrorType("ERR_INVALID_ARG_VALUE",(function(r,i){var u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:"is invalid";void 0===p&&(p=o(9539));var g=p.inspect(i);return g.length>128&&(g="".concat(g.slice(0,128),"...")),"The argument '".concat(r,"' ").concat(u,". Received ").concat(g)}),TypeError,RangeError),createErrorType("ERR_INVALID_RETURN_VALUE",(function(r,i,o){var u;return u=o&&o.constructor&&o.constructor.name?"instance of ".concat(o.constructor.name):"type ".concat(_typeof(o)),"Expected ".concat(r,' to be returned from the "').concat(i,'"')+" function but got ".concat(u,".")}),TypeError),createErrorType("ERR_MISSING_ARGS",(function(){for(var r=arguments.length,i=new Array(r),p=0;p<r;p++)i[p]=arguments[p];void 0===u&&(u=o(9282)),u(i.length>0,"At least one arg needs to be specified");var g="The ",y=i.length;switch(i=i.map((function(r){return'"'.concat(r,'"')})),y){case 1:g+="".concat(i[0]," argument");break;case 2:g+="".concat(i[0]," and ").concat(i[1]," arguments");break;default:g+=i.slice(0,y-1).join(", "),g+=", and ".concat(i[y-1]," arguments")}return"".concat(g," must be specified")}),TypeError),r.exports.codes=g},9158:function(r,i,o){"use strict";function _slicedToArray(r,i){return function _arrayWithHoles(r){if(Array.isArray(r))return r}(r)||function _iterableToArrayLimit(r,i){var o=[],u=!0,p=!1,g=void 0;try{for(var y,w=r[Symbol.iterator]();!(u=(y=w.next()).done)&&(o.push(y.value),!i||o.length!==i);u=!0);}catch(r){p=!0,g=r}finally{try{u||null==w.return||w.return()}finally{if(p)throw g}}return o}(r,i)||function _nonIterableRest(){throw new TypeError("Invalid attempt to destructure non-iterable instance")}()}function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function _typeof(r){return typeof r}:function _typeof(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}var u=void 0!==/a/g.flags,p=function arrayFromSet(r){var i=[];return r.forEach((function(r){return i.push(r)})),i},g=function arrayFromMap(r){var i=[];return r.forEach((function(r,o){return i.push([o,r])})),i},y=Object.is?Object.is:o(609),w=Object.getOwnPropertySymbols?Object.getOwnPropertySymbols:function(){return[]},_=Number.isNaN?Number.isNaN:o(360);function uncurryThis(r){return r.call.bind(r)}var x=uncurryThis(Object.prototype.hasOwnProperty),k=uncurryThis(Object.prototype.propertyIsEnumerable),P=uncurryThis(Object.prototype.toString),O=o(9539).types,I=O.isAnyArrayBuffer,B=O.isArrayBufferView,R=O.isDate,D=O.isMap,N=O.isRegExp,W=O.isSet,U=O.isNativeError,j=O.isBoxedPrimitive,G=O.isNumberObject,X=O.isStringObject,K=O.isBooleanObject,Y=O.isBigIntObject,J=O.isSymbolObject,$=O.isFloat32Array,ee=O.isFloat64Array;function isNonIndex(r){if(0===r.length||r.length>10)return!0;for(var i=0;i<r.length;i++){var o=r.charCodeAt(i);if(o<48||o>57)return!0}return 10===r.length&&r>=Math.pow(2,32)}function getOwnNonIndexProperties(r){return Object.keys(r).filter(isNonIndex).concat(w(r).filter(Object.prototype.propertyIsEnumerable.bind(r)))}function compare(r,i){if(r===i)return 0;for(var o=r.length,u=i.length,p=0,g=Math.min(o,u);p<g;++p)if(r[p]!==i[p]){o=r[p],u=i[p];break}return o<u?-1:u<o?1:0}function innerDeepEqual(r,i,o,p){if(r===i)return 0!==r||(!o||y(r,i));if(o){if("object"!==_typeof(r))return"number"==typeof r&&_(r)&&_(i);if("object"!==_typeof(i)||null===r||null===i)return!1;if(Object.getPrototypeOf(r)!==Object.getPrototypeOf(i))return!1}else{if(null===r||"object"!==_typeof(r))return(null===i||"object"!==_typeof(i))&&r==i;if(null===i||"object"!==_typeof(i))return!1}var g=P(r);if(g!==P(i))return!1;if(Array.isArray(r)){if(r.length!==i.length)return!1;var w=getOwnNonIndexProperties(r),x=getOwnNonIndexProperties(i);return w.length===x.length&&keyCheck(r,i,o,p,1,w)}if("[object Object]"===g&&(!D(r)&&D(i)||!W(r)&&W(i)))return!1;if(R(r)){if(!R(i)||Date.prototype.getTime.call(r)!==Date.prototype.getTime.call(i))return!1}else if(N(r)){if(!N(i)||!function areSimilarRegExps(r,i){return u?r.source===i.source&&r.flags===i.flags:RegExp.prototype.toString.call(r)===RegExp.prototype.toString.call(i)}(r,i))return!1}else if(U(r)||r instanceof Error){if(r.message!==i.message||r.name!==i.name)return!1}else{if(B(r)){if(o||!$(r)&&!ee(r)){if(!function areSimilarTypedArrays(r,i){return r.byteLength===i.byteLength&&0===compare(new Uint8Array(r.buffer,r.byteOffset,r.byteLength),new Uint8Array(i.buffer,i.byteOffset,i.byteLength))}(r,i))return!1}else if(!function areSimilarFloatArrays(r,i){if(r.byteLength!==i.byteLength)return!1;for(var o=0;o<r.byteLength;o++)if(r[o]!==i[o])return!1;return!0}(r,i))return!1;var k=getOwnNonIndexProperties(r),O=getOwnNonIndexProperties(i);return k.length===O.length&&keyCheck(r,i,o,p,0,k)}if(W(r))return!(!W(i)||r.size!==i.size)&&keyCheck(r,i,o,p,2);if(D(r))return!(!D(i)||r.size!==i.size)&&keyCheck(r,i,o,p,3);if(I(r)){if(!function areEqualArrayBuffers(r,i){return r.byteLength===i.byteLength&&0===compare(new Uint8Array(r),new Uint8Array(i))}(r,i))return!1}else if(j(r)&&!function isEqualBoxedPrimitive(r,i){return G(r)?G(i)&&y(Number.prototype.valueOf.call(r),Number.prototype.valueOf.call(i)):X(r)?X(i)&&String.prototype.valueOf.call(r)===String.prototype.valueOf.call(i):K(r)?K(i)&&Boolean.prototype.valueOf.call(r)===Boolean.prototype.valueOf.call(i):Y(r)?Y(i)&&BigInt.prototype.valueOf.call(r)===BigInt.prototype.valueOf.call(i):J(i)&&Symbol.prototype.valueOf.call(r)===Symbol.prototype.valueOf.call(i)}(r,i))return!1}return keyCheck(r,i,o,p,0)}function getEnumerables(r,i){return i.filter((function(i){return k(r,i)}))}function keyCheck(r,i,o,u,p,g){if(5===arguments.length){g=Object.keys(r);var y=Object.keys(i);if(g.length!==y.length)return!1}for(var _=0;_<g.length;_++)if(!x(i,g[_]))return!1;if(o&&5===arguments.length){var P=w(r);if(0!==P.length){var O=0;for(_=0;_<P.length;_++){var I=P[_];if(k(r,I)){if(!k(i,I))return!1;g.push(I),O++}else if(k(i,I))return!1}var B=w(i);if(P.length!==B.length&&getEnumerables(i,B).length!==O)return!1}else{var R=w(i);if(0!==R.length&&0!==getEnumerables(i,R).length)return!1}}if(0===g.length&&(0===p||1===p&&0===r.length||0===r.size))return!0;if(void 0===u)u={val1:new Map,val2:new Map,position:0};else{var D=u.val1.get(r);if(void 0!==D){var N=u.val2.get(i);if(void 0!==N)return D===N}u.position++}u.val1.set(r,u.position),u.val2.set(i,u.position);var W=objEquiv(r,i,o,g,u,p);return u.val1.delete(r),u.val2.delete(i),W}function setHasEqualElement(r,i,o,u){for(var g=p(r),y=0;y<g.length;y++){var w=g[y];if(innerDeepEqual(i,w,o,u))return r.delete(w),!0}return!1}function findLooseMatchingPrimitives(r){switch(_typeof(r)){case"undefined":return null;case"object":return;case"symbol":return!1;case"string":r=+r;case"number":if(_(r))return!1}return!0}function setMightHaveLoosePrim(r,i,o){var u=findLooseMatchingPrimitives(o);return null!=u?u:i.has(u)&&!r.has(u)}function mapMightHaveLoosePrim(r,i,o,u,p){var g=findLooseMatchingPrimitives(o);if(null!=g)return g;var y=i.get(g);return!(void 0===y&&!i.has(g)||!innerDeepEqual(u,y,!1,p))&&(!r.has(g)&&innerDeepEqual(u,y,!1,p))}function mapHasEqualEntry(r,i,o,u,g,y){for(var w=p(r),_=0;_<w.length;_++){var x=w[_];if(innerDeepEqual(o,x,g,y)&&innerDeepEqual(u,i.get(x),g,y))return r.delete(x),!0}return!1}function objEquiv(r,i,o,u,y,w){var _=0;if(2===w){if(!function setEquiv(r,i,o,u){for(var g=null,y=p(r),w=0;w<y.length;w++){var _=y[w];if("object"===_typeof(_)&&null!==_)null===g&&(g=new Set),g.add(_);else if(!i.has(_)){if(o)return!1;if(!setMightHaveLoosePrim(r,i,_))return!1;null===g&&(g=new Set),g.add(_)}}if(null!==g){for(var x=p(i),k=0;k<x.length;k++){var P=x[k];if("object"===_typeof(P)&&null!==P){if(!setHasEqualElement(g,P,o,u))return!1}else if(!o&&!r.has(P)&&!setHasEqualElement(g,P,o,u))return!1}return 0===g.size}return!0}(r,i,o,y))return!1}else if(3===w){if(!function mapEquiv(r,i,o,u){for(var p=null,y=g(r),w=0;w<y.length;w++){var _=_slicedToArray(y[w],2),x=_[0],k=_[1];if("object"===_typeof(x)&&null!==x)null===p&&(p=new Set),p.add(x);else{var P=i.get(x);if(void 0===P&&!i.has(x)||!innerDeepEqual(k,P,o,u)){if(o)return!1;if(!mapMightHaveLoosePrim(r,i,x,k,u))return!1;null===p&&(p=new Set),p.add(x)}}}if(null!==p){for(var O=g(i),I=0;I<O.length;I++){var B=_slicedToArray(O[I],2),R=(x=B[0],B[1]);if("object"===_typeof(x)&&null!==x){if(!mapHasEqualEntry(p,r,x,R,o,u))return!1}else if(!(o||r.has(x)&&innerDeepEqual(r.get(x),R,!1,u)||mapHasEqualEntry(p,r,x,R,!1,u)))return!1}return 0===p.size}return!0}(r,i,o,y))return!1}else if(1===w)for(;_<r.length;_++){if(!x(r,_)){if(x(i,_))return!1;for(var k=Object.keys(r);_<k.length;_++){var P=k[_];if(!x(i,P)||!innerDeepEqual(r[P],i[P],o,y))return!1}return k.length===Object.keys(i).length}if(!x(i,_)||!innerDeepEqual(r[_],i[_],o,y))return!1}for(_=0;_<u.length;_++){var O=u[_];if(!innerDeepEqual(r[O],i[O],o,y))return!1}return!0}r.exports={isDeepEqual:function isDeepEqual(r,i){return innerDeepEqual(r,i,false)},isDeepStrictEqual:function isDeepStrictEqual(r,i){return innerDeepEqual(r,i,true)}}},2055:function(r,i){"use strict";i.OP=0,i.CL=1,i.CP=2,i.QU=3,i.GL=4,i.NS=5,i.EX=6,i.SY=7,i.IS=8,i.PR=9,i.PO=10,i.NU=11,i.AL=12,i.HL=13,i.ID=14,i.IN=15,i.HY=16,i.BA=17,i.BB=18,i.B2=19,i.ZW=20,i.CM=21,i.WJ=22,i.H2=23,i.H3=24,i.JL=25,i.JV=26,i.JT=27,i.RI=28,i.EB=29,i.EM=30,i.ZWJ=31,i.CB=32,i.AI=33,i.BK=34,i.CJ=35,i.CR=36,i.LF=37,i.NL=38,i.SA=39,i.SG=40,i.SP=41,i.XX=42},8383:function(r,i){"use strict";i.DI_BRK=0,i.IN_BRK=1,i.CI_BRK=2,i.CP_BRK=3,i.PR_BRK=4,i.pairTable=[[4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,3,4,4,4,4,4,4,4,4,4,4,4],[0,4,4,1,1,4,4,4,4,1,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,4,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[4,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,1,0,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,0,0,1,1,1,1,1,1,1,0,0,4,2,4,1,1,1,1,1,0,1,1,1,0],[1,4,4,1,1,1,4,4,4,0,0,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,0,1,4,4,4,0,0,1,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,0,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,4,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,1,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,1,1,1,1,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,1,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,1,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,1,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,0,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0]]},5106:function(r,i,o){"use strict";o(9601),i.EncodeStream=o(9126),i.DecodeStream=o(3030),i.Array=o(1988),i.LazyArray=o(6768),i.Bitfield=o(3425),i.Boolean=o(9024),i.Buffer=o(5250),i.Enum=o(3100),i.Optional=o(9541),i.Reserved=o(7468),i.String=o(1466),i.Struct=o(1219),i.VersionedStruct=o(3585);var u=o(6610),p=o(6462),g=o(8011);Object.assign(i,u,p,g)},1988:function(r,i,o){"use strict";function _createForOfIteratorHelperLoose(r,i){var o="undefined"!=typeof Symbol&&r[Symbol.iterator]||r["@@iterator"];if(o)return(o=o.call(r)).next.bind(o);if(Array.isArray(r)||(o=function _unsupportedIterableToArray(r,i){if(!r)return;if("string"==typeof r)return _arrayLikeToArray(r,i);var o=Object.prototype.toString.call(r).slice(8,-1);"Object"===o&&r.constructor&&(o=r.constructor.name);if("Map"===o||"Set"===o)return Array.from(r);if("Arguments"===o||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(o))return _arrayLikeToArray(r,i)}(r))||i&&r&&"number"==typeof r.length){o&&(r=o);var u=0;return function(){return u>=r.length?{done:!0}:{done:!1,value:r[u++]}}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}function _arrayLikeToArray(r,i){(null==i||i>r.length)&&(i=r.length);for(var o=0,u=new Array(i);o<i;o++)u[o]=r[o];return u}o(7042),o(1539),o(8309),o(1038),o(8783),o(4916),o(2526),o(1817),o(2165),o(6992),o(3948);var u=o(6462).Number,p=o(6610),g=function(){function ArrayT(r,i,o){void 0===o&&(o="count"),this.type=r,this.length=i,this.lengthType=o}var r=ArrayT.prototype;return r.decode=function decode(r,i){var o,g=r.pos,y=[],w=i;if(null!=this.length&&(o=p.resolveLength(this.length,r,i)),this.length instanceof u&&(Object.defineProperties(y,{parent:{value:i},_startOffset:{value:g},_currentOffset:{value:0,writable:!0},_length:{value:o}}),w=y),null==o||"bytes"===this.lengthType)for(var _=null!=o?r.pos+o:(null!=i?i._length:void 0)?i._startOffset+i._length:r.length;r.pos<_;)y.push(this.type.decode(r,w));else for(var x=0,k=o;x<k;x++)y.push(this.type.decode(r,w));return y},r.size=function size(r,i){if(!r)return this.type.size(null,i)*p.resolveLength(this.length,null,i);var size=0;this.length instanceof u&&(size+=this.length.size(),i={parent:i});for(var o,g=_createForOfIteratorHelperLoose(r);!(o=g()).done;){var y=o.value;size+=this.type.size(y,i)}return size},r.encode=function encode(r,i,o){var p=o;this.length instanceof u&&((p={pointers:[],startOffset:r.pos,parent:o}).pointerOffset=r.pos+this.size(i,p),this.length.encode(r,i.length));for(var g,y=_createForOfIteratorHelperLoose(i);!(g=y()).done;){var w=g.value;this.type.encode(r,w,p)}if(this.length instanceof u)for(var _=0;_<p.pointers.length;){var x=p.pointers[_++];x.type.encode(r,x.val)}},ArrayT}();r.exports=g},3425:function(r,i,o){"use strict";o(2087);var u=function(){function Bitfield(r,i){void 0===i&&(i=[]),this.type=r,this.flags=i}var r=Bitfield.prototype;return r.decode=function decode(r){for(var i=this.type.decode(r),o={},u=0;u<this.flags.length;u++){var p=this.flags[u];null!=p&&(o[p]=!!(i&1<<u))}return o},r.size=function size(){return this.type.size()},r.encode=function encode(r,i){for(var o=0,u=0;u<this.flags.length;u++){var p=this.flags[u];null!=p&&i[p]&&(o|=1<<u)}return this.type.encode(r,o)},Bitfield}();r.exports=u},9024:function(r){"use strict";var i=function(){function BooleanT(r){this.type=r}var r=BooleanT.prototype;return r.decode=function decode(r,i){return!!this.type.decode(r,i)},r.size=function size(r,i){return this.type.size(r,i)},r.encode=function encode(r,i,o){return this.type.encode(r,+i,o)},BooleanT}();r.exports=i},5250:function(r,i,o){"use strict";var u=o(6610),p=o(6462).Number,g=function(){function BufferT(r){this.length=r}var r=BufferT.prototype;return r.decode=function decode(r,i){var o=u.resolveLength(this.length,r,i);return r.readBuffer(o)},r.size=function size(r,i){return r?r.length:u.resolveLength(this.length,null,i)},r.encode=function encode(r,i,o){return this.length instanceof p&&this.length.encode(r,i.length),r.writeBuffer(i)},BufferT}();r.exports=g},3030:function(r,i,o){"use strict";var u,p=o(8823).Buffer;o(1539),o(9714),o(7042),o(4916),o(5306);try{u=o(4914)}catch(r){}var g=function(){function DecodeStream(r){this.buffer=r,this.pos=0,this.length=this.buffer.length}var r=DecodeStream.prototype;return r.readString=function readString(r,i){switch(void 0===i&&(i="ascii"),i){case"utf16le":case"ucs2":case"utf8":case"ascii":return this.buffer.toString(i,this.pos,this.pos+=r);case"utf16be":for(var o=p.from(this.readBuffer(r)),g=0,y=o.length-1;g<y;g+=2){var w=o[g];o[g]=o[g+1],o[g+1]=w}return o.toString("utf16le");default:if(o=this.readBuffer(r),u)try{return u.decode(o,i)}catch(r){}return o}},r.readBuffer=function readBuffer(r){return this.buffer.slice(this.pos,this.pos+=r)},r.readUInt24BE=function readUInt24BE(){return(this.readUInt16BE()<<8)+this.readUInt8()},r.readUInt24LE=function readUInt24LE(){return this.readUInt16LE()+(this.readUInt8()<<16)},r.readInt24BE=function readInt24BE(){return(this.readInt16BE()<<8)+this.readUInt8()},r.readInt24LE=function readInt24LE(){return this.readUInt16LE()+(this.readInt8()<<16)},DecodeStream}();g.TYPES={UInt8:1,UInt16:2,UInt24:3,UInt32:4,Int8:1,Int16:2,Int24:3,Int32:4,Float:4,Double:8};var y=function _loop(r){if("read"===r.slice(0,4)){var i=g.TYPES[r.replace(/read|[BL]E/g,"")];g.prototype[r]=function(){var o=this.buffer[r](this.pos);return this.pos+=i,o}}};for(var w in p.prototype)y(w);r.exports=g},9126:function(r,i,o){"use strict";var u,p=o(8823).Buffer;function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(7042),o(3290),o(4916),o(5306);var g=o(2830),y=o(3030);try{u=o(4914)}catch(r){}var w=function(r){function EncodeStream(i){var o;return void 0===i&&(i=65536),(o=r.apply(this,arguments)||this).buffer=p.alloc(i),o.bufferOffset=0,o.pos=0,o}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(EncodeStream,r);var i=EncodeStream.prototype;return i._read=function _read(){},i.ensure=function ensure(r){if(this.bufferOffset+r>this.buffer.length)return this.flush()},i.flush=function flush(){if(this.bufferOffset>0)return this.push(p.from(this.buffer.slice(0,this.bufferOffset))),this.bufferOffset=0},i.writeBuffer=function writeBuffer(r){return this.flush(),this.push(r),this.pos+=r.length},i.writeString=function writeString(r,i){switch(void 0===i&&(i="ascii"),i){case"utf16le":case"ucs2":case"utf8":case"ascii":return this.writeBuffer(p.from(r,i));case"utf16be":for(var o=p.from(r,"utf16le"),g=0,y=o.length-1;g<y;g+=2){var w=o[g];o[g]=o[g+1],o[g+1]=w}return this.writeBuffer(o);default:if(u)return this.writeBuffer(u.encode(r,i));throw new Error("Install iconv-lite to enable additional string encodings.")}},i.writeUInt24BE=function writeUInt24BE(r){return this.ensure(3),this.buffer[this.bufferOffset++]=r>>>16&255,this.buffer[this.bufferOffset++]=r>>>8&255,this.buffer[this.bufferOffset++]=255&r,this.pos+=3},i.writeUInt24LE=function writeUInt24LE(r){return this.ensure(3),this.buffer[this.bufferOffset++]=255&r,this.buffer[this.bufferOffset++]=r>>>8&255,this.buffer[this.bufferOffset++]=r>>>16&255,this.pos+=3},i.writeInt24BE=function writeInt24BE(r){return r>=0?this.writeUInt24BE(r):this.writeUInt24BE(r+16777215+1)},i.writeInt24LE=function writeInt24LE(r){return r>=0?this.writeUInt24LE(r):this.writeUInt24LE(r+16777215+1)},i.fill=function fill(r,i){if(i<this.buffer.length)return this.ensure(i),this.buffer.fill(r,this.bufferOffset,this.bufferOffset+i),this.bufferOffset+=i,this.pos+=i;var o=p.alloc(i);return o.fill(r),this.writeBuffer(o)},i.end=function end(){return this.flush(),this.push(null)},EncodeStream}(g.Readable),_=function _loop(r){if("write"===r.slice(0,5)){var i=+y.TYPES[r.replace(/write|[BL]E/g,"")];w.prototype[r]=function(o){return this.ensure(i),this.buffer[r](o,this.bufferOffset),this.bufferOffset+=i,this.pos+=i}}};for(var x in p.prototype)_(x);r.exports=w},3100:function(r){"use strict";var i=function(){function Enum(r,i){void 0===i&&(i=[]),this.type=r,this.options=i}var r=Enum.prototype;return r.decode=function decode(r){var i=this.type.decode(r);return this.options[i]||i},r.size=function size(){return this.type.size()},r.encode=function encode(r,i){var o=this.options.indexOf(i);if(-1===o)throw new Error("Unknown option in enum: "+i);return this.type.encode(r,o)},Enum}();r.exports=i},6768:function(r,i,o){"use strict";function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(1539),o(8674);var u=o(1988),p=o(6462).Number,g=o(6610),y=o(9539).inspect,w=function(r){function LazyArrayT(){return r.apply(this,arguments)||this}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(LazyArrayT,r);var i=LazyArrayT.prototype;return i.decode=function decode(r,i){var o=r.pos,u=g.resolveLength(this.length,r,i);this.length instanceof p&&(i={parent:i,_startOffset:o,_currentOffset:0,_length:u});var y=new _(this.type,u,r,i);return r.pos+=u*this.type.size(null,i),y},i.size=function size(i,o){return i instanceof _&&(i=i.toArray()),r.prototype.size.call(this,i,o)},i.encode=function encode(i,o,u){return o instanceof _&&(o=o.toArray()),r.prototype.encode.call(this,i,o,u)},LazyArrayT}(u),_=function(){function LazyArray(r,i,o,u){this.type=r,this.length=i,this.stream=o,this.ctx=u,this.base=this.stream.pos,this.items=[]}var r=LazyArray.prototype;return r.get=function get(r){if(!(r<0||r>=this.length)){if(null==this.items[r]){var i=this.stream.pos;this.stream.pos=this.base+this.type.size(null,this.ctx)*r,this.items[r]=this.type.decode(this.stream,this.ctx),this.stream.pos=i}return this.items[r]}},r.toArray=function toArray(){for(var r=[],i=0,o=this.length;i<o;i++)r.push(this.get(i));return r},r.inspect=function inspect(){return y(this.toArray())},LazyArray}();r.exports=w},6462:function(r,i,o){"use strict";function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}var u=o(3030),p=function(){function NumberT(r,i){void 0===i&&(i="BE"),this.type=r,this.endian=i,this.fn=this.type,"8"!==this.type[this.type.length-1]&&(this.fn+=this.endian)}var r=NumberT.prototype;return r.size=function size(){return u.TYPES[this.type]},r.decode=function decode(r){return r["read"+this.fn]()},r.encode=function encode(r,i){return r["write"+this.fn](i)},NumberT}();i.Number=p,i.uint8=new p("UInt8"),i.uint16be=i.uint16=new p("UInt16","BE"),i.uint16le=new p("UInt16","LE"),i.uint24be=i.uint24=new p("UInt24","BE"),i.uint24le=new p("UInt24","LE"),i.uint32be=i.uint32=new p("UInt32","BE"),i.uint32le=new p("UInt32","LE"),i.int8=new p("Int8"),i.int16be=i.int16=new p("Int16","BE"),i.int16le=new p("Int16","LE"),i.int24be=i.int24=new p("Int24","BE"),i.int24le=new p("Int24","LE"),i.int32be=i.int32=new p("Int32","BE"),i.int32le=new p("Int32","LE"),i.floatbe=i.float=new p("Float","BE"),i.floatle=new p("Float","LE"),i.doublebe=i.double=new p("Double","BE"),i.doublele=new p("Double","LE");var g=function(r){function Fixed(i,o,u){var p;return void 0===u&&(u=i>>1),(p=r.call(this,"Int"+i,o)||this)._point=1<<u,p}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(Fixed,r);var i=Fixed.prototype;return i.decode=function decode(i){return r.prototype.decode.call(this,i)/this._point},i.encode=function encode(i,o){return r.prototype.encode.call(this,i,o*this._point|0)},Fixed}(p);i.Fixed=g,i.fixed16be=i.fixed16=new g(16,"BE"),i.fixed16le=new g(16,"LE"),i.fixed32be=i.fixed32=new g(32,"BE"),i.fixed32le=new g(32,"LE")},9541:function(r){"use strict";var i=function(){function Optional(r,i){void 0===i&&(i=!0),this.type=r,this.condition=i}var r=Optional.prototype;return r.decode=function decode(r,i){var o=this.condition;if("function"==typeof o&&(o=o.call(i,i)),o)return this.type.decode(r,i)},r.size=function size(r,i){var o=this.condition;return"function"==typeof o&&(o=o.call(i,i)),o?this.type.size(r,i):0},r.encode=function encode(r,i,o){var u=this.condition;if("function"==typeof u&&(u=u.call(o,o)),u)return this.type.encode(r,i,o)},Optional}();r.exports=i},8011:function(r,i,o){"use strict";var u=o(6610),p=function(){function Pointer(r,i,o){if(void 0===o&&(o={}),this.offsetType=r,this.type=i,this.options=o,"void"===this.type&&(this.type=null),null==this.options.type&&(this.options.type="local"),null==this.options.allowNull&&(this.options.allowNull=!0),null==this.options.nullValue&&(this.options.nullValue=0),null==this.options.lazy&&(this.options.lazy=!1),this.options.relativeTo){if("function"!=typeof this.options.relativeTo)throw new Error("relativeTo option must be a function");this.relativeToGetter=o.relativeTo}}var r=Pointer.prototype;return r.decode=function decode(r,i){var o,p=this,g=this.offsetType.decode(r,i);if(g===this.options.nullValue&&this.options.allowNull)return null;switch(this.options.type){case"local":o=i._startOffset;break;case"immediate":o=r.pos-this.offsetType.size();break;case"parent":o=i.parent._startOffset;break;default:for(var y=i;y.parent;)y=y.parent;o=y._startOffset||0}this.options.relativeTo&&(o+=this.relativeToGetter(i));var w=g+o;if(null!=this.type){var _=null,x=function decodeValue(){if(null!=_)return _;var o=r.pos;return r.pos=w,_=p.type.decode(r,i),r.pos=o,_};return this.options.lazy?new u.PropertyDescriptor({get:x}):x()}return w},r.size=function size(r,i){var o=i;switch(this.options.type){case"local":case"immediate":break;case"parent":i=i.parent;break;default:for(;i.parent;)i=i.parent}var u=this.type;if(null==u){if(!(r instanceof g))throw new Error("Must be a VoidPointer");u=r.type,r=r.value}return r&&i&&(i.pointerSize+=u.size(r,o)),this.offsetType.size()},r.encode=function encode(r,i,o){var u,p=o;if(null!=i){switch(this.options.type){case"local":u=o.startOffset;break;case"immediate":u=r.pos+this.offsetType.size(i,p);break;case"parent":u=(o=o.parent).startOffset;break;default:for(u=0;o.parent;)o=o.parent}this.options.relativeTo&&(u+=this.relativeToGetter(p.val)),this.offsetType.encode(r,o.pointerOffset-u);var y=this.type;if(null==y){if(!(i instanceof g))throw new Error("Must be a VoidPointer");y=i.type,i=i.value}return o.pointers.push({type:y,val:i,parent:p}),o.pointerOffset+=y.size(i,p)}this.offsetType.encode(r,this.options.nullValue)},Pointer}(),g=function VoidPointer(r,i){this.type=r,this.value=i};i.Pointer=p,i.VoidPointer=g},7468:function(r,i,o){"use strict";o(3290);var u=o(6610),p=function(){function Reserved(r,i){void 0===i&&(i=1),this.type=r,this.count=i}var r=Reserved.prototype;return r.decode=function decode(r,i){r.pos+=this.size(null,i)},r.size=function size(r,i){var o=u.resolveLength(this.count,null,i);return this.type.size()*o},r.encode=function encode(r,i,o){return r.fill(0,this.size(i,o))},Reserved}();r.exports=p},1466:function(r,i,o){"use strict";var u=o(8823).Buffer,p=o(6462).Number,g=o(6610),y=function(){function StringT(r,i){void 0===i&&(i="ascii"),this.length=r,this.encoding=i}var r=StringT.prototype;return r.decode=function decode(r,i){var o,u;if(null!=this.length)o=g.resolveLength(this.length,r,i);else{var p;for(p=r.buffer,o=r.length,u=r.pos;u<o&&0!==p[u];)++u;o=u-r.pos}var y=this.encoding;"function"==typeof y&&(y=y.call(i,i)||"ascii");var w=r.readString(o,y);return null==this.length&&r.pos<r.length&&r.pos++,w},r.size=function size(r,i){if(!r)return g.resolveLength(this.length,null,i);var o=this.encoding;"function"==typeof o&&(o=o.call(null!=i?i.val:void 0,null!=i?i.val:void 0)||"ascii"),"utf16be"===o&&(o="utf16le");var size=u.byteLength(r,o);return this.length instanceof p&&(size+=this.length.size()),null==this.length&&size++,size},r.encode=function encode(r,i,o){var g=this.encoding;if("function"==typeof g&&(g=g.call(null!=o?o.val:void 0,null!=o?o.val:void 0)||"ascii"),this.length instanceof p&&this.length.encode(r,u.byteLength(i,g)),r.writeString(i,g),null==this.length)return r.writeUInt8(0)},StringT}();r.exports=y},1219:function(r,i,o){"use strict";var u=o(6610),p=function(){function Struct(r){void 0===r&&(r={}),this.fields=r}var r=Struct.prototype;return r.decode=function decode(r,i,o){void 0===o&&(o=0);var u=this._setup(r,i,o);return this._parseFields(r,u,this.fields),null!=this.process&&this.process.call(u,r),u},r._setup=function _setup(r,i,o){var u={};return Object.defineProperties(u,{parent:{value:i},_startOffset:{value:r.pos},_currentOffset:{value:0,writable:!0},_length:{value:o}}),u},r._parseFields=function _parseFields(r,i,o){for(var p in o){var g,y=o[p];void 0!==(g="function"==typeof y?y.call(i,i):y.decode(r,i))&&(g instanceof u.PropertyDescriptor?Object.defineProperty(i,p,g):i[p]=g),i._currentOffset=r.pos-i._startOffset}},r.size=function size(r,i,o){null==r&&(r={}),null==o&&(o=!0);var u={parent:i,val:r,pointerSize:0},size=0;for(var p in this.fields){var g=this.fields[p];null!=g.size&&(size+=g.size(r[p],u))}return o&&(size+=u.pointerSize),size},r.encode=function encode(r,i,o){var u;null!=this.preEncode&&this.preEncode.call(i,r);var p={pointers:[],startOffset:r.pos,parent:o,val:i,pointerSize:0};for(var g in p.pointerOffset=r.pos+this.size(i,p,!1),this.fields)null!=(u=this.fields[g]).encode&&u.encode(r,i[g],p);for(var y=0;y<p.pointers.length;){var w=p.pointers[y++];w.type.encode(r,w.val,w.parent)}},Struct}();r.exports=p},3585:function(r,i,o){"use strict";function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(1539),o(4916),o(3123);var u=o(1219),p=function(r){function VersionedStruct(i,o){var u;return void 0===o&&(o={}),(u=r.call(this)||this).type=i,u.versions=o,"string"==typeof i&&(u.versionPath=i.split(".")),u}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(VersionedStruct,r);var i=VersionedStruct.prototype;return i.decode=function decode(r,i,o){void 0===o&&(o=0);var u=this._setup(r,i,o);"string"==typeof this.type?u.version=function getPath(r,i){return i.reduce((function(r,i){return r&&r[i]}),r)}(i,this.versionPath):u.version=this.type.decode(r),this.versions.header&&this._parseFields(r,u,this.versions.header);var p=this.versions[u.version];if(null==p)throw new Error("Unknown version "+u.version);return p instanceof VersionedStruct?p.decode(r,i):(this._parseFields(r,u,p),null!=this.process&&this.process.call(u,r),u)},i.size=function size(r,i,o){var u,p;if(void 0===o&&(o=!0),!r)throw new Error("Not a fixed size");var g={parent:i,val:r,pointerSize:0},size=0;if("string"!=typeof this.type&&(size+=this.type.size(r.version,g)),this.versions.header)for(u in this.versions.header)null!=(p=this.versions.header[u]).size&&(size+=p.size(r[u],g));var y=this.versions[r.version];if(null==y)throw new Error("Unknown version "+r.version);for(u in y)null!=(p=y[u]).size&&(size+=p.size(r[u],g));return o&&(size+=g.pointerSize),size},i.encode=function encode(r,i,o){var u,p;null!=this.preEncode&&this.preEncode.call(i,r);var g={pointers:[],startOffset:r.pos,parent:o,val:i,pointerSize:0};if(g.pointerOffset=r.pos+this.size(i,g,!1),"string"!=typeof this.type&&this.type.encode(r,i.version),this.versions.header)for(u in this.versions.header)null!=(p=this.versions.header[u]).encode&&p.encode(r,i[u],g);var y=this.versions[i.version];for(u in y)null!=(p=y[u]).encode&&p.encode(r,i[u],g);for(var w=0;w<g.pointers.length;){var _=g.pointers[w++];_.type.encode(r,_.val,_.parent)}},VersionedStruct}(u);r.exports=p},6610:function(r,i,o){"use strict";var u=o(6462).Number;i.resolveLength=function(r,i,o){var p;if("number"==typeof r?p=r:"function"==typeof r?p=r.call(o,o):o&&"string"==typeof r?p=o[r]:i&&r instanceof u&&(p=r.decode(i)),isNaN(p))throw new Error("Not a fixed size");return p};i.PropertyDescriptor=function PropertyDescriptor(r){for(var i in void 0===r&&(r={}),this.enumerable=!0,this.configurable=!0,r){var o=r[i];this[i]=o}}},8823:function(r,i,o){"use strict";function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,u.key,u)}}function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(2526),o(1817),o(1539),o(6992),o(2472),o(2990),o(8927),o(3105),o(5035),o(4345),o(7174),o(2846),o(4731),o(7209),o(6319),o(8867),o(7789),o(3739),o(9368),o(4483),o(2056),o(3462),o(678),o(7462),o(3824),o(5021),o(2974),o(5016),o(7803),o(6649),o(6078),o(3290),o(7042),o(2222),o(9714),o(3210),o(4916),o(5306),o(6699),o(2023),o(9653),o(3753),o(545),o(8309),o(3161),o(3123);var u=o(9742),p=o(645),g="function"==typeof Symbol&&"function"==typeof Symbol.for?Symbol.for("nodejs.util.inspect.custom"):null;i.Buffer=Buffer,i.SlowBuffer=function SlowBuffer(r){+r!=r&&(r=0);return Buffer.alloc(+r)},i.INSPECT_MAX_BYTES=50;var y=2147483647;function createBuffer(r){if(r>y)throw new RangeError('The value "'+r+'" is invalid for option "size"');var i=new Uint8Array(r);return Object.setPrototypeOf(i,Buffer.prototype),i}function Buffer(r,i,o){if("number"==typeof r){if("string"==typeof i)throw new TypeError('The "string" argument must be of type string. Received type number');return allocUnsafe(r)}return from(r,i,o)}function from(r,i,o){if("string"==typeof r)return function fromString(r,i){"string"==typeof i&&""!==i||(i="utf8");if(!Buffer.isEncoding(i))throw new TypeError("Unknown encoding: "+i);var o=0|byteLength(r,i),u=createBuffer(o),p=u.write(r,i);p!==o&&(u=u.slice(0,p));return u}(r,i);if(ArrayBuffer.isView(r))return function fromArrayView(r){if(isInstance(r,Uint8Array)){var i=new Uint8Array(r);return fromArrayBuffer(i.buffer,i.byteOffset,i.byteLength)}return fromArrayLike(r)}(r);if(null==r)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r);if(isInstance(r,ArrayBuffer)||r&&isInstance(r.buffer,ArrayBuffer))return fromArrayBuffer(r,i,o);if("undefined"!=typeof SharedArrayBuffer&&(isInstance(r,SharedArrayBuffer)||r&&isInstance(r.buffer,SharedArrayBuffer)))return fromArrayBuffer(r,i,o);if("number"==typeof r)throw new TypeError('The "value" argument must not be of type number. Received type number');var u=r.valueOf&&r.valueOf();if(null!=u&&u!==r)return Buffer.from(u,i,o);var p=function fromObject(r){if(Buffer.isBuffer(r)){var i=0|checked(r.length),o=createBuffer(i);return 0===o.length||r.copy(o,0,0,i),o}if(void 0!==r.length)return"number"!=typeof r.length||numberIsNaN(r.length)?createBuffer(0):fromArrayLike(r);if("Buffer"===r.type&&Array.isArray(r.data))return fromArrayLike(r.data)}(r);if(p)return p;if("undefined"!=typeof Symbol&&null!=Symbol.toPrimitive&&"function"==typeof r[Symbol.toPrimitive])return Buffer.from(r[Symbol.toPrimitive]("string"),i,o);throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r)}function assertSize(r){if("number"!=typeof r)throw new TypeError('"size" argument must be of type number');if(r<0)throw new RangeError('The value "'+r+'" is invalid for option "size"')}function allocUnsafe(r){return assertSize(r),createBuffer(r<0?0:0|checked(r))}function fromArrayLike(r){for(var i=r.length<0?0:0|checked(r.length),o=createBuffer(i),u=0;u<i;u+=1)o[u]=255&r[u];return o}function fromArrayBuffer(r,i,o){if(i<0||r.byteLength<i)throw new RangeError('"offset" is outside of buffer bounds');if(r.byteLength<i+(o||0))throw new RangeError('"length" is outside of buffer bounds');var u;return u=void 0===i&&void 0===o?new Uint8Array(r):void 0===o?new Uint8Array(r,i):new Uint8Array(r,i,o),Object.setPrototypeOf(u,Buffer.prototype),u}function checked(r){if(r>=y)throw new RangeError("Attempt to allocate Buffer larger than maximum size: 0x"+y.toString(16)+" bytes");return 0|r}function byteLength(r,i){if(Buffer.isBuffer(r))return r.length;if(ArrayBuffer.isView(r)||isInstance(r,ArrayBuffer))return r.byteLength;if("string"!=typeof r)throw new TypeError('The "string" argument must be one of type string, Buffer, or ArrayBuffer. Received type '+typeof r);var o=r.length,u=arguments.length>2&&!0===arguments[2];if(!u&&0===o)return 0;for(var p=!1;;)switch(i){case"ascii":case"latin1":case"binary":return o;case"utf8":case"utf-8":return utf8ToBytes(r).length;case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return 2*o;case"hex":return o>>>1;case"base64":return base64ToBytes(r).length;default:if(p)return u?-1:utf8ToBytes(r).length;i=(""+i).toLowerCase(),p=!0}}function slowToString(r,i,o){var u=!1;if((void 0===i||i<0)&&(i=0),i>this.length)return"";if((void 0===o||o>this.length)&&(o=this.length),o<=0)return"";if((o>>>=0)<=(i>>>=0))return"";for(r||(r="utf8");;)switch(r){case"hex":return hexSlice(this,i,o);case"utf8":case"utf-8":return utf8Slice(this,i,o);case"ascii":return asciiSlice(this,i,o);case"latin1":case"binary":return latin1Slice(this,i,o);case"base64":return base64Slice(this,i,o);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return utf16leSlice(this,i,o);default:if(u)throw new TypeError("Unknown encoding: "+r);r=(r+"").toLowerCase(),u=!0}}function swap(r,i,o){var u=r[i];r[i]=r[o],r[o]=u}function bidirectionalIndexOf(r,i,o,u,p){if(0===r.length)return-1;if("string"==typeof o?(u=o,o=0):o>2147483647?o=2147483647:o<-2147483648&&(o=-2147483648),numberIsNaN(o=+o)&&(o=p?0:r.length-1),o<0&&(o=r.length+o),o>=r.length){if(p)return-1;o=r.length-1}else if(o<0){if(!p)return-1;o=0}if("string"==typeof i&&(i=Buffer.from(i,u)),Buffer.isBuffer(i))return 0===i.length?-1:arrayIndexOf(r,i,o,u,p);if("number"==typeof i)return i&=255,"function"==typeof Uint8Array.prototype.indexOf?p?Uint8Array.prototype.indexOf.call(r,i,o):Uint8Array.prototype.lastIndexOf.call(r,i,o):arrayIndexOf(r,[i],o,u,p);throw new TypeError("val must be string, number or Buffer")}function arrayIndexOf(r,i,o,u,p){var g,y=1,w=r.length,_=i.length;if(void 0!==u&&("ucs2"===(u=String(u).toLowerCase())||"ucs-2"===u||"utf16le"===u||"utf-16le"===u)){if(r.length<2||i.length<2)return-1;y=2,w/=2,_/=2,o/=2}function read(r,i){return 1===y?r[i]:r.readUInt16BE(i*y)}if(p){var x=-1;for(g=o;g<w;g++)if(read(r,g)===read(i,-1===x?0:g-x)){if(-1===x&&(x=g),g-x+1===_)return x*y}else-1!==x&&(g-=g-x),x=-1}else for(o+_>w&&(o=w-_),g=o;g>=0;g--){for(var k=!0,P=0;P<_;P++)if(read(r,g+P)!==read(i,P)){k=!1;break}if(k)return g}return-1}function hexWrite(r,i,o,u){o=Number(o)||0;var p=r.length-o;u?(u=Number(u))>p&&(u=p):u=p;var g,y=i.length;for(u>y/2&&(u=y/2),g=0;g<u;++g){var w=parseInt(i.substr(2*g,2),16);if(numberIsNaN(w))return g;r[o+g]=w}return g}function utf8Write(r,i,o,u){return blitBuffer(utf8ToBytes(i,r.length-o),r,o,u)}function asciiWrite(r,i,o,u){return blitBuffer(function asciiToBytes(r){for(var i=[],o=0;o<r.length;++o)i.push(255&r.charCodeAt(o));return i}(i),r,o,u)}function base64Write(r,i,o,u){return blitBuffer(base64ToBytes(i),r,o,u)}function ucs2Write(r,i,o,u){return blitBuffer(function utf16leToBytes(r,i){for(var o,u,p,g=[],y=0;y<r.length&&!((i-=2)<0);++y)u=(o=r.charCodeAt(y))>>8,p=o%256,g.push(p),g.push(u);return g}(i,r.length-o),r,o,u)}function base64Slice(r,i,o){return 0===i&&o===r.length?u.fromByteArray(r):u.fromByteArray(r.slice(i,o))}function utf8Slice(r,i,o){o=Math.min(r.length,o);for(var u=[],p=i;p<o;){var g=r[p],y=null,_=g>239?4:g>223?3:g>191?2:1;if(p+_<=o){var x=void 0,k=void 0,P=void 0,O=void 0;switch(_){case 1:g<128&&(y=g);break;case 2:128==(192&(x=r[p+1]))&&(O=(31&g)<<6|63&x)>127&&(y=O);break;case 3:x=r[p+1],k=r[p+2],128==(192&x)&&128==(192&k)&&(O=(15&g)<<12|(63&x)<<6|63&k)>2047&&(O<55296||O>57343)&&(y=O);break;case 4:x=r[p+1],k=r[p+2],P=r[p+3],128==(192&x)&&128==(192&k)&&128==(192&P)&&(O=(15&g)<<18|(63&x)<<12|(63&k)<<6|63&P)>65535&&O<1114112&&(y=O)}}null===y?(y=65533,_=1):y>65535&&(y-=65536,u.push(y>>>10&1023|55296),y=56320|1023&y),u.push(y),p+=_}return function decodeCodePointsArray(r){var i=r.length;if(i<=w)return String.fromCharCode.apply(String,r);var o="",u=0;for(;u<i;)o+=String.fromCharCode.apply(String,r.slice(u,u+=w));return o}(u)}i.kMaxLength=y,Buffer.TYPED_ARRAY_SUPPORT=function typedArraySupport(){try{var r=new Uint8Array(1),i={foo:function foo(){return 42}};return Object.setPrototypeOf(i,Uint8Array.prototype),Object.setPrototypeOf(r,i),42===r.foo()}catch(r){return!1}}(),!Buffer.TYPED_ARRAY_SUPPORT&&"undefined"!=typeof console&&console.error,Object.defineProperty(Buffer.prototype,"parent",{enumerable:!0,get:function get(){if(Buffer.isBuffer(this))return this.buffer}}),Object.defineProperty(Buffer.prototype,"offset",{enumerable:!0,get:function get(){if(Buffer.isBuffer(this))return this.byteOffset}}),Buffer.poolSize=8192,Buffer.from=function(r,i,o){return from(r,i,o)},Object.setPrototypeOf(Buffer.prototype,Uint8Array.prototype),Object.setPrototypeOf(Buffer,Uint8Array),Buffer.alloc=function(r,i,o){return function alloc(r,i,o){return assertSize(r),r<=0?createBuffer(r):void 0!==i?"string"==typeof o?createBuffer(r).fill(i,o):createBuffer(r).fill(i):createBuffer(r)}(r,i,o)},Buffer.allocUnsafe=function(r){return allocUnsafe(r)},Buffer.allocUnsafeSlow=function(r){return allocUnsafe(r)},Buffer.isBuffer=function isBuffer(r){return null!=r&&!0===r._isBuffer&&r!==Buffer.prototype},Buffer.compare=function compare(r,i){if(isInstance(r,Uint8Array)&&(r=Buffer.from(r,r.offset,r.byteLength)),isInstance(i,Uint8Array)&&(i=Buffer.from(i,i.offset,i.byteLength)),!Buffer.isBuffer(r)||!Buffer.isBuffer(i))throw new TypeError('The "buf1", "buf2" arguments must be one of type Buffer or Uint8Array');if(r===i)return 0;for(var o=r.length,u=i.length,p=0,g=Math.min(o,u);p<g;++p)if(r[p]!==i[p]){o=r[p],u=i[p];break}return o<u?-1:u<o?1:0},Buffer.isEncoding=function isEncoding(r){switch(String(r).toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"latin1":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return!0;default:return!1}},Buffer.concat=function concat(r,i){if(!Array.isArray(r))throw new TypeError('"list" argument must be an Array of Buffers');if(0===r.length)return Buffer.alloc(0);var o;if(void 0===i)for(i=0,o=0;o<r.length;++o)i+=r[o].length;var u=Buffer.allocUnsafe(i),p=0;for(o=0;o<r.length;++o){var g=r[o];if(isInstance(g,Uint8Array))p+g.length>u.length?(Buffer.isBuffer(g)||(g=Buffer.from(g)),g.copy(u,p)):Uint8Array.prototype.set.call(u,g,p);else{if(!Buffer.isBuffer(g))throw new TypeError('"list" argument must be an Array of Buffers');g.copy(u,p)}p+=g.length}return u},Buffer.byteLength=byteLength,Buffer.prototype._isBuffer=!0,Buffer.prototype.swap16=function swap16(){var r=this.length;if(r%2!=0)throw new RangeError("Buffer size must be a multiple of 16-bits");for(var i=0;i<r;i+=2)swap(this,i,i+1);return this},Buffer.prototype.swap32=function swap32(){var r=this.length;if(r%4!=0)throw new RangeError("Buffer size must be a multiple of 32-bits");for(var i=0;i<r;i+=4)swap(this,i,i+3),swap(this,i+1,i+2);return this},Buffer.prototype.swap64=function swap64(){var r=this.length;if(r%8!=0)throw new RangeError("Buffer size must be a multiple of 64-bits");for(var i=0;i<r;i+=8)swap(this,i,i+7),swap(this,i+1,i+6),swap(this,i+2,i+5),swap(this,i+3,i+4);return this},Buffer.prototype.toString=function toString(){var r=this.length;return 0===r?"":0===arguments.length?utf8Slice(this,0,r):slowToString.apply(this,arguments)},Buffer.prototype.toLocaleString=Buffer.prototype.toString,Buffer.prototype.equals=function equals(r){if(!Buffer.isBuffer(r))throw new TypeError("Argument must be a Buffer");return this===r||0===Buffer.compare(this,r)},Buffer.prototype.inspect=function inspect(){var r="",o=i.INSPECT_MAX_BYTES;return r=this.toString("hex",0,o).replace(/(.{2})/g,"$1 ").trim(),this.length>o&&(r+=" ... "),"<Buffer "+r+">"},g&&(Buffer.prototype[g]=Buffer.prototype.inspect),Buffer.prototype.compare=function compare(r,i,o,u,p){if(isInstance(r,Uint8Array)&&(r=Buffer.from(r,r.offset,r.byteLength)),!Buffer.isBuffer(r))throw new TypeError('The "target" argument must be one of type Buffer or Uint8Array. Received type '+typeof r);if(void 0===i&&(i=0),void 0===o&&(o=r?r.length:0),void 0===u&&(u=0),void 0===p&&(p=this.length),i<0||o>r.length||u<0||p>this.length)throw new RangeError("out of range index");if(u>=p&&i>=o)return 0;if(u>=p)return-1;if(i>=o)return 1;if(this===r)return 0;for(var g=(p>>>=0)-(u>>>=0),y=(o>>>=0)-(i>>>=0),w=Math.min(g,y),_=this.slice(u,p),x=r.slice(i,o),k=0;k<w;++k)if(_[k]!==x[k]){g=_[k],y=x[k];break}return g<y?-1:y<g?1:0},Buffer.prototype.includes=function includes(r,i,o){return-1!==this.indexOf(r,i,o)},Buffer.prototype.indexOf=function indexOf(r,i,o){return bidirectionalIndexOf(this,r,i,o,!0)},Buffer.prototype.lastIndexOf=function lastIndexOf(r,i,o){return bidirectionalIndexOf(this,r,i,o,!1)},Buffer.prototype.write=function write(r,i,o,u){if(void 0===i)u="utf8",o=this.length,i=0;else if(void 0===o&&"string"==typeof i)u=i,o=this.length,i=0;else{if(!isFinite(i))throw new Error("Buffer.write(string, encoding, offset[, length]) is no longer supported");i>>>=0,isFinite(o)?(o>>>=0,void 0===u&&(u="utf8")):(u=o,o=void 0)}var p=this.length-i;if((void 0===o||o>p)&&(o=p),r.length>0&&(o<0||i<0)||i>this.length)throw new RangeError("Attempt to write outside buffer bounds");u||(u="utf8");for(var g=!1;;)switch(u){case"hex":return hexWrite(this,r,i,o);case"utf8":case"utf-8":return utf8Write(this,r,i,o);case"ascii":case"latin1":case"binary":return asciiWrite(this,r,i,o);case"base64":return base64Write(this,r,i,o);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return ucs2Write(this,r,i,o);default:if(g)throw new TypeError("Unknown encoding: "+u);u=(""+u).toLowerCase(),g=!0}},Buffer.prototype.toJSON=function toJSON(){return{type:"Buffer",data:Array.prototype.slice.call(this._arr||this,0)}};var w=4096;function asciiSlice(r,i,o){var u="";o=Math.min(r.length,o);for(var p=i;p<o;++p)u+=String.fromCharCode(127&r[p]);return u}function latin1Slice(r,i,o){var u="";o=Math.min(r.length,o);for(var p=i;p<o;++p)u+=String.fromCharCode(r[p]);return u}function hexSlice(r,i,o){var u=r.length;(!i||i<0)&&(i=0),(!o||o<0||o>u)&&(o=u);for(var p="",g=i;g<o;++g)p+=k[r[g]];return p}function utf16leSlice(r,i,o){for(var u=r.slice(i,o),p="",g=0;g<u.length-1;g+=2)p+=String.fromCharCode(u[g]+256*u[g+1]);return p}function checkOffset(r,i,o){if(r%1!=0||r<0)throw new RangeError("offset is not uint");if(r+i>o)throw new RangeError("Trying to access beyond buffer length")}function checkInt(r,i,o,u,p,g){if(!Buffer.isBuffer(r))throw new TypeError('"buffer" argument must be a Buffer instance');if(i>p||i<g)throw new RangeError('"value" argument is out of bounds');if(o+u>r.length)throw new RangeError("Index out of range")}function wrtBigUInt64LE(r,i,o,u,p){checkIntBI(i,u,p,r,o,7);var g=Number(i&BigInt(4294967295));r[o++]=g,g>>=8,r[o++]=g,g>>=8,r[o++]=g,g>>=8,r[o++]=g;var y=Number(i>>BigInt(32)&BigInt(4294967295));return r[o++]=y,y>>=8,r[o++]=y,y>>=8,r[o++]=y,y>>=8,r[o++]=y,o}function wrtBigUInt64BE(r,i,o,u,p){checkIntBI(i,u,p,r,o,7);var g=Number(i&BigInt(4294967295));r[o+7]=g,g>>=8,r[o+6]=g,g>>=8,r[o+5]=g,g>>=8,r[o+4]=g;var y=Number(i>>BigInt(32)&BigInt(4294967295));return r[o+3]=y,y>>=8,r[o+2]=y,y>>=8,r[o+1]=y,y>>=8,r[o]=y,o+8}function checkIEEE754(r,i,o,u,p,g){if(o+u>r.length)throw new RangeError("Index out of range");if(o<0)throw new RangeError("Index out of range")}function writeFloat(r,i,o,u,g){return i=+i,o>>>=0,g||checkIEEE754(r,0,o,4),p.write(r,i,o,u,23,4),o+4}function writeDouble(r,i,o,u,g){return i=+i,o>>>=0,g||checkIEEE754(r,0,o,8),p.write(r,i,o,u,52,8),o+8}Buffer.prototype.slice=function slice(r,i){var o=this.length;(r=~~r)<0?(r+=o)<0&&(r=0):r>o&&(r=o),(i=void 0===i?o:~~i)<0?(i+=o)<0&&(i=0):i>o&&(i=o),i<r&&(i=r);var u=this.subarray(r,i);return Object.setPrototypeOf(u,Buffer.prototype),u},Buffer.prototype.readUintLE=Buffer.prototype.readUIntLE=function readUIntLE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=this[r],p=1,g=0;++g<i&&(p*=256);)u+=this[r+g]*p;return u},Buffer.prototype.readUintBE=Buffer.prototype.readUIntBE=function readUIntBE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=this[r+--i],p=1;i>0&&(p*=256);)u+=this[r+--i]*p;return u},Buffer.prototype.readUint8=Buffer.prototype.readUInt8=function readUInt8(r,i){return r>>>=0,i||checkOffset(r,1,this.length),this[r]},Buffer.prototype.readUint16LE=Buffer.prototype.readUInt16LE=function readUInt16LE(r,i){return r>>>=0,i||checkOffset(r,2,this.length),this[r]|this[r+1]<<8},Buffer.prototype.readUint16BE=Buffer.prototype.readUInt16BE=function readUInt16BE(r,i){return r>>>=0,i||checkOffset(r,2,this.length),this[r]<<8|this[r+1]},Buffer.prototype.readUint32LE=Buffer.prototype.readUInt32LE=function readUInt32LE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),(this[r]|this[r+1]<<8|this[r+2]<<16)+16777216*this[r+3]},Buffer.prototype.readUint32BE=Buffer.prototype.readUInt32BE=function readUInt32BE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),16777216*this[r]+(this[r+1]<<16|this[r+2]<<8|this[r+3])},Buffer.prototype.readBigUInt64LE=defineBigIntMethod((function readBigUInt64LE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=i+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,24),p=this[++r]+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+o*Math.pow(2,24);return BigInt(u)+(BigInt(p)<<BigInt(32))})),Buffer.prototype.readBigUInt64BE=defineBigIntMethod((function readBigUInt64BE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=i*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+this[++r],p=this[++r]*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+o;return(BigInt(u)<<BigInt(32))+BigInt(p)})),Buffer.prototype.readIntLE=function readIntLE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=this[r],p=1,g=0;++g<i&&(p*=256);)u+=this[r+g]*p;return u>=(p*=128)&&(u-=Math.pow(2,8*i)),u},Buffer.prototype.readIntBE=function readIntBE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=i,p=1,g=this[r+--u];u>0&&(p*=256);)g+=this[r+--u]*p;return g>=(p*=128)&&(g-=Math.pow(2,8*i)),g},Buffer.prototype.readInt8=function readInt8(r,i){return r>>>=0,i||checkOffset(r,1,this.length),128&this[r]?-1*(255-this[r]+1):this[r]},Buffer.prototype.readInt16LE=function readInt16LE(r,i){r>>>=0,i||checkOffset(r,2,this.length);var o=this[r]|this[r+1]<<8;return 32768&o?4294901760|o:o},Buffer.prototype.readInt16BE=function readInt16BE(r,i){r>>>=0,i||checkOffset(r,2,this.length);var o=this[r+1]|this[r]<<8;return 32768&o?4294901760|o:o},Buffer.prototype.readInt32LE=function readInt32LE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),this[r]|this[r+1]<<8|this[r+2]<<16|this[r+3]<<24},Buffer.prototype.readInt32BE=function readInt32BE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),this[r]<<24|this[r+1]<<16|this[r+2]<<8|this[r+3]},Buffer.prototype.readBigInt64LE=defineBigIntMethod((function readBigInt64LE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=this[r+4]+this[r+5]*Math.pow(2,8)+this[r+6]*Math.pow(2,16)+(o<<24);return(BigInt(u)<<BigInt(32))+BigInt(i+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,24))})),Buffer.prototype.readBigInt64BE=defineBigIntMethod((function readBigInt64BE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=(i<<24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+this[++r];return(BigInt(u)<<BigInt(32))+BigInt(this[++r]*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+o)})),Buffer.prototype.readFloatLE=function readFloatLE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),p.read(this,r,!0,23,4)},Buffer.prototype.readFloatBE=function readFloatBE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),p.read(this,r,!1,23,4)},Buffer.prototype.readDoubleLE=function readDoubleLE(r,i){return r>>>=0,i||checkOffset(r,8,this.length),p.read(this,r,!0,52,8)},Buffer.prototype.readDoubleBE=function readDoubleBE(r,i){return r>>>=0,i||checkOffset(r,8,this.length),p.read(this,r,!1,52,8)},Buffer.prototype.writeUintLE=Buffer.prototype.writeUIntLE=function writeUIntLE(r,i,o,u){(r=+r,i>>>=0,o>>>=0,u)||checkInt(this,r,i,o,Math.pow(2,8*o)-1,0);var p=1,g=0;for(this[i]=255&r;++g<o&&(p*=256);)this[i+g]=r/p&255;return i+o},Buffer.prototype.writeUintBE=Buffer.prototype.writeUIntBE=function writeUIntBE(r,i,o,u){(r=+r,i>>>=0,o>>>=0,u)||checkInt(this,r,i,o,Math.pow(2,8*o)-1,0);var p=o-1,g=1;for(this[i+p]=255&r;--p>=0&&(g*=256);)this[i+p]=r/g&255;return i+o},Buffer.prototype.writeUint8=Buffer.prototype.writeUInt8=function writeUInt8(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,1,255,0),this[i]=255&r,i+1},Buffer.prototype.writeUint16LE=Buffer.prototype.writeUInt16LE=function writeUInt16LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,65535,0),this[i]=255&r,this[i+1]=r>>>8,i+2},Buffer.prototype.writeUint16BE=Buffer.prototype.writeUInt16BE=function writeUInt16BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,65535,0),this[i]=r>>>8,this[i+1]=255&r,i+2},Buffer.prototype.writeUint32LE=Buffer.prototype.writeUInt32LE=function writeUInt32LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,4294967295,0),this[i+3]=r>>>24,this[i+2]=r>>>16,this[i+1]=r>>>8,this[i]=255&r,i+4},Buffer.prototype.writeUint32BE=Buffer.prototype.writeUInt32BE=function writeUInt32BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,4294967295,0),this[i]=r>>>24,this[i+1]=r>>>16,this[i+2]=r>>>8,this[i+3]=255&r,i+4},Buffer.prototype.writeBigUInt64LE=defineBigIntMethod((function writeBigUInt64LE(r,i){return void 0===i&&(i=0),wrtBigUInt64LE(this,r,i,BigInt(0),BigInt("0xffffffffffffffff"))})),Buffer.prototype.writeBigUInt64BE=defineBigIntMethod((function writeBigUInt64BE(r,i){return void 0===i&&(i=0),wrtBigUInt64BE(this,r,i,BigInt(0),BigInt("0xffffffffffffffff"))})),Buffer.prototype.writeIntLE=function writeIntLE(r,i,o,u){if(r=+r,i>>>=0,!u){var p=Math.pow(2,8*o-1);checkInt(this,r,i,o,p-1,-p)}var g=0,y=1,w=0;for(this[i]=255&r;++g<o&&(y*=256);)r<0&&0===w&&0!==this[i+g-1]&&(w=1),this[i+g]=(r/y>>0)-w&255;return i+o},Buffer.prototype.writeIntBE=function writeIntBE(r,i,o,u){if(r=+r,i>>>=0,!u){var p=Math.pow(2,8*o-1);checkInt(this,r,i,o,p-1,-p)}var g=o-1,y=1,w=0;for(this[i+g]=255&r;--g>=0&&(y*=256);)r<0&&0===w&&0!==this[i+g+1]&&(w=1),this[i+g]=(r/y>>0)-w&255;return i+o},Buffer.prototype.writeInt8=function writeInt8(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,1,127,-128),r<0&&(r=255+r+1),this[i]=255&r,i+1},Buffer.prototype.writeInt16LE=function writeInt16LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,32767,-32768),this[i]=255&r,this[i+1]=r>>>8,i+2},Buffer.prototype.writeInt16BE=function writeInt16BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,32767,-32768),this[i]=r>>>8,this[i+1]=255&r,i+2},Buffer.prototype.writeInt32LE=function writeInt32LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,2147483647,-2147483648),this[i]=255&r,this[i+1]=r>>>8,this[i+2]=r>>>16,this[i+3]=r>>>24,i+4},Buffer.prototype.writeInt32BE=function writeInt32BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,2147483647,-2147483648),r<0&&(r=4294967295+r+1),this[i]=r>>>24,this[i+1]=r>>>16,this[i+2]=r>>>8,this[i+3]=255&r,i+4},Buffer.prototype.writeBigInt64LE=defineBigIntMethod((function writeBigInt64LE(r,i){return void 0===i&&(i=0),wrtBigUInt64LE(this,r,i,-BigInt("0x8000000000000000"),BigInt("0x7fffffffffffffff"))})),Buffer.prototype.writeBigInt64BE=defineBigIntMethod((function writeBigInt64BE(r,i){return void 0===i&&(i=0),wrtBigUInt64BE(this,r,i,-BigInt("0x8000000000000000"),BigInt("0x7fffffffffffffff"))})),Buffer.prototype.writeFloatLE=function writeFloatLE(r,i,o){return writeFloat(this,r,i,!0,o)},Buffer.prototype.writeFloatBE=function writeFloatBE(r,i,o){return writeFloat(this,r,i,!1,o)},Buffer.prototype.writeDoubleLE=function writeDoubleLE(r,i,o){return writeDouble(this,r,i,!0,o)},Buffer.prototype.writeDoubleBE=function writeDoubleBE(r,i,o){return writeDouble(this,r,i,!1,o)},Buffer.prototype.copy=function copy(r,i,o,u){if(!Buffer.isBuffer(r))throw new TypeError("argument should be a Buffer");if(o||(o=0),u||0===u||(u=this.length),i>=r.length&&(i=r.length),i||(i=0),u>0&&u<o&&(u=o),u===o)return 0;if(0===r.length||0===this.length)return 0;if(i<0)throw new RangeError("targetStart out of bounds");if(o<0||o>=this.length)throw new RangeError("Index out of range");if(u<0)throw new RangeError("sourceEnd out of bounds");u>this.length&&(u=this.length),r.length-i<u-o&&(u=r.length-i+o);var p=u-o;return this===r&&"function"==typeof Uint8Array.prototype.copyWithin?this.copyWithin(i,o,u):Uint8Array.prototype.set.call(r,this.subarray(o,u),i),p},Buffer.prototype.fill=function fill(r,i,o,u){if("string"==typeof r){if("string"==typeof i?(u=i,i=0,o=this.length):"string"==typeof o&&(u=o,o=this.length),void 0!==u&&"string"!=typeof u)throw new TypeError("encoding must be a string");if("string"==typeof u&&!Buffer.isEncoding(u))throw new TypeError("Unknown encoding: "+u);if(1===r.length){var p=r.charCodeAt(0);("utf8"===u&&p<128||"latin1"===u)&&(r=p)}}else"number"==typeof r?r&=255:"boolean"==typeof r&&(r=Number(r));if(i<0||this.length<i||this.length<o)throw new RangeError("Out of range index");if(o<=i)return this;var g;if(i>>>=0,o=void 0===o?this.length:o>>>0,r||(r=0),"number"==typeof r)for(g=i;g<o;++g)this[g]=r;else{var y=Buffer.isBuffer(r)?r:Buffer.from(r,u),w=y.length;if(0===w)throw new TypeError('The value "'+r+'" is invalid for argument "value"');for(g=0;g<o-i;++g)this[g+i]=y[g%w]}return this};var _={};function E(r,i,o){_[r]=function(o){function NodeError(){var u;return u=o.call(this)||this,Object.defineProperty(_assertThisInitialized(u),"message",{value:i.apply(_assertThisInitialized(u),arguments),writable:!0,configurable:!0}),u.name=u.name+" ["+r+"]",u.stack,delete u.name,u}return function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(NodeError,o),NodeError.prototype.toString=function toString(){return this.name+" ["+r+"]: "+this.message},function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),Object.defineProperty(r,"prototype",{writable:!1}),r}(NodeError,[{key:"code",get:function get(){return r},set:function set(r){Object.defineProperty(this,"code",{configurable:!0,enumerable:!0,value:r,writable:!0})}}]),NodeError}(o)}function addNumericalSeparator(r){for(var i="",o=r.length,u="-"===r[0]?1:0;o>=u+4;o-=3)i="_"+r.slice(o-3,o)+i;return""+r.slice(0,o)+i}function checkIntBI(r,i,o,u,p,g){if(r>o||r<i){var y,w="bigint"==typeof i?"n":"";throw y=g>3?0===i||i===BigInt(0)?">= 0"+w+" and < 2"+w+" ** "+8*(g+1)+w:">= -(2"+w+" ** "+(8*(g+1)-1)+w+") and < 2 ** "+(8*(g+1)-1)+w:">= "+i+w+" and <= "+o+w,new _.ERR_OUT_OF_RANGE("value",y,r)}!function checkBounds(r,i,o){validateNumber(i,"offset"),void 0!==r[i]&&void 0!==r[i+o]||boundsError(i,r.length-(o+1))}(u,p,g)}function validateNumber(r,i){if("number"!=typeof r)throw new _.ERR_INVALID_ARG_TYPE(i,"number",r)}function boundsError(r,i,o){if(Math.floor(r)!==r)throw validateNumber(r,o),new _.ERR_OUT_OF_RANGE(o||"offset","an integer",r);if(i<0)throw new _.ERR_BUFFER_OUT_OF_BOUNDS;throw new _.ERR_OUT_OF_RANGE(o||"offset",">= "+(o?1:0)+" and <= "+i,r)}E("ERR_BUFFER_OUT_OF_BOUNDS",(function(r){return r?r+" is outside of buffer bounds":"Attempt to access memory outside buffer bounds"}),RangeError),E("ERR_INVALID_ARG_TYPE",(function(r,i){return'The "'+r+'" argument must be of type number. Received type '+typeof i}),TypeError),E("ERR_OUT_OF_RANGE",(function(r,i,o){var u='The value of "'+r+'" is out of range.',p=o;return Number.isInteger(o)&&Math.abs(o)>Math.pow(2,32)?p=addNumericalSeparator(String(o)):"bigint"==typeof o&&(p=String(o),(o>Math.pow(BigInt(2),BigInt(32))||o<-Math.pow(BigInt(2),BigInt(32)))&&(p=addNumericalSeparator(p)),p+="n"),u+=" It must be "+i+". Received "+p}),RangeError);var x=/[^+/0-9A-Za-z-_]/g;function utf8ToBytes(r,i){var o;i=i||1/0;for(var u=r.length,p=null,g=[],y=0;y<u;++y){if((o=r.charCodeAt(y))>55295&&o<57344){if(!p){if(o>56319){(i-=3)>-1&&g.push(239,191,189);continue}if(y+1===u){(i-=3)>-1&&g.push(239,191,189);continue}p=o;continue}if(o<56320){(i-=3)>-1&&g.push(239,191,189),p=o;continue}o=65536+(p-55296<<10|o-56320)}else p&&(i-=3)>-1&&g.push(239,191,189);if(p=null,o<128){if((i-=1)<0)break;g.push(o)}else if(o<2048){if((i-=2)<0)break;g.push(o>>6|192,63&o|128)}else if(o<65536){if((i-=3)<0)break;g.push(o>>12|224,o>>6&63|128,63&o|128)}else{if(!(o<1114112))throw new Error("Invalid code point");if((i-=4)<0)break;g.push(o>>18|240,o>>12&63|128,o>>6&63|128,63&o|128)}}return g}function base64ToBytes(r){return u.toByteArray(function base64clean(r){if((r=(r=r.split("=")[0]).trim().replace(x,"")).length<2)return"";for(;r.length%4!=0;)r+="=";return r}(r))}function blitBuffer(r,i,o,u){var p;for(p=0;p<u&&!(p+o>=i.length||p>=r.length);++p)i[p+o]=r[p];return p}function isInstance(r,i){return r instanceof i||null!=r&&null!=r.constructor&&null!=r.constructor.name&&r.constructor.name===i.name}function numberIsNaN(r){return r!=r}var k=function(){for(var r="0123456789abcdef",i=new Array(256),o=0;o<16;++o)for(var u=16*o,p=0;p<16;++p)i[u+p]=r[o]+r[p];return i}();function defineBigIntMethod(r){return"undefined"==typeof BigInt?BufferBigIntNotDefined:r}function BufferBigIntNotDefined(){throw new Error("BigInt not supported")}},477:function(r,i,o){"use strict";o(7803),o(1539),r.exports="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof DataView},2094:function(r,i,o){"use strict";var u,p,g,y=o(477),w=o(9781),_=o(7854),x=o(614),k=o(111),P=o(2597),O=o(648),I=o(6330),B=o(8880),R=o(1320),D=o(3070).f,N=o(7976),W=o(9518),U=o(7674),j=o(5112),G=o(9711),X=_.Int8Array,K=X&&X.prototype,Y=_.Uint8ClampedArray,J=Y&&Y.prototype,$=X&&W(X),ee=K&&W(K),te=Object.prototype,re=_.TypeError,ne=j("toStringTag"),ie=G("TYPED_ARRAY_TAG"),oe=G("TYPED_ARRAY_CONSTRUCTOR"),ae=y&&!!U&&"Opera"!==O(_.opera),se=!1,ce={Int8Array:1,Uint8Array:1,Uint8ClampedArray:1,Int16Array:2,Uint16Array:2,Int32Array:4,Uint32Array:4,Float32Array:4,Float64Array:8},le={BigInt64Array:8,BigUint64Array:8},ue=function isTypedArray(r){if(!k(r))return!1;var i=O(r);return P(ce,i)||P(le,i)};for(u in ce)(g=(p=_[u])&&p.prototype)?B(g,oe,p):ae=!1;for(u in le)(g=(p=_[u])&&p.prototype)&&B(g,oe,p);if((!ae||!x($)||$===Function.prototype)&&($=function TypedArray(){throw re("Incorrect invocation")},ae))for(u in ce)_[u]&&U(_[u],$);if((!ae||!ee||ee===te)&&(ee=$.prototype,ae))for(u in ce)_[u]&&U(_[u].prototype,ee);if(ae&&W(J)!==ee&&U(J,ee),w&&!P(ee,ne))for(u in se=!0,D(ee,ne,{get:function get(){return k(this)?this[ie]:void 0}}),ce)_[u]&&B(_[u],ie,u);r.exports={NATIVE_ARRAY_BUFFER_VIEWS:ae,TYPED_ARRAY_CONSTRUCTOR:oe,TYPED_ARRAY_TAG:se&&ie,aTypedArray:function aTypedArray(r){if(ue(r))return r;throw re("Target is not a typed array")},aTypedArrayConstructor:function aTypedArrayConstructor(r){if(x(r)&&(!U||N($,r)))return r;throw re(I(r)+" is not a typed array constructor")},exportTypedArrayMethod:function exportTypedArrayMethod(r,i,o){if(w){if(o)for(var u in ce){var p=_[u];if(p&&P(p.prototype,r))try{delete p.prototype[r]}catch(r){}}ee[r]&&!o||R(ee,r,o?i:ae&&K[r]||i)}},exportTypedArrayStaticMethod:function exportTypedArrayStaticMethod(r,i,o){var u,p;if(w){if(U){if(o)for(u in ce)if((p=_[u])&&P(p,r))try{delete p[r]}catch(r){}if($[r]&&!o)return;try{return R($,r,o?i:ae&&$[r]||i)}catch(r){}}for(u in ce)!(p=_[u])||p[r]&&!o||R(p,r,i)}},isView:function isView(r){if(!k(r))return!1;var i=O(r);return"DataView"===i||P(ce,i)||P(le,i)},isTypedArray:ue,TypedArray:$,TypedArrayPrototype:ee}},2091:function(r,i,o){"use strict";o(8309);var u=o(7854),p=o(1702),g=o(9781),y=o(477),w=o(6530),_=o(8880),x=o(2248),k=o(7293),P=o(5787),O=o(9303),I=o(7466),B=o(7067),R=o(1179),D=o(9518),N=o(7674),W=o(8006).f,U=o(3070).f,j=o(1285),G=o(206),X=o(8003),K=o(9909),Y=w.PROPER,J=w.CONFIGURABLE,$=K.get,ee=K.set,te="ArrayBuffer",re="DataView",ne="Wrong index",ie=u.ArrayBuffer,oe=ie,ae=oe&&oe.prototype,se=u.DataView,ce=se&&se.prototype,le=Object.prototype,ue=u.Array,fe=u.RangeError,he=p(j),de=p([].reverse),pe=R.pack,ge=R.unpack,ye=function packInt8(r){return[255&r]},ve=function packInt16(r){return[255&r,r>>8&255]},me=function packInt32(r){return[255&r,r>>8&255,r>>16&255,r>>24&255]},be=function unpackInt32(r){return r[3]<<24|r[2]<<16|r[1]<<8|r[0]},we=function packFloat32(r){return pe(r,23,4)},_e=function packFloat64(r){return pe(r,52,8)},Se=function addGetter(r,i){U(r.prototype,i,{get:function get(){return $(this)[i]}})},xe=function get(r,i,o,u){var p=B(o),g=$(r);if(p+i>g.byteLength)throw fe(ne);var y=$(g.buffer).bytes,w=p+g.byteOffset,_=G(y,w,w+i);return u?_:de(_)},Ce=function set(r,i,o,u,p,g){var y=B(o),w=$(r);if(y+i>w.byteLength)throw fe(ne);for(var _=$(w.buffer).bytes,x=y+w.byteOffset,k=u(+p),P=0;P<i;P++)_[x+P]=k[g?P:i-P-1]};if(y){var ke=Y&&ie.name!==te;if(k((function(){ie(1)}))&&k((function(){new ie(-1)}))&&!k((function(){return new ie,new ie(1.5),new ie(NaN),ke&&!J})))ke&&J&&_(ie,"name",te);else{(oe=function ArrayBuffer(r){return P(this,ae),new ie(B(r))}).prototype=ae;for(var Pe,Ae=W(ie),Ee=0;Ae.length>Ee;)(Pe=Ae[Ee++])in oe||_(oe,Pe,ie[Pe]);ae.constructor=oe}N&&D(ce)!==le&&N(ce,le);var Te=new se(new oe(2)),Oe=p(ce.setInt8);Te.setInt8(0,2147483648),Te.setInt8(1,2147483649),!Te.getInt8(0)&&Te.getInt8(1)||x(ce,{setInt8:function setInt8(r,i){Oe(this,r,i<<24>>24)},setUint8:function setUint8(r,i){Oe(this,r,i<<24>>24)}},{unsafe:!0})}else ae=(oe=function ArrayBuffer(r){P(this,ae);var i=B(r);ee(this,{bytes:he(ue(i),0),byteLength:i}),g||(this.byteLength=i)}).prototype,ce=(se=function DataView(r,i,o){P(this,ce),P(r,ae);var u=$(r).byteLength,p=O(i);if(p<0||p>u)throw fe("Wrong offset");if(p+(o=void 0===o?u-p:I(o))>u)throw fe("Wrong length");ee(this,{buffer:r,byteLength:o,byteOffset:p}),g||(this.buffer=r,this.byteLength=o,this.byteOffset=p)}).prototype,g&&(Se(oe,"byteLength"),Se(se,"buffer"),Se(se,"byteLength"),Se(se,"byteOffset")),x(ce,{getInt8:function getInt8(r){return xe(this,1,r)[0]<<24>>24},getUint8:function getUint8(r){return xe(this,1,r)[0]},getInt16:function getInt16(r){var i=xe(this,2,r,arguments.length>1?arguments[1]:void 0);return(i[1]<<8|i[0])<<16>>16},getUint16:function getUint16(r){var i=xe(this,2,r,arguments.length>1?arguments[1]:void 0);return i[1]<<8|i[0]},getInt32:function getInt32(r){return be(xe(this,4,r,arguments.length>1?arguments[1]:void 0))},getUint32:function getUint32(r){return be(xe(this,4,r,arguments.length>1?arguments[1]:void 0))>>>0},getFloat32:function getFloat32(r){return ge(xe(this,4,r,arguments.length>1?arguments[1]:void 0),23)},getFloat64:function getFloat64(r){return ge(xe(this,8,r,arguments.length>1?arguments[1]:void 0),52)},setInt8:function setInt8(r,i){Ce(this,1,r,ye,i)},setUint8:function setUint8(r,i){Ce(this,1,r,ye,i)},setInt16:function setInt16(r,i){Ce(this,2,r,ve,i,arguments.length>2?arguments[2]:void 0)},setUint16:function setUint16(r,i){Ce(this,2,r,ve,i,arguments.length>2?arguments[2]:void 0)},setInt32:function setInt32(r,i){Ce(this,4,r,me,i,arguments.length>2?arguments[2]:void 0)},setUint32:function setUint32(r,i){Ce(this,4,r,me,i,arguments.length>2?arguments[2]:void 0)},setFloat32:function setFloat32(r,i){Ce(this,4,r,we,i,arguments.length>2?arguments[2]:void 0)},setFloat64:function setFloat64(r,i){Ce(this,8,r,_e,i,arguments.length>2?arguments[2]:void 0)}});X(oe,te),X(se,re),r.exports={ArrayBuffer:oe,DataView:se}},7803:function(r,i,o){"use strict";var u=o(2109),p=o(7854),g=o(2091),y=o(6340),w="ArrayBuffer",_=g.ArrayBuffer;u({global:!0,forced:p.ArrayBuffer!==_},{ArrayBuffer:_}),y(w)},194:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(7508),o(3440),o(3839),o(1582),function(){var r=u,i=r.lib.BlockCipher,o=r.algo,p=[],g=[],y=[],w=[],_=[],x=[],k=[],P=[],O=[],I=[];!function(){for(var r=[],i=0;i<256;i++)r[i]=i<128?i<<1:i<<1^283;var o=0,u=0;for(i=0;i<256;i++){var B=u^u<<1^u<<2^u<<3^u<<4;B=B>>>8^255&B^99,p[o]=B,g[B]=o;var R=r[o],D=r[R],N=r[D],W=257*r[B]^16843008*B;y[o]=W<<24|W>>>8,w[o]=W<<16|W>>>16,_[o]=W<<8|W>>>24,x[o]=W,W=16843009*N^65537*D^257*R^16843008*o,k[B]=W<<24|W>>>8,P[B]=W<<16|W>>>16,O[B]=W<<8|W>>>24,I[B]=W,o?(o=R^r[r[r[N^R]]],u^=r[r[u]]):o=u=1}}();var B=[0,1,2,4,8,16,32,64,128,27,54],R=o.AES=i.extend({_doReset:function _doReset(){if(!this._nRounds||this._keyPriorReset!==this._key){for(var r=this._keyPriorReset=this._key,i=r.words,o=r.sigBytes/4,u=4*((this._nRounds=o+6)+1),g=this._keySchedule=[],y=0;y<u;y++)y<o?g[y]=i[y]:(x=g[y-1],y%o?o>6&&y%o==4&&(x=p[x>>>24]<<24|p[x>>>16&255]<<16|p[x>>>8&255]<<8|p[255&x]):(x=p[(x=x<<8|x>>>24)>>>24]<<24|p[x>>>16&255]<<16|p[x>>>8&255]<<8|p[255&x],x^=B[y/o|0]<<24),g[y]=g[y-o]^x);for(var w=this._invKeySchedule=[],_=0;_<u;_++){if(y=u-_,_%4)var x=g[y];else x=g[y-4];w[_]=_<4||y<=4?x:k[p[x>>>24]]^P[p[x>>>16&255]]^O[p[x>>>8&255]]^I[p[255&x]]}}},encryptBlock:function encryptBlock(r,i){this._doCryptBlock(r,i,this._keySchedule,y,w,_,x,p)},decryptBlock:function decryptBlock(r,i){var o=r[i+1];r[i+1]=r[i+3],r[i+3]=o,this._doCryptBlock(r,i,this._invKeySchedule,k,P,O,I,g),o=r[i+1],r[i+1]=r[i+3],r[i+3]=o},_doCryptBlock:function _doCryptBlock(r,i,o,u,p,g,y,w){for(var _=this._nRounds,x=r[i]^o[0],k=r[i+1]^o[1],P=r[i+2]^o[2],O=r[i+3]^o[3],I=4,B=1;B<_;B++){var R=u[x>>>24]^p[k>>>16&255]^g[P>>>8&255]^y[255&O]^o[I++],D=u[k>>>24]^p[P>>>16&255]^g[O>>>8&255]^y[255&x]^o[I++],N=u[P>>>24]^p[O>>>16&255]^g[x>>>8&255]^y[255&k]^o[I++],W=u[O>>>24]^p[x>>>16&255]^g[k>>>8&255]^y[255&P]^o[I++];x=R,k=D,P=N,O=W}R=(w[x>>>24]<<24|w[k>>>16&255]<<16|w[P>>>8&255]<<8|w[255&O])^o[I++],D=(w[k>>>24]<<24|w[P>>>16&255]<<16|w[O>>>8&255]<<8|w[255&x])^o[I++],N=(w[P>>>24]<<24|w[O>>>16&255]<<16|w[x>>>8&255]<<8|w[255&k])^o[I++],W=(w[O>>>24]<<24|w[x>>>16&255]<<16|w[k>>>8&255]<<8|w[255&P])^o[I++],r[i]=R,r[i+1]=D,r[i+2]=N,r[i+3]=W},keySize:8});r.AES=i._createHelper(R)}(),u.AES)},1582:function(r,i,o){"use strict";var u;o(7042),o(2222),o(1539),o(9714),o(561),r.exports=(u=o(757),o(3839),void(u.lib.Cipher||function(r){var i=u,o=i.lib,p=o.Base,g=o.WordArray,y=o.BufferedBlockAlgorithm,w=i.enc,_=(w.Utf8,w.Base64),x=i.algo.EvpKDF,k=o.Cipher=y.extend({cfg:p.extend(),createEncryptor:function createEncryptor(r,i){return this.create(this._ENC_XFORM_MODE,r,i)},createDecryptor:function createDecryptor(r,i){return this.create(this._DEC_XFORM_MODE,r,i)},init:function init(r,i,o){this.cfg=this.cfg.extend(o),this._xformMode=r,this._key=i,this.reset()},reset:function reset(){y.reset.call(this),this._doReset()},process:function process(r){return this._append(r),this._process()},finalize:function finalize(r){return r&&this._append(r),this._doFinalize()},keySize:4,ivSize:4,_ENC_XFORM_MODE:1,_DEC_XFORM_MODE:2,_createHelper:function(){function selectCipherStrategy(r){return"string"==typeof r?U:N}return function(r){return{encrypt:function encrypt(i,o,u){return selectCipherStrategy(o).encrypt(r,i,o,u)},decrypt:function decrypt(i,o,u){return selectCipherStrategy(o).decrypt(r,i,o,u)}}}}()}),P=(o.StreamCipher=k.extend({_doFinalize:function _doFinalize(){return this._process(!0)},blockSize:1}),i.mode={}),O=o.BlockCipherMode=p.extend({createEncryptor:function createEncryptor(r,i){return this.Encryptor.create(r,i)},createDecryptor:function createDecryptor(r,i){return this.Decryptor.create(r,i)},init:function init(r,i){this._cipher=r,this._iv=i}}),I=P.CBC=function(){var i=O.extend();function xorBlock(i,o,u){var p,g=this._iv;g?(p=g,this._iv=r):p=this._prevBlock;for(var y=0;y<u;y++)i[o+y]^=p[y]}return i.Encryptor=i.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize;xorBlock.call(this,r,i,u),o.encryptBlock(r,i),this._prevBlock=r.slice(i,i+u)}}),i.Decryptor=i.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=r.slice(i,i+u);o.decryptBlock(r,i),xorBlock.call(this,r,i,u),this._prevBlock=p}}),i}(),B=(i.pad={}).Pkcs7={pad:function pad(r,i){for(var o=4*i,u=o-r.sigBytes%o,p=u<<24|u<<16|u<<8|u,y=[],w=0;w<u;w+=4)y.push(p);var _=g.create(y,u);r.concat(_)},unpad:function unpad(r){var i=255&r.words[r.sigBytes-1>>>2];r.sigBytes-=i}},R=(o.BlockCipher=k.extend({cfg:k.cfg.extend({mode:I,padding:B}),reset:function reset(){var r;k.reset.call(this);var i=this.cfg,o=i.iv,u=i.mode;this._xformMode==this._ENC_XFORM_MODE?r=u.createEncryptor:(r=u.createDecryptor,this._minBufferSize=1),this._mode&&this._mode.__creator==r?this._mode.init(this,o&&o.words):(this._mode=r.call(u,this,o&&o.words),this._mode.__creator=r)},_doProcessBlock:function _doProcessBlock(r,i){this._mode.processBlock(r,i)},_doFinalize:function _doFinalize(){var r,i=this.cfg.padding;return this._xformMode==this._ENC_XFORM_MODE?(i.pad(this._data,this.blockSize),r=this._process(!0)):(r=this._process(!0),i.unpad(r)),r},blockSize:4}),o.CipherParams=p.extend({init:function init(r){this.mixIn(r)},toString:function toString(r){return(r||this.formatter).stringify(this)}})),D=(i.format={}).OpenSSL={stringify:function stringify(r){var i=r.ciphertext,o=r.salt;return(o?g.create([1398893684,1701076831]).concat(o).concat(i):i).toString(_)},parse:function parse(r){var i,o=_.parse(r),u=o.words;return 1398893684==u[0]&&1701076831==u[1]&&(i=g.create(u.slice(2,4)),u.splice(0,4),o.sigBytes-=16),R.create({ciphertext:o,salt:i})}},N=o.SerializableCipher=p.extend({cfg:p.extend({format:D}),encrypt:function encrypt(r,i,o,u){u=this.cfg.extend(u);var p=r.createEncryptor(o,u),g=p.finalize(i),y=p.cfg;return R.create({ciphertext:g,key:o,iv:y.iv,algorithm:r,mode:y.mode,padding:y.padding,blockSize:r.blockSize,formatter:u.format})},decrypt:function decrypt(r,i,o,u){return u=this.cfg.extend(u),i=this._parse(i,u.format),r.createDecryptor(o,u).finalize(i.ciphertext)},_parse:function _parse(r,i){return"string"==typeof r?i.parse(r,this):r}}),W=(i.kdf={}).OpenSSL={execute:function execute(r,i,o,u){u||(u=g.random(8));var p=x.create({keySize:i+o}).compute(r,u),y=g.create(p.words.slice(i),4*o);return p.sigBytes=4*i,R.create({key:p,iv:y,salt:u})}},U=o.PasswordBasedCipher=N.extend({cfg:N.cfg.extend({kdf:W}),encrypt:function encrypt(r,i,o,u){var p=(u=this.cfg.extend(u)).kdf.execute(o,r.keySize,r.ivSize);u.iv=p.iv;var g=N.encrypt.call(this,r,i,p.key,u);return g.mixIn(p),g},decrypt:function decrypt(r,i,o,u){u=this.cfg.extend(u),i=this._parse(i,u.format);var p=u.kdf.execute(o,r.keySize,r.ivSize,i.salt);return u.iv=p.iv,N.decrypt.call(this,r,i,p.key,u)}})}()))},757:function(r,i,o){"use strict";var u;o(5743),o(6992),o(1539),o(9135),o(2990),o(8927),o(3105),o(5035),o(4345),o(7174),o(2846),o(4731),o(7209),o(6319),o(8867),o(7789),o(3739),o(9368),o(4483),o(2056),o(3462),o(678),o(7462),o(3824),o(5021),o(2974),o(5016),o(9714),o(7042),o(9600),o(2222),o(561),r.exports=(u=u||function(r,i){var u;if("undefined"!=typeof window&&window.crypto&&(u=window.crypto),"undefined"!=typeof self&&self.crypto&&(u=self.crypto),"undefined"!=typeof globalThis&&globalThis.crypto&&(u=globalThis.crypto),!u&&"undefined"!=typeof window&&window.msCrypto&&(u=window.msCrypto),!u&&void 0!==o.g&&o.g.crypto&&(u=o.g.crypto),!u)try{u=o(2480)}catch(r){}var p=function cryptoSecureRandomInt(){if(u){if("function"==typeof u.getRandomValues)try{return u.getRandomValues(new Uint32Array(1))[0]}catch(r){}if("function"==typeof u.randomBytes)try{return u.randomBytes(4).readInt32LE()}catch(r){}}throw new Error("Native crypto module could not be used to get secure random number.")},g=Object.create||function(){function F(){}return function(r){var i;return F.prototype=r,i=new F,F.prototype=null,i}}(),y={},w=y.lib={},_=w.Base={extend:function extend(r){var i=g(this);return r&&i.mixIn(r),i.hasOwnProperty("init")&&this.init!==i.init||(i.init=function(){i.$super.init.apply(this,arguments)}),i.init.prototype=i,i.$super=this,i},create:function create(){var r=this.extend();return r.init.apply(r,arguments),r},init:function init(){},mixIn:function mixIn(r){for(var i in r)r.hasOwnProperty(i)&&(this[i]=r[i]);r.hasOwnProperty("toString")&&(this.toString=r.toString)},clone:function clone(){return this.init.prototype.extend(this)}},x=w.WordArray=_.extend({init:function init(r,o){r=this.words=r||[],this.sigBytes=o!=i?o:4*r.length},toString:function toString(r){return(r||P).stringify(this)},concat:function concat(r){var i=this.words,o=r.words,u=this.sigBytes,p=r.sigBytes;if(this.clamp(),u%4)for(var g=0;g<p;g++){var y=o[g>>>2]>>>24-g%4*8&255;i[u+g>>>2]|=y<<24-(u+g)%4*8}else for(var w=0;w<p;w+=4)i[u+w>>>2]=o[w>>>2];return this.sigBytes+=p,this},clamp:function clamp(){var i=this.words,o=this.sigBytes;i[o>>>2]&=4294967295<<32-o%4*8,i.length=r.ceil(o/4)},clone:function clone(){var clone=_.clone.call(this);return clone.words=this.words.slice(0),clone},random:function random(r){for(var i=[],o=0;o<r;o+=4)i.push(p());return new x.init(i,r)}}),k=y.enc={},P=k.Hex={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p++){var g=i[p>>>2]>>>24-p%4*8&255;u.push((g>>>4).toString(16)),u.push((15&g).toString(16))}return u.join("")},parse:function parse(r){for(var i=r.length,o=[],u=0;u<i;u+=2)o[u>>>3]|=parseInt(r.substr(u,2),16)<<24-u%8*4;return new x.init(o,i/2)}},O=k.Latin1={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p++){var g=i[p>>>2]>>>24-p%4*8&255;u.push(String.fromCharCode(g))}return u.join("")},parse:function parse(r){for(var i=r.length,o=[],u=0;u<i;u++)o[u>>>2]|=(255&r.charCodeAt(u))<<24-u%4*8;return new x.init(o,i)}},I=k.Utf8={stringify:function stringify(r){try{return decodeURIComponent(escape(O.stringify(r)))}catch(r){throw new Error("Malformed UTF-8 data")}},parse:function parse(r){return O.parse(unescape(encodeURIComponent(r)))}},B=w.BufferedBlockAlgorithm=_.extend({reset:function reset(){this._data=new x.init,this._nDataBytes=0},_append:function _append(r){"string"==typeof r&&(r=I.parse(r)),this._data.concat(r),this._nDataBytes+=r.sigBytes},_process:function _process(i){var o,u=this._data,p=u.words,g=u.sigBytes,y=this.blockSize,w=g/(4*y),_=(w=i?r.ceil(w):r.max((0|w)-this._minBufferSize,0))*y,k=r.min(4*_,g);if(_){for(var P=0;P<_;P+=y)this._doProcessBlock(p,P);o=p.splice(0,_),u.sigBytes-=k}return new x.init(o,k)},clone:function clone(){var clone=_.clone.call(this);return clone._data=this._data.clone(),clone},_minBufferSize:0}),R=(w.Hasher=B.extend({cfg:_.extend(),init:function init(r){this.cfg=this.cfg.extend(r),this.reset()},reset:function reset(){B.reset.call(this),this._doReset()},update:function update(r){return this._append(r),this._process(),this},finalize:function finalize(r){return r&&this._append(r),this._doFinalize()},blockSize:16,_createHelper:function _createHelper(r){return function(i,o){return new r.init(o).finalize(i)}},_createHmacHelper:function _createHmacHelper(r){return function(i,o){return new R.HMAC.init(r,o).finalize(i)}}}),y.algo={});return y}(Math),u)},7508:function(r,i,o){"use strict";var u;o(9600),r.exports=(u=o(757),function(){var r=u,i=r.lib.WordArray;function parseLoop(r,o,u){for(var p=[],g=0,y=0;y<o;y++)if(y%4){var w=u[r.charCodeAt(y-1)]<<y%4*2|u[r.charCodeAt(y)]>>>6-y%4*2;p[g>>>2]|=w<<24-g%4*8,g++}return i.create(p,g)}r.enc.Base64={stringify:function stringify(r){var i=r.words,o=r.sigBytes,u=this._map;r.clamp();for(var p=[],g=0;g<o;g+=3)for(var y=(i[g>>>2]>>>24-g%4*8&255)<<16|(i[g+1>>>2]>>>24-(g+1)%4*8&255)<<8|i[g+2>>>2]>>>24-(g+2)%4*8&255,w=0;w<4&&g+.75*w<o;w++)p.push(u.charAt(y>>>6*(3-w)&63));var _=u.charAt(64);if(_)for(;p.length%4;)p.push(_);return p.join("")},parse:function parse(r){var i=r.length,o=this._map,u=this._reverseMap;if(!u){u=this._reverseMap=[];for(var p=0;p<o.length;p++)u[o.charCodeAt(p)]=p}var g=o.charAt(64);if(g){var y=r.indexOf(g);-1!==y&&(i=y)}return parseLoop(r,i,u)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="}}(),u.enc.Base64)},7590:function(r,i,o){"use strict";var u;o(9600),r.exports=(u=o(757),function(){var r=u,i=r.lib.WordArray;function parseLoop(r,o,u){for(var p=[],g=0,y=0;y<o;y++)if(y%4){var w=u[r.charCodeAt(y-1)]<<y%4*2|u[r.charCodeAt(y)]>>>6-y%4*2;p[g>>>2]|=w<<24-g%4*8,g++}return i.create(p,g)}r.enc.Base64url={stringify:function stringify(r,i){void 0===i&&(i=!0);var o=r.words,u=r.sigBytes,p=i?this._safe_map:this._map;r.clamp();for(var g=[],y=0;y<u;y+=3)for(var w=(o[y>>>2]>>>24-y%4*8&255)<<16|(o[y+1>>>2]>>>24-(y+1)%4*8&255)<<8|o[y+2>>>2]>>>24-(y+2)%4*8&255,_=0;_<4&&y+.75*_<u;_++)g.push(p.charAt(w>>>6*(3-_)&63));var x=p.charAt(64);if(x)for(;g.length%4;)g.push(x);return g.join("")},parse:function parse(r,i){void 0===i&&(i=!0);var o=r.length,u=i?this._safe_map:this._map,p=this._reverseMap;if(!p){p=this._reverseMap=[];for(var g=0;g<u.length;g++)p[u.charCodeAt(g)]=g}var y=u.charAt(64);if(y){var w=r.indexOf(y);-1!==w&&(o=w)}return parseLoop(r,o,p)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",_safe_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_"}}(),u.enc.Base64url)},4978:function(r,i,o){"use strict";var u;o(9600),r.exports=(u=o(757),function(){var r=u,i=r.lib.WordArray,o=r.enc;function swapEndian(r){return r<<8&4278255360|r>>>8&16711935}o.Utf16=o.Utf16BE={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p+=2){var g=i[p>>>2]>>>16-p%4*8&65535;u.push(String.fromCharCode(g))}return u.join("")},parse:function parse(r){for(var o=r.length,u=[],p=0;p<o;p++)u[p>>>1]|=r.charCodeAt(p)<<16-p%2*16;return i.create(u,2*o)}},o.Utf16LE={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p+=2){var g=swapEndian(i[p>>>2]>>>16-p%4*8&65535);u.push(String.fromCharCode(g))}return u.join("")},parse:function parse(r){for(var o=r.length,u=[],p=0;p<o;p++)u[p>>>1]|=swapEndian(r.charCodeAt(p)<<16-p%2*16);return i.create(u,2*o)}}}(),u.enc.Utf16)},3839:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k;o(2222),r.exports=(k=o(757),o(9865),o(6727),p=(u=k).lib,g=p.Base,y=p.WordArray,w=u.algo,_=w.MD5,x=w.EvpKDF=g.extend({cfg:g.extend({keySize:4,hasher:_,iterations:1}),init:function init(r){this.cfg=this.cfg.extend(r)},compute:function compute(r,i){for(var o,u=this.cfg,p=u.hasher.create(),g=y.create(),w=g.words,_=u.keySize,x=u.iterations;w.length<_;){o&&p.update(o),o=p.update(r).finalize(i),p.reset();for(var k=1;k<x;k++)o=p.finalize(o),p.reset();g.concat(o)}return g.sigBytes=4*_,g}}),u.EvpKDF=function(r,i,o){return x.create(o).compute(r,i)},k.EvpKDF)},8942:function(r,i,o){"use strict";var u,p,g,y;o(1539),o(9714),r.exports=(y=o(757),o(1582),p=(u=y).lib.CipherParams,g=u.enc.Hex,u.format.Hex={stringify:function stringify(r){return r.ciphertext.toString(g)},parse:function parse(r){var i=g.parse(r);return p.create({ciphertext:i})}},y.format.Hex)},6727:function(r,i,o){"use strict";var u,p,g,y;o(2222),r.exports=(u=o(757),g=(p=u).lib.Base,y=p.enc.Utf8,void(p.algo.HMAC=g.extend({init:function init(r,i){r=this._hasher=new r.init,"string"==typeof i&&(i=y.parse(i));var o=r.blockSize,u=4*o;i.sigBytes>u&&(i=r.finalize(i)),i.clamp();for(var p=this._oKey=i.clone(),g=this._iKey=i.clone(),w=p.words,_=g.words,x=0;x<o;x++)w[x]^=1549556828,_[x]^=909522486;p.sigBytes=g.sigBytes=u,this.reset()},reset:function reset(){var r=this._hasher;r.reset(),r.update(this._iKey)},update:function update(r){return this._hasher.update(r),this},finalize:function finalize(r){var i=this._hasher,o=i.finalize(r);return i.reset(),i.finalize(this._oKey.clone().concat(o))}})))},5153:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(2601),o(1947),o(4978),o(7508),o(7590),o(3440),o(9865),o(8921),o(6876),o(7991),o(8122),o(8342),o(8714),o(6727),o(3486),o(3839),o(1582),o(702),o(2362),o(4412),o(5720),o(3518),o(6362),o(4431),o(8800),o(3992),o(649),o(8942),o(194),o(8437),o(4640),o(5323),o(4363),u)},1947:function(r,i,o){"use strict";var u;o(7803),o(1539),o(6992),o(2472),o(2990),o(8927),o(3105),o(5035),o(4345),o(7174),o(2846),o(4731),o(7209),o(6319),o(8867),o(7789),o(3739),o(9368),o(4483),o(2056),o(3462),o(678),o(7462),o(3824),o(5021),o(2974),o(5016),o(7145),o(9743),o(5109),o(8255),o(5125),o(9135),o(4197),o(6495),r.exports=(u=o(757),function(){if("function"==typeof ArrayBuffer){var r=u.lib.WordArray,i=r.init,o=r.init=function(r){if(r instanceof ArrayBuffer&&(r=new Uint8Array(r)),(r instanceof Int8Array||"undefined"!=typeof Uint8ClampedArray&&r instanceof Uint8ClampedArray||r instanceof Int16Array||r instanceof Uint16Array||r instanceof Int32Array||r instanceof Uint32Array||r instanceof Float32Array||r instanceof Float64Array)&&(r=new Uint8Array(r.buffer,r.byteOffset,r.byteLength)),r instanceof Uint8Array){for(var o=r.byteLength,u=[],p=0;p<o;p++)u[p>>>2]|=r[p]<<24-p%4*8;i.call(this,u,o)}else i.apply(this,arguments)};o.prototype=r}}(),u.lib.WordArray)},3440:function(r,i,o){"use strict";var u;r.exports=(u=o(757),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=[];!function(){for(var i=0;i<64;i++)w[i]=4294967296*r.abs(r.sin(i+1))|0}();var _=y.MD5=g.extend({_doReset:function _doReset(){this._hash=new p.init([1732584193,4023233417,2562383102,271733878])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=0;o<16;o++){var u=i+o,p=r[u];r[u]=16711935&(p<<8|p>>>24)|4278255360&(p<<24|p>>>8)}var g=this._hash.words,y=r[i+0],_=r[i+1],x=r[i+2],k=r[i+3],P=r[i+4],O=r[i+5],I=r[i+6],B=r[i+7],R=r[i+8],D=r[i+9],N=r[i+10],W=r[i+11],U=r[i+12],j=r[i+13],G=r[i+14],X=r[i+15],K=g[0],Y=g[1],J=g[2],$=g[3];K=FF(K,Y,J,$,y,7,w[0]),$=FF($,K,Y,J,_,12,w[1]),J=FF(J,$,K,Y,x,17,w[2]),Y=FF(Y,J,$,K,k,22,w[3]),K=FF(K,Y,J,$,P,7,w[4]),$=FF($,K,Y,J,O,12,w[5]),J=FF(J,$,K,Y,I,17,w[6]),Y=FF(Y,J,$,K,B,22,w[7]),K=FF(K,Y,J,$,R,7,w[8]),$=FF($,K,Y,J,D,12,w[9]),J=FF(J,$,K,Y,N,17,w[10]),Y=FF(Y,J,$,K,W,22,w[11]),K=FF(K,Y,J,$,U,7,w[12]),$=FF($,K,Y,J,j,12,w[13]),J=FF(J,$,K,Y,G,17,w[14]),K=GG(K,Y=FF(Y,J,$,K,X,22,w[15]),J,$,_,5,w[16]),$=GG($,K,Y,J,I,9,w[17]),J=GG(J,$,K,Y,W,14,w[18]),Y=GG(Y,J,$,K,y,20,w[19]),K=GG(K,Y,J,$,O,5,w[20]),$=GG($,K,Y,J,N,9,w[21]),J=GG(J,$,K,Y,X,14,w[22]),Y=GG(Y,J,$,K,P,20,w[23]),K=GG(K,Y,J,$,D,5,w[24]),$=GG($,K,Y,J,G,9,w[25]),J=GG(J,$,K,Y,k,14,w[26]),Y=GG(Y,J,$,K,R,20,w[27]),K=GG(K,Y,J,$,j,5,w[28]),$=GG($,K,Y,J,x,9,w[29]),J=GG(J,$,K,Y,B,14,w[30]),K=HH(K,Y=GG(Y,J,$,K,U,20,w[31]),J,$,O,4,w[32]),$=HH($,K,Y,J,R,11,w[33]),J=HH(J,$,K,Y,W,16,w[34]),Y=HH(Y,J,$,K,G,23,w[35]),K=HH(K,Y,J,$,_,4,w[36]),$=HH($,K,Y,J,P,11,w[37]),J=HH(J,$,K,Y,B,16,w[38]),Y=HH(Y,J,$,K,N,23,w[39]),K=HH(K,Y,J,$,j,4,w[40]),$=HH($,K,Y,J,y,11,w[41]),J=HH(J,$,K,Y,k,16,w[42]),Y=HH(Y,J,$,K,I,23,w[43]),K=HH(K,Y,J,$,D,4,w[44]),$=HH($,K,Y,J,U,11,w[45]),J=HH(J,$,K,Y,X,16,w[46]),K=II(K,Y=HH(Y,J,$,K,x,23,w[47]),J,$,y,6,w[48]),$=II($,K,Y,J,B,10,w[49]),J=II(J,$,K,Y,G,15,w[50]),Y=II(Y,J,$,K,O,21,w[51]),K=II(K,Y,J,$,U,6,w[52]),$=II($,K,Y,J,k,10,w[53]),J=II(J,$,K,Y,N,15,w[54]),Y=II(Y,J,$,K,_,21,w[55]),K=II(K,Y,J,$,R,6,w[56]),$=II($,K,Y,J,X,10,w[57]),J=II(J,$,K,Y,I,15,w[58]),Y=II(Y,J,$,K,j,21,w[59]),K=II(K,Y,J,$,P,6,w[60]),$=II($,K,Y,J,W,10,w[61]),J=II(J,$,K,Y,x,15,w[62]),Y=II(Y,J,$,K,D,21,w[63]),g[0]=g[0]+K|0,g[1]=g[1]+Y|0,g[2]=g[2]+J|0,g[3]=g[3]+$|0},_doFinalize:function _doFinalize(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;o[p>>>5]|=128<<24-p%32;var g=r.floor(u/4294967296),y=u;o[15+(p+64>>>9<<4)]=16711935&(g<<8|g>>>24)|4278255360&(g<<24|g>>>8),o[14+(p+64>>>9<<4)]=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),i.sigBytes=4*(o.length+1),this._process();for(var w=this._hash,_=w.words,x=0;x<4;x++){var k=_[x];_[x]=16711935&(k<<8|k>>>24)|4278255360&(k<<24|k>>>8)}return w},clone:function clone(){var clone=g.clone.call(this);return clone._hash=this._hash.clone(),clone}});function FF(r,i,o,u,p,g,y){var w=r+(i&o|~i&u)+p+y;return(w<<g|w>>>32-g)+i}function GG(r,i,o,u,p,g,y){var w=r+(i&u|o&~u)+p+y;return(w<<g|w>>>32-g)+i}function HH(r,i,o,u,p,g,y){var w=r+(i^o^u)+p+y;return(w<<g|w>>>32-g)+i}function II(r,i,o,u,p,g,y){var w=r+(o^(i|~u))+p+y;return(w<<g|w>>>32-g)+i}i.MD5=g._createHelper(_),i.HmacMD5=g._createHmacHelper(_)}(Math),u.MD5)},702:function(r,i,o){"use strict";var u;o(7042),r.exports=(u=o(757),o(1582),u.mode.CFB=function(){var r=u.lib.BlockCipherMode.extend();function generateKeystreamAndEncrypt(r,i,o,u){var p,g=this._iv;g?(p=g.slice(0),this._iv=void 0):p=this._prevBlock,u.encryptBlock(p,0);for(var y=0;y<o;y++)r[i+y]^=p[y]}return r.Encryptor=r.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize;generateKeystreamAndEncrypt.call(this,r,i,u,o),this._prevBlock=r.slice(i,i+u)}}),r.Decryptor=r.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=r.slice(i,i+u);generateKeystreamAndEncrypt.call(this,r,i,u,o),this._prevBlock=p}}),r}(),u.mode.CFB)},4412:function(r,i,o){"use strict";var u;o(7042),r.exports=(u=o(757),o(1582),u.mode.CTRGladman=function(){var r=u.lib.BlockCipherMode.extend();function incWord(r){if(255==(r>>24&255)){var i=r>>16&255,o=r>>8&255,u=255&r;255===i?(i=0,255===o?(o=0,255===u?u=0:++u):++o):++i,r=0,r+=i<<16,r+=o<<8,r+=u}else r+=1<<24;return r}function incCounter(r){return 0===(r[0]=incWord(r[0]))&&(r[1]=incWord(r[1])),r}var i=r.Encryptor=r.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=this._iv,g=this._counter;p&&(g=this._counter=p.slice(0),this._iv=void 0),incCounter(g);var y=g.slice(0);o.encryptBlock(y,0);for(var w=0;w<u;w++)r[i+w]^=y[w]}});return r.Decryptor=i,r}(),u.mode.CTRGladman)},2362:function(r,i,o){"use strict";var u,p,g;o(7042),r.exports=(g=o(757),o(1582),g.mode.CTR=(u=g.lib.BlockCipherMode.extend(),p=u.Encryptor=u.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=this._iv,g=this._counter;p&&(g=this._counter=p.slice(0),this._iv=void 0);var y=g.slice(0);o.encryptBlock(y,0),g[u-1]=g[u-1]+1|0;for(var w=0;w<u;w++)r[i+w]^=y[w]}}),u.Decryptor=p,u),g.mode.CTR)},3518:function(r,i,o){"use strict";var u,p;r.exports=(p=o(757),o(1582),p.mode.ECB=((u=p.lib.BlockCipherMode.extend()).Encryptor=u.extend({processBlock:function processBlock(r,i){this._cipher.encryptBlock(r,i)}}),u.Decryptor=u.extend({processBlock:function processBlock(r,i){this._cipher.decryptBlock(r,i)}}),u),p.mode.ECB)},5720:function(r,i,o){"use strict";var u,p,g;o(7042),r.exports=(g=o(757),o(1582),g.mode.OFB=(u=g.lib.BlockCipherMode.extend(),p=u.Encryptor=u.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=this._iv,g=this._keystream;p&&(g=this._keystream=p.slice(0),this._iv=void 0),o.encryptBlock(g,0);for(var y=0;y<u;y++)r[i+y]^=g[y]}}),u.Decryptor=p,u),g.mode.OFB)},6362:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(1582),u.pad.AnsiX923={pad:function pad(r,i){var o=r.sigBytes,u=4*i,p=u-o%u,g=o+p-1;r.clamp(),r.words[g>>>2]|=p<<24-g%4*8,r.sigBytes+=p},unpad:function unpad(r){var i=255&r.words[r.sigBytes-1>>>2];r.sigBytes-=i}},u.pad.Ansix923)},4431:function(r,i,o){"use strict";var u;o(2222),r.exports=(u=o(757),o(1582),u.pad.Iso10126={pad:function pad(r,i){var o=4*i,p=o-r.sigBytes%o;r.concat(u.lib.WordArray.random(p-1)).concat(u.lib.WordArray.create([p<<24],1))},unpad:function unpad(r){var i=255&r.words[r.sigBytes-1>>>2];r.sigBytes-=i}},u.pad.Iso10126)},8800:function(r,i,o){"use strict";var u;o(2222),r.exports=(u=o(757),o(1582),u.pad.Iso97971={pad:function pad(r,i){r.concat(u.lib.WordArray.create([2147483648],1)),u.pad.ZeroPadding.pad(r,i)},unpad:function unpad(r){u.pad.ZeroPadding.unpad(r),r.sigBytes--}},u.pad.Iso97971)},649:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(1582),u.pad.NoPadding={pad:function pad(){},unpad:function unpad(){}},u.pad.NoPadding)},3992:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(1582),u.pad.ZeroPadding={pad:function pad(r,i){var o=4*i;r.clamp(),r.sigBytes+=o-(r.sigBytes%o||o)},unpad:function unpad(r){var i=r.words,o=r.sigBytes-1;for(o=r.sigBytes-1;o>=0;o--)if(i[o>>>2]>>>24-o%4*8&255){r.sigBytes=o+1;break}}},u.pad.ZeroPadding)},3486:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k,P;o(2222),r.exports=(P=o(757),o(9865),o(6727),p=(u=P).lib,g=p.Base,y=p.WordArray,w=u.algo,_=w.SHA1,x=w.HMAC,k=w.PBKDF2=g.extend({cfg:g.extend({keySize:4,hasher:_,iterations:1}),init:function init(r){this.cfg=this.cfg.extend(r)},compute:function compute(r,i){for(var o=this.cfg,u=x.create(o.hasher,r),p=y.create(),g=y.create([1]),w=p.words,_=g.words,k=o.keySize,P=o.iterations;w.length<k;){var O=u.update(i).finalize(g);u.reset();for(var I=O.words,B=I.length,R=O,D=1;D<P;D++){R=u.finalize(R),u.reset();for(var N=R.words,W=0;W<B;W++)I[W]^=N[W]}p.concat(O),_[0]++}return p.sigBytes=4*k,p}}),u.PBKDF2=function(r,i,o){return k.create(o).compute(r,i)},P.PBKDF2)},4363:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(7508),o(3440),o(3839),o(1582),function(){var r=u,i=r.lib.StreamCipher,o=r.algo,p=[],g=[],y=[],w=o.RabbitLegacy=i.extend({_doReset:function _doReset(){var r=this._key.words,i=this.cfg.iv,o=this._X=[r[0],r[3]<<16|r[2]>>>16,r[1],r[0]<<16|r[3]>>>16,r[2],r[1]<<16|r[0]>>>16,r[3],r[2]<<16|r[1]>>>16],u=this._C=[r[2]<<16|r[2]>>>16,4294901760&r[0]|65535&r[1],r[3]<<16|r[3]>>>16,4294901760&r[1]|65535&r[2],r[0]<<16|r[0]>>>16,4294901760&r[2]|65535&r[3],r[1]<<16|r[1]>>>16,4294901760&r[3]|65535&r[0]];this._b=0;for(var p=0;p<4;p++)nextState.call(this);for(p=0;p<8;p++)u[p]^=o[p+4&7];if(i){var g=i.words,y=g[0],w=g[1],_=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),x=16711935&(w<<8|w>>>24)|4278255360&(w<<24|w>>>8),k=_>>>16|4294901760&x,P=x<<16|65535&_;for(u[0]^=_,u[1]^=k,u[2]^=x,u[3]^=P,u[4]^=_,u[5]^=k,u[6]^=x,u[7]^=P,p=0;p<4;p++)nextState.call(this)}},_doProcessBlock:function _doProcessBlock(r,i){var o=this._X;nextState.call(this),p[0]=o[0]^o[5]>>>16^o[3]<<16,p[1]=o[2]^o[7]>>>16^o[5]<<16,p[2]=o[4]^o[1]>>>16^o[7]<<16,p[3]=o[6]^o[3]>>>16^o[1]<<16;for(var u=0;u<4;u++)p[u]=16711935&(p[u]<<8|p[u]>>>24)|4278255360&(p[u]<<24|p[u]>>>8),r[i+u]^=p[u]},blockSize:4,ivSize:2});function nextState(){for(var r=this._X,i=this._C,o=0;o<8;o++)g[o]=i[o];for(i[0]=i[0]+1295307597+this._b|0,i[1]=i[1]+3545052371+(i[0]>>>0<g[0]>>>0?1:0)|0,i[2]=i[2]+886263092+(i[1]>>>0<g[1]>>>0?1:0)|0,i[3]=i[3]+1295307597+(i[2]>>>0<g[2]>>>0?1:0)|0,i[4]=i[4]+3545052371+(i[3]>>>0<g[3]>>>0?1:0)|0,i[5]=i[5]+886263092+(i[4]>>>0<g[4]>>>0?1:0)|0,i[6]=i[6]+1295307597+(i[5]>>>0<g[5]>>>0?1:0)|0,i[7]=i[7]+3545052371+(i[6]>>>0<g[6]>>>0?1:0)|0,this._b=i[7]>>>0<g[7]>>>0?1:0,o=0;o<8;o++){var u=r[o]+i[o],p=65535&u,w=u>>>16,_=((p*p>>>17)+p*w>>>15)+w*w,x=((4294901760&u)*u|0)+((65535&u)*u|0);y[o]=_^x}r[0]=y[0]+(y[7]<<16|y[7]>>>16)+(y[6]<<16|y[6]>>>16)|0,r[1]=y[1]+(y[0]<<8|y[0]>>>24)+y[7]|0,r[2]=y[2]+(y[1]<<16|y[1]>>>16)+(y[0]<<16|y[0]>>>16)|0,r[3]=y[3]+(y[2]<<8|y[2]>>>24)+y[1]|0,r[4]=y[4]+(y[3]<<16|y[3]>>>16)+(y[2]<<16|y[2]>>>16)|0,r[5]=y[5]+(y[4]<<8|y[4]>>>24)+y[3]|0,r[6]=y[6]+(y[5]<<16|y[5]>>>16)+(y[4]<<16|y[4]>>>16)|0,r[7]=y[7]+(y[6]<<8|y[6]>>>24)+y[5]|0}r.RabbitLegacy=i._createHelper(w)}(),u.RabbitLegacy)},5323:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(7508),o(3440),o(3839),o(1582),function(){var r=u,i=r.lib.StreamCipher,o=r.algo,p=[],g=[],y=[],w=o.Rabbit=i.extend({_doReset:function _doReset(){for(var r=this._key.words,i=this.cfg.iv,o=0;o<4;o++)r[o]=16711935&(r[o]<<8|r[o]>>>24)|4278255360&(r[o]<<24|r[o]>>>8);var u=this._X=[r[0],r[3]<<16|r[2]>>>16,r[1],r[0]<<16|r[3]>>>16,r[2],r[1]<<16|r[0]>>>16,r[3],r[2]<<16|r[1]>>>16],p=this._C=[r[2]<<16|r[2]>>>16,4294901760&r[0]|65535&r[1],r[3]<<16|r[3]>>>16,4294901760&r[1]|65535&r[2],r[0]<<16|r[0]>>>16,4294901760&r[2]|65535&r[3],r[1]<<16|r[1]>>>16,4294901760&r[3]|65535&r[0]];for(this._b=0,o=0;o<4;o++)nextState.call(this);for(o=0;o<8;o++)p[o]^=u[o+4&7];if(i){var g=i.words,y=g[0],w=g[1],_=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),x=16711935&(w<<8|w>>>24)|4278255360&(w<<24|w>>>8),k=_>>>16|4294901760&x,P=x<<16|65535&_;for(p[0]^=_,p[1]^=k,p[2]^=x,p[3]^=P,p[4]^=_,p[5]^=k,p[6]^=x,p[7]^=P,o=0;o<4;o++)nextState.call(this)}},_doProcessBlock:function _doProcessBlock(r,i){var o=this._X;nextState.call(this),p[0]=o[0]^o[5]>>>16^o[3]<<16,p[1]=o[2]^o[7]>>>16^o[5]<<16,p[2]=o[4]^o[1]>>>16^o[7]<<16,p[3]=o[6]^o[3]>>>16^o[1]<<16;for(var u=0;u<4;u++)p[u]=16711935&(p[u]<<8|p[u]>>>24)|4278255360&(p[u]<<24|p[u]>>>8),r[i+u]^=p[u]},blockSize:4,ivSize:2});function nextState(){for(var r=this._X,i=this._C,o=0;o<8;o++)g[o]=i[o];for(i[0]=i[0]+1295307597+this._b|0,i[1]=i[1]+3545052371+(i[0]>>>0<g[0]>>>0?1:0)|0,i[2]=i[2]+886263092+(i[1]>>>0<g[1]>>>0?1:0)|0,i[3]=i[3]+1295307597+(i[2]>>>0<g[2]>>>0?1:0)|0,i[4]=i[4]+3545052371+(i[3]>>>0<g[3]>>>0?1:0)|0,i[5]=i[5]+886263092+(i[4]>>>0<g[4]>>>0?1:0)|0,i[6]=i[6]+1295307597+(i[5]>>>0<g[5]>>>0?1:0)|0,i[7]=i[7]+3545052371+(i[6]>>>0<g[6]>>>0?1:0)|0,this._b=i[7]>>>0<g[7]>>>0?1:0,o=0;o<8;o++){var u=r[o]+i[o],p=65535&u,w=u>>>16,_=((p*p>>>17)+p*w>>>15)+w*w,x=((4294901760&u)*u|0)+((65535&u)*u|0);y[o]=_^x}r[0]=y[0]+(y[7]<<16|y[7]>>>16)+(y[6]<<16|y[6]>>>16)|0,r[1]=y[1]+(y[0]<<8|y[0]>>>24)+y[7]|0,r[2]=y[2]+(y[1]<<16|y[1]>>>16)+(y[0]<<16|y[0]>>>16)|0,r[3]=y[3]+(y[2]<<8|y[2]>>>24)+y[1]|0,r[4]=y[4]+(y[3]<<16|y[3]>>>16)+(y[2]<<16|y[2]>>>16)|0,r[5]=y[5]+(y[4]<<8|y[4]>>>24)+y[3]|0,r[6]=y[6]+(y[5]<<16|y[5]>>>16)+(y[4]<<16|y[4]>>>16)|0,r[7]=y[7]+(y[6]<<8|y[6]>>>24)+y[5]|0}r.Rabbit=i._createHelper(w)}(),u.Rabbit)},4640:function(r,i,o){"use strict";var u;o(1539),o(8674),r.exports=(u=o(757),o(7508),o(3440),o(3839),o(1582),function(){var r=u,i=r.lib.StreamCipher,o=r.algo,p=o.RC4=i.extend({_doReset:function _doReset(){for(var r=this._key,i=r.words,o=r.sigBytes,u=this._S=[],p=0;p<256;p++)u[p]=p;p=0;for(var g=0;p<256;p++){var y=p%o,w=i[y>>>2]>>>24-y%4*8&255;g=(g+u[p]+w)%256;var _=u[p];u[p]=u[g],u[g]=_}this._i=this._j=0},_doProcessBlock:function _doProcessBlock(r,i){r[i]^=generateKeystreamWord.call(this)},keySize:8,ivSize:0});function generateKeystreamWord(){for(var r=this._S,i=this._i,o=this._j,u=0,p=0;p<4;p++){o=(o+r[i=(i+1)%256])%256;var g=r[i];r[i]=r[o],r[o]=g,u|=r[(r[i]+r[o])%256]<<24-8*p}return this._i=i,this._j=o,u}r.RC4=i._createHelper(p);var g=o.RC4Drop=p.extend({cfg:p.cfg.extend({drop:192}),_doReset:function _doReset(){p._doReset.call(this);for(var r=this.cfg.drop;r>0;r--)generateKeystreamWord.call(this)}});r.RC4Drop=i._createHelper(g)}(),u.RC4)},8714:function(r,i,o){"use strict";var u;r.exports=(u=o(757),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=p.create([0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,7,4,13,1,10,6,15,3,12,0,9,5,2,14,11,8,3,10,14,4,9,15,8,1,2,7,0,6,13,11,5,12,1,9,11,10,0,8,12,4,13,3,7,15,14,5,6,2,4,0,5,9,7,12,2,10,14,1,3,8,11,6,15,13]),_=p.create([5,14,7,0,9,2,11,4,13,6,15,8,1,10,3,12,6,11,3,7,0,13,5,10,14,15,8,12,4,9,1,2,15,5,1,3,7,14,6,9,11,8,12,2,10,0,4,13,8,6,4,1,3,11,15,0,5,12,2,13,9,7,10,14,12,15,10,4,1,5,8,7,6,2,13,14,0,3,9,11]),x=p.create([11,14,15,12,5,8,7,9,11,13,14,15,6,7,9,8,7,6,8,13,11,9,7,15,7,12,15,9,11,7,13,12,11,13,6,7,14,9,13,15,14,8,13,6,5,12,7,5,11,12,14,15,14,15,9,8,9,14,5,6,8,6,5,12,9,15,5,11,6,8,13,12,5,12,13,14,11,8,5,6]),k=p.create([8,9,9,11,13,15,15,5,7,7,8,11,14,14,12,6,9,13,15,7,12,8,9,11,7,7,12,7,6,15,13,11,9,7,15,11,8,6,6,14,12,13,5,14,13,13,7,5,15,5,8,11,14,14,6,14,6,9,12,9,12,5,15,8,8,5,12,9,12,5,14,6,8,13,6,5,15,13,11,11]),P=p.create([0,1518500249,1859775393,2400959708,2840853838]),O=p.create([1352829926,1548603684,1836072691,2053994217,0]),I=y.RIPEMD160=g.extend({_doReset:function _doReset(){this._hash=p.create([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=0;o<16;o++){var u=i+o,p=r[u];r[u]=16711935&(p<<8|p>>>24)|4278255360&(p<<24|p>>>8)}var g,y,I,B,R,D,N,W,U,j,G,X=this._hash.words,K=P.words,Y=O.words,J=w.words,$=_.words,ee=x.words,te=k.words;for(D=g=X[0],N=y=X[1],W=I=X[2],U=B=X[3],j=R=X[4],o=0;o<80;o+=1)G=g+r[i+J[o]]|0,G+=o<16?f1(y,I,B)+K[0]:o<32?f2(y,I,B)+K[1]:o<48?f3(y,I,B)+K[2]:o<64?f4(y,I,B)+K[3]:f5(y,I,B)+K[4],G=(G=rotl(G|=0,ee[o]))+R|0,g=R,R=B,B=rotl(I,10),I=y,y=G,G=D+r[i+$[o]]|0,G+=o<16?f5(N,W,U)+Y[0]:o<32?f4(N,W,U)+Y[1]:o<48?f3(N,W,U)+Y[2]:o<64?f2(N,W,U)+Y[3]:f1(N,W,U)+Y[4],G=(G=rotl(G|=0,te[o]))+j|0,D=j,j=U,U=rotl(W,10),W=N,N=G;G=X[1]+I+U|0,X[1]=X[2]+B+j|0,X[2]=X[3]+R+D|0,X[3]=X[4]+g+N|0,X[4]=X[0]+y+W|0,X[0]=G},_doFinalize:function _doFinalize(){var r=this._data,i=r.words,o=8*this._nDataBytes,u=8*r.sigBytes;i[u>>>5]|=128<<24-u%32,i[14+(u+64>>>9<<4)]=16711935&(o<<8|o>>>24)|4278255360&(o<<24|o>>>8),r.sigBytes=4*(i.length+1),this._process();for(var p=this._hash,g=p.words,y=0;y<5;y++){var w=g[y];g[y]=16711935&(w<<8|w>>>24)|4278255360&(w<<24|w>>>8)}return p},clone:function clone(){var clone=g.clone.call(this);return clone._hash=this._hash.clone(),clone}});function f1(r,i,o){return r^i^o}function f2(r,i,o){return r&i|~r&o}function f3(r,i,o){return(r|~i)^o}function f4(r,i,o){return r&o|i&~o}function f5(r,i,o){return r^(i|~o)}function rotl(r,i){return r<<i|r>>>32-i}i.RIPEMD160=g._createHelper(I),i.HmacRIPEMD160=g._createHmacHelper(I)}(Math),u.RIPEMD160)},9865:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k;r.exports=(k=o(757),p=(u=k).lib,g=p.WordArray,y=p.Hasher,w=u.algo,_=[],x=w.SHA1=y.extend({_doReset:function _doReset(){this._hash=new g.init([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],w=o[4],x=0;x<80;x++){if(x<16)_[x]=0|r[i+x];else{var k=_[x-3]^_[x-8]^_[x-14]^_[x-16];_[x]=k<<1|k>>>31}var P=(u<<5|u>>>27)+w+_[x];P+=x<20?1518500249+(p&g|~p&y):x<40?1859775393+(p^g^y):x<60?(p&g|p&y|g&y)-1894007588:(p^g^y)-899497514,w=y,y=g,g=p<<30|p>>>2,p=u,u=P}o[0]=o[0]+u|0,o[1]=o[1]+p|0,o[2]=o[2]+g|0,o[3]=o[3]+y|0,o[4]=o[4]+w|0},_doFinalize:function _doFinalize(){var r=this._data,i=r.words,o=8*this._nDataBytes,u=8*r.sigBytes;return i[u>>>5]|=128<<24-u%32,i[14+(u+64>>>9<<4)]=Math.floor(o/4294967296),i[15+(u+64>>>9<<4)]=o,r.sigBytes=4*i.length,this._process(),this._hash},clone:function clone(){var clone=y.clone.call(this);return clone._hash=this._hash.clone(),clone}}),u.SHA1=y._createHelper(x),u.HmacSHA1=y._createHmacHelper(x),k.SHA1)},6876:function(r,i,o){"use strict";var u,p,g,y,w,_;r.exports=(_=o(757),o(8921),p=(u=_).lib.WordArray,g=u.algo,y=g.SHA256,w=g.SHA224=y.extend({_doReset:function _doReset(){this._hash=new p.init([3238371032,914150663,812702999,4144912697,4290775857,1750603025,1694076839,3204075428])},_doFinalize:function _doFinalize(){var r=y._doFinalize.call(this);return r.sigBytes-=4,r}}),u.SHA224=y._createHelper(w),u.HmacSHA224=y._createHmacHelper(w),_.SHA224)},8921:function(r,i,o){"use strict";var u;o(7042),r.exports=(u=o(757),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=[],_=[];!function(){function isPrime(i){for(var o=r.sqrt(i),u=2;u<=o;u++)if(!(i%u))return!1;return!0}function getFractionalBits(r){return 4294967296*(r-(0|r))|0}for(var i=2,o=0;o<64;)isPrime(i)&&(o<8&&(w[o]=getFractionalBits(r.pow(i,.5))),_[o]=getFractionalBits(r.pow(i,1/3)),o++),i++}();var x=[],k=y.SHA256=g.extend({_doReset:function _doReset(){this._hash=new p.init(w.slice(0))},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],w=o[4],k=o[5],P=o[6],O=o[7],I=0;I<64;I++){if(I<16)x[I]=0|r[i+I];else{var B=x[I-15],R=(B<<25|B>>>7)^(B<<14|B>>>18)^B>>>3,D=x[I-2],N=(D<<15|D>>>17)^(D<<13|D>>>19)^D>>>10;x[I]=R+x[I-7]+N+x[I-16]}var W=u&p^u&g^p&g,U=(u<<30|u>>>2)^(u<<19|u>>>13)^(u<<10|u>>>22),j=O+((w<<26|w>>>6)^(w<<21|w>>>11)^(w<<7|w>>>25))+(w&k^~w&P)+_[I]+x[I];O=P,P=k,k=w,w=y+j|0,y=g,g=p,p=u,u=j+(U+W)|0}o[0]=o[0]+u|0,o[1]=o[1]+p|0,o[2]=o[2]+g|0,o[3]=o[3]+y|0,o[4]=o[4]+w|0,o[5]=o[5]+k|0,o[6]=o[6]+P|0,o[7]=o[7]+O|0},_doFinalize:function _doFinalize(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;return o[p>>>5]|=128<<24-p%32,o[14+(p+64>>>9<<4)]=r.floor(u/4294967296),o[15+(p+64>>>9<<4)]=u,i.sigBytes=4*o.length,this._process(),this._hash},clone:function clone(){var clone=g.clone.call(this);return clone._hash=this._hash.clone(),clone}});i.SHA256=g._createHelper(k),i.HmacSHA256=g._createHmacHelper(k)}(Math),u.SHA256)},8342:function(r,i,o){"use strict";var u;o(7042),r.exports=(u=o(757),o(2601),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.x64.Word,w=i.algo,_=[],x=[],k=[];!function(){for(var r=1,i=0,o=0;o<24;o++){_[r+5*i]=(o+1)*(o+2)/2%64;var u=(2*r+3*i)%5;r=i%5,i=u}for(r=0;r<5;r++)for(i=0;i<5;i++)x[r+5*i]=i+(2*r+3*i)%5*5;for(var p=1,g=0;g<24;g++){for(var w=0,P=0,O=0;O<7;O++){if(1&p){var I=(1<<O)-1;I<32?P^=1<<I:w^=1<<I-32}128&p?p=p<<1^113:p<<=1}k[g]=y.create(w,P)}}();var P=[];!function(){for(var r=0;r<25;r++)P[r]=y.create()}();var O=w.SHA3=g.extend({cfg:g.cfg.extend({outputLength:512}),_doReset:function _doReset(){for(var r=this._state=[],i=0;i<25;i++)r[i]=new y.init;this.blockSize=(1600-2*this.cfg.outputLength)/32},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._state,u=this.blockSize/2,p=0;p<u;p++){var g=r[i+2*p],y=r[i+2*p+1];g=16711935&(g<<8|g>>>24)|4278255360&(g<<24|g>>>8),y=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),(te=o[p]).high^=y,te.low^=g}for(var w=0;w<24;w++){for(var O=0;O<5;O++){for(var I=0,B=0,R=0;R<5;R++)I^=(te=o[O+5*R]).high,B^=te.low;var D=P[O];D.high=I,D.low=B}for(O=0;O<5;O++){var N=P[(O+4)%5],W=P[(O+1)%5],U=W.high,j=W.low;for(I=N.high^(U<<1|j>>>31),B=N.low^(j<<1|U>>>31),R=0;R<5;R++)(te=o[O+5*R]).high^=I,te.low^=B}for(var G=1;G<25;G++){var X=(te=o[G]).high,K=te.low,Y=_[G];Y<32?(I=X<<Y|K>>>32-Y,B=K<<Y|X>>>32-Y):(I=K<<Y-32|X>>>64-Y,B=X<<Y-32|K>>>64-Y);var J=P[x[G]];J.high=I,J.low=B}var $=P[0],ee=o[0];for($.high=ee.high,$.low=ee.low,O=0;O<5;O++)for(R=0;R<5;R++){var te=o[G=O+5*R],re=P[G],ne=P[(O+1)%5+5*R],ie=P[(O+2)%5+5*R];te.high=re.high^~ne.high&ie.high,te.low=re.low^~ne.low&ie.low}te=o[0];var oe=k[w];te.high^=oe.high,te.low^=oe.low}},_doFinalize:function _doFinalize(){var i=this._data,o=i.words,u=(this._nDataBytes,8*i.sigBytes),g=32*this.blockSize;o[u>>>5]|=1<<24-u%32,o[(r.ceil((u+1)/g)*g>>>5)-1]|=128,i.sigBytes=4*o.length,this._process();for(var y=this._state,w=this.cfg.outputLength/8,_=w/8,x=[],k=0;k<_;k++){var P=y[k],O=P.high,I=P.low;O=16711935&(O<<8|O>>>24)|4278255360&(O<<24|O>>>8),I=16711935&(I<<8|I>>>24)|4278255360&(I<<24|I>>>8),x.push(I),x.push(O)}return new p.init(x,w)},clone:function clone(){for(var clone=g.clone.call(this),r=clone._state=this._state.slice(0),i=0;i<25;i++)r[i]=r[i].clone();return clone}});i.SHA3=g._createHelper(O),i.HmacSHA3=g._createHmacHelper(O)}(Math),u.SHA3)},8122:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k;r.exports=(k=o(757),o(2601),o(7991),p=(u=k).x64,g=p.Word,y=p.WordArray,w=u.algo,_=w.SHA512,x=w.SHA384=_.extend({_doReset:function _doReset(){this._hash=new y.init([new g.init(3418070365,3238371032),new g.init(1654270250,914150663),new g.init(2438529370,812702999),new g.init(355462360,4144912697),new g.init(1731405415,4290775857),new g.init(2394180231,1750603025),new g.init(3675008525,1694076839),new g.init(1203062813,3204075428)])},_doFinalize:function _doFinalize(){var r=_._doFinalize.call(this);return r.sigBytes-=16,r}}),u.SHA384=_._createHelper(x),u.HmacSHA384=_._createHmacHelper(x),k.SHA384)},7991:function(r,i,o){"use strict";var u;r.exports=(u=o(757),o(2601),function(){var r=u,i=r.lib.Hasher,o=r.x64,p=o.Word,g=o.WordArray,y=r.algo;function X64Word_create(){return p.create.apply(p,arguments)}var w=[X64Word_create(1116352408,3609767458),X64Word_create(1899447441,602891725),X64Word_create(3049323471,3964484399),X64Word_create(3921009573,2173295548),X64Word_create(961987163,4081628472),X64Word_create(1508970993,3053834265),X64Word_create(2453635748,2937671579),X64Word_create(2870763221,3664609560),X64Word_create(3624381080,2734883394),X64Word_create(310598401,1164996542),X64Word_create(607225278,1323610764),X64Word_create(1426881987,3590304994),X64Word_create(1925078388,4068182383),X64Word_create(2162078206,991336113),X64Word_create(2614888103,633803317),X64Word_create(3248222580,3479774868),X64Word_create(3835390401,2666613458),X64Word_create(4022224774,944711139),X64Word_create(264347078,2341262773),X64Word_create(604807628,2007800933),X64Word_create(770255983,1495990901),X64Word_create(1249150122,1856431235),X64Word_create(1555081692,3175218132),X64Word_create(1996064986,2198950837),X64Word_create(2554220882,3999719339),X64Word_create(2821834349,766784016),X64Word_create(2952996808,2566594879),X64Word_create(3210313671,3203337956),X64Word_create(3336571891,1034457026),X64Word_create(3584528711,2466948901),X64Word_create(113926993,3758326383),X64Word_create(338241895,168717936),X64Word_create(666307205,1188179964),X64Word_create(773529912,1546045734),X64Word_create(1294757372,1522805485),X64Word_create(1396182291,2643833823),X64Word_create(1695183700,2343527390),X64Word_create(1986661051,1014477480),X64Word_create(2177026350,1206759142),X64Word_create(2456956037,344077627),X64Word_create(2730485921,1290863460),X64Word_create(2820302411,3158454273),X64Word_create(3259730800,3505952657),X64Word_create(3345764771,106217008),X64Word_create(3516065817,3606008344),X64Word_create(3600352804,1432725776),X64Word_create(4094571909,1467031594),X64Word_create(275423344,851169720),X64Word_create(430227734,3100823752),X64Word_create(506948616,1363258195),X64Word_create(659060556,3750685593),X64Word_create(883997877,3785050280),X64Word_create(958139571,3318307427),X64Word_create(1322822218,3812723403),X64Word_create(1537002063,2003034995),X64Word_create(1747873779,3602036899),X64Word_create(1955562222,1575990012),X64Word_create(2024104815,1125592928),X64Word_create(2227730452,2716904306),X64Word_create(2361852424,442776044),X64Word_create(2428436474,593698344),X64Word_create(2756734187,3733110249),X64Word_create(3204031479,2999351573),X64Word_create(3329325298,3815920427),X64Word_create(3391569614,3928383900),X64Word_create(3515267271,566280711),X64Word_create(3940187606,3454069534),X64Word_create(4118630271,4000239992),X64Word_create(116418474,1914138554),X64Word_create(174292421,2731055270),X64Word_create(289380356,3203993006),X64Word_create(460393269,320620315),X64Word_create(685471733,587496836),X64Word_create(852142971,1086792851),X64Word_create(1017036298,365543100),X64Word_create(1126000580,2618297676),X64Word_create(1288033470,3409855158),X64Word_create(1501505948,4234509866),X64Word_create(1607167915,987167468),X64Word_create(1816402316,1246189591)],_=[];!function(){for(var r=0;r<80;r++)_[r]=X64Word_create()}();var x=y.SHA512=i.extend({_doReset:function _doReset(){this._hash=new g.init([new p.init(1779033703,4089235720),new p.init(3144134277,2227873595),new p.init(1013904242,4271175723),new p.init(2773480762,1595750129),new p.init(1359893119,2917565137),new p.init(2600822924,725511199),new p.init(528734635,4215389547),new p.init(1541459225,327033209)])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],x=o[4],k=o[5],P=o[6],O=o[7],I=u.high,B=u.low,R=p.high,D=p.low,N=g.high,W=g.low,U=y.high,j=y.low,G=x.high,X=x.low,K=k.high,Y=k.low,J=P.high,$=P.low,ee=O.high,te=O.low,re=I,ne=B,ie=R,oe=D,ae=N,se=W,ce=U,le=j,ue=G,fe=X,he=K,de=Y,pe=J,ge=$,ye=ee,ve=te,me=0;me<80;me++){var be,we,_e=_[me];if(me<16)we=_e.high=0|r[i+2*me],be=_e.low=0|r[i+2*me+1];else{var Se=_[me-15],xe=Se.high,Ce=Se.low,ke=(xe>>>1|Ce<<31)^(xe>>>8|Ce<<24)^xe>>>7,Pe=(Ce>>>1|xe<<31)^(Ce>>>8|xe<<24)^(Ce>>>7|xe<<25),Ae=_[me-2],Ee=Ae.high,Te=Ae.low,Oe=(Ee>>>19|Te<<13)^(Ee<<3|Te>>>29)^Ee>>>6,Ie=(Te>>>19|Ee<<13)^(Te<<3|Ee>>>29)^(Te>>>6|Ee<<26),Be=_[me-7],Fe=Be.high,Le=Be.low,Re=_[me-16],De=Re.high,Me=Re.low;we=(we=(we=ke+Fe+((be=Pe+Le)>>>0<Pe>>>0?1:0))+Oe+((be+=Ie)>>>0<Ie>>>0?1:0))+De+((be+=Me)>>>0<Me>>>0?1:0),_e.high=we,_e.low=be}var Ne,We=ue&he^~ue&pe,Ue=fe&de^~fe&ge,ze=re&ie^re&ae^ie&ae,je=ne&oe^ne&se^oe&se,Ge=(re>>>28|ne<<4)^(re<<30|ne>>>2)^(re<<25|ne>>>7),Ve=(ne>>>28|re<<4)^(ne<<30|re>>>2)^(ne<<25|re>>>7),He=(ue>>>14|fe<<18)^(ue>>>18|fe<<14)^(ue<<23|fe>>>9),qe=(fe>>>14|ue<<18)^(fe>>>18|ue<<14)^(fe<<23|ue>>>9),Xe=w[me],Ze=Xe.high,Ke=Xe.low,Ye=ye+He+((Ne=ve+qe)>>>0<ve>>>0?1:0),Je=Ve+je;ye=pe,ve=ge,pe=he,ge=de,he=ue,de=fe,ue=ce+(Ye=(Ye=(Ye=Ye+We+((Ne+=Ue)>>>0<Ue>>>0?1:0))+Ze+((Ne+=Ke)>>>0<Ke>>>0?1:0))+we+((Ne+=be)>>>0<be>>>0?1:0))+((fe=le+Ne|0)>>>0<le>>>0?1:0)|0,ce=ae,le=se,ae=ie,se=oe,ie=re,oe=ne,re=Ye+(Ge+ze+(Je>>>0<Ve>>>0?1:0))+((ne=Ne+Je|0)>>>0<Ne>>>0?1:0)|0}B=u.low=B+ne,u.high=I+re+(B>>>0<ne>>>0?1:0),D=p.low=D+oe,p.high=R+ie+(D>>>0<oe>>>0?1:0),W=g.low=W+se,g.high=N+ae+(W>>>0<se>>>0?1:0),j=y.low=j+le,y.high=U+ce+(j>>>0<le>>>0?1:0),X=x.low=X+fe,x.high=G+ue+(X>>>0<fe>>>0?1:0),Y=k.low=Y+de,k.high=K+he+(Y>>>0<de>>>0?1:0),$=P.low=$+ge,P.high=J+pe+($>>>0<ge>>>0?1:0),te=O.low=te+ve,O.high=ee+ye+(te>>>0<ve>>>0?1:0)},_doFinalize:function _doFinalize(){var r=this._data,i=r.words,o=8*this._nDataBytes,u=8*r.sigBytes;return i[u>>>5]|=128<<24-u%32,i[30+(u+128>>>10<<5)]=Math.floor(o/4294967296),i[31+(u+128>>>10<<5)]=o,r.sigBytes=4*i.length,this._process(),this._hash.toX32()},clone:function clone(){var clone=i.clone.call(this);return clone._hash=this._hash.clone(),clone},blockSize:32});r.SHA512=i._createHelper(x),r.HmacSHA512=i._createHmacHelper(x)}(),u.SHA512)},8437:function(r,i,o){"use strict";var u;o(7042),r.exports=(u=o(757),o(7508),o(3440),o(3839),o(1582),function(){var r=u,i=r.lib,o=i.WordArray,p=i.BlockCipher,g=r.algo,y=[57,49,41,33,25,17,9,1,58,50,42,34,26,18,10,2,59,51,43,35,27,19,11,3,60,52,44,36,63,55,47,39,31,23,15,7,62,54,46,38,30,22,14,6,61,53,45,37,29,21,13,5,28,20,12,4],w=[14,17,11,24,1,5,3,28,15,6,21,10,23,19,12,4,26,8,16,7,27,20,13,2,41,52,31,37,47,55,30,40,51,45,33,48,44,49,39,56,34,53,46,42,50,36,29,32],_=[1,2,4,6,8,10,12,14,15,17,19,21,23,25,27,28],x=[{0:8421888,268435456:32768,536870912:8421378,805306368:2,1073741824:512,1342177280:8421890,1610612736:8389122,1879048192:8388608,2147483648:514,2415919104:8389120,2684354560:33280,2952790016:8421376,3221225472:32770,3489660928:8388610,3758096384:0,4026531840:33282,134217728:0,402653184:8421890,671088640:33282,939524096:32768,1207959552:8421888,1476395008:512,1744830464:8421378,2013265920:2,2281701376:8389120,2550136832:33280,2818572288:8421376,3087007744:8389122,3355443200:8388610,3623878656:32770,3892314112:514,4160749568:8388608,1:32768,268435457:2,536870913:8421888,805306369:8388608,1073741825:8421378,1342177281:33280,1610612737:512,1879048193:8389122,2147483649:8421890,2415919105:8421376,2684354561:8388610,2952790017:33282,3221225473:514,3489660929:8389120,3758096385:32770,4026531841:0,134217729:8421890,402653185:8421376,671088641:8388608,939524097:512,1207959553:32768,1476395009:8388610,1744830465:2,2013265921:33282,2281701377:32770,2550136833:8389122,2818572289:514,3087007745:8421888,3355443201:8389120,3623878657:0,3892314113:33280,4160749569:8421378},{0:1074282512,16777216:16384,33554432:524288,50331648:1074266128,67108864:1073741840,83886080:1074282496,100663296:1073758208,117440512:16,134217728:540672,150994944:1073758224,167772160:1073741824,184549376:540688,201326592:524304,218103808:0,234881024:16400,251658240:1074266112,8388608:1073758208,25165824:540688,41943040:16,58720256:1073758224,75497472:1074282512,92274688:1073741824,109051904:524288,125829120:1074266128,142606336:524304,159383552:0,176160768:16384,192937984:1074266112,209715200:1073741840,226492416:540672,243269632:1074282496,260046848:16400,268435456:0,285212672:1074266128,301989888:1073758224,318767104:1074282496,335544320:1074266112,352321536:16,369098752:540688,385875968:16384,402653184:16400,419430400:524288,436207616:524304,452984832:1073741840,469762048:540672,486539264:1073758208,503316480:1073741824,520093696:1074282512,276824064:540688,293601280:524288,310378496:1074266112,327155712:16384,343932928:1073758208,360710144:1074282512,377487360:16,394264576:1073741824,411041792:1074282496,427819008:1073741840,444596224:1073758224,461373440:524304,478150656:0,494927872:16400,511705088:1074266128,528482304:540672},{0:260,1048576:0,2097152:67109120,3145728:65796,4194304:65540,5242880:67108868,6291456:67174660,7340032:67174400,8388608:67108864,9437184:67174656,10485760:65792,11534336:67174404,12582912:67109124,13631488:65536,14680064:4,15728640:256,524288:67174656,1572864:67174404,2621440:0,3670016:67109120,4718592:67108868,5767168:65536,6815744:65540,7864320:260,8912896:4,9961472:256,11010048:67174400,12058624:65796,13107200:65792,14155776:67109124,15204352:67174660,16252928:67108864,16777216:67174656,17825792:65540,18874368:65536,19922944:67109120,20971520:256,22020096:67174660,23068672:67108868,24117248:0,25165824:67109124,26214400:67108864,27262976:4,28311552:65792,29360128:67174400,30408704:260,31457280:65796,32505856:67174404,17301504:67108864,18350080:260,19398656:67174656,20447232:0,21495808:65540,22544384:67109120,23592960:256,24641536:67174404,25690112:65536,26738688:67174660,27787264:65796,28835840:67108868,29884416:67109124,30932992:67174400,31981568:4,33030144:65792},{0:2151682048,65536:2147487808,131072:4198464,196608:2151677952,262144:0,327680:4198400,393216:2147483712,458752:4194368,524288:2147483648,589824:4194304,655360:64,720896:2147487744,786432:2151678016,851968:4160,917504:4096,983040:2151682112,32768:2147487808,98304:64,163840:2151678016,229376:2147487744,294912:4198400,360448:2151682112,425984:0,491520:2151677952,557056:4096,622592:2151682048,688128:4194304,753664:4160,819200:2147483648,884736:4194368,950272:4198464,1015808:2147483712,1048576:4194368,1114112:4198400,1179648:2147483712,1245184:0,1310720:4160,1376256:2151678016,1441792:2151682048,1507328:2147487808,1572864:2151682112,1638400:2147483648,1703936:2151677952,1769472:4198464,1835008:2147487744,1900544:4194304,1966080:64,2031616:4096,1081344:2151677952,1146880:2151682112,1212416:0,1277952:4198400,1343488:4194368,1409024:2147483648,1474560:2147487808,1540096:64,1605632:2147483712,1671168:4096,1736704:2147487744,1802240:2151678016,1867776:4160,1933312:2151682048,1998848:4194304,2064384:4198464},{0:128,4096:17039360,8192:262144,12288:536870912,16384:537133184,20480:16777344,24576:553648256,28672:262272,32768:16777216,36864:537133056,40960:536871040,45056:553910400,49152:553910272,53248:0,57344:17039488,61440:553648128,2048:17039488,6144:553648256,10240:128,14336:17039360,18432:262144,22528:537133184,26624:553910272,30720:536870912,34816:537133056,38912:0,43008:553910400,47104:16777344,51200:536871040,55296:553648128,59392:16777216,63488:262272,65536:262144,69632:128,73728:536870912,77824:553648256,81920:16777344,86016:553910272,90112:537133184,94208:16777216,98304:553910400,102400:553648128,106496:17039360,110592:537133056,114688:262272,118784:536871040,122880:0,126976:17039488,67584:553648256,71680:16777216,75776:17039360,79872:537133184,83968:536870912,88064:17039488,92160:128,96256:553910272,100352:262272,104448:553910400,108544:0,112640:553648128,116736:16777344,120832:262144,124928:537133056,129024:536871040},{0:268435464,256:8192,512:270532608,768:270540808,1024:268443648,1280:2097152,1536:2097160,1792:268435456,2048:0,2304:268443656,2560:2105344,2816:8,3072:270532616,3328:2105352,3584:8200,3840:270540800,128:270532608,384:270540808,640:8,896:2097152,1152:2105352,1408:268435464,1664:268443648,1920:8200,2176:2097160,2432:8192,2688:268443656,2944:270532616,3200:0,3456:270540800,3712:2105344,3968:268435456,4096:268443648,4352:270532616,4608:270540808,4864:8200,5120:2097152,5376:268435456,5632:268435464,5888:2105344,6144:2105352,6400:0,6656:8,6912:270532608,7168:8192,7424:268443656,7680:270540800,7936:2097160,4224:8,4480:2105344,4736:2097152,4992:268435464,5248:268443648,5504:8200,5760:270540808,6016:270532608,6272:270540800,6528:270532616,6784:8192,7040:2105352,7296:2097160,7552:0,7808:268435456,8064:268443656},{0:1048576,16:33555457,32:1024,48:1049601,64:34604033,80:0,96:1,112:34603009,128:33555456,144:1048577,160:33554433,176:34604032,192:34603008,208:1025,224:1049600,240:33554432,8:34603009,24:0,40:33555457,56:34604032,72:1048576,88:33554433,104:33554432,120:1025,136:1049601,152:33555456,168:34603008,184:1048577,200:1024,216:34604033,232:1,248:1049600,256:33554432,272:1048576,288:33555457,304:34603009,320:1048577,336:33555456,352:34604032,368:1049601,384:1025,400:34604033,416:1049600,432:1,448:0,464:34603008,480:33554433,496:1024,264:1049600,280:33555457,296:34603009,312:1,328:33554432,344:1048576,360:1025,376:34604032,392:33554433,408:34603008,424:0,440:34604033,456:1049601,472:1024,488:33555456,504:1048577},{0:134219808,1:131072,2:134217728,3:32,4:131104,5:134350880,6:134350848,7:2048,8:134348800,9:134219776,10:133120,11:134348832,12:2080,13:0,14:134217760,15:133152,2147483648:2048,2147483649:134350880,2147483650:134219808,2147483651:134217728,2147483652:134348800,2147483653:133120,2147483654:133152,2147483655:32,2147483656:134217760,2147483657:2080,2147483658:131104,2147483659:134350848,2147483660:0,2147483661:134348832,2147483662:134219776,2147483663:131072,16:133152,17:134350848,18:32,19:2048,20:134219776,21:134217760,22:134348832,23:131072,24:0,25:131104,26:134348800,27:134219808,28:134350880,29:133120,30:2080,31:134217728,2147483664:131072,2147483665:2048,2147483666:134348832,2147483667:133152,2147483668:32,2147483669:134348800,2147483670:134217728,2147483671:134219808,2147483672:134350880,2147483673:134217760,2147483674:134219776,2147483675:0,2147483676:133120,2147483677:2080,2147483678:131104,2147483679:134350848}],k=[4160749569,528482304,33030144,2064384,129024,8064,504,2147483679],P=g.DES=p.extend({_doReset:function _doReset(){for(var r=this._key.words,i=[],o=0;o<56;o++){var u=y[o]-1;i[o]=r[u>>>5]>>>31-u%32&1}for(var p=this._subKeys=[],g=0;g<16;g++){var x=p[g]=[],k=_[g];for(o=0;o<24;o++)x[o/6|0]|=i[(w[o]-1+k)%28]<<31-o%6,x[4+(o/6|0)]|=i[28+(w[o+24]-1+k)%28]<<31-o%6;for(x[0]=x[0]<<1|x[0]>>>31,o=1;o<7;o++)x[o]=x[o]>>>4*(o-1)+3;x[7]=x[7]<<5|x[7]>>>27}var P=this._invSubKeys=[];for(o=0;o<16;o++)P[o]=p[15-o]},encryptBlock:function encryptBlock(r,i){this._doCryptBlock(r,i,this._subKeys)},decryptBlock:function decryptBlock(r,i){this._doCryptBlock(r,i,this._invSubKeys)},_doCryptBlock:function _doCryptBlock(r,i,o){this._lBlock=r[i],this._rBlock=r[i+1],exchangeLR.call(this,4,252645135),exchangeLR.call(this,16,65535),exchangeRL.call(this,2,858993459),exchangeRL.call(this,8,16711935),exchangeLR.call(this,1,1431655765);for(var u=0;u<16;u++){for(var p=o[u],g=this._lBlock,y=this._rBlock,w=0,_=0;_<8;_++)w|=x[_][((y^p[_])&k[_])>>>0];this._lBlock=y,this._rBlock=g^w}var P=this._lBlock;this._lBlock=this._rBlock,this._rBlock=P,exchangeLR.call(this,1,1431655765),exchangeRL.call(this,8,16711935),exchangeRL.call(this,2,858993459),exchangeLR.call(this,16,65535),exchangeLR.call(this,4,252645135),r[i]=this._lBlock,r[i+1]=this._rBlock},keySize:2,ivSize:2,blockSize:2});function exchangeLR(r,i){var o=(this._lBlock>>>r^this._rBlock)&i;this._rBlock^=o,this._lBlock^=o<<r}function exchangeRL(r,i){var o=(this._rBlock>>>r^this._lBlock)&i;this._lBlock^=o,this._rBlock^=o<<r}r.DES=p._createHelper(P);var O=g.TripleDES=p.extend({_doReset:function _doReset(){var r=this._key.words;if(2!==r.length&&4!==r.length&&r.length<6)throw new Error("Invalid key length - 3DES requires the key length to be 64, 128, 192 or >192.");var i=r.slice(0,2),u=r.length<4?r.slice(0,2):r.slice(2,4),p=r.length<6?r.slice(0,2):r.slice(4,6);this._des1=P.createEncryptor(o.create(i)),this._des2=P.createEncryptor(o.create(u)),this._des3=P.createEncryptor(o.create(p))},encryptBlock:function encryptBlock(r,i){this._des1.encryptBlock(r,i),this._des2.decryptBlock(r,i),this._des3.encryptBlock(r,i)},decryptBlock:function decryptBlock(r,i){this._des3.decryptBlock(r,i),this._des2.encryptBlock(r,i),this._des1.decryptBlock(r,i)},keySize:6,ivSize:2,blockSize:2});r.TripleDES=p._createHelper(O)}(),u.TripleDES)},2601:function(r,i,o){"use strict";var u;o(7042),r.exports=(u=o(757),function(r){var i=u,o=i.lib,p=o.Base,g=o.WordArray,y=i.x64={};y.Word=p.extend({init:function init(r,i){this.high=r,this.low=i}}),y.WordArray=p.extend({init:function init(i,o){i=this.words=i||[],this.sigBytes=o!=r?o:8*i.length},toX32:function toX32(){for(var r=this.words,i=r.length,o=[],u=0;u<i;u++){var p=r[u];o.push(p.high),o.push(p.low)}return g.create(o,this.sigBytes)},clone:function clone(){for(var clone=p.clone.call(this),r=clone.words=this.words.slice(0),i=r.length,o=0;o<i;o++)r[o]=r[o].clone();return clone}})}(),u)},9811:function(r,i,o){"use strict";function _createForOfIteratorHelperLoose(r,i){var o="undefined"!=typeof Symbol&&r[Symbol.iterator]||r["@@iterator"];if(o)return(o=o.call(r)).next.bind(o);if(Array.isArray(r)||(o=function _unsupportedIterableToArray(r,i){if(!r)return;if("string"==typeof r)return _arrayLikeToArray(r,i);var o=Object.prototype.toString.call(r).slice(8,-1);"Object"===o&&r.constructor&&(o=r.constructor.name);if("Map"===o||"Set"===o)return Array.from(r);if("Arguments"===o||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(o))return _arrayLikeToArray(r,i)}(r))||i&&r&&"number"==typeof r.length){o&&(r=o);var u=0;return function(){return u>=r.length?{done:!0}:{done:!1,value:r[u++]}}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}function _arrayLikeToArray(r,i){(null==i||i>r.length)&&(i=r.length);for(var o=0,u=new Array(i);o<i;o++)u[o]=r[o];return u}o(8309),o(1038),o(5666),o(4916),o(4723),o(2165),o(6992),o(1539),o(8783),o(3948),o(2526),o(1817),o(7042);var u=function(){function StateMachine(r){this.stateTable=r.stateTable,this.accepting=r.accepting,this.tags=r.tags}var r=StateMachine.prototype;return r.match=function match(r){var i,o=this;return(i={})[Symbol.iterator]=regeneratorRuntime.mark((function _callee(){var i,u,p,g,y,w;return regeneratorRuntime.wrap((function _callee$(_){for(;;)switch(_.prev=_.next){case 0:i=1,u=null,p=null,g=null,y=0;case 5:if(!(y<r.length)){_.next=21;break}if(w=r[y],g=i,0!==(i=o.stateTable[i][w])){_.next=15;break}if(!(null!=u&&null!=p&&p>=u)){_.next=13;break}return _.next=13,[u,p,o.tags[g]];case 13:i=o.stateTable[1][w],u=null;case 15:0!==i&&null==u&&(u=y),o.accepting[i]&&(p=y),0===i&&(i=1);case 18:y++,_.next=5;break;case 21:if(!(null!=u&&null!=p&&p>=u)){_.next=24;break}return _.next=24,[u,p,o.tags[i]];case 24:case"end":return _.stop()}}),_callee)})),i},r.apply=function apply(r,i){for(var o,u=_createForOfIteratorHelperLoose(this.match(r));!(o=u()).done;)for(var p,g=o.value,y=g[0],w=g[1],_=_createForOfIteratorHelperLoose(g[2]);!(p=_()).done;){var x=p.value;"function"==typeof i[x]&&i[x](y,w,r.slice(y,w+1))}},StateMachine}();r.exports=u},8478:function(r,i,o){"use strict";var u=o(8823).Buffer;o(7042),o(6699);var p=o(3857),g=o(2635);r.exports=function(){function PNG(r){var i;for(this.data=r,this.pos=8,this.palette=[],this.imgData=[],this.transparency={},this.text={};;){var o=this.readUInt32(),p="";for(i=0;i<4;i++)p+=String.fromCharCode(this.data[this.pos++]);switch(p){case"IHDR":this.width=this.readUInt32(),this.height=this.readUInt32(),this.bits=this.data[this.pos++],this.colorType=this.data[this.pos++],this.compressionMethod=this.data[this.pos++],this.filterMethod=this.data[this.pos++],this.interlaceMethod=this.data[this.pos++];break;case"PLTE":this.palette=this.read(o);break;case"IDAT":for(i=0;i<o;i++)this.imgData.push(this.data[this.pos++]);break;case"tRNS":switch(this.transparency={},this.colorType){case 3:this.transparency.indexed=this.read(o);var g=255-this.transparency.indexed.length;if(g>0)for(i=0;i<g;i++)this.transparency.indexed.push(255);break;case 0:this.transparency.grayscale=this.read(o)[0];break;case 2:this.transparency.rgb=this.read(o)}break;case"tEXt":var y=this.read(o),w=y.indexOf(0),_=String.fromCharCode.apply(String,y.slice(0,w));this.text[_]=String.fromCharCode.apply(String,y.slice(w+1));break;case"IEND":switch(this.colorType){case 0:case 3:case 4:this.colors=1;break;case 2:case 6:this.colors=3}this.hasAlphaChannel=[4,6].includes(this.colorType);var x=this.colors+(this.hasAlphaChannel?1:0);switch(this.pixelBitlength=this.bits*x,this.colors){case 1:this.colorSpace="DeviceGray";break;case 3:this.colorSpace="DeviceRGB"}return void(this.imgData=new u(this.imgData));default:this.pos+=o}if(this.pos+=4,this.pos>this.data.length)throw new Error("Incomplete or corrupt PNG file")}}PNG.decode=function decode(r,i){return p.readFile(r,(function(r,o){return new PNG(o).decode((function(r){return i(r)}))}))},PNG.load=function load(r){return new PNG(p.readFileSync(r))};var r=PNG.prototype;return r.read=function read(r){for(var i=new Array(r),o=0;o<r;o++)i[o]=this.data[this.pos++];return i},r.readUInt32=function readUInt32(){return this.data[this.pos++]<<24|this.data[this.pos++]<<16|this.data[this.pos++]<<8|this.data[this.pos++]},r.readUInt16=function readUInt16(){return this.data[this.pos++]<<8|this.data[this.pos++]},r.decodePixels=function decodePixels(r){var i=this;return g.inflate(this.imgData,(function(o,p){if(o)throw o;var g=i.width,y=i.height,w=i.pixelBitlength/8,_=new u(g*y*w),x=p.length,k=0;function pass(r,i,o,P,O){void 0===O&&(O=!1);for(var I=Math.ceil((g-r)/o),B=Math.ceil((y-i)/P),R=w*I,D=O?_:new u(R*B),N=0,W=0;N<B&&k<x;){var U,j,G,X,K;switch(p[k++]){case 0:for(G=0;G<R;G++)D[W++]=p[k++];break;case 1:for(G=0;G<R;G++)U=p[k++],X=G<w?0:D[W-w],D[W++]=(U+X)%256;break;case 2:for(G=0;G<R;G++)U=p[k++],j=(G-G%w)/w,K=N&&D[(N-1)*R+j*w+G%w],D[W++]=(K+U)%256;break;case 3:for(G=0;G<R;G++)U=p[k++],j=(G-G%w)/w,X=G<w?0:D[W-w],K=N&&D[(N-1)*R+j*w+G%w],D[W++]=(U+Math.floor((X+K)/2))%256;break;case 4:for(G=0;G<R;G++){var Y,J;U=p[k++],j=(G-G%w)/w,X=G<w?0:D[W-w],0===N?K=J=0:(K=D[(N-1)*R+j*w+G%w],J=j&&D[(N-1)*R+(j-1)*w+G%w]);var $=X+K-J,ee=Math.abs($-X),te=Math.abs($-K),re=Math.abs($-J);Y=ee<=te&&ee<=re?X:te<=re?K:J,D[W++]=(U+Y)%256}break;default:throw new Error("Invalid filter algorithm: "+p[k-1])}if(!O){var ne=((i+N*P)*g+r)*w,ie=N*R;for(G=0;G<I;G++){for(var oe=0;oe<w;oe++)_[ne++]=D[ie++];ne+=(o-1)*w}}N++}}return 1===i.interlaceMethod?(pass(0,0,8,8),pass(4,0,8,8),pass(0,4,4,8),pass(2,0,4,4),pass(0,2,2,4),pass(1,0,2,2),pass(0,1,1,2)):pass(0,0,1,1,!0),r(_)}))},r.decodePalette=function decodePalette(){for(var r=this.palette,i=r.length,o=this.transparency.indexed||[],p=new u(o.length+i),g=0,y=0,w=0;w<i;w+=3){var _;p[g++]=r[w],p[g++]=r[w+1],p[g++]=r[w+2],p[g++]=null!=(_=o[y++])?_:255}return p},r.copyToImageData=function copyToImageData(r,i){var o,u,p=this.colors,g=null,y=this.hasAlphaChannel;this.palette.length&&(g=this._decodedPalette||(this._decodedPalette=this.decodePalette()),p=4,y=!0);var w=r.data||r,_=w.length,x=g||i,k=o=0;if(1===p)for(;k<_;){u=g?4*i[k/4]:o;var P=x[u++];w[k++]=P,w[k++]=P,w[k++]=P,w[k++]=y?x[u++]:255,o=u}else for(;k<_;)u=g?4*i[k/4]:o,w[k++]=x[u++],w[k++]=x[u++],w[k++]=x[u++],w[k++]=y?x[u++]:255,o=u},r.decode=function decode(r){var i=this,o=new u(this.width*this.height*4);return this.decodePixels((function(u){return i.copyToImageData(o,u),r(o)}))},PNG}()},1750:function(r,i,o){"use strict";o(3290);var u=o(8823),p=u.Buffer;function copyProps(r,i){for(var o in r)i[o]=r[o]}function SafeBuffer(r,i,o){return p(r,i,o)}p.from&&p.alloc&&p.allocUnsafe&&p.allocUnsafeSlow?r.exports=u:(copyProps(u,i),i.Buffer=SafeBuffer),copyProps(p,SafeBuffer),SafeBuffer.from=function(r,i,o){if("number"==typeof r)throw new TypeError("Argument must not be a number");return p(r,i,o)},SafeBuffer.alloc=function(r,i,o){if("number"!=typeof r)throw new TypeError("Argument must be a number");var u=p(r);return void 0!==i?"string"==typeof o?u.fill(i,o):u.fill(i):u.fill(0),u},SafeBuffer.allocUnsafe=function(r){if("number"!=typeof r)throw new TypeError("Argument must be a number");return p(r)},SafeBuffer.allocUnsafeSlow=function(r){if("number"!=typeof r)throw new TypeError("Argument must be a number");return u.SlowBuffer(r)}},7103:function(r,i,o){"use strict";var u=o(4155);o(8145),o(6992),o(1539),o(2472),o(2990),o(8927),o(3105),o(5035),o(4345),o(7174),o(2846),o(4731),o(7209),o(6319),o(8867),o(7789),o(3739),o(9368),o(4483),o(2056),o(3462),o(678),o(7462),o(3824),o(5021),o(2974),o(5016),o(3290);var p,g=o(8823),y=g.Buffer,w={};for(p in g)g.hasOwnProperty(p)&&"SlowBuffer"!==p&&"Buffer"!==p&&(w[p]=g[p]);var _=w.Buffer={};for(p in y)y.hasOwnProperty(p)&&"allocUnsafe"!==p&&"allocUnsafeSlow"!==p&&(_[p]=y[p]);if(w.Buffer.prototype=y.prototype,_.from&&_.from!==Uint8Array.from||(_.from=function(r,i,o){if("number"==typeof r)throw new TypeError('The "value" argument must not be of type number. Received type '+typeof r);if(r&&void 0===r.length)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r);return y(r,i,o)}),_.alloc||(_.alloc=function(r,i,o){if("number"!=typeof r)throw new TypeError('The "size" argument must be of type number. Received type '+typeof r);if(r<0||r>=2*(1<<30))throw new RangeError('The value "'+r+'" is invalid for option "size"');var u=y(r);return i&&0!==i.length?"string"==typeof o?u.fill(i,o):u.fill(i):u.fill(0),u}),!w.kStringMaxLength)try{w.kStringMaxLength=u.binding("buffer").kStringMaxLength}catch(r){}w.constants||(w.constants={MAX_LENGTH:w.kMaxLength},w.kStringMaxLength&&(w.constants.MAX_STRING_LENGTH=w.kStringMaxLength)),r.exports=w},3361:function(r,i,o){"use strict";function ownKeys(r,i){var o=Object.keys(r);if(Object.getOwnPropertySymbols){var u=Object.getOwnPropertySymbols(r);i&&(u=u.filter((function(i){return Object.getOwnPropertyDescriptor(r,i).enumerable}))),o.push.apply(o,u)}return o}function _defineProperty(r,i,o){return i in r?Object.defineProperty(r,i,{value:o,enumerable:!0,configurable:!0,writable:!0}):r[i]=o,r}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,u.key,u)}}o(7941),o(2526),o(7327),o(1539),o(5003),o(4747),o(9337),o(7042);var u=o(8823).Buffer,p=o(9862).inspect,g=p&&p.custom||"inspect";r.exports=function(){function BufferList(){!function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,BufferList),this.head=null,this.tail=null,this.length=0}return function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),r}(BufferList,[{key:"push",value:function push(r){var i={data:r,next:null};this.length>0?this.tail.next=i:this.head=i,this.tail=i,++this.length}},{key:"unshift",value:function unshift(r){var i={data:r,next:this.head};0===this.length&&(this.tail=i),this.head=i,++this.length}},{key:"shift",value:function shift(){if(0!==this.length){var r=this.head.data;return 1===this.length?this.head=this.tail=null:this.head=this.head.next,--this.length,r}}},{key:"clear",value:function clear(){this.head=this.tail=null,this.length=0}},{key:"join",value:function join(r){if(0===this.length)return"";for(var i=this.head,o=""+i.data;i=i.next;)o+=r+i.data;return o}},{key:"concat",value:function concat(r){if(0===this.length)return u.alloc(0);for(var i,o,p,g=u.allocUnsafe(r>>>0),y=this.head,w=0;y;)i=y.data,o=g,p=w,u.prototype.copy.call(i,o,p),w+=y.data.length,y=y.next;return g}},{key:"consume",value:function consume(r,i){var o;return r<this.head.data.length?(o=this.head.data.slice(0,r),this.head.data=this.head.data.slice(r)):o=r===this.head.data.length?this.shift():i?this._getString(r):this._getBuffer(r),o}},{key:"first",value:function first(){return this.head.data}},{key:"_getString",value:function _getString(r){var i=this.head,o=1,u=i.data;for(r-=u.length;i=i.next;){var p=i.data,g=r>p.length?p.length:r;if(g===p.length?u+=p:u+=p.slice(0,r),0===(r-=g)){g===p.length?(++o,i.next?this.head=i.next:this.head=this.tail=null):(this.head=i,i.data=p.slice(g));break}++o}return this.length-=o,u}},{key:"_getBuffer",value:function _getBuffer(r){var i=u.allocUnsafe(r),o=this.head,p=1;for(o.data.copy(i),r-=o.data.length;o=o.next;){var g=o.data,y=r>g.length?g.length:r;if(g.copy(i,i.length-r,0,y),0===(r-=y)){y===g.length?(++p,o.next?this.head=o.next:this.head=this.tail=null):(this.head=o,o.data=g.slice(y));break}++p}return this.length-=p,i}},{key:g,value:function value(r,i){return p(this,function _objectSpread(r){for(var i=1;i<arguments.length;i++){var o=null!=arguments[i]?arguments[i]:{};i%2?ownKeys(Object(o),!0).forEach((function(i){_defineProperty(r,i,o[i])})):Object.getOwnPropertyDescriptors?Object.defineProperties(r,Object.getOwnPropertyDescriptors(o)):ownKeys(Object(o)).forEach((function(i){Object.defineProperty(r,i,Object.getOwnPropertyDescriptor(o,i))}))}return r}({},i,{depth:0,customInspect:!1}))}}]),BufferList}()},1680:function(r,i,o){"use strict";var u=o(8823).Buffer;Object.defineProperty(i,"__esModule",{value:!0}),i.default=void 0,o(7941),o(2526),o(7327),o(1539),o(5003),o(4747),o(9337),o(489),o(2419),o(1817),o(2165),o(6992),o(8783),o(3948),o(1038),o(7042),o(8309),o(4916),o(2707),o(2222),o(9600),o(9714),o(5306),o(1249),o(9841),o(4953),o(6977),o(6699),o(5192),o(9653),o(3123),o(4723),o(8734),o(2472),o(2990),o(8927),o(3105),o(5035),o(4345),o(7174),o(2846),o(4731),o(7209),o(6319),o(8867),o(7789),o(3739),o(9368),o(4483),o(2056),o(3462),o(678),o(7462),o(3824),o(5021),o(2974),o(5016),o(7803),o(3290),o(9601),o(3210),o(9254),o(7397);var p=_interopRequireDefault(o(2830)),g=_interopRequireDefault(o(2635)),y=_interopRequireDefault(o(5153)),w=_interopRequireDefault(o(1917)),_=o(7187),x=_interopRequireDefault(o(7337)),k=_interopRequireDefault(o(8478));function _interopRequireDefault(r){return r&&r.__esModule?r:{default:r}}var P=o(3857);function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,u.key,u)}}function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),r}function _defineProperty(r,i,o){return i in r?Object.defineProperty(r,i,{value:o,enumerable:!0,configurable:!0,writable:!0}):r[i]=o,r}function ownKeys(r,i){var o=Object.keys(r);if(Object.getOwnPropertySymbols){var u=Object.getOwnPropertySymbols(r);i&&(u=u.filter((function(i){return Object.getOwnPropertyDescriptor(r,i).enumerable}))),o.push.apply(o,u)}return o}function _objectSpread2(r){for(var i=1;i<arguments.length;i++){var o=null!=arguments[i]?arguments[i]:{};i%2?ownKeys(Object(o),!0).forEach((function(i){_defineProperty(r,i,o[i])})):Object.getOwnPropertyDescriptors?Object.defineProperties(r,Object.getOwnPropertyDescriptors(o)):ownKeys(Object(o)).forEach((function(i){Object.defineProperty(r,i,Object.getOwnPropertyDescriptor(o,i))}))}return r}function _inherits(r,i){if("function"!=typeof i&&null!==i)throw new TypeError("Super expression must either be null or a function");r.prototype=Object.create(i&&i.prototype,{constructor:{value:r,writable:!0,configurable:!0}}),i&&_setPrototypeOf(r,i)}function _getPrototypeOf(r){return _getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf:function _getPrototypeOf(r){return r.__proto__||Object.getPrototypeOf(r)},_getPrototypeOf(r)}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}function _possibleConstructorReturn(r,i){return!i||"object"!=typeof i&&"function"!=typeof i?_assertThisInitialized(r):i}function _createSuper(r){var i=function _isNativeReflectConstruct(){if("undefined"==typeof Reflect||!Reflect.construct)return!1;if(Reflect.construct.sham)return!1;if("function"==typeof Proxy)return!0;try{return Date.prototype.toString.call(Reflect.construct(Date,[],(function(){}))),!0}catch(r){return!1}}();return function _createSuperInternal(){var o,u=_getPrototypeOf(r);if(i){var p=_getPrototypeOf(this).constructor;o=Reflect.construct(u,arguments,p)}else o=u.apply(this,arguments);return _possibleConstructorReturn(this,o)}}function _slicedToArray(r,i){return function _arrayWithHoles(r){if(Array.isArray(r))return r}(r)||function _iterableToArrayLimit(r,i){if("undefined"==typeof Symbol||!(Symbol.iterator in Object(r)))return;var o=[],u=!0,p=!1,g=void 0;try{for(var y,w=r[Symbol.iterator]();!(u=(y=w.next()).done)&&(o.push(y.value),!i||o.length!==i);u=!0);}catch(r){p=!0,g=r}finally{try{u||null==w.return||w.return()}finally{if(p)throw g}}return o}(r,i)||_unsupportedIterableToArray(r,i)||function _nonIterableRest(){throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}()}function _toConsumableArray(r){return function _arrayWithoutHoles(r){if(Array.isArray(r))return _arrayLikeToArray(r)}(r)||function _iterableToArray(r){if("undefined"!=typeof Symbol&&Symbol.iterator in Object(r))return Array.from(r)}(r)||_unsupportedIterableToArray(r)||function _nonIterableSpread(){throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}()}function _unsupportedIterableToArray(r,i){if(r){if("string"==typeof r)return _arrayLikeToArray(r,i);var o=Object.prototype.toString.call(r).slice(8,-1);return"Object"===o&&r.constructor&&(o=r.constructor.name),"Map"===o||"Set"===o?Array.from(r):"Arguments"===o||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(o)?_arrayLikeToArray(r,i):void 0}}function _arrayLikeToArray(r,i){(null==i||i>r.length)&&(i=r.length);for(var o=0,u=new Array(i);o<i;o++)u[o]=r[o];return u}function _createForOfIteratorHelper(r,i){var o;if("undefined"==typeof Symbol||null==r[Symbol.iterator]){if(Array.isArray(r)||(o=_unsupportedIterableToArray(r))||i&&r&&"number"==typeof r.length){o&&(r=o);var u=0,p=function F(){};return{s:p,n:function n(){return u>=r.length?{done:!0}:{done:!1,value:r[u++]}},e:function e(r){throw r},f:p}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}var g,y=!0,w=!1;return{s:function s(){o=r[Symbol.iterator]()},n:function n(){var r=o.next();return y=r.done,r},e:function e(r){w=!0,g=r},f:function f(){try{y||null==o.return||o.return()}finally{if(w)throw g}}}}var O=function(){function PDFAbstractReference(){_classCallCheck(this,PDFAbstractReference)}return _createClass(PDFAbstractReference,[{key:"toString",value:function toString(){throw new Error("Must be implemented by subclasses")}}]),PDFAbstractReference}(),I=function(){function PDFTree(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{};_classCallCheck(this,PDFTree),this._items={},this.limits="boolean"!=typeof r.limits||r.limits}return _createClass(PDFTree,[{key:"add",value:function add(r,i){return this._items[r]=i}},{key:"get",value:function get(r){return this._items[r]}},{key:"toString",value:function toString(){var r=this,i=Object.keys(this._items).sort((function(i,o){return r._compareKeys(i,o)})),o=["<<"];if(this.limits&&i.length>1){var u=i[0],p=i[i.length-1];o.push("  /Limits ".concat(W.convert([this._dataForKey(u),this._dataForKey(p)])))}o.push("  /".concat(this._keysName()," ["));var g,y=_createForOfIteratorHelper(i);try{for(y.s();!(g=y.n()).done;){var w=g.value;o.push("    ".concat(W.convert(this._dataForKey(w))," ").concat(W.convert(this._items[w])))}}catch(r){y.e(r)}finally{y.f()}return o.push("]"),o.push(">>"),o.join("\n")}},{key:"_compareKeys",value:function _compareKeys(){throw new Error("Must be implemented by subclasses")}},{key:"_keysName",value:function _keysName(){throw new Error("Must be implemented by subclasses")}},{key:"_dataForKey",value:function _dataForKey(){throw new Error("Must be implemented by subclasses")}}]),PDFTree}(),B=function pad(r,i){return(Array(i+1).join("0")+r).slice(-i)},R=/[\n\r\t\b\f()\\]/g,D={"\n":"\\n","\r":"\\r","\t":"\\t","\b":"\\b","\f":"\\f","\\":"\\\\","(":"\\(",")":"\\)"},N=function swapBytes(r){var i=r.length;if(1&i)throw new Error("Buffer length must be even");for(var o=0,u=i-1;o<u;o+=2){var p=r[o];r[o]=r[o+1],r[o+1]=p}return r},W=function(){function PDFObject(){_classCallCheck(this,PDFObject)}return _createClass(PDFObject,null,[{key:"convert",value:function convert(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:null;if("string"==typeof r)return"/".concat(r);if(r instanceof String){for(var o,p=r,g=!1,y=0,w=p.length;y<w;y++)if(p.charCodeAt(y)>127){g=!0;break}return o=g?N(u.from("\ufeff".concat(p),"utf16le")):u.from(p.valueOf(),"ascii"),p=(p=i?i(o).toString("binary"):o.toString("binary")).replace(R,(function(r){return D[r]})),"(".concat(p,")")}if(u.isBuffer(r))return"<".concat(r.toString("hex"),">");if(r instanceof O||r instanceof I)return r.toString();if(r instanceof Date){var _="D:".concat(B(r.getUTCFullYear(),4))+B(r.getUTCMonth()+1,2)+B(r.getUTCDate(),2)+B(r.getUTCHours(),2)+B(r.getUTCMinutes(),2)+B(r.getUTCSeconds(),2)+"Z";return i&&(_=(_=i(u.from(_,"ascii")).toString("binary")).replace(R,(function(r){return D[r]}))),"(".concat(_,")")}if(Array.isArray(r)){var x=r.map((function(r){return PDFObject.convert(r,i)})).join(" ");return"[".concat(x,"]")}if("[object Object]"==={}.toString.call(r)){var k=["<<"];for(var P in r){var W=r[P];k.push("/".concat(P," ").concat(PDFObject.convert(W,i)))}return k.push(">>"),k.join("\n")}return"number"==typeof r?PDFObject.number(r):"".concat(r)}},{key:"number",value:function number(r){if(r>-1e21&&r<1e21)return Math.round(1e6*r)/1e6;throw new Error("unsupported number: ".concat(r))}}]),PDFObject}(),U=function(r){_inherits(PDFReference,r);var i=_createSuper(PDFReference);function PDFReference(r,o){var u,p=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};return _classCallCheck(this,PDFReference),(u=i.call(this)).document=r,u.id=o,u.data=p,u.gen=0,u.compress=u.document.compress&&!u.data.Filter,u.uncompressedLength=0,u.buffer=[],u}return _createClass(PDFReference,[{key:"write",value:function write(r){if(u.isBuffer(r)||(r=u.from(r+"\n","binary")),this.uncompressedLength+=r.length,null==this.data.Length&&(this.data.Length=0),this.buffer.push(r),this.data.Length+=r.length,this.compress)return this.data.Filter="FlateDecode"}},{key:"end",value:function end(r){return r&&this.write(r),this.finalize()}},{key:"finalize",value:function finalize(){this.offset=this.document._offset;var r=this.document._security?this.document._security.getEncryptFn(this.id,this.gen):null;this.buffer.length&&(this.buffer=u.concat(this.buffer),this.compress&&(this.buffer=g.default.deflateSync(this.buffer)),r&&(this.buffer=r(this.buffer)),this.data.Length=this.buffer.length),this.document._write("".concat(this.id," ").concat(this.gen," obj")),this.document._write(W.convert(this.data,r)),this.buffer.length&&(this.document._write("stream"),this.document._write(this.buffer),this.buffer=[],this.document._write("\nendstream")),this.document._write("endobj"),this.document._refEnd(this)}},{key:"toString",value:function toString(){return"".concat(this.id," ").concat(this.gen," R")}}]),PDFReference}(O),j={top:72,left:72,bottom:72,right:72},G={"4A0":[4767.87,6740.79],"2A0":[3370.39,4767.87],A0:[2383.94,3370.39],A1:[1683.78,2383.94],A2:[1190.55,1683.78],A3:[841.89,1190.55],A4:[595.28,841.89],A5:[419.53,595.28],A6:[297.64,419.53],A7:[209.76,297.64],A8:[147.4,209.76],A9:[104.88,147.4],A10:[73.7,104.88],B0:[2834.65,4008.19],B1:[2004.09,2834.65],B2:[1417.32,2004.09],B3:[1000.63,1417.32],B4:[708.66,1000.63],B5:[498.9,708.66],B6:[354.33,498.9],B7:[249.45,354.33],B8:[175.75,249.45],B9:[124.72,175.75],B10:[87.87,124.72],C0:[2599.37,3676.54],C1:[1836.85,2599.37],C2:[1298.27,1836.85],C3:[918.43,1298.27],C4:[649.13,918.43],C5:[459.21,649.13],C6:[323.15,459.21],C7:[229.61,323.15],C8:[161.57,229.61],C9:[113.39,161.57],C10:[79.37,113.39],RA0:[2437.8,3458.27],RA1:[1729.13,2437.8],RA2:[1218.9,1729.13],RA3:[864.57,1218.9],RA4:[609.45,864.57],SRA0:[2551.18,3628.35],SRA1:[1814.17,2551.18],SRA2:[1275.59,1814.17],SRA3:[907.09,1275.59],SRA4:[637.8,907.09],EXECUTIVE:[521.86,756],FOLIO:[612,936],LEGAL:[612,1008],LETTER:[612,792],TABLOID:[792,1224]},X=function(){function PDFPage(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};_classCallCheck(this,PDFPage),this.document=r,this.size=i.size||"letter",this.layout=i.layout||"portrait","number"==typeof i.margin?this.margins={top:i.margin,left:i.margin,bottom:i.margin,right:i.margin}:this.margins=i.margins||j;var o=Array.isArray(this.size)?this.size:G[this.size.toUpperCase()];this.width=o["portrait"===this.layout?0:1],this.height=o["portrait"===this.layout?1:0],this.content=this.document.ref(),this.resources=this.document.ref({ProcSet:["PDF","Text","ImageB","ImageC","ImageI"]}),this.dictionary=this.document.ref({Type:"Page",Parent:this.document._root.data.Pages,MediaBox:[0,0,this.width,this.height],Contents:this.content,Resources:this.resources}),this.markings=[]}return _createClass(PDFPage,[{key:"maxY",value:function maxY(){return this.height-this.margins.bottom}},{key:"write",value:function write(r){return this.content.write(r)}},{key:"end",value:function end(){return this.dictionary.end(),this.resources.end(),this.content.end()}},{key:"fonts",get:function get(){var r=this.resources.data;return null!=r.Font?r.Font:r.Font={}}},{key:"xobjects",get:function get(){var r=this.resources.data;return null!=r.XObject?r.XObject:r.XObject={}}},{key:"ext_gstates",get:function get(){var r=this.resources.data;return null!=r.ExtGState?r.ExtGState:r.ExtGState={}}},{key:"patterns",get:function get(){var r=this.resources.data;return null!=r.Pattern?r.Pattern:r.Pattern={}}},{key:"colorSpaces",get:function get(){var r=this.resources.data;return r.ColorSpace||(r.ColorSpace={})}},{key:"annotations",get:function get(){var r=this.dictionary.data;return null!=r.Annots?r.Annots:r.Annots=[]}},{key:"structParentTreeKey",get:function get(){var r=this.dictionary.data;return null!=r.StructParents?r.StructParents:r.StructParents=this.document.createStructParentTreeNextKey()}}]),PDFPage}(),K=function(r){_inherits(PDFNameTree,r);var i=_createSuper(PDFNameTree);function PDFNameTree(){return _classCallCheck(this,PDFNameTree),i.apply(this,arguments)}return _createClass(PDFNameTree,[{key:"_compareKeys",value:function _compareKeys(r,i){return r.localeCompare(i)}},{key:"_keysName",value:function _keysName(){return"Names"}},{key:"_dataForKey",value:function _dataForKey(r){return new String(r)}}]),PDFNameTree}(I);function inRange(r,i){if(r<i[0])return!1;for(var o=0,u=i.length/2;o<=u;){var p=Math.floor((o+u)/2),g=2*p;if(r>=i[g]&&r<=i[g+1])return!0;r>i[g+1]?o=p+1:u=p-1}return!1}var Y=[545,545,564,591,686,687,751,767,848,863,880,883,886,889,891,893,895,899,907,907,909,909,930,930,975,975,1015,1023,1159,1159,1231,1231,1270,1271,1274,1279,1296,1328,1367,1368,1376,1376,1416,1416,1419,1424,1442,1442,1466,1466,1477,1487,1515,1519,1525,1547,1549,1562,1564,1566,1568,1568,1595,1599,1622,1631,1774,1775,1791,1791,1806,1806,1837,1839,1867,1919,1970,2304,2308,2308,2362,2363,2382,2383,2389,2391,2417,2432,2436,2436,2445,2446,2449,2450,2473,2473,2481,2481,2483,2485,2490,2491,2493,2493,2501,2502,2505,2506,2510,2518,2520,2523,2526,2526,2532,2533,2555,2561,2563,2564,2571,2574,2577,2578,2601,2601,2609,2609,2612,2612,2615,2615,2618,2619,2621,2621,2627,2630,2633,2634,2638,2648,2653,2653,2655,2661,2677,2688,2692,2692,2700,2700,2702,2702,2706,2706,2729,2729,2737,2737,2740,2740,2746,2747,2758,2758,2762,2762,2766,2767,2769,2783,2785,2789,2800,2816,2820,2820,2829,2830,2833,2834,2857,2857,2865,2865,2868,2869,2874,2875,2884,2886,2889,2890,2894,2901,2904,2907,2910,2910,2914,2917,2929,2945,2948,2948,2955,2957,2961,2961,2966,2968,2971,2971,2973,2973,2976,2978,2981,2983,2987,2989,2998,2998,3002,3005,3011,3013,3017,3017,3022,3030,3032,3046,3059,3072,3076,3076,3085,3085,3089,3089,3113,3113,3124,3124,3130,3133,3141,3141,3145,3145,3150,3156,3159,3167,3170,3173,3184,3201,3204,3204,3213,3213,3217,3217,3241,3241,3252,3252,3258,3261,3269,3269,3273,3273,3278,3284,3287,3293,3295,3295,3298,3301,3312,3329,3332,3332,3341,3341,3345,3345,3369,3369,3386,3389,3396,3397,3401,3401,3406,3414,3416,3423,3426,3429,3440,3457,3460,3460,3479,3481,3506,3506,3516,3516,3518,3519,3527,3529,3531,3534,3541,3541,3543,3543,3552,3569,3573,3584,3643,3646,3676,3712,3715,3715,3717,3718,3721,3721,3723,3724,3726,3731,3736,3736,3744,3744,3748,3748,3750,3750,3752,3753,3756,3756,3770,3770,3774,3775,3781,3781,3783,3783,3790,3791,3802,3803,3806,3839,3912,3912,3947,3952,3980,3983,3992,3992,4029,4029,4045,4046,4048,4095,4130,4130,4136,4136,4139,4139,4147,4149,4154,4159,4186,4255,4294,4303,4345,4346,4348,4351,4442,4446,4515,4519,4602,4607,4615,4615,4679,4679,4681,4681,4686,4687,4695,4695,4697,4697,4702,4703,4743,4743,4745,4745,4750,4751,4783,4783,4785,4785,4790,4791,4799,4799,4801,4801,4806,4807,4815,4815,4823,4823,4847,4847,4879,4879,4881,4881,4886,4887,4895,4895,4935,4935,4955,4960,4989,5023,5109,5120,5751,5759,5789,5791,5873,5887,5901,5901,5909,5919,5943,5951,5972,5983,5997,5997,6001,6001,6004,6015,6109,6111,6122,6143,6159,6159,6170,6175,6264,6271,6314,7679,7836,7839,7930,7935,7958,7959,7966,7967,8006,8007,8014,8015,8024,8024,8026,8026,8028,8028,8030,8030,8062,8063,8117,8117,8133,8133,8148,8149,8156,8156,8176,8177,8181,8181,8191,8191,8275,8278,8280,8286,8292,8297,8306,8307,8335,8351,8370,8399,8427,8447,8507,8508,8524,8530,8580,8591,9167,9215,9255,9279,9291,9311,9471,9471,9748,9749,9752,9752,9854,9855,9866,9984,9989,9989,9994,9995,10024,10024,10060,10060,10062,10062,10067,10069,10071,10071,10079,10080,10133,10135,10160,10160,10175,10191,10220,10223,11008,11903,11930,11930,12020,12031,12246,12271,12284,12287,12352,12352,12439,12440,12544,12548,12589,12592,12687,12687,12728,12783,12829,12831,12868,12880,12924,12926,13004,13007,13055,13055,13175,13178,13278,13279,13311,13311,19894,19967,40870,40959,42125,42127,42183,44031,55204,55295,64046,64047,64107,64255,64263,64274,64280,64284,64311,64311,64317,64317,64319,64319,64322,64322,64325,64325,64434,64466,64832,64847,64912,64913,64968,64975,65021,65023,65040,65055,65060,65071,65095,65096,65107,65107,65127,65127,65132,65135,65141,65141,65277,65278,65280,65280,65471,65473,65480,65481,65488,65489,65496,65497,65501,65503,65511,65511,65519,65528,65536,66303,66335,66335,66340,66351,66379,66559,66598,66599,66638,118783,119030,119039,119079,119081,119262,119807,119893,119893,119965,119965,119968,119969,119971,119972,119975,119976,119981,119981,119994,119994,119996,119996,120001,120001,120004,120004,120070,120070,120075,120076,120085,120085,120093,120093,120122,120122,120127,120127,120133,120133,120135,120137,120145,120145,120484,120487,120778,120781,120832,131069,173783,194559,195102,196605,196608,262141,262144,327677,327680,393213,393216,458749,458752,524285,524288,589821,589824,655357,655360,720893,720896,786429,786432,851965,851968,917501,917504,917504,917506,917535,917632,983037],J=function isUnassignedCodePoint(r){return inRange(r,Y)},$=[173,173,847,847,6150,6150,6155,6155,6156,6156,6157,6157,8203,8203,8204,8204,8205,8205,8288,8288,65024,65024,65025,65025,65026,65026,65027,65027,65028,65028,65029,65029,65030,65030,65031,65031,65032,65032,65033,65033,65034,65034,65035,65035,65036,65036,65037,65037,65038,65038,65039,65039,65279,65279],ee=[160,160,5760,5760,8192,8192,8193,8193,8194,8194,8195,8195,8196,8196,8197,8197,8198,8198,8199,8199,8200,8200,8201,8201,8202,8202,8203,8203,8239,8239,8287,8287,12288,12288],te=[128,159,1757,1757,1807,1807,6158,6158,8204,8204,8205,8205,8232,8232,8233,8233,8288,8288,8289,8289,8290,8290,8291,8291,8298,8303,65279,65279,65529,65532,119155,119162],re=[64976,65007,65534,65535,131070,131071,196606,196607,262142,262143,327678,327679,393214,393215,458750,458751,524286,524287,589822,589823,655358,655359,720894,720895,786430,786431,851966,851967,917502,917503,983038,983039,1114110,1114111],ne=[0,31,127,127,832,832,833,833,8206,8206,8207,8207,8234,8234,8235,8235,8236,8236,8237,8237,8238,8238,8298,8298,8299,8299,8300,8300,8301,8301,8302,8302,8303,8303,12272,12283,55296,57343,57344,63743,65529,65529,65530,65530,65531,65531,65532,65532,65533,65533,917505,917505,917536,917631,983040,1048573,1048576,1114109],ie=function isProhibitedCharacter(r){return inRange(r,ee)||inRange(r,ne)||inRange(r,te)||inRange(r,re)},oe=[1470,1470,1472,1472,1475,1475,1488,1514,1520,1524,1563,1563,1567,1567,1569,1594,1600,1610,1645,1647,1649,1749,1757,1757,1765,1766,1786,1790,1792,1805,1808,1808,1810,1836,1920,1957,1969,1969,8207,8207,64285,64285,64287,64296,64298,64310,64312,64316,64318,64318,64320,64321,64323,64324,64326,64433,64467,64829,64848,64911,64914,64967,65008,65020,65136,65140,65142,65276],ae=function isBidirectionalRAL(r){return inRange(r,oe)},se=[65,90,97,122,170,170,181,181,186,186,192,214,216,246,248,544,546,563,592,685,688,696,699,705,720,721,736,740,750,750,890,890,902,902,904,906,908,908,910,929,931,974,976,1013,1024,1154,1162,1230,1232,1269,1272,1273,1280,1295,1329,1366,1369,1375,1377,1415,1417,1417,2307,2307,2309,2361,2365,2368,2377,2380,2384,2384,2392,2401,2404,2416,2434,2435,2437,2444,2447,2448,2451,2472,2474,2480,2482,2482,2486,2489,2494,2496,2503,2504,2507,2508,2519,2519,2524,2525,2527,2529,2534,2545,2548,2554,2565,2570,2575,2576,2579,2600,2602,2608,2610,2611,2613,2614,2616,2617,2622,2624,2649,2652,2654,2654,2662,2671,2674,2676,2691,2691,2693,2699,2701,2701,2703,2705,2707,2728,2730,2736,2738,2739,2741,2745,2749,2752,2761,2761,2763,2764,2768,2768,2784,2784,2790,2799,2818,2819,2821,2828,2831,2832,2835,2856,2858,2864,2866,2867,2870,2873,2877,2878,2880,2880,2887,2888,2891,2892,2903,2903,2908,2909,2911,2913,2918,2928,2947,2947,2949,2954,2958,2960,2962,2965,2969,2970,2972,2972,2974,2975,2979,2980,2984,2986,2990,2997,2999,3001,3006,3007,3009,3010,3014,3016,3018,3020,3031,3031,3047,3058,3073,3075,3077,3084,3086,3088,3090,3112,3114,3123,3125,3129,3137,3140,3168,3169,3174,3183,3202,3203,3205,3212,3214,3216,3218,3240,3242,3251,3253,3257,3262,3262,3264,3268,3271,3272,3274,3275,3285,3286,3294,3294,3296,3297,3302,3311,3330,3331,3333,3340,3342,3344,3346,3368,3370,3385,3390,3392,3398,3400,3402,3404,3415,3415,3424,3425,3430,3439,3458,3459,3461,3478,3482,3505,3507,3515,3517,3517,3520,3526,3535,3537,3544,3551,3570,3572,3585,3632,3634,3635,3648,3654,3663,3675,3713,3714,3716,3716,3719,3720,3722,3722,3725,3725,3732,3735,3737,3743,3745,3747,3749,3749,3751,3751,3754,3755,3757,3760,3762,3763,3773,3773,3776,3780,3782,3782,3792,3801,3804,3805,3840,3863,3866,3892,3894,3894,3896,3896,3902,3911,3913,3946,3967,3967,3973,3973,3976,3979,4030,4037,4039,4044,4047,4047,4096,4129,4131,4135,4137,4138,4140,4140,4145,4145,4152,4152,4160,4183,4256,4293,4304,4344,4347,4347,4352,4441,4447,4514,4520,4601,4608,4614,4616,4678,4680,4680,4682,4685,4688,4694,4696,4696,4698,4701,4704,4742,4744,4744,4746,4749,4752,4782,4784,4784,4786,4789,4792,4798,4800,4800,4802,4805,4808,4814,4816,4822,4824,4846,4848,4878,4880,4880,4882,4885,4888,4894,4896,4934,4936,4954,4961,4988,5024,5108,5121,5750,5761,5786,5792,5872,5888,5900,5902,5905,5920,5937,5941,5942,5952,5969,5984,5996,5998,6e3,6016,6070,6078,6085,6087,6088,6100,6106,6108,6108,6112,6121,6160,6169,6176,6263,6272,6312,7680,7835,7840,7929,7936,7957,7960,7965,7968,8005,8008,8013,8016,8023,8025,8025,8027,8027,8029,8029,8031,8061,8064,8116,8118,8124,8126,8126,8130,8132,8134,8140,8144,8147,8150,8155,8160,8172,8178,8180,8182,8188,8206,8206,8305,8305,8319,8319,8450,8450,8455,8455,8458,8467,8469,8469,8473,8477,8484,8484,8486,8486,8488,8488,8490,8493,8495,8497,8499,8505,8509,8511,8517,8521,8544,8579,9014,9082,9109,9109,9372,9449,12293,12295,12321,12329,12337,12341,12344,12348,12353,12438,12445,12447,12449,12538,12540,12543,12549,12588,12593,12686,12688,12727,12784,12828,12832,12867,12896,12923,12927,12976,12992,13003,13008,13054,13056,13174,13179,13277,13280,13310,13312,19893,19968,40869,40960,42124,44032,55203,55296,64045,64048,64106,64256,64262,64275,64279,65313,65338,65345,65370,65382,65470,65474,65479,65482,65487,65490,65495,65498,65500,66304,66334,66336,66339,66352,66378,66560,66597,66600,66637,118784,119029,119040,119078,119082,119142,119146,119154,119171,119172,119180,119209,119214,119261,119808,119892,119894,119964,119966,119967,119970,119970,119973,119974,119977,119980,119982,119993,119995,119995,119997,12e4,120002,120003,120005,120069,120071,120074,120077,120084,120086,120092,120094,120121,120123,120126,120128,120132,120134,120134,120138,120144,120146,120483,120488,120777,131072,173782,194560,195101,983040,1048573,1048576,1114109],ce=function isBidirectionalL(r){return inRange(r,se)},le=function isNonASCIISpaceCharacter(r){return inRange(r,ee)},ue=function isCommonlyMappedToNothing(r){return inRange(r,$)},fe=function getCodePoint(r){return r.codePointAt(0)},he=function first(r){return r[0]},de=function last(r){return r[r.length-1]};function toCodePoints(r){for(var i=[],o=r.length,u=0;u<o;u+=1){var p=r.charCodeAt(u);if(p>=55296&&p<=56319&&o>u+1){var g=r.charCodeAt(u+1);if(g>=56320&&g<=57343){i.push(1024*(p-55296)+g-56320+65536),u+=1;continue}}i.push(p)}return i}function saslprep(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if("string"!=typeof r)throw new TypeError("Expected string.");if(0===r.length)return"";var o=toCodePoints(r).map((function(r){return le(r)?32:r})).filter((function(r){return!ue(r)})),u=String.fromCodePoint.apply(null,o).normalize("NFKC"),p=toCodePoints(u),g=p.some(ie);if(g)throw new Error("Prohibited character, see https://tools.ietf.org/html/rfc4013#section-2.3");if(!0!==i.allowUnassigned){var y=p.some(J);if(y)throw new Error("Unassigned code point, see https://tools.ietf.org/html/rfc4013#section-2.5")}var w=p.some(ae),_=p.some(ce);if(w&&_)throw new Error("String must not contain RandALCat and LCat at the same time, see https://tools.ietf.org/html/rfc3454#section-6");var x=ae(fe(he(u))),k=ae(fe(de(u)));if(w&&(!x||!k))throw new Error("Bidirectional RandALCat character must be the first and the last character of the string, see https://tools.ietf.org/html/rfc3454#section-6");return u}var pe=function(){function PDFSecurity(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if(_classCallCheck(this,PDFSecurity),!i.ownerPassword&&!i.userPassword)throw new Error("None of owner password and user password is defined.");this.document=r,this._setupEncryption(i)}return _createClass(PDFSecurity,null,[{key:"generateFileID",value:function generateFileID(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i="".concat(r.CreationDate.getTime(),"\n");for(var o in r)r.hasOwnProperty(o)&&(i+="".concat(o,": ").concat(r[o].valueOf(),"\n"));return wordArrayToBuffer(y.default.MD5(i))}},{key:"generateRandomWordArray",value:function generateRandomWordArray(r){return y.default.lib.WordArray.random(r)}},{key:"create",value:function create(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return i.ownerPassword||i.userPassword?new PDFSecurity(r,i):null}}]),_createClass(PDFSecurity,[{key:"_setupEncryption",value:function _setupEncryption(r){switch(r.pdfVersion){case"1.4":case"1.5":this.version=2;break;case"1.6":case"1.7":this.version=4;break;case"1.7ext3":this.version=5;break;default:this.version=1}var i={Filter:"Standard"};switch(this.version){case 1:case 2:case 4:this._setupEncryptionV1V2V4(this.version,i,r);break;case 5:this._setupEncryptionV5(i,r)}this.dictionary=this.document.ref(i)}},{key:"_setupEncryptionV1V2V4",value:function _setupEncryptionV1V2V4(r,i,o){var u,p;switch(r){case 1:u=2,this.keyBits=40,p=function getPermissionsR2(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=-64;r.printing&&(i|=4);r.modifying&&(i|=8);r.copying&&(i|=16);r.annotating&&(i|=32);return i}(o.permissions);break;case 2:u=3,this.keyBits=128,p=getPermissionsR3(o.permissions);break;case 4:u=4,this.keyBits=128,p=getPermissionsR3(o.permissions)}var g,w=processPasswordR2R3R4(o.userPassword),_=o.ownerPassword?processPasswordR2R3R4(o.ownerPassword):w,x=function getOwnerPasswordR2R3R4(r,i,o,u){for(var p=u,g=r>=3?51:1,w=0;w<g;w++)p=y.default.MD5(p);var _=p.clone();_.sigBytes=i/8;var x=o;g=r>=3?20:1;for(var k=0;k<g;k++){for(var P=Math.ceil(_.sigBytes/4),O=0;O<P;O++)_.words[O]=p.words[O]^(k|k<<8|k<<16|k<<24);x=y.default.RC4.encrypt(x,_).ciphertext}return x}(u,this.keyBits,w,_);this.encryptionKey=function getEncryptionKeyR2R3R4(r,i,o,u,p,g){for(var w=u.clone().concat(p).concat(y.default.lib.WordArray.create([lsbFirstWord(g)],4)).concat(y.default.lib.WordArray.create(o)),_=r>=3?51:1,x=0;x<_;x++)(w=y.default.MD5(w)).sigBytes=i/8;return w}(u,this.keyBits,this.document._id,w,x,p),g=2===u?function getUserPasswordR2(r){return y.default.RC4.encrypt(processPasswordR2R3R4(),r).ciphertext}(this.encryptionKey):function getUserPasswordR3R4(r,i){for(var o=i.clone(),u=y.default.MD5(processPasswordR2R3R4().concat(y.default.lib.WordArray.create(r))),p=0;p<20;p++){for(var g=Math.ceil(o.sigBytes/4),w=0;w<g;w++)o.words[w]=i.words[w]^(p|p<<8|p<<16|p<<24);u=y.default.RC4.encrypt(u,o).ciphertext}return u.concat(y.default.lib.WordArray.create(null,16))}(this.document._id,this.encryptionKey),i.V=r,r>=2&&(i.Length=this.keyBits),4===r&&(i.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV2",Length:this.keyBits/8}},i.StmF="StdCF",i.StrF="StdCF"),i.R=u,i.O=wordArrayToBuffer(x),i.U=wordArrayToBuffer(g),i.P=p}},{key:"_setupEncryptionV5",value:function _setupEncryptionV5(r,i){this.keyBits=256;var o=getPermissionsR3(i.permissions),u=processPasswordR5(i.userPassword),p=i.ownerPassword?processPasswordR5(i.ownerPassword):u;this.encryptionKey=function getEncryptionKeyR5(r){return r(32)}(PDFSecurity.generateRandomWordArray);var g=function getUserPasswordR5(r,i){var o=i(8),u=i(8);return y.default.SHA256(r.clone().concat(o)).concat(o).concat(u)}(u,PDFSecurity.generateRandomWordArray),w=function getUserEncryptionKeyR5(r,i,o){var u=y.default.SHA256(r.clone().concat(i)),p={mode:y.default.mode.CBC,padding:y.default.pad.NoPadding,iv:y.default.lib.WordArray.create(null,16)};return y.default.AES.encrypt(o,u,p).ciphertext}(u,y.default.lib.WordArray.create(g.words.slice(10,12),8),this.encryptionKey),_=function getOwnerPasswordR5(r,i,o){var u=o(8),p=o(8);return y.default.SHA256(r.clone().concat(u).concat(i)).concat(u).concat(p)}(p,g,PDFSecurity.generateRandomWordArray),x=function getOwnerEncryptionKeyR5(r,i,o,u){var p=y.default.SHA256(r.clone().concat(i).concat(o)),g={mode:y.default.mode.CBC,padding:y.default.pad.NoPadding,iv:y.default.lib.WordArray.create(null,16)};return y.default.AES.encrypt(u,p,g).ciphertext}(p,y.default.lib.WordArray.create(_.words.slice(10,12),8),g,this.encryptionKey),k=function getEncryptedPermissionsR5(r,i,o){var u=y.default.lib.WordArray.create([lsbFirstWord(r),4294967295,1415668834],12).concat(o(4)),p={mode:y.default.mode.ECB,padding:y.default.pad.NoPadding};return y.default.AES.encrypt(u,i,p).ciphertext}(o,this.encryptionKey,PDFSecurity.generateRandomWordArray);r.V=5,r.Length=this.keyBits,r.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV3",Length:this.keyBits/8}},r.StmF="StdCF",r.StrF="StdCF",r.R=5,r.O=wordArrayToBuffer(_),r.OE=wordArrayToBuffer(x),r.U=wordArrayToBuffer(g),r.UE=wordArrayToBuffer(w),r.P=o,r.Perms=wordArrayToBuffer(k)}},{key:"getEncryptFn",value:function getEncryptFn(r,i){var o,u;if(this.version<5&&(o=this.encryptionKey.clone().concat(y.default.lib.WordArray.create([(255&r)<<24|(65280&r)<<8|r>>8&65280|255&i,(65280&i)<<16],5))),1===this.version||2===this.version){var p=y.default.MD5(o);return p.sigBytes=Math.min(16,this.keyBits/8+5),function(r){return wordArrayToBuffer(y.default.RC4.encrypt(y.default.lib.WordArray.create(r),p).ciphertext)}}u=4===this.version?y.default.MD5(o.concat(y.default.lib.WordArray.create([1933667412],4))):this.encryptionKey;var g=PDFSecurity.generateRandomWordArray(16),w={mode:y.default.mode.CBC,padding:y.default.pad.Pkcs7,iv:g};return function(r){return wordArrayToBuffer(g.clone().concat(y.default.AES.encrypt(y.default.lib.WordArray.create(r),u,w).ciphertext))}}},{key:"end",value:function end(){this.dictionary.end()}}]),PDFSecurity}();function getPermissionsR3(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=-3904;return"lowResolution"===r.printing&&(i|=4),"highResolution"===r.printing&&(i|=2052),r.modifying&&(i|=8),r.copying&&(i|=16),r.annotating&&(i|=32),r.fillingForms&&(i|=256),r.contentAccessibility&&(i|=512),r.documentAssembly&&(i|=1024),i}function processPasswordR2R3R4(){for(var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"",i=u.alloc(32),o=r.length,p=0;p<o&&p<32;){var g=r.charCodeAt(p);if(g>255)throw new Error("Password contains one or more invalid characters.");i[p]=g,p++}for(;p<32;)i[p]=_e[p-o],p++;return y.default.lib.WordArray.create(i)}function processPasswordR5(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"";r=unescape(encodeURIComponent(saslprep(r)));for(var i=Math.min(127,r.length),o=u.alloc(i),p=0;p<i;p++)o[p]=r.charCodeAt(p);return y.default.lib.WordArray.create(o)}function lsbFirstWord(r){return(255&r)<<24|(65280&r)<<8|r>>8&65280|r>>24&255}function wordArrayToBuffer(r){for(var i=[],o=0;o<r.sigBytes;o++)i.push(r.words[Math.floor(o/4)]>>8*(3-o%4)&255);return u.from(i)}var ge,ye,ve,me,be,we,_e=[40,191,78,94,78,117,138,65,100,0,78,86,255,250,1,8,46,46,0,182,208,104,62,128,47,12,169,254,100,83,105,122],Se=W.number,xe=function(){function PDFGradient(r){_classCallCheck(this,PDFGradient),this.doc=r,this.stops=[],this.embedded=!1,this.transform=[1,0,0,1,0,0]}return _createClass(PDFGradient,[{key:"stop",value:function stop(r,i,o){if(null==o&&(o=1),i=this.doc._normalizeColor(i),0===this.stops.length)if(3===i.length)this._colorSpace="DeviceRGB";else if(4===i.length)this._colorSpace="DeviceCMYK";else{if(1!==i.length)throw new Error("Unknown color space");this._colorSpace="DeviceGray"}else if("DeviceRGB"===this._colorSpace&&3!==i.length||"DeviceCMYK"===this._colorSpace&&4!==i.length||"DeviceGray"===this._colorSpace&&1!==i.length)throw new Error("All gradient stops must use the same color space");return o=Math.max(0,Math.min(1,o)),this.stops.push([r,i,o]),this}},{key:"setTransform",value:function setTransform(r,i,o,u,p,g){return this.transform=[r,i,o,u,p,g],this}},{key:"embed",value:function embed(r){var i,o=this.stops.length;if(0!==o){this.embedded=!0,this.matrix=r;var u=this.stops[o-1];u[0]<1&&this.stops.push([1,u[1],u[2]]);for(var p=[],g=[],y=[],w=0;w<o-1;w++)g.push(0,1),w+2!==o&&p.push(this.stops[w+1][0]),i=this.doc.ref({FunctionType:2,Domain:[0,1],C0:this.stops[w+0][1],C1:this.stops[w+1][1],N:1}),y.push(i),i.end();1===o?i=y[0]:(i=this.doc.ref({FunctionType:3,Domain:[0,1],Functions:y,Bounds:p,Encode:g})).end(),this.id="Sh".concat(++this.doc._gradCount);var _=this.shader(i);_.end();var x=this.doc.ref({Type:"Pattern",PatternType:2,Shading:_,Matrix:this.matrix.map(Se)});if(x.end(),this.stops.some((function(r){return r[2]<1}))){var k=this.opacityGradient();k._colorSpace="DeviceGray";var P,O=_createForOfIteratorHelper(this.stops);try{for(O.s();!(P=O.n()).done;){var I=P.value;k.stop(I[0],[I[2]])}}catch(r){O.e(r)}finally{O.f()}k=k.embed(this.matrix);var B=[0,0,this.doc.page.width,this.doc.page.height],R=this.doc.ref({Type:"XObject",Subtype:"Form",FormType:1,BBox:B,Group:{Type:"Group",S:"Transparency",CS:"DeviceGray"},Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:k}}});R.write("/Pattern cs /Sh1 scn"),R.end("".concat(B.join(" ")," re f"));var D=this.doc.ref({Type:"ExtGState",SMask:{Type:"Mask",S:"Luminosity",G:R}});D.end();var N=this.doc.ref({Type:"Pattern",PatternType:1,PaintType:1,TilingType:2,BBox:B,XStep:B[2],YStep:B[3],Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:x},ExtGState:{Gs1:D}}});N.write("/Gs1 gs /Pattern cs /Sh1 scn"),N.end("".concat(B.join(" ")," re f")),this.doc.page.patterns[this.id]=N}else this.doc.page.patterns[this.id]=x;return x}}},{key:"apply",value:function apply(r){var i=_slicedToArray(this.doc._ctm,6),o=i[0],u=i[1],p=i[2],g=i[3],y=i[4],w=i[5],_=_slicedToArray(this.transform,6),x=_[0],k=_[1],P=_[2],O=_[3],I=_[4],B=_[5],R=[o*x+p*k,u*x+g*k,o*P+p*O,u*P+g*O,o*I+p*B+y,u*I+g*B+w];this.embedded&&R.join(" ")===this.matrix.join(" ")||this.embed(R),this.doc._setColorSpace("Pattern",r);var D=r?"SCN":"scn";return this.doc.addContent("/".concat(this.id," ").concat(D))}}]),PDFGradient}(),Ce={PDFGradient:xe,PDFLinearGradient:function(r){_inherits(PDFLinearGradient,r);var i=_createSuper(PDFLinearGradient);function PDFLinearGradient(r,o,u,p,g){var y;return _classCallCheck(this,PDFLinearGradient),(y=i.call(this,r)).x1=o,y.y1=u,y.x2=p,y.y2=g,y}return _createClass(PDFLinearGradient,[{key:"shader",value:function shader(r){return this.doc.ref({ShadingType:2,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.x2,this.y2],Function:r,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFLinearGradient(this.doc,this.x1,this.y1,this.x2,this.y2)}}]),PDFLinearGradient}(xe),PDFRadialGradient:function(r){_inherits(PDFRadialGradient,r);var i=_createSuper(PDFRadialGradient);function PDFRadialGradient(r,o,u,p,g,y,w){var _;return _classCallCheck(this,PDFRadialGradient),(_=i.call(this,r)).doc=r,_.x1=o,_.y1=u,_.r1=p,_.x2=g,_.y2=y,_.r2=w,_}return _createClass(PDFRadialGradient,[{key:"shader",value:function shader(r){return this.doc.ref({ShadingType:3,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.r1,this.x2,this.y2,this.r2],Function:r,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFRadialGradient(this.doc,this.x1,this.y1,this.r1,this.x2,this.y2,this.r2)}}]),PDFRadialGradient}(xe)},ke=["DeviceCMYK","DeviceRGB"],Pe=function(){function PDFTilingPattern(r,i,o,u,p){_classCallCheck(this,PDFTilingPattern),this.doc=r,this.bBox=i,this.xStep=o,this.yStep=u,this.stream=p}return _createClass(PDFTilingPattern,[{key:"createPattern",value:function createPattern(){var r=this.doc.ref();r.end();var i=_slicedToArray(this.doc._ctm,6),o=i[0],u=i[1],p=i[2],g=i[3],y=[1*o+0*p,1*u+0*g,0*o+1*p,0*u+1*g,0*o+0*p+i[4],0*u+0*g+i[5]],w=this.doc.ref({Type:"Pattern",PatternType:1,PaintType:2,TilingType:2,BBox:this.bBox,XStep:this.xStep,YStep:this.yStep,Matrix:y.map((function(r){return+r.toFixed(5)})),Resources:r});return w.end(this.stream),w}},{key:"embedPatternColorSpaces",value:function embedPatternColorSpaces(){var r=this;ke.forEach((function(i){var o=r.getPatternColorSpaceId(i);if(!r.doc.page.colorSpaces[o]){var u=r.doc.ref(["Pattern",i]);u.end(),r.doc.page.colorSpaces[o]=u}}))}},{key:"getPatternColorSpaceId",value:function getPatternColorSpaceId(r){return"CsP".concat(r)}},{key:"embed",value:function embed(){this.id||(this.doc._patternCount=this.doc._patternCount+1,this.id="P"+this.doc._patternCount,this.pattern=this.createPattern()),this.doc.page.patterns[this.id]||(this.doc.page.patterns[this.id]=this.pattern)}},{key:"apply",value:function apply(r,i){this.embedPatternColorSpaces(),this.embed();var o=this.doc._normalizeColor(i);if(!o)throw Error("invalid pattern color. (value: ".concat(i,")"));var u=this.getPatternColorSpaceId(this.doc._getColorSpace(o));this.doc._setColorSpace(u,r);var p=r?"SCN":"scn";return this.doc.addContent("".concat(o.join(" ")," /").concat(this.id," ").concat(p))}}]),PDFTilingPattern}(),Ae=Ce.PDFGradient,Ee=Ce.PDFLinearGradient,Te=Ce.PDFRadialGradient,Oe={PDFTilingPattern:Pe}.PDFTilingPattern,Ie={initColor:function initColor(){return this._opacityRegistry={},this._opacityCount=0,this._patternCount=0,this._gradCount=0},_normalizeColor:function _normalizeColor(r){if("string"==typeof r)if("#"===r.charAt(0)){4===r.length&&(r=r.replace(/#([0-9A-F])([0-9A-F])([0-9A-F])/i,"#$1$1$2$2$3$3"));var i=parseInt(r.slice(1),16);r=[i>>16,i>>8&255,255&i]}else Be[r]&&(r=Be[r]);return Array.isArray(r)?(3===r.length?r=r.map((function(r){return r/255})):4===r.length&&(r=r.map((function(r){return r/100}))),r):null},_setColor:function _setColor(r,i){return r instanceof Ae?(r.apply(i),!0):Array.isArray(r)&&r[0]instanceof Oe?(r[0].apply(i,r[1]),!0):this._setColorCore(r,i)},_setColorCore:function _setColorCore(r,i){if(!(r=this._normalizeColor(r)))return!1;var o=i?"SCN":"scn",u=this._getColorSpace(r);return this._setColorSpace(u,i),r=r.join(" "),this.addContent("".concat(r," ").concat(o)),!0},_setColorSpace:function _setColorSpace(r,i){var o=i?"CS":"cs";return this.addContent("/".concat(r," ").concat(o))},_getColorSpace:function _getColorSpace(r){return 4===r.length?"DeviceCMYK":"DeviceRGB"},fillColor:function fillColor(r,i){return this._setColor(r,!1)&&this.fillOpacity(i),this._fillColor=[r,i],this},strokeColor:function strokeColor(r,i){return this._setColor(r,!0)&&this.strokeOpacity(i),this},opacity:function opacity(r){return this._doOpacity(r,r),this},fillOpacity:function fillOpacity(r){return this._doOpacity(r,null),this},strokeOpacity:function strokeOpacity(r){return this._doOpacity(null,r),this},_doOpacity:function _doOpacity(r,i){var o,u;if(null!=r||null!=i){null!=r&&(r=Math.max(0,Math.min(1,r))),null!=i&&(i=Math.max(0,Math.min(1,i)));var p="".concat(r,"_").concat(i);if(this._opacityRegistry[p]){var g=_slicedToArray(this._opacityRegistry[p],2);o=g[0],u=g[1]}else{o={Type:"ExtGState"},null!=r&&(o.ca=r),null!=i&&(o.CA=i),(o=this.ref(o)).end();var y=++this._opacityCount;u="Gs".concat(y),this._opacityRegistry[p]=[o,u]}return this.page.ext_gstates[u]=o,this.addContent("/".concat(u," gs"))}},linearGradient:function linearGradient(r,i,o,u){return new Ee(this,r,i,o,u)},radialGradient:function radialGradient(r,i,o,u,p,g){return new Te(this,r,i,o,u,p,g)},pattern:function pattern(r,i,o,u){return new Oe(this,r,i,o,u)}},Be={aliceblue:[240,248,255],antiquewhite:[250,235,215],aqua:[0,255,255],aquamarine:[127,255,212],azure:[240,255,255],beige:[245,245,220],bisque:[255,228,196],black:[0,0,0],blanchedalmond:[255,235,205],blue:[0,0,255],blueviolet:[138,43,226],brown:[165,42,42],burlywood:[222,184,135],cadetblue:[95,158,160],chartreuse:[127,255,0],chocolate:[210,105,30],coral:[255,127,80],cornflowerblue:[100,149,237],cornsilk:[255,248,220],crimson:[220,20,60],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgoldenrod:[184,134,11],darkgray:[169,169,169],darkgreen:[0,100,0],darkgrey:[169,169,169],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkseagreen:[143,188,143],darkslateblue:[72,61,139],darkslategray:[47,79,79],darkslategrey:[47,79,79],darkturquoise:[0,206,209],darkviolet:[148,0,211],deeppink:[255,20,147],deepskyblue:[0,191,255],dimgray:[105,105,105],dimgrey:[105,105,105],dodgerblue:[30,144,255],firebrick:[178,34,34],floralwhite:[255,250,240],forestgreen:[34,139,34],fuchsia:[255,0,255],gainsboro:[220,220,220],ghostwhite:[248,248,255],gold:[255,215,0],goldenrod:[218,165,32],gray:[128,128,128],grey:[128,128,128],green:[0,128,0],greenyellow:[173,255,47],honeydew:[240,255,240],hotpink:[255,105,180],indianred:[205,92,92],indigo:[75,0,130],ivory:[255,255,240],khaki:[240,230,140],lavender:[230,230,250],lavenderblush:[255,240,245],lawngreen:[124,252,0],lemonchiffon:[255,250,205],lightblue:[173,216,230],lightcoral:[240,128,128],lightcyan:[224,255,255],lightgoldenrodyellow:[250,250,210],lightgray:[211,211,211],lightgreen:[144,238,144],lightgrey:[211,211,211],lightpink:[255,182,193],lightsalmon:[255,160,122],lightseagreen:[32,178,170],lightskyblue:[135,206,250],lightslategray:[119,136,153],lightslategrey:[119,136,153],lightsteelblue:[176,196,222],lightyellow:[255,255,224],lime:[0,255,0],limegreen:[50,205,50],linen:[250,240,230],magenta:[255,0,255],maroon:[128,0,0],mediumaquamarine:[102,205,170],mediumblue:[0,0,205],mediumorchid:[186,85,211],mediumpurple:[147,112,219],mediumseagreen:[60,179,113],mediumslateblue:[123,104,238],mediumspringgreen:[0,250,154],mediumturquoise:[72,209,204],mediumvioletred:[199,21,133],midnightblue:[25,25,112],mintcream:[245,255,250],mistyrose:[255,228,225],moccasin:[255,228,181],navajowhite:[255,222,173],navy:[0,0,128],oldlace:[253,245,230],olive:[128,128,0],olivedrab:[107,142,35],orange:[255,165,0],orangered:[255,69,0],orchid:[218,112,214],palegoldenrod:[238,232,170],palegreen:[152,251,152],paleturquoise:[175,238,238],palevioletred:[219,112,147],papayawhip:[255,239,213],peachpuff:[255,218,185],peru:[205,133,63],pink:[255,192,203],plum:[221,160,221],powderblue:[176,224,230],purple:[128,0,128],red:[255,0,0],rosybrown:[188,143,143],royalblue:[65,105,225],saddlebrown:[139,69,19],salmon:[250,128,114],sandybrown:[244,164,96],seagreen:[46,139,87],seashell:[255,245,238],sienna:[160,82,45],silver:[192,192,192],skyblue:[135,206,235],slateblue:[106,90,205],slategray:[112,128,144],slategrey:[112,128,144],snow:[255,250,250],springgreen:[0,255,127],steelblue:[70,130,180],tan:[210,180,140],teal:[0,128,128],thistle:[216,191,216],tomato:[255,99,71],turquoise:[64,224,208],violet:[238,130,238],wheat:[245,222,179],white:[255,255,255],whitesmoke:[245,245,245],yellow:[255,255,0],yellowgreen:[154,205,50]};ge=ye=ve=me=be=we=0;var Fe={A:7,a:7,C:6,c:6,H:1,h:1,L:2,l:2,M:2,m:2,Q:4,q:4,S:4,s:4,T:2,t:2,V:1,v:1,Z:0,z:0},Le={M:function M(r,i){return ge=i[0],ye=i[1],ve=me=null,be=ge,we=ye,r.moveTo(ge,ye)},m:function m(r,i){return ge+=i[0],ye+=i[1],ve=me=null,be=ge,we=ye,r.moveTo(ge,ye)},C:function C(r,i){return ge=i[4],ye=i[5],ve=i[2],me=i[3],r.bezierCurveTo.apply(r,_toConsumableArray(i))},c:function c(r,i){return r.bezierCurveTo(i[0]+ge,i[1]+ye,i[2]+ge,i[3]+ye,i[4]+ge,i[5]+ye),ve=ge+i[2],me=ye+i[3],ge+=i[4],ye+=i[5]},S:function S(r,i){return null===ve&&(ve=ge,me=ye),r.bezierCurveTo(ge-(ve-ge),ye-(me-ye),i[0],i[1],i[2],i[3]),ve=i[0],me=i[1],ge=i[2],ye=i[3]},s:function s(r,i){return null===ve&&(ve=ge,me=ye),r.bezierCurveTo(ge-(ve-ge),ye-(me-ye),ge+i[0],ye+i[1],ge+i[2],ye+i[3]),ve=ge+i[0],me=ye+i[1],ge+=i[2],ye+=i[3]},Q:function Q(r,i){return ve=i[0],me=i[1],ge=i[2],ye=i[3],r.quadraticCurveTo(i[0],i[1],ge,ye)},q:function q(r,i){return r.quadraticCurveTo(i[0]+ge,i[1]+ye,i[2]+ge,i[3]+ye),ve=ge+i[0],me=ye+i[1],ge+=i[2],ye+=i[3]},T:function T(r,i){return null===ve?(ve=ge,me=ye):(ve=ge-(ve-ge),me=ye-(me-ye)),r.quadraticCurveTo(ve,me,i[0],i[1]),ve=ge-(ve-ge),me=ye-(me-ye),ge=i[0],ye=i[1]},t:function t(r,i){return null===ve?(ve=ge,me=ye):(ve=ge-(ve-ge),me=ye-(me-ye)),r.quadraticCurveTo(ve,me,ge+i[0],ye+i[1]),ge+=i[0],ye+=i[1]},A:function A(r,i){return Re(r,ge,ye,i),ge=i[5],ye=i[6]},a:function a(r,i){return i[5]+=ge,i[6]+=ye,Re(r,ge,ye,i),ge=i[5],ye=i[6]},L:function L(r,i){return ge=i[0],ye=i[1],ve=me=null,r.lineTo(ge,ye)},l:function l(r,i){return ge+=i[0],ye+=i[1],ve=me=null,r.lineTo(ge,ye)},H:function H(r,i){return ge=i[0],ve=me=null,r.lineTo(ge,ye)},h:function h(r,i){return ge+=i[0],ve=me=null,r.lineTo(ge,ye)},V:function V(r,i){return ye=i[0],ve=me=null,r.lineTo(ge,ye)},v:function v(r,i){return ye+=i[0],ve=me=null,r.lineTo(ge,ye)},Z:function Z(r){return r.closePath(),ge=be,ye=we},z:function z(r){return r.closePath(),ge=be,ye=we}},Re=function solveArc(r,i,o,u){var p,g=_slicedToArray(u,7),y=g[0],w=g[1],_=g[2],x=g[3],k=g[4],P=g[5],O=g[6],I=_createForOfIteratorHelper(De(P,O,y,w,x,k,_,i,o));try{for(I.s();!(p=I.n()).done;){var B=p.value,R=Me.apply(void 0,_toConsumableArray(B));r.bezierCurveTo.apply(r,_toConsumableArray(R))}}catch(r){I.e(r)}finally{I.f()}},De=function arcToSegments(r,i,o,u,p,g,y,w,_){var x=y*(Math.PI/180),k=Math.sin(x),P=Math.cos(x);o=Math.abs(o),u=Math.abs(u);var O=(ve=P*(w-r)*.5+k*(_-i)*.5)*ve/(o*o)+(me=P*(_-i)*.5-k*(w-r)*.5)*me/(u*u);O>1&&(o*=O=Math.sqrt(O),u*=O);var I=P/o,B=k/o,R=-k/u,D=P/u,N=I*w+B*_,W=R*w+D*_,U=I*r+B*i,j=R*r+D*i,G=1/((U-N)*(U-N)+(j-W)*(j-W))-.25;G<0&&(G=0);var X=Math.sqrt(G);g===p&&(X=-X);var K=.5*(N+U)-X*(j-W),Y=.5*(W+j)+X*(U-N),J=Math.atan2(W-Y,N-K),$=Math.atan2(j-Y,U-K)-J;$<0&&1===g?$+=2*Math.PI:$>0&&0===g&&($-=2*Math.PI);for(var ee=Math.ceil(Math.abs($/(.5*Math.PI+.001))),te=[],re=0;re<ee;re++){var ne=J+re*$/ee,ie=J+(re+1)*$/ee;te[re]=[K,Y,ne,ie,o,u,k,P]}return te},Me=function segmentToBezier(r,i,o,u,p,g,y,w){var _=w*p,x=-y*g,k=y*p,P=w*g,O=.5*(u-o),I=8/3*Math.sin(.5*O)*Math.sin(.5*O)/Math.sin(O),B=r+Math.cos(o)-I*Math.sin(o),R=i+Math.sin(o)+I*Math.cos(o),D=r+Math.cos(u),N=i+Math.sin(u),W=D+I*Math.sin(u),U=N-I*Math.cos(u);return[_*B+x*R,k*B+P*R,_*W+x*U,k*W+P*U,_*D+x*N,k*D+P*N]},Ne=function(){function SVGPath(){_classCallCheck(this,SVGPath)}return _createClass(SVGPath,null,[{key:"apply",value:function apply(r,i){var o=function parse(r){var i,o,u=[],p=[],g="",y=!1,w=0,_=_createForOfIteratorHelper(r);try{for(_.s();!(o=_.n()).done;){var x=o.value;if(null!=Fe[x])w=Fe[x],i&&(g.length>0&&(p[p.length]=+g),u[u.length]={cmd:i,args:p},p=[],g="",y=!1),i=x;else if([" ",","].includes(x)||"-"===x&&g.length>0&&"e"!==g[g.length-1]||"."===x&&y){if(0===g.length)continue;p.length===w?(u[u.length]={cmd:i,args:p},p=[+g],"M"===i&&(i="L"),"m"===i&&(i="l")):p[p.length]=+g,y="."===x,g=["-","."].includes(x)?x:""}else g+=x,"."===x&&(y=!0)}}catch(r){_.e(r)}finally{_.f()}return g.length>0&&(p.length===w?(u[u.length]={cmd:i,args:p},p=[+g],"M"===i&&(i="L"),"m"===i&&(i="l")):p[p.length]=+g),u[u.length]={cmd:i,args:p},u}(i);!function apply(r,i){ge=ye=ve=me=be=we=0;for(var o=0;o<r.length;o++){var u=r[o];"function"==typeof Le[u.cmd]&&Le[u.cmd](i,u.args)}}(o,r)}}]),SVGPath}(),We=W.number,Ue=(Math.sqrt(2)-1)/3*4,ze={initVector:function initVector(){return this._ctm=[1,0,0,1,0,0],this._ctmStack=[]},save:function save(){return this._ctmStack.push(this._ctm.slice()),this.addContent("q")},restore:function restore(){return this._ctm=this._ctmStack.pop()||[1,0,0,1,0,0],this.addContent("Q")},closePath:function closePath(){return this.addContent("h")},lineWidth:function lineWidth(r){return this.addContent("".concat(We(r)," w"))},_CAP_STYLES:{BUTT:0,ROUND:1,SQUARE:2},lineCap:function lineCap(r){return"string"==typeof r&&(r=this._CAP_STYLES[r.toUpperCase()]),this.addContent("".concat(r," J"))},_JOIN_STYLES:{MITER:0,ROUND:1,BEVEL:2},lineJoin:function lineJoin(r){return"string"==typeof r&&(r=this._JOIN_STYLES[r.toUpperCase()]),this.addContent("".concat(r," j"))},miterLimit:function miterLimit(r){return this.addContent("".concat(We(r)," M"))},dash:function dash(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=r;Array.isArray(r)||(r=[r,i.space||r]);var u=r.every((function(r){return Number.isFinite(r)&&r>0}));if(!u)throw new Error("dash(".concat(JSON.stringify(o),", ").concat(JSON.stringify(i),") invalid, lengths must be numeric and greater than zero"));return r=r.map(We).join(" "),this.addContent("[".concat(r,"] ").concat(We(i.phase||0)," d"))},undash:function undash(){return this.addContent("[] 0 d")},moveTo:function moveTo(r,i){return this.addContent("".concat(We(r)," ").concat(We(i)," m"))},lineTo:function lineTo(r,i){return this.addContent("".concat(We(r)," ").concat(We(i)," l"))},bezierCurveTo:function bezierCurveTo(r,i,o,u,p,g){return this.addContent("".concat(We(r)," ").concat(We(i)," ").concat(We(o)," ").concat(We(u)," ").concat(We(p)," ").concat(We(g)," c"))},quadraticCurveTo:function quadraticCurveTo(r,i,o,u){return this.addContent("".concat(We(r)," ").concat(We(i)," ").concat(We(o)," ").concat(We(u)," v"))},rect:function rect(r,i,o,u){return this.addContent("".concat(We(r)," ").concat(We(i)," ").concat(We(o)," ").concat(We(u)," re"))},roundedRect:function roundedRect(r,i,o,u,p){null==p&&(p=0);var g=(p=Math.min(p,.5*o,.5*u))*(1-Ue);return this.moveTo(r+p,i),this.lineTo(r+o-p,i),this.bezierCurveTo(r+o-g,i,r+o,i+g,r+o,i+p),this.lineTo(r+o,i+u-p),this.bezierCurveTo(r+o,i+u-g,r+o-g,i+u,r+o-p,i+u),this.lineTo(r+p,i+u),this.bezierCurveTo(r+g,i+u,r,i+u-g,r,i+u-p),this.lineTo(r,i+p),this.bezierCurveTo(r,i+g,r+g,i,r+p,i),this.closePath()},ellipse:function ellipse(r,i,o,u){null==u&&(u=o);var p=o*Ue,g=u*Ue,y=(r-=o)+2*o,w=(i-=u)+2*u,_=r+o,x=i+u;return this.moveTo(r,x),this.bezierCurveTo(r,x-g,_-p,i,_,i),this.bezierCurveTo(_+p,i,y,x-g,y,x),this.bezierCurveTo(y,x+g,_+p,w,_,w),this.bezierCurveTo(_-p,w,r,x+g,r,x),this.closePath()},circle:function circle(r,i,o){return this.ellipse(r,i,o)},arc:function arc(r,i,o,u,p,g){null==g&&(g=!1);var y=2*Math.PI,w=.5*Math.PI,_=p-u;if(Math.abs(_)>y)_=y;else if(0!==_&&g!==_<0){_=(g?-1:1)*y+_}var x=Math.ceil(Math.abs(_)/w),k=_/x,P=k/w*Ue*o,O=u,I=-Math.sin(O)*P,B=Math.cos(O)*P,R=r+Math.cos(O)*o,D=i+Math.sin(O)*o;this.moveTo(R,D);for(var N=0;N<x;N++){var W=R+I,U=D+B;O+=k,R=r+Math.cos(O)*o,D=i+Math.sin(O)*o;var j=R-(I=-Math.sin(O)*P),G=D-(B=Math.cos(O)*P);this.bezierCurveTo(W,U,j,G,R,D)}return this},polygon:function polygon(){for(var r=arguments.length,i=new Array(r),o=0;o<r;o++)i[o]=arguments[o];this.moveTo.apply(this,_toConsumableArray(i.shift()||[]));for(var u=0,p=i;u<p.length;u++){var g=p[u];this.lineTo.apply(this,_toConsumableArray(g||[]))}return this.closePath()},path:function path(r){return Ne.apply(this,r),this},_windingRule:function _windingRule(r){return/even-?odd/.test(r)?"*":""},fill:function fill(r,i){return/(even-?odd)|(non-?zero)/.test(r)&&(i=r,r=null),r&&this.fillColor(r),this.addContent("f".concat(this._windingRule(i)))},stroke:function stroke(r){return r&&this.strokeColor(r),this.addContent("S")},fillAndStroke:function fillAndStroke(r,i,o){null==i&&(i=r);var u=/(even-?odd)|(non-?zero)/;return u.test(r)&&(o=r,r=null),u.test(i)&&(o=i,i=r),r&&(this.fillColor(r),this.strokeColor(i)),this.addContent("B".concat(this._windingRule(o)))},clip:function clip(r){return this.addContent("W".concat(this._windingRule(r)," n"))},transform:function transform(r,i,o,u,p,g){var y=this._ctm,w=_slicedToArray(y,6),_=w[0],x=w[1],k=w[2],P=w[3],O=w[4],I=w[5];y[0]=_*r+k*i,y[1]=x*r+P*i,y[2]=_*o+k*u,y[3]=x*o+P*u,y[4]=_*p+k*g+O,y[5]=x*p+P*g+I;var B=[r,i,o,u,p,g].map((function(r){return We(r)})).join(" ");return this.addContent("".concat(B," cm"))},translate:function translate(r,i){return this.transform(1,0,0,1,r,i)},rotate:function rotate(r){var i,o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=r*Math.PI/180,p=Math.cos(u),g=Math.sin(u),y=i=0;if(null!=o.origin){var w=_slicedToArray(o.origin,2),_=(y=w[0])*p-(i=w[1])*g,x=y*g+i*p;y-=_,i-=x}return this.transform(p,g,-g,p,y,i)},scale:function scale(r,i){var o,u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};null==i&&(i=r),"object"==typeof i&&(u=i,i=r);var p=o=0;if(null!=u.origin){var g=_slicedToArray(u.origin,2);p=g[0],o=g[1],p-=r*p,o-=i*o}return this.transform(r,0,0,i,p,o)}},je={402:131,8211:150,8212:151,8216:145,8217:146,8218:130,8220:147,8221:148,8222:132,8224:134,8225:135,8226:149,8230:133,8364:128,8240:137,8249:139,8250:155,710:136,8482:153,338:140,339:156,732:152,352:138,353:154,376:159,381:142,382:158},Ge=".notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n  \nspace         exclam         quotedbl       numbersign\ndollar        percent        ampersand      quotesingle\nparenleft     parenright     asterisk       plus\ncomma         hyphen         period         slash\nzero          one            two            three\nfour          five           six            seven\neight         nine           colon          semicolon\nless          equal          greater        question\n  \nat            A              B              C\nD             E              F              G\nH             I              J              K\nL             M              N              O\nP             Q              R              S\nT             U              V              W\nX             Y              Z              bracketleft\nbackslash     bracketright   asciicircum    underscore\n  \ngrave         a              b              c\nd             e              f              g\nh             i              j              k\nl             m              n              o\np             q              r              s\nt             u              v              w\nx             y              z              braceleft\nbar           braceright     asciitilde     .notdef\n  \nEuro          .notdef        quotesinglbase florin\nquotedblbase  ellipsis       dagger         daggerdbl\ncircumflex    perthousand    Scaron         guilsinglleft\nOE            .notdef        Zcaron         .notdef\n.notdef       quoteleft      quoteright     quotedblleft\nquotedblright bullet         endash         emdash\ntilde         trademark      scaron         guilsinglright\noe            .notdef        zcaron         ydieresis\n  \nspace         exclamdown     cent           sterling\ncurrency      yen            brokenbar      section\ndieresis      copyright      ordfeminine    guillemotleft\nlogicalnot    hyphen         registered     macron\ndegree        plusminus      twosuperior    threesuperior\nacute         mu             paragraph      periodcentered\ncedilla       onesuperior    ordmasculine   guillemotright\nonequarter    onehalf        threequarters  questiondown\n  \nAgrave        Aacute         Acircumflex    Atilde\nAdieresis     Aring          AE             Ccedilla\nEgrave        Eacute         Ecircumflex    Edieresis\nIgrave        Iacute         Icircumflex    Idieresis\nEth           Ntilde         Ograve         Oacute\nOcircumflex   Otilde         Odieresis      multiply\nOslash        Ugrave         Uacute         Ucircumflex\nUdieresis     Yacute         Thorn          germandbls\n  \nagrave        aacute         acircumflex    atilde\nadieresis     aring          ae             ccedilla\negrave        eacute         ecircumflex    edieresis\nigrave        iacute         icircumflex    idieresis\neth           ntilde         ograve         oacute\nocircumflex   otilde         odieresis      divide\noslash        ugrave         uacute         ucircumflex\nudieresis     yacute         thorn          ydieresis".split(/\s+/),Ve=function(){function AFMFont(r){_classCallCheck(this,AFMFont),this.contents=r,this.attributes={},this.glyphWidths={},this.boundingBoxes={},this.kernPairs={},this.parse(),this.charWidths=new Array(256);for(var i=0;i<=255;i++)this.charWidths[i]=this.glyphWidths[Ge[i]];this.bbox=this.attributes.FontBBox.split(/\s+/).map((function(r){return+r})),this.ascender=+(this.attributes.Ascender||0),this.descender=+(this.attributes.Descender||0),this.xHeight=+(this.attributes.XHeight||0),this.capHeight=+(this.attributes.CapHeight||0),this.lineGap=this.bbox[3]-this.bbox[1]-(this.ascender-this.descender)}return _createClass(AFMFont,null,[{key:"open",value:function open(r){return new AFMFont(P.readFileSync(r,"utf8"))}}]),_createClass(AFMFont,[{key:"parse",value:function parse(){var r,i="",o=_createForOfIteratorHelper(this.contents.split("\n"));try{for(o.s();!(r=o.n()).done;){var u,p,g=r.value;if(u=g.match(/^Start(\w+)/))i=u[1];else if(u=g.match(/^End(\w+)/))i="";else switch(i){case"FontMetrics":var y=(u=g.match(/(^\w+)\s+(.*)/))[1],w=u[2];(p=this.attributes[y])?(Array.isArray(p)||(p=this.attributes[y]=[p]),p.push(w)):this.attributes[y]=w;break;case"CharMetrics":if(!/^CH?\s/.test(g))continue;var _=g.match(/\bN\s+(\.?\w+)\s*;/)[1];this.glyphWidths[_]=+g.match(/\bWX\s+(\d+)\s*;/)[1];break;case"KernPairs":(u=g.match(/^KPX\s+(\.?\w+)\s+(\.?\w+)\s+(-?\d+)/))&&(this.kernPairs[u[1]+"\0"+u[2]]=parseInt(u[3]))}}}catch(r){o.e(r)}finally{o.f()}}},{key:"encodeText",value:function encodeText(r){for(var i=[],o=0,u=r.length;o<u;o++){var p=r.charCodeAt(o);p=je[p]||p,i.push(p.toString(16))}return i}},{key:"glyphsForString",value:function glyphsForString(r){for(var i=[],o=0,u=r.length;o<u;o++){var p=r.charCodeAt(o);i.push(this.characterToGlyph(p))}return i}},{key:"characterToGlyph",value:function characterToGlyph(r){return Ge[je[r]||r]||".notdef"}},{key:"widthOfGlyph",value:function widthOfGlyph(r){return this.glyphWidths[r]||0}},{key:"getKernPair",value:function getKernPair(r,i){return this.kernPairs[r+"\0"+i]||0}},{key:"advancesForGlyphs",value:function advancesForGlyphs(r){for(var i=[],o=0;o<r.length;o++){var u=r[o],p=r[o+1];i.push(this.widthOfGlyph(u)+this.getKernPair(u,p))}return i}}]),AFMFont}(),He=function(){function PDFFont(){_classCallCheck(this,PDFFont)}return _createClass(PDFFont,[{key:"encode",value:function encode(){throw new Error("Must be implemented by subclasses")}},{key:"widthOfString",value:function widthOfString(){throw new Error("Must be implemented by subclasses")}},{key:"ref",value:function ref(){return null!=this.dictionary?this.dictionary:this.dictionary=this.document.ref()}},{key:"finalize",value:function finalize(){if(!this.embedded&&null!=this.dictionary)return this.embed(),this.embedded=!0}},{key:"embed",value:function embed(){throw new Error("Must be implemented by subclasses")}},{key:"lineHeight",value:function lineHeight(r,i){null==i&&(i=!1);var o=i?this.lineGap:0;return(this.ascender+o-this.descender)/1e3*r}}]),PDFFont}(),qe={Courier:function Courier(){return P.readFileSync("//data/Courier.afm","utf8")},"Courier-Bold":function CourierBold(){return P.readFileSync("//data/Courier-Bold.afm","utf8")},"Courier-Oblique":function CourierOblique(){return P.readFileSync("//data/Courier-Oblique.afm","utf8")},"Courier-BoldOblique":function CourierBoldOblique(){return P.readFileSync("//data/Courier-BoldOblique.afm","utf8")},Helvetica:function Helvetica(){return P.readFileSync("//data/Helvetica.afm","utf8")},"Helvetica-Bold":function HelveticaBold(){return P.readFileSync("//data/Helvetica-Bold.afm","utf8")},"Helvetica-Oblique":function HelveticaOblique(){return P.readFileSync("//data/Helvetica-Oblique.afm","utf8")},"Helvetica-BoldOblique":function HelveticaBoldOblique(){return P.readFileSync("//data/Helvetica-BoldOblique.afm","utf8")},"Times-Roman":function TimesRoman(){return P.readFileSync("//data/Times-Roman.afm","utf8")},"Times-Bold":function TimesBold(){return P.readFileSync("//data/Times-Bold.afm","utf8")},"Times-Italic":function TimesItalic(){return P.readFileSync("//data/Times-Italic.afm","utf8")},"Times-BoldItalic":function TimesBoldItalic(){return P.readFileSync("//data/Times-BoldItalic.afm","utf8")},Symbol:function Symbol(){return P.readFileSync("//data/Symbol.afm","utf8")},ZapfDingbats:function ZapfDingbats(){return P.readFileSync("//data/ZapfDingbats.afm","utf8")}},Xe=function(r){_inherits(StandardFont,r);var i=_createSuper(StandardFont);function StandardFont(r,o,u){var p;_classCallCheck(this,StandardFont),(p=i.call(this)).document=r,p.name=o,p.id=u,p.font=new Ve(qe[p.name]());var g=p.font;return p.ascender=g.ascender,p.descender=g.descender,p.bbox=g.bbox,p.lineGap=g.lineGap,p.xHeight=g.xHeight,p.capHeight=g.capHeight,p}return _createClass(StandardFont,[{key:"embed",value:function embed(){return this.dictionary.data={Type:"Font",BaseFont:this.name,Subtype:"Type1",Encoding:"WinAnsiEncoding"},this.dictionary.end()}},{key:"encode",value:function encode(r){for(var i=this.font.encodeText(r),o=this.font.glyphsForString("".concat(r)),u=this.font.advancesForGlyphs(o),p=[],g=0;g<o.length;g++){var y=o[g];p.push({xAdvance:u[g],yAdvance:0,xOffset:0,yOffset:0,advanceWidth:this.font.widthOfGlyph(y)})}return[i,p]}},{key:"widthOfString",value:function widthOfString(r,i){var o,u=this.font.glyphsForString("".concat(r)),p=0,g=_createForOfIteratorHelper(this.font.advancesForGlyphs(u));try{for(g.s();!(o=g.n()).done;){p+=o.value}}catch(r){g.e(r)}finally{g.f()}return p*(i/1e3)}}],[{key:"isStandardFont",value:function isStandardFont(r){return r in qe}}]),StandardFont}(He),Ze=function toHex(r){return"0000".concat(r.toString(16)).slice(-4)},Ke=function(r){_inherits(EmbeddedFont,r);var i=_createSuper(EmbeddedFont);function EmbeddedFont(r,o,u){var p;return _classCallCheck(this,EmbeddedFont),(p=i.call(this)).document=r,p.font=o,p.id=u,p.subset=p.font.createSubset(),p.unicode=[[0]],p.widths=[p.font.getGlyph(0).advanceWidth],p.name=p.font.postscriptName,p.scale=1e3/p.font.unitsPerEm,p.ascender=p.font.ascent*p.scale,p.descender=p.font.descent*p.scale,p.xHeight=p.font.xHeight*p.scale,p.capHeight=p.font.capHeight*p.scale,p.lineGap=p.font.lineGap*p.scale,p.bbox=p.font.bbox,!1!==r.options.fontLayoutCache&&(p.layoutCache=Object.create(null)),p}return _createClass(EmbeddedFont,[{key:"layoutRun",value:function layoutRun(r,i){for(var o=this.font.layout(r,i),u=0;u<o.positions.length;u++){var p=o.positions[u];for(var g in p)p[g]*=this.scale;p.advanceWidth=o.glyphs[u].advanceWidth*this.scale}return o}},{key:"layoutCached",value:function layoutCached(r){if(!this.layoutCache)return this.layoutRun(r);var i;if(i=this.layoutCache[r])return i;var o=this.layoutRun(r);return this.layoutCache[r]=o,o}},{key:"layout",value:function layout(r,i,o){if(i)return this.layoutRun(r,i);for(var u=o?null:[],p=o?null:[],g=0,y=0,w=0;w<=r.length;){var _;if(w===r.length&&y<w||(_=r.charAt(w),[" ","\t"].includes(_))){var x=this.layoutCached(r.slice(y,++w));o||(u=u.concat(x.glyphs),p=p.concat(x.positions)),g+=x.advanceWidth,y=w}else w++}return{glyphs:u,positions:p,advanceWidth:g}}},{key:"encode",value:function encode(r,i){for(var o=this.layout(r,i),u=o.glyphs,p=o.positions,g=[],y=0;y<u.length;y++){var w=u[y],_=this.subset.includeGlyph(w.id);g.push("0000".concat(_.toString(16)).slice(-4)),null==this.widths[_]&&(this.widths[_]=w.advanceWidth*this.scale),null==this.unicode[_]&&(this.unicode[_]=w.codePoints)}return[g,p]}},{key:"widthOfString",value:function widthOfString(r,i,o){return this.layout(r,o,!0).advanceWidth*(i/1e3)}},{key:"embed",value:function embed(){var r=this,i=null!=this.subset.cff,o=this.document.ref();i&&(o.data.Subtype="CIDFontType0C"),this.subset.encodeStream().on("data",(function(r){return o.write(r)})).on("end",(function(){return o.end()}));var u=((null!=this.font["OS/2"]?this.font["OS/2"].sFamilyClass:void 0)||0)>>8,p=0;this.font.post.isFixedPitch&&(p|=1),1<=u&&u<=7&&(p|=2),p|=4,10===u&&(p|=8),this.font.head.macStyle.italic&&(p|=64);var g=[1,2,3,4,5,6].map((function(i){return String.fromCharCode((r.id.charCodeAt(i)||73)+17)})).join("")+"+"+this.font.postscriptName,y=this.font.bbox,w=this.document.ref({Type:"FontDescriptor",FontName:g,Flags:p,FontBBox:[y.minX*this.scale,y.minY*this.scale,y.maxX*this.scale,y.maxY*this.scale],ItalicAngle:this.font.italicAngle,Ascent:this.ascender,Descent:this.descender,CapHeight:(this.font.capHeight||this.font.ascent)*this.scale,XHeight:(this.font.xHeight||0)*this.scale,StemV:0});i?w.data.FontFile3=o:w.data.FontFile2=o,w.end();var _={Type:"Font",Subtype:"CIDFontType0",BaseFont:g,CIDSystemInfo:{Registry:new String("Adobe"),Ordering:new String("Identity"),Supplement:0},FontDescriptor:w,W:[0,this.widths]};i||(_.Subtype="CIDFontType2",_.CIDToGIDMap="Identity");var x=this.document.ref(_);return x.end(),this.dictionary.data={Type:"Font",Subtype:"Type0",BaseFont:g,Encoding:"Identity-H",DescendantFonts:[x],ToUnicode:this.toUnicodeCmap()},this.dictionary.end()}},{key:"toUnicodeCmap",value:function toUnicodeCmap(){var r,i=this.document.ref(),o=[],u=_createForOfIteratorHelper(this.unicode);try{for(u.s();!(r=u.n()).done;){var p,g=r.value,y=[],w=_createForOfIteratorHelper(g);try{for(w.s();!(p=w.n()).done;){var _=p.value;_>65535&&(_-=65536,y.push(Ze(_>>>10&1023|55296)),_=56320|1023&_),y.push(Ze(_))}}catch(r){w.e(r)}finally{w.f()}o.push("<".concat(y.join(" "),">"))}}catch(r){u.e(r)}finally{u.f()}return i.end("/CIDInit /ProcSet findresource begin\n12 dict begin\nbegincmap\n/CIDSystemInfo <<\n  /Registry (Adobe)\n  /Ordering (UCS)\n  /Supplement 0\n>> def\n/CMapName /Adobe-Identity-UCS def\n/CMapType 2 def\n1 begincodespacerange\n<0000><ffff>\nendcodespacerange\n1 beginbfrange\n<0000> <".concat(Ze(o.length-1),"> [").concat(o.join(" "),"]\nendbfrange\nendcmap\nCMapName currentdict /CMap defineresource pop\nend\nend")),i}}]),EmbeddedFont}(He),Ye=function(){function PDFFontFactory(){_classCallCheck(this,PDFFontFactory)}return _createClass(PDFFontFactory,null,[{key:"open",value:function open(r,i,o,p){var g;if("string"==typeof i){if(Xe.isStandardFont(i))return new Xe(r,i,p);i=P.readFileSync(i)}if(u.isBuffer(i)?g=w.default.create(i,o):i instanceof Uint8Array?g=w.default.create(u.from(i),o):i instanceof ArrayBuffer&&(g=w.default.create(u.from(new Uint8Array(i)),o)),null==g)throw new Error("Not a supported font format or standard PDF font.");return new Ke(r,g,p)}}]),PDFFontFactory}(),Je={initFonts:function initFonts(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"Helvetica";this._fontFamilies={},this._fontCount=0,this._fontSize=12,this._font=null,this._registeredFonts={},r&&this.font(r)},font:function font(r,i,o){var u,font;if("number"==typeof i&&(o=i,i=null),"string"==typeof r&&this._registeredFonts[r]){u=r;var p=this._registeredFonts[r];r=p.src,i=p.family}else"string"!=typeof(u=i||r)&&(u=null);if(null!=o&&this.fontSize(o),font=this._fontFamilies[u])return this._font=font,this;var g="F".concat(++this._fontCount);return this._font=Ye.open(this,r,i,g),(font=this._fontFamilies[this._font.name])?(this._font=font,this):(u&&(this._fontFamilies[u]=this._font),this._font.name&&(this._fontFamilies[this._font.name]=this._font),this)},fontSize:function fontSize(r){return this._fontSize=r,this},currentLineHeight:function currentLineHeight(r){return null==r&&(r=!1),this._font.lineHeight(this._fontSize,r)},registerFont:function registerFont(r,i,o){return this._registeredFonts[r]={src:i,family:o},this}},Qe=function(r){_inherits(LineWrapper,r);var i=_createSuper(LineWrapper);function LineWrapper(r,o){var u;return _classCallCheck(this,LineWrapper),(u=i.call(this)).document=r,u.indent=o.indent||0,u.characterSpacing=o.characterSpacing||0,u.wordSpacing=0===o.wordSpacing,u.columns=o.columns||1,u.columnGap=null!=o.columnGap?o.columnGap:18,u.lineWidth=(o.width-u.columnGap*(u.columns-1))/u.columns,u.spaceLeft=u.lineWidth,u.startX=u.document.x,u.startY=u.document.y,u.column=1,u.ellipsis=o.ellipsis,u.continuedX=0,u.features=o.features,null!=o.height?(u.height=o.height,u.maxY=u.startY+o.height):u.maxY=u.document.page.maxY(),u.on("firstLine",(function(r){var i=u.continuedX||u.indent;return u.document.x+=i,u.lineWidth-=i,u.once("line",(function(){if(u.document.x-=i,u.lineWidth+=i,r.continued&&!u.continuedX&&(u.continuedX=u.indent),!r.continued)return u.continuedX=0}))})),u.on("lastLine",(function(r){var i=r.align;return"justify"===i&&(r.align="left"),u.lastLine=!0,u.once("line",(function(){return u.document.y+=r.paragraphGap||0,r.align=i,u.lastLine=!1}))})),u}return _createClass(LineWrapper,[{key:"wordWidth",value:function wordWidth(r){return this.document.widthOfString(r,this)+this.characterSpacing+this.wordSpacing}},{key:"eachWord",value:function eachWord(r,i){for(var o,u=new x.default(r),p=null,g=Object.create(null);o=u.nextBreak();){var y,w=r.slice((null!=p?p.position:void 0)||0,o.position),_=null!=g[w]?g[w]:g[w]=this.wordWidth(w);if(_>this.lineWidth+this.continuedX)for(var k=p,P={};w.length;){var O,I;_>this.spaceLeft?(O=Math.ceil(this.spaceLeft/(_/w.length)),I=(_=this.wordWidth(w.slice(0,O)))<=this.spaceLeft&&O<w.length):O=w.length;for(var B=_>this.spaceLeft&&O>0;B||I;)B?B=(_=this.wordWidth(w.slice(0,--O)))>this.spaceLeft&&O>0:(B=(_=this.wordWidth(w.slice(0,++O)))>this.spaceLeft&&O>0,I=_<=this.spaceLeft&&O<w.length);if(0===O&&this.spaceLeft===this.lineWidth&&(O=1),P.required=o.required||O<w.length,y=i(w.slice(0,O),_,P,k),k={required:!1},w=w.slice(O),_=this.wordWidth(w),!1===y)break}else y=i(w,_,o,p);if(!1===y)break;p=o}}},{key:"wrap",value:function wrap(r,i){var o=this;null!=i.indent&&(this.indent=i.indent),null!=i.characterSpacing&&(this.characterSpacing=i.characterSpacing),null!=i.wordSpacing&&(this.wordSpacing=i.wordSpacing),null!=i.ellipsis&&(this.ellipsis=i.ellipsis);var u=this.document.y+this.document.currentLineHeight(!0);(this.document.y>this.maxY||u>this.maxY)&&this.nextSection();var p="",g=0,y=0,w=0,_=this.document.y,x=function emitLine(){return i.textWidth=g+o.wordSpacing*(y-1),i.wordCount=y,i.lineWidth=o.lineWidth,_=o.document.y,o.emit("line",p,i,o),w++};return this.emit("sectionStart",i,this),this.eachWord(r,(function(r,u,w,_){if((null==_||_.required)&&(o.emit("firstLine",i,o),o.spaceLeft=o.lineWidth),u<=o.spaceLeft&&(p+=r,g+=u,y++),w.required||u>o.spaceLeft){var k=o.document.currentLineHeight(!0);if(null!=o.height&&o.ellipsis&&o.document.y+2*k>o.maxY&&o.column>=o.columns){for(!0===o.ellipsis&&(o.ellipsis="…"),p=p.replace(/\s+$/,""),g=o.wordWidth(p+o.ellipsis);p&&g>o.lineWidth;)p=p.slice(0,-1).replace(/\s+$/,""),g=o.wordWidth(p+o.ellipsis);g<=o.lineWidth&&(p+=o.ellipsis),g=o.wordWidth(p)}if(w.required&&(u>o.spaceLeft&&(x(),p=r,g=u,y=1),o.emit("lastLine",i,o)),x(),o.document.y+k>o.maxY)if(!o.nextSection())return y=0,p="",!1;return w.required?(o.spaceLeft=o.lineWidth,p="",g=0,y=0):(o.spaceLeft=o.lineWidth-u,p=r,g=u,y=1)}return o.spaceLeft-=u})),y>0&&(this.emit("lastLine",i,this),x()),this.emit("sectionEnd",i,this),!0===i.continued?(w>1&&(this.continuedX=0),this.continuedX+=i.textWidth||0,this.document.y=_):this.document.x=this.startX}},{key:"nextSection",value:function nextSection(r){if(this.emit("sectionEnd",r,this),++this.column>this.columns){if(null!=this.height)return!1;var i;if(this.document.continueOnNewPage(),this.column=1,this.startY=this.document.page.margins.top,this.maxY=this.document.page.maxY(),this.document.x=this.startX,this.document._fillColor)(i=this.document).fillColor.apply(i,_toConsumableArray(this.document._fillColor));this.emit("pageBreak",r,this)}else this.document.x+=this.lineWidth+this.columnGap,this.document.y=this.startY,this.emit("columnBreak",r,this);return this.emit("sectionStart",r,this),!0}}]),LineWrapper}(_.EventEmitter),$e=W.number,et={initText:function initText(){return this._line=this._line.bind(this),this.x=0,this.y=0,this._lineGap=0},lineGap:function lineGap(r){return this._lineGap=r,this},moveDown:function moveDown(r){return null==r&&(r=1),this.y+=this.currentLineHeight(!0)*r+this._lineGap,this},moveUp:function moveUp(r){return null==r&&(r=1),this.y-=this.currentLineHeight(!0)*r+this._lineGap,this},_text:function _text(r,i,o,u,p){var g=this;u=this._initOptions(i,o,u),r=null==r?"":"".concat(r),u.wordSpacing&&(r=r.replace(/\s{2,}/g," "));var y=function addStructure(){u.structParent&&u.structParent.add(g.struct(u.structType||"P",[g.markStructureContent(u.structType||"P")]))};if(u.width){var w=this._wrapper;w||((w=new Qe(this,u)).on("line",p),w.on("firstLine",y)),this._wrapper=u.continued?w:null,this._textOptions=u.continued?u:null,w.wrap(r,u)}else{var _,x=_createForOfIteratorHelper(r.split("\n"));try{for(x.s();!(_=x.n()).done;){var k=_.value;y(),p(k,u)}}catch(r){x.e(r)}finally{x.f()}}return this},text:function text(r,i,o,u){return this._text(r,i,o,u,this._line)},widthOfString:function widthOfString(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return this._font.widthOfString(r,this._fontSize,i.features)+(i.characterSpacing||0)*(r.length-1)},heightOfString:function heightOfString(r,i){var o=this,u=this.x,p=this.y;(i=this._initOptions(i)).height=1/0;var g=i.lineGap||this._lineGap||0;this._text(r,this.x,this.y,i,(function(){return o.y+=o.currentLineHeight(!0)+g}));var y=this.y-p;return this.x=u,this.y=p,y},list:function list(r,i,o,u,p){var g=this,y=(u=this._initOptions(i,o,u)).listType||"bullet",w=Math.round(this._font.ascender/1e3*this._fontSize),_=w/2,x=u.bulletRadius||w/3,k=u.textIndent||("bullet"===y?5*x:2*w),P=u.bulletIndent||("bullet"===y?8*x:2*w),O=1,I=[],B=[],R=[];!function flatten(r){for(var i=1,o=0;o<r.length;o++){var u=r[o];Array.isArray(u)?(O++,flatten(u),O--):(I.push(u),B.push(O),"bullet"!==y&&R.push(i++))}}(r);(p=new Qe(this,u)).on("line",this._line),O=1;var D=0;return p.on("firstLine",(function(){var r,i,o,w,I;if(u.structParent)if(u.structTypes){var N=_slicedToArray(u.structTypes,3);i=N[0],o=N[1],w=N[2]}else i="LI",o="Lbl",w="LBody";if(i?(r=g.struct(i),u.structParent.add(r)):u.structParent&&(r=u.structParent),(I=B[D++])!==O){var W=P*(I-O);g.x+=W,p.lineWidth-=W,O=I}switch(r&&(o||w)&&r.add(g.struct(o||w,[g.markStructureContent(o||w)])),y){case"bullet":g.circle(g.x-k+x,g.y+_,x),g.fill();break;case"numbered":case"lettered":var U=function label(r){switch(y){case"numbered":return"".concat(r,".");case"lettered":var i=String.fromCharCode((r-1)%26+65),o=Math.floor((r-1)/26+1),u=Array(o+1).join(i);return"".concat(u,".")}}(R[D-1]);g._fragment(U,g.x-k,g.y,u)}r&&o&&w&&r.add(g.struct(w,[g.markStructureContent(w)])),r&&r!==u.structParent&&r.end()})),p.on("sectionStart",(function(){var r=k+P*(O-1);return g.x+=r,p.lineWidth-=r})),p.on("sectionEnd",(function(){var r=k+P*(O-1);return g.x-=r,p.lineWidth+=r})),p.wrap(I.join("\n"),u),this},_initOptions:function _initOptions(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=arguments.length>1?arguments[1]:void 0,o=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};"object"==typeof r&&(o=r,r=null);var u=Object.assign({},o);if(this._textOptions)for(var p in this._textOptions){var g=this._textOptions[p];"continued"!==p&&void 0===u[p]&&(u[p]=g)}return null!=r&&(this.x=r),null!=i&&(this.y=i),!1!==u.lineBreak&&(null==u.width&&(u.width=this.page.width-this.x-this.page.margins.right),u.width=Math.max(u.width,0)),u.columns||(u.columns=0),null==u.columnGap&&(u.columnGap=18),u},_line:function _line(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=arguments.length>2?arguments[2]:void 0;this._fragment(r,this.x,this.y,i);var u=i.lineGap||this._lineGap||0;return o?this.y+=this.currentLineHeight(!0)+u:this.x+=this.widthOfString(r)},_fragment:function _fragment(r,i,o,u){var p,g,y,w,_,x,k=this;if(0!==(r="".concat(r).replace(/\n/g,"")).length){var P=u.align||"left",O=u.wordSpacing||0,I=u.characterSpacing||0;if(u.width)switch(P){case"right":_=this.widthOfString(r.replace(/\s+$/,""),u),i+=u.lineWidth-_;break;case"center":i+=u.lineWidth/2-u.textWidth/2;break;case"justify":x=r.trim().split(/\s+/),_=this.widthOfString(r.replace(/\s+/g,""),u);var B=this.widthOfString(" ")+I;O=Math.max(0,(u.lineWidth-_)/Math.max(1,x.length-1)-B)}if("number"==typeof u.baseline)p=-u.baseline;else{switch(u.baseline){case"svg-middle":p=.5*this._font.xHeight;break;case"middle":case"svg-central":p=.5*(this._font.descender+this._font.ascender);break;case"bottom":case"ideographic":p=this._font.descender;break;case"alphabetic":p=0;break;case"mathematical":p=.5*this._font.ascender;break;case"hanging":p=.8*this._font.ascender;break;default:p=this._font.ascender}p=p/1e3*this._fontSize}var R,D=u.textWidth+O*(u.wordCount-1)+I*(r.length-1);if(null!=u.link&&this.link(i,o,D,this.currentLineHeight(),u.link),null!=u.goTo&&this.goTo(i,o,D,this.currentLineHeight(),u.goTo),null!=u.destination&&this.addNamedDestination(u.destination,"XYZ",i,o,null),u.underline){this.save(),u.stroke||this.strokeColor.apply(this,_toConsumableArray(this._fillColor||[]));var N=this._fontSize<10?.5:Math.floor(this._fontSize/10);this.lineWidth(N);var W=o+this.currentLineHeight()-N;this.moveTo(i,W),this.lineTo(i+D,W),this.stroke(),this.restore()}if(u.strike){this.save(),u.stroke||this.strokeColor.apply(this,_toConsumableArray(this._fillColor||[]));var U=this._fontSize<10?.5:Math.floor(this._fontSize/10);this.lineWidth(U);var j=o+this.currentLineHeight()/2;this.moveTo(i,j),this.lineTo(i+D,j),this.stroke(),this.restore()}if(this.save(),u.oblique)R="number"==typeof u.oblique?-Math.tan(u.oblique*Math.PI/180):-.25,this.transform(1,0,0,1,i,o),this.transform(1,0,R,1,-R*p,0),this.transform(1,0,0,1,-i,-o);this.transform(1,0,0,-1,0,this.page.height),o=this.page.height-o-p,null==this.page.fonts[this._font.id]&&(this.page.fonts[this._font.id]=this._font.ref()),this.addContent("BT"),this.addContent("1 0 0 1 ".concat($e(i)," ").concat($e(o)," Tm")),this.addContent("/".concat(this._font.id," ").concat($e(this._fontSize)," Tf"));var G=u.fill&&u.stroke?2:u.stroke?1:0;if(G&&this.addContent("".concat(G," Tr")),I&&this.addContent("".concat($e(I)," Tc")),O){x=r.trim().split(/\s+/),O+=this.widthOfString(" ")+I,O*=1e3/this._fontSize,g=[],w=[];var X,K=_createForOfIteratorHelper(x);try{for(K.s();!(X=K.n()).done;){var Y=X.value,J=_slicedToArray(this._font.encode(Y,u.features),2),$=J[0],ee=J[1];g=g.concat($),w=w.concat(ee);var te={},re=w[w.length-1];for(var ne in re){var ie=re[ne];te[ne]=ie}te.xAdvance+=O,w[w.length-1]=te}}catch(r){K.e(r)}finally{K.f()}}else{var oe=_slicedToArray(this._font.encode(r,u.features),2);g=oe[0],w=oe[1]}var ae=this._fontSize/1e3,se=[],ce=0,le=!1,ue=function addSegment(r){if(ce<r){var i=g.slice(ce,r).join(""),o=w[r-1].xAdvance-w[r-1].advanceWidth;se.push("<".concat(i,"> ").concat($e(-o)))}return ce=r},fe=function flush(r){if(ue(r),se.length>0)return k.addContent("[".concat(se.join(" "),"] TJ")),se.length=0};for(y=0;y<w.length;y++){var he=w[y];he.xOffset||he.yOffset?(fe(y),this.addContent("1 0 0 1 ".concat($e(i+he.xOffset*ae)," ").concat($e(o+he.yOffset*ae)," Tm")),fe(y+1),le=!0):(le&&(this.addContent("1 0 0 1 ".concat($e(i)," ").concat($e(o)," Tm")),le=!1),he.xAdvance-he.advanceWidth!=0&&ue(y+1)),i+=he.xAdvance*ae}return fe(y),this.addContent("ET"),this.restore()}}},tt=[65472,65473,65474,65475,65477,65478,65479,65480,65481,65482,65483,65484,65485,65486,65487],rt={1:"DeviceGray",3:"DeviceRGB",4:"DeviceCMYK"},nt=function(){function JPEG(r,i){var o;if(_classCallCheck(this,JPEG),this.data=r,this.label=i,65496!==this.data.readUInt16BE(0))throw"SOI not found in JPEG";for(var u=2;u<this.data.length&&(o=this.data.readUInt16BE(u),u+=2,!tt.includes(o));)u+=this.data.readUInt16BE(u);if(!tt.includes(o))throw"Invalid JPEG.";u+=2,this.bits=this.data[u++],this.height=this.data.readUInt16BE(u),u+=2,this.width=this.data.readUInt16BE(u),u+=2;var p=this.data[u++];this.colorSpace=rt[p],this.obj=null}return _createClass(JPEG,[{key:"embed",value:function embed(r){if(!this.obj)return this.obj=r.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:this.bits,Width:this.width,Height:this.height,ColorSpace:this.colorSpace,Filter:"DCTDecode"}),"DeviceCMYK"===this.colorSpace&&(this.obj.data.Decode=[1,0,1,0,1,0,1,0]),this.obj.end(this.data),this.data=null}}]),JPEG}(),it=function(){function PNGImage(r,i){_classCallCheck(this,PNGImage),this.label=i,this.image=new k.default(r),this.width=this.image.width,this.height=this.image.height,this.imgData=this.image.imgData,this.obj=null}return _createClass(PNGImage,[{key:"embed",value:function embed(r){var i=!1;if(this.document=r,!this.obj){var o=this.image.hasAlphaChannel,p=1===this.image.interlaceMethod;if(this.obj=this.document.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:o?8:this.image.bits,Width:this.width,Height:this.height,Filter:"FlateDecode"}),!o){var g=this.document.ref({Predictor:p?1:15,Colors:this.image.colors,BitsPerComponent:this.image.bits,Columns:this.width});this.obj.data.DecodeParms=g,g.end()}if(0===this.image.palette.length)this.obj.data.ColorSpace=this.image.colorSpace;else{var y=this.document.ref();y.end(u.from(this.image.palette)),this.obj.data.ColorSpace=["Indexed","DeviceRGB",this.image.palette.length/3-1,y]}if(null!=this.image.transparency.grayscale){var w=this.image.transparency.grayscale;this.obj.data.Mask=[w,w]}else if(this.image.transparency.rgb){var _,x=[],k=_createForOfIteratorHelper(this.image.transparency.rgb);try{for(k.s();!(_=k.n()).done;){var P=_.value;x.push(P,P)}}catch(r){k.e(r)}finally{k.f()}this.obj.data.Mask=x}else{if(this.image.transparency.indexed)return i=!0,this.loadIndexedAlphaChannel();if(o)return i=!0,this.splitAlphaChannel()}if(p&&!i)return this.decodeData();this.finalize()}}},{key:"finalize",value:function finalize(){if(this.alphaChannel){var r=this.document.ref({Type:"XObject",Subtype:"Image",Height:this.height,Width:this.width,BitsPerComponent:8,Filter:"FlateDecode",ColorSpace:"DeviceGray",Decode:[0,1]});r.end(this.alphaChannel),this.obj.data.SMask=r}return this.obj.end(this.imgData),this.image=null,this.imgData=null}},{key:"splitAlphaChannel",value:function splitAlphaChannel(){var r=this;return this.image.decodePixels((function(i){for(var o,p,y=r.image.colors,w=r.width*r.height,_=u.alloc(w*y),x=u.alloc(w),k=p=o=0,P=i.length,O=16===r.image.bits?1:0;k<P;){for(var I=0;I<y;I++)_[p++]=i[k++],k+=O;x[o++]=i[k++],k+=O}return r.imgData=g.default.deflateSync(_),r.alphaChannel=g.default.deflateSync(x),r.finalize()}))}},{key:"loadIndexedAlphaChannel",value:function loadIndexedAlphaChannel(){var r=this,i=this.image.transparency.indexed;return this.image.decodePixels((function(o){for(var p=u.alloc(r.width*r.height),y=0,w=0,_=o.length;w<_;w++)p[y++]=i[o[w]];return r.alphaChannel=g.default.deflateSync(p),r.finalize()}))}},{key:"decodeData",value:function decodeData(){var r=this;this.image.decodePixels((function(i){r.imgData=g.default.deflateSync(i),r.finalize()}))}}]),PNGImage}(),ot=function(){function PDFImage(){_classCallCheck(this,PDFImage)}return _createClass(PDFImage,null,[{key:"open",value:function open(r,i){var o;if(u.isBuffer(r))o=r;else if(r instanceof ArrayBuffer)o=u.from(new Uint8Array(r));else{var p;if(p=/^data:.+;base64,(.*)$/.exec(r))o=u.from(p[1],"base64");else if(!(o=P.readFileSync(r)))return}if(255===o[0]&&216===o[1])return new nt(o,i);if(137===o[0]&&"PNG"===o.toString("ascii",1,4))return new it(o,i);throw new Error("Unknown image format.")}}]),PDFImage}(),at={initImages:function initImages(){return this._imageRegistry={},this._imageCount=0},image:function image(r,i,o){var u,p,g,image,y,w,_,x=arguments.length>3&&void 0!==arguments[3]?arguments[3]:{};"object"==typeof i&&(x=i,i=null),i=null!=(w=null!=i?i:x.x)?w:this.x,o=null!=(_=null!=o?o:x.y)?_:this.y,"string"==typeof r&&(image=this._imageRegistry[r]),image||(image=r.width&&r.height?r:this.openImage(r)),image.obj||image.embed(this),null==this.page.xobjects[image.label]&&(this.page.xobjects[image.label]=image.obj);var k=x.width||image.width,P=x.height||image.height;if(x.width&&!x.height){var O=k/image.width;k=image.width*O,P=image.height*O}else if(x.height&&!x.width){var I=P/image.height;k=image.width*I,P=image.height*I}else if(x.scale)k=image.width*x.scale,P=image.height*x.scale;else if(x.fit){var B=_slicedToArray(x.fit,2);p=(g=B[0])/(u=B[1]),(y=image.width/image.height)>p?(k=g,P=g/y):(P=u,k=u*y)}else if(x.cover){var R=_slicedToArray(x.cover,2);p=(g=R[0])/(u=R[1]),(y=image.width/image.height)>p?(P=u,k=u*y):(k=g,P=g/y)}return(x.fit||x.cover)&&("center"===x.align?i=i+g/2-k/2:"right"===x.align&&(i=i+g-k),"center"===x.valign?o=o+u/2-P/2:"bottom"===x.valign&&(o=o+u-P)),null!=x.link&&this.link(i,o,k,P,x.link),null!=x.goTo&&this.goTo(i,o,k,P,x.goTo),null!=x.destination&&this.addNamedDestination(x.destination,"XYZ",i,o,null),this.y===o&&(this.y+=P),this.save(),this.transform(k,0,0,-P,i,o+P),this.addContent("/".concat(image.label," Do")),this.restore(),this},openImage:function openImage(r){var i;return"string"==typeof r&&(i=this._imageRegistry[r]),i||(i=ot.open(r,"I".concat(++this._imageCount)),"string"==typeof r&&(this._imageRegistry[r]=i)),i}},st={annotate:function annotate(r,i,o,u,p){for(var g in p.Type="Annot",p.Rect=this._convertRect(r,i,o,u),p.Border=[0,0,0],"Link"===p.Subtype&&void 0===p.F&&(p.F=4),"Link"!==p.Subtype&&null==p.C&&(p.C=this._normalizeColor(p.color||[0,0,0])),delete p.color,"string"==typeof p.Dest&&(p.Dest=new String(p.Dest)),p){var y=p[g];p[g[0].toUpperCase()+g.slice(1)]=y}var w=this.ref(p);return this.page.annotations.push(w),w.end(),this},note:function note(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="Text",g.Contents=new String(p),g.Name="Comment",null==g.color&&(g.color=[243,223,92]),this.annotate(r,i,o,u,g)},goTo:function goTo(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="Link",g.A=this.ref({S:"GoTo",D:new String(p)}),g.A.end(),this.annotate(r,i,o,u,g)},link:function link(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};if(g.Subtype="Link","number"==typeof p){var y=this._root.data.Pages.data;if(!(p>=0&&p<y.Kids.length))throw new Error("The document has no page ".concat(p));g.A=this.ref({S:"GoTo",D:[y.Kids[p],"XYZ",null,null,null]}),g.A.end()}else g.A=this.ref({S:"URI",URI:new String(p)}),g.A.end();return this.annotate(r,i,o,u,g)},_markup:function _markup(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{},g=this._convertRect(r,i,o,u),y=_slicedToArray(g,4),w=y[0],_=y[1],x=y[2],k=y[3];return p.QuadPoints=[w,k,x,k,w,_,x,_],p.Contents=new String,this.annotate(r,i,o,u,p)},highlight:function highlight(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Highlight",null==p.color&&(p.color=[241,238,148]),this._markup(r,i,o,u,p)},underline:function underline(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Underline",this._markup(r,i,o,u,p)},strike:function strike(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="StrikeOut",this._markup(r,i,o,u,p)},lineAnnotation:function lineAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Line",p.Contents=new String,p.L=[r,this.page.height-i,o,this.page.height-u],this.annotate(r,i,o,u,p)},rectAnnotation:function rectAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Square",p.Contents=new String,this.annotate(r,i,o,u,p)},ellipseAnnotation:function ellipseAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Circle",p.Contents=new String,this.annotate(r,i,o,u,p)},textAnnotation:function textAnnotation(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="FreeText",g.Contents=new String(p),g.DA=new String,this.annotate(r,i,o,u,g)},fileAnnotation:function fileAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{},g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{},y=this.file(p.src,Object.assign({hidden:!0},p));return g.Subtype="FileAttachment",g.FS=y,g.Contents?g.Contents=new String(g.Contents):y.data.Desc&&(g.Contents=y.data.Desc),this.annotate(r,i,o,u,g)},_convertRect:function _convertRect(r,i,o,u){var p=i;i+=u;var g=r+o,y=_slicedToArray(this._ctm,6),w=y[0],_=y[1],x=y[2],k=y[3],P=y[4],O=y[5];return[r=w*r+x*i+P,i=_*r+k*i+O,g=w*g+x*p+P,p=_*g+k*p+O]}},ct=function(){function PDFOutline(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{expanded:!1};_classCallCheck(this,PDFOutline),this.document=r,this.options=p,this.outlineData={},null!==u&&(this.outlineData.Dest=[u.dictionary,"Fit"]),null!==i&&(this.outlineData.Parent=i),null!==o&&(this.outlineData.Title=new String(o)),this.dictionary=this.document.ref(this.outlineData),this.children=[]}return _createClass(PDFOutline,[{key:"addItem",value:function addItem(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{expanded:!1},o=new PDFOutline(this.document,this.dictionary,r,this.document.page,i);return this.children.push(o),o}},{key:"endOutline",value:function endOutline(){if(this.children.length>0){this.options.expanded&&(this.outlineData.Count=this.children.length);var r=this.children[0],i=this.children[this.children.length-1];this.outlineData.First=r.dictionary,this.outlineData.Last=i.dictionary;for(var o=0,u=this.children.length;o<u;o++){var p=this.children[o];o>0&&(p.outlineData.Prev=this.children[o-1].dictionary),o<this.children.length-1&&(p.outlineData.Next=this.children[o+1].dictionary),p.endOutline()}}return this.dictionary.end()}}]),PDFOutline}(),lt={initOutline:function initOutline(){return this.outline=new ct(this,null,null,null)},endOutline:function endOutline(){if(this.outline.endOutline(),this.outline.children.length>0)return this._root.data.Outlines=this.outline.dictionary,this._root.data.PageMode="UseOutlines"}},ut=function(){function PDFStructureContent(r,i){_classCallCheck(this,PDFStructureContent),this.refs=[{pageRef:r,mcid:i}]}return _createClass(PDFStructureContent,[{key:"push",value:function push(r){var i=this;r.refs.forEach((function(r){return i.refs.push(r)}))}}]),PDFStructureContent}(),ft=function(){function PDFStructureElement(r,i){var o=this,u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{},p=arguments.length>3&&void 0!==arguments[3]?arguments[3]:null;_classCallCheck(this,PDFStructureElement),this.document=r,this._attached=!1,this._ended=!1,this._flushed=!1,this.dictionary=r.ref({S:i});var g=this.dictionary.data;(Array.isArray(u)||this._isValidChild(u))&&(p=u,u={}),void 0!==u.title&&(g.T=new String(u.title)),void 0!==u.lang&&(g.Lang=new String(u.lang)),void 0!==u.alt&&(g.Alt=new String(u.alt)),void 0!==u.expanded&&(g.E=new String(u.expanded)),void 0!==u.actual&&(g.ActualText=new String(u.actual)),this._children=[],p&&(Array.isArray(p)||(p=[p]),p.forEach((function(r){return o.add(r)})),this.end())}return _createClass(PDFStructureElement,[{key:"add",value:function add(r){if(this._ended)throw new Error("Cannot add child to already-ended structure element");if(!this._isValidChild(r))throw new Error("Invalid structure element child");return r instanceof PDFStructureElement&&(r.setParent(this.dictionary),this._attached&&r.setAttached()),r instanceof ut&&this._addContentToParentTree(r),"function"==typeof r&&this._attached&&(r=this._contentForClosure(r)),this._children.push(r),this}},{key:"_addContentToParentTree",value:function _addContentToParentTree(r){var i=this;r.refs.forEach((function(r){var o=r.pageRef,u=r.mcid;i.document.getStructParentTree().get(o.data.StructParents)[u]=i.dictionary}))}},{key:"setParent",value:function setParent(r){if(this.dictionary.data.P)throw new Error("Structure element added to more than one parent");this.dictionary.data.P=r,this._flush()}},{key:"setAttached",value:function setAttached(){var r=this;this._attached||(this._children.forEach((function(i,o){i instanceof PDFStructureElement&&i.setAttached(),"function"==typeof i&&(r._children[o]=r._contentForClosure(i))})),this._attached=!0,this._flush())}},{key:"end",value:function end(){this._ended||(this._children.filter((function(r){return r instanceof PDFStructureElement})).forEach((function(r){return r.end()})),this._ended=!0,this._flush())}},{key:"_isValidChild",value:function _isValidChild(r){return r instanceof PDFStructureElement||r instanceof ut||"function"==typeof r}},{key:"_contentForClosure",value:function _contentForClosure(r){var i=this.document.markStructureContent(this.dictionary.data.S);return r(),this.document.endMarkedContent(),this._addContentToParentTree(i),i}},{key:"_isFlushable",value:function _isFlushable(){return!(!this.dictionary.data.P||!this._ended)&&this._children.every((function(r){return"function"!=typeof r&&(!(r instanceof PDFStructureElement)||r._isFlushable())}))}},{key:"_flush",value:function _flush(){var r=this;!this._flushed&&this._isFlushable()&&(this.dictionary.data.K=[],this._children.forEach((function(i){return r._flushChild(i)})),this.dictionary.end(),this._children=[],this.dictionary.data.K=null,this._flushed=!0)}},{key:"_flushChild",value:function _flushChild(r){var i=this;r instanceof PDFStructureElement&&this.dictionary.data.K.push(r.dictionary),r instanceof ut&&r.refs.forEach((function(r){var o=r.pageRef,u=r.mcid;i.dictionary.data.Pg||(i.dictionary.data.Pg=o),i.dictionary.data.Pg===o?i.dictionary.data.K.push(u):i.dictionary.data.K.push({Type:"MCR",Pg:o,MCID:u})}))}}]),PDFStructureElement}(),ht=function(r){_inherits(PDFNumberTree,r);var i=_createSuper(PDFNumberTree);function PDFNumberTree(){return _classCallCheck(this,PDFNumberTree),i.apply(this,arguments)}return _createClass(PDFNumberTree,[{key:"_compareKeys",value:function _compareKeys(r,i){return parseInt(r)-parseInt(i)}},{key:"_keysName",value:function _keysName(){return"Nums"}},{key:"_dataForKey",value:function _dataForKey(r){return parseInt(r)}}]),PDFNumberTree}(I),dt={initMarkings:function initMarkings(r){this.structChildren=[],r.tagged&&(this.getMarkInfoDictionary().data.Marked=!0,this.getStructTreeRoot())},markContent:function markContent(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:null;if("Artifact"===r||i&&i.mcid){var o=0;for(this.page.markings.forEach((function(r){(o||r.structContent||"Artifact"===r.tag)&&o++}));o--;)this.endMarkedContent()}if(!i)return this.page.markings.push({tag:r}),this.addContent("/".concat(r," BMC")),this;this.page.markings.push({tag:r,options:i});var u={};return void 0!==i.mcid&&(u.MCID=i.mcid),"Artifact"===r&&("string"==typeof i.type&&(u.Type=i.type),Array.isArray(i.bbox)&&(u.BBox=[i.bbox[0],this.page.height-i.bbox[3],i.bbox[2],this.page.height-i.bbox[1]]),Array.isArray(i.attached)&&i.attached.every((function(r){return"string"==typeof r}))&&(u.Attached=i.attached)),"Span"===r&&(i.lang&&(u.Lang=new String(i.lang)),i.alt&&(u.Alt=new String(i.alt)),i.expanded&&(u.E=new String(i.expanded)),i.actual&&(u.ActualText=new String(i.actual))),this.addContent("/".concat(r," ").concat(W.convert(u)," BDC")),this},markStructureContent:function markStructureContent(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=this.getStructParentTree().get(this.page.structParentTreeKey),u=o.length;o.push(null),this.markContent(r,_objectSpread2(_objectSpread2({},i),{},{mcid:u}));var p=new ut(this.page.dictionary,u);return this.page.markings.slice(-1)[0].structContent=p,p},endMarkedContent:function endMarkedContent(){return this.page.markings.pop(),this.addContent("EMC"),this},struct:function struct(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=arguments.length>2&&void 0!==arguments[2]?arguments[2]:null;return new ft(this,r,i,o)},addStructure:function addStructure(r){var i=this.getStructTreeRoot();return r.setParent(i),r.setAttached(),this.structChildren.push(r),i.data.K||(i.data.K=[]),i.data.K.push(r.dictionary),this},initPageMarkings:function initPageMarkings(r){var i=this;r.forEach((function(r){if(r.structContent){var o=r.structContent,u=i.markStructureContent(r.tag,r.options);o.push(u),i.page.markings.slice(-1)[0].structContent=o}else i.markContent(r.tag,r.options)}))},endPageMarkings:function endPageMarkings(r){var i=r.markings;return i.forEach((function(){return r.write("EMC")})),r.markings=[],i},getMarkInfoDictionary:function getMarkInfoDictionary(){return this._root.data.MarkInfo||(this._root.data.MarkInfo=this.ref({})),this._root.data.MarkInfo},getStructTreeRoot:function getStructTreeRoot(){return this._root.data.StructTreeRoot||(this._root.data.StructTreeRoot=this.ref({Type:"StructTreeRoot",ParentTree:new ht,ParentTreeNextKey:0})),this._root.data.StructTreeRoot},getStructParentTree:function getStructParentTree(){return this.getStructTreeRoot().data.ParentTree},createStructParentTreeNextKey:function createStructParentTreeNextKey(){this.getMarkInfoDictionary();var r=this.getStructTreeRoot(),i=r.data.ParentTreeNextKey++;return r.data.ParentTree.add(i,[]),i},endMarkings:function endMarkings(){var r=this._root.data.StructTreeRoot;r&&(r.end(),this.structChildren.forEach((function(r){return r.end()}))),this._root.data.MarkInfo&&this._root.data.MarkInfo.end()}},pt={readOnly:1,required:2,noExport:4,multiline:4096,password:8192,toggleToOffButton:16384,radioButton:32768,pushButton:65536,combo:131072,edit:262144,sort:524288,multiSelect:2097152,noSpell:4194304},gt={left:0,center:1,right:2},yt={value:"V",defaultValue:"DV"},vt={zip:"0",zipPlus4:"1",zip4:"1",phone:"2",ssn:"3"},mt={nDec:0,sepComma:!1,negStyle:"MinusBlack",currency:"",currencyPrepend:!0},bt={nDec:0,sepComma:!1},wt={initForm:function initForm(){if(!this._font)throw new Error("Must set a font before calling initForm method");this._acroform={fonts:{},defaultFont:this._font.name},this._acroform.fonts[this._font.id]=this._font.ref();var r={Fields:[],NeedAppearances:!0,DA:new String("/".concat(this._font.id," 0 Tf 0 g")),DR:{Font:{}}};r.DR.Font[this._font.id]=this._font.ref();var i=this.ref(r);return this._root.data.AcroForm=i,this},endAcroForm:function endAcroForm(){var r=this;if(this._root.data.AcroForm){if(!Object.keys(this._acroform.fonts).length&&!this._acroform.defaultFont)throw new Error("No fonts specified for PDF form");var i=this._root.data.AcroForm.data.DR.Font;Object.keys(this._acroform.fonts).forEach((function(o){i[o]=r._acroform.fonts[o]})),this._root.data.AcroForm.data.Fields.forEach((function(i){r._endChild(i)})),this._root.data.AcroForm.end()}return this},_endChild:function _endChild(r){var i=this;return Array.isArray(r.data.Kids)&&(r.data.Kids.forEach((function(r){i._endChild(r)})),r.end()),this},formField:function formField(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=this._fieldDict(r,null,i),u=this.ref(o);return this._addToParent(u),u},formAnnotation:function formAnnotation(r,i,o,u,p,g){var y=arguments.length>6&&void 0!==arguments[6]?arguments[6]:{},w=this._fieldDict(r,i,y);w.Subtype="Widget",void 0===w.F&&(w.F=4),this.annotate(o,u,p,g,w);var _=this.page.annotations[this.page.annotations.length-1];return this._addToParent(_)},formText:function formText(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"text",i,o,u,p,g)},formPushButton:function formPushButton(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"pushButton",i,o,u,p,g)},formCombo:function formCombo(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"combo",i,o,u,p,g)},formList:function formList(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"list",i,o,u,p,g)},formRadioButton:function formRadioButton(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"radioButton",i,o,u,p,g)},formCheckbox:function formCheckbox(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"checkbox",i,o,u,p,g)},_addToParent:function _addToParent(r){var i=r.data.Parent;return i?(i.data.Kids||(i.data.Kids=[]),i.data.Kids.push(r)):this._root.data.AcroForm.data.Fields.push(r),this},_fieldDict:function _fieldDict(r,i){var o=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};if(!this._acroform)throw new Error("Call document.initForms() method before adding form elements to document");var u=Object.assign({},o);return null!==i&&(u=this._resolveType(i,o)),u=this._resolveFlags(u),u=this._resolveJustify(u),u=this._resolveFont(u),u=this._resolveStrings(u),u=this._resolveColors(u),(u=this._resolveFormat(u)).T=new String(r),u.parent&&(u.Parent=u.parent,delete u.parent),u},_resolveType:function _resolveType(r,i){if("text"===r)i.FT="Tx";else if("pushButton"===r)i.FT="Btn",i.pushButton=!0;else if("radioButton"===r)i.FT="Btn",i.radioButton=!0;else if("checkbox"===r)i.FT="Btn";else if("combo"===r)i.FT="Ch",i.combo=!0;else{if("list"!==r)throw new Error("Invalid form annotation type '".concat(r,"'"));i.FT="Ch"}return i},_resolveFormat:function _resolveFormat(r){var i=r.format;if(i&&i.type){var o,u,p="";if(void 0!==vt[i.type])o="AFSpecial_Keystroke",u="AFSpecial_Format",p=vt[i.type];else{var g=i.type.charAt(0).toUpperCase()+i.type.slice(1);if(o="AF".concat(g,"_Keystroke"),u="AF".concat(g,"_Format"),"date"===i.type)o+="Ex",p=String(i.param);else if("time"===i.type)p=String(i.param);else if("number"===i.type){var y=Object.assign({},mt,i);p=String([String(y.nDec),y.sepComma?"0":"1",'"'+y.negStyle+'"',"null",'"'+y.currency+'"',String(y.currencyPrepend)].join(","))}else if("percent"===i.type){var w=Object.assign({},bt,i);p=String([String(w.nDec),w.sepComma?"0":"1"].join(","))}}r.AA=r.AA?r.AA:{},r.AA.K={S:"JavaScript",JS:new String("".concat(o,"(").concat(p,");"))},r.AA.F={S:"JavaScript",JS:new String("".concat(u,"(").concat(p,");"))}}return delete r.format,r},_resolveColors:function _resolveColors(r){var i=this._normalizeColor(r.backgroundColor);return i&&(r.MK||(r.MK={}),r.MK.BG=i),(i=this._normalizeColor(r.borderColor))&&(r.MK||(r.MK={}),r.MK.BC=i),delete r.backgroundColor,delete r.borderColor,r},_resolveFlags:function _resolveFlags(r){var i=0;return Object.keys(r).forEach((function(o){pt[o]&&(i|=pt[o],delete r[o])})),0!==i&&(r.Ff=r.Ff?r.Ff:0,r.Ff|=i),r},_resolveJustify:function _resolveJustify(r){var i=0;return void 0!==r.align&&("number"==typeof gt[r.align]&&(i=gt[r.align]),delete r.align),0!==i&&(r.Q=i),r},_resolveFont:function _resolveFont(r){if(null===this._acroform.fonts[this._font.id]&&(this._acroform.fonts[this._font.id]=this._font.ref()),this._acroform.defaultFont!==this._font.name){r.DR={Font:{}};var i=r.fontSize||0;r.DR.Font[this._font.id]=this._font.ref(),r.DA=new String("/".concat(this._font.id," ").concat(i," Tf 0 g"))}return r},_resolveStrings:function _resolveStrings(r){var i=[];function appendChoices(r){if(Array.isArray(r))for(var o=0;o<r.length;o++)"string"==typeof r[o]?i.push(new String(r[o])):i.push(r[o])}return appendChoices(r.Opt),r.select&&(appendChoices(r.select),delete r.select),i.length&&(r.Opt=i),Object.keys(yt).forEach((function(i){void 0!==r[i]&&(r[yt[i]]=r[i],delete r[i])})),["V","DV"].forEach((function(i){"string"==typeof r[i]&&(r[i]=new String(r[i]))})),r.MK&&r.MK.CA&&(r.MK.CA=new String(r.MK.CA)),r.label&&(r.MK=r.MK?r.MK:{},r.MK.CA=new String(r.label),delete r.label),r}},_t={file:function file(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};i.name=i.name||r;var o,p={Type:"EmbeddedFile",Params:{}};if(!r)throw new Error("No src specified");if(u.isBuffer(r))o=r;else if(r instanceof ArrayBuffer)o=u.from(new Uint8Array(r));else{var g;if(g=/^data:(.*);base64,(.*)$/.exec(r))g[1]&&(p.Subtype=g[1].replace("/","#2F")),o=u.from(g[2],"base64");else{if(!(o=P.readFileSync(r)))throw new Error("Could not read contents of file at filepath ".concat(r));var w=P.statSync(r),_=w.birthtime,x=w.ctime;p.Params.CreationDate=_,p.Params.ModDate=x}}i.creationDate instanceof Date&&(p.Params.CreationDate=i.creationDate),i.modifiedDate instanceof Date&&(p.Params.ModDate=i.modifiedDate),i.type&&(p.Subtype=i.type.replace("/","#2F"));var k,O=y.default.MD5(y.default.lib.WordArray.create(new Uint8Array(o)));p.Params.CheckSum=new String(O),p.Params.Size=o.byteLength,this._fileRegistry||(this._fileRegistry={});var file=this._fileRegistry[i.name];file&&isEqual(p,file)?k=file.ref:((k=this.ref(p)).end(o),this._fileRegistry[i.name]=_objectSpread2(_objectSpread2({},p),{},{ref:k}));var I={Type:"Filespec",F:new String(i.name),EF:{F:k},UF:new String(i.name)};i.description&&(I.Desc=new String(i.description));var B=this.ref(I);return B.end(),i.hidden||this.addNamedEmbeddedFile(i.name,B),B}};function isEqual(r,i){return r.Subtype===i.Subtype&&r.Params.CheckSum.toString()===i.Params.CheckSum.toString()&&r.Params.Size===i.Params.Size&&r.Params.CreationDate===i.Params.CreationDate&&r.Params.ModDate===i.Params.ModDate}var St=function(r){_inherits(PDFDocument,r);var i=_createSuper(PDFDocument);function PDFDocument(){var r,o=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{};switch(_classCallCheck(this,PDFDocument),(r=i.call(this,o)).options=o,o.pdfVersion){case"1.4":r.version=1.4;break;case"1.5":r.version=1.5;break;case"1.6":r.version=1.6;break;case"1.7":case"1.7ext3":r.version=1.7;break;default:r.version=1.3}r.compress=null==r.options.compress||r.options.compress,r._pageBuffer=[],r._pageBufferStart=0,r._offsets=[],r._waiting=0,r._ended=!1,r._offset=0;var u=r.ref({Type:"Pages",Count:0,Kids:[]}),p=r.ref({Dests:new K});if(r._root=r.ref({Type:"Catalog",Pages:u,Names:p}),r.options.lang&&(r._root.data.Lang=new String(r.options.lang)),r.page=null,r.initColor(),r.initVector(),r.initFonts(o.font),r.initText(),r.initImages(),r.initOutline(),r.initMarkings(o),r.info={Producer:"PDFKit",Creator:"PDFKit",CreationDate:new Date},r.options.info)for(var g in r.options.info){var y=r.options.info[g];r.info[g]=y}return r.options.displayTitle&&(r._root.data.ViewerPreferences=r.ref({DisplayDocTitle:!0})),r._id=pe.generateFileID(r.info),r._security=pe.create(_assertThisInitialized(r),o),r._write("%PDF-".concat(r.version)),r._write("%ÿÿÿÿ"),!1!==r.options.autoFirstPage&&r.addPage(),r}return _createClass(PDFDocument,[{key:"addPage",value:function addPage(r){null==r&&(r=this.options),this.options.bufferPages||this.flushPages(),this.page=new X(this,r),this._pageBuffer.push(this.page);var i=this._root.data.Pages.data;return i.Kids.push(this.page.dictionary),i.Count++,this.x=this.page.margins.left,this.y=this.page.margins.top,this._ctm=[1,0,0,1,0,0],this.transform(1,0,0,-1,0,this.page.height),this.emit("pageAdded"),this}},{key:"continueOnNewPage",value:function continueOnNewPage(r){var i=this.endPageMarkings(this.page);return this.addPage(r),this.initPageMarkings(i),this}},{key:"bufferedPageRange",value:function bufferedPageRange(){return{start:this._pageBufferStart,count:this._pageBuffer.length}}},{key:"switchToPage",value:function switchToPage(r){var i;if(!(i=this._pageBuffer[r-this._pageBufferStart]))throw new Error("switchToPage(".concat(r,") out of bounds, current buffer covers pages ").concat(this._pageBufferStart," to ").concat(this._pageBufferStart+this._pageBuffer.length-1));return this.page=i}},{key:"flushPages",value:function flushPages(){var r=this._pageBuffer;this._pageBuffer=[],this._pageBufferStart+=r.length;var i,o=_createForOfIteratorHelper(r);try{for(o.s();!(i=o.n()).done;){var u=i.value;this.endPageMarkings(u),u.end()}}catch(r){o.e(r)}finally{o.f()}}},{key:"addNamedDestination",value:function addNamedDestination(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];0===o.length&&(o=["XYZ",null,null,null]),"XYZ"===o[0]&&null!==o[2]&&(o[2]=this.page.height-o[2]),o.unshift(this.page.dictionary),this._root.data.Names.data.Dests.add(r,o)}},{key:"addNamedEmbeddedFile",value:function addNamedEmbeddedFile(r,i){this._root.data.Names.data.EmbeddedFiles||(this._root.data.Names.data.EmbeddedFiles=new K({limits:!1})),this._root.data.Names.data.EmbeddedFiles.add(r,i)}},{key:"addNamedJavaScript",value:function addNamedJavaScript(r,i){this._root.data.Names.data.JavaScript||(this._root.data.Names.data.JavaScript=new K);var o={JS:new String(i),S:"JavaScript"};this._root.data.Names.data.JavaScript.add(r,o)}},{key:"ref",value:function ref(r){var ref=new U(this,this._offsets.length+1,r);return this._offsets.push(null),this._waiting++,ref}},{key:"_read",value:function _read(){}},{key:"_write",value:function _write(r){return u.isBuffer(r)||(r=u.from(r+"\n","binary")),this.push(r),this._offset+=r.length}},{key:"addContent",value:function addContent(r){return this.page.write(r),this}},{key:"_refEnd",value:function _refEnd(r){if(this._offsets[r.id-1]=r.offset,0==--this._waiting&&this._ended)return this._finalize(),this._ended=!1}},{key:"write",value:function write(r,i){new Error("PDFDocument#write is deprecated, and will be removed in a future version of PDFKit. Please pipe the document into a Node stream.");return this.pipe(P.createWriteStream(r)),this.end(),this.once("end",i)}},{key:"end",value:function end(){for(var r in this.flushPages(),this._info=this.ref(),this.info){var i=this.info[r];"string"==typeof i&&(i=new String(i));var o=this.ref(i);o.end(),this._info.data[r]=o}for(var u in this._info.end(),this._fontFamilies){this._fontFamilies[u].finalize()}return this.endOutline(),this.endMarkings(),this._root.end(),this._root.data.Pages.end(),this._root.data.Names.end(),this.endAcroForm(),this._root.data.ViewerPreferences&&this._root.data.ViewerPreferences.end(),this._security&&this._security.end(),0===this._waiting?this._finalize():this._ended=!0}},{key:"_finalize",value:function _finalize(){var r=this._offset;this._write("xref"),this._write("0 ".concat(this._offsets.length+1)),this._write("0000000000 65535 f ");var i,o=_createForOfIteratorHelper(this._offsets);try{for(o.s();!(i=o.n()).done;){var u=i.value;u="0000000000".concat(u).slice(-10),this._write(u+" 00000 n ")}}catch(r){o.e(r)}finally{o.f()}var p={Size:this._offsets.length+1,Root:this._root,Info:this._info,ID:[this._id,this._id]};return this._security&&(p.Encrypt=this._security.dictionary),this._write("trailer"),this._write(W.convert(p)),this._write("startxref"),this._write("".concat(r)),this._write("%%EOF"),this.push(null)}},{key:"toString",value:function toString(){return"[object PDFDocument]"}}]),PDFDocument}(p.default.Readable),xt=function mixin(r){Object.assign(St.prototype,r)};xt(Ie),xt(ze),xt(Je),xt(et),xt(at),xt(st),xt(lt),xt(dt),xt(wt),xt(_t),St.LineWrapper=Qe;var Ct=St;i.default=Ct},9160:function(r,i,o){"use strict";Object.defineProperty(i,"__esModule",{value:!0}),i.default=void 0,o(4256);var u=_interopRequireDefault(o(4781)),p=_interopRequireDefault(o(9742));function _interopRequireDefault(r){return r&&r.__esModule?r:{default:r}}var g={categories:["Cc","Zs","Po","Sc","Ps","Pe","Sm","Pd","Nd","Lu","Sk","Pc","Ll","So","Lo","Pi","Cf","No","Pf","Lt","Lm","Mn","Me","Mc","Nl","Zl","Zp","Cs","Co"],combiningClasses:["Not_Reordered","Above","Above_Right","Below","Attached_Above_Right","Attached_Below","Overlay","Iota_Subscript","Double_Below","Double_Above","Below_Right","Above_Left","CCC10","CCC11","CCC12","CCC13","CCC14","CCC15","CCC16","CCC17","CCC18","CCC19","CCC20","CCC21","CCC22","CCC23","CCC24","CCC25","CCC30","CCC31","CCC32","CCC27","CCC28","CCC29","CCC33","CCC34","CCC35","CCC36","Nukta","Virama","CCC84","CCC91","CCC103","CCC107","CCC118","CCC122","CCC129","CCC130","CCC132","Attached_Above","Below_Left","Left","Kana_Voicing","CCC26","Right"],scripts:["Common","Latin","Bopomofo","Inherited","Greek","Coptic","Cyrillic","Armenian","Hebrew","Arabic","Syriac","Thaana","Nko","Samaritan","Mandaic","Devanagari","Bengali","Gurmukhi","Gujarati","Oriya","Tamil","Telugu","Kannada","Malayalam","Sinhala","Thai","Lao","Tibetan","Myanmar","Georgian","Hangul","Ethiopic","Cherokee","Canadian_Aboriginal","Ogham","Runic","Tagalog","Hanunoo","Buhid","Tagbanwa","Khmer","Mongolian","Limbu","Tai_Le","New_Tai_Lue","Buginese","Tai_Tham","Balinese","Sundanese","Batak","Lepcha","Ol_Chiki","Braille","Glagolitic","Tifinagh","Han","Hiragana","Katakana","Yi","Lisu","Vai","Bamum","Syloti_Nagri","Phags_Pa","Saurashtra","Kayah_Li","Rejang","Javanese","Cham","Tai_Viet","Meetei_Mayek","null","Linear_B","Lycian","Carian","Old_Italic","Gothic","Old_Permic","Ugaritic","Old_Persian","Deseret","Shavian","Osmanya","Osage","Elbasan","Caucasian_Albanian","Linear_A","Cypriot","Imperial_Aramaic","Palmyrene","Nabataean","Hatran","Phoenician","Lydian","Meroitic_Hieroglyphs","Meroitic_Cursive","Kharoshthi","Old_South_Arabian","Old_North_Arabian","Manichaean","Avestan","Inscriptional_Parthian","Inscriptional_Pahlavi","Psalter_Pahlavi","Old_Turkic","Old_Hungarian","Hanifi_Rohingya","Old_Sogdian","Sogdian","Elymaic","Brahmi","Kaithi","Sora_Sompeng","Chakma","Mahajani","Sharada","Khojki","Multani","Khudawadi","Grantha","Newa","Tirhuta","Siddham","Modi","Takri","Ahom","Dogra","Warang_Citi","Nandinagari","Zanabazar_Square","Soyombo","Pau_Cin_Hau","Bhaiksuki","Marchen","Masaram_Gondi","Gunjala_Gondi","Makasar","Cuneiform","Egyptian_Hieroglyphs","Anatolian_Hieroglyphs","Mro","Bassa_Vah","Pahawh_Hmong","Medefaidrin","Miao","Tangut","Nushu","Duployan","SignWriting","Nyiakeng_Puachue_Hmong","Wancho","Mende_Kikakui","Adlam"],eaw:["N","Na","A","W","H","F"]},y="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",w=Math.log2||function(r){return Math.log(r)/Math.LN2},_=function bits(r){return w(r)+1|0},x=function buildUnicodeProperties(r,i){var o=_(r.categories.length-1),u=_(r.combiningClasses.length-1),p=_(r.scripts.length-1),g=_(r.eaw.length-1),y=u+p+g+10,w=p+g+10,x=g+10,k=(1<<o)-1,P=(1<<u)-1,O=(1<<p)-1,I=(1<<g)-1,B=function getCategory(o){var u=i.get(o);return r.categories[u>>y&k]};return{getCategory:B,getCombiningClass:function getCombiningClass(o){var u=i.get(o);return r.combiningClasses[u>>w&P]},getScript:function getScript(o){var u=i.get(o);return r.scripts[u>>x&O]},getEastAsianWidth:function getEastAsianWidth(o){var u=i.get(o);return r.eaw[u>>10&I]},getNumericValue:function getNumericValue(r){var o=i.get(r),u=1023&o;if(0===u)return null;if(u<=50)return u-1;if(u<480)return((u>>4)-12)/(1+(15&u));if(u<768){o=(u>>5)-14;for(var p=2+(31&u);p>0;)o*=10,p--;return o}o=(u>>2)-191;for(var g=1+(3&u);g>0;)o*=60,g--;return o},isAlphabetic:function isAlphabetic(r){var i=B(r);return"Lu"===i||"Ll"===i||"Lt"===i||"Lm"===i||"Lo"===i||"Nl"===i},isDigit:function isDigit(r){return"Nd"===B(r)},isPunctuation:function isPunctuation(r){var i=B(r);return"Pc"===i||"Pd"===i||"Pe"===i||"Pf"===i||"Pi"===i||"Po"===i||"Ps"===i},isLowerCase:function isLowerCase(r){return"Ll"===B(r)},isUpperCase:function isUpperCase(r){return"Lu"===B(r)},isTitleCase:function isTitleCase(r){return"Lt"===B(r)},isWhiteSpace:function isWhiteSpace(r){var i=B(r);return"Zs"===i||"Zl"===i||"Zp"===i},isBaseForm:function isBaseForm(r){var i=B(r);return"Nd"===i||"No"===i||"Nl"===i||"Lu"===i||"Ll"===i||"Lt"===i||"Lm"===i||"Lo"===i||"Me"===i||"Mc"===i},isMark:function isMark(r){var i=B(r);return"Mn"===i||"Me"===i||"Mc"===i}}}(g,new u.default(p.default.toByteArray(y))),k=x;i.default=k},4781:function(r,i,o){"use strict";o(7042),o(6992),o(1539),o(2472),o(2990),o(8927),o(3105),o(5035),o(4345),o(7174),o(2846),o(4731),o(7209),o(6319),o(8867),o(7789),o(3739),o(9368),o(4483),o(2056),o(3462),o(678),o(7462),o(3824),o(5021),o(2974),o(5016),o(9135);var u=o(311),p=o(1753).swap32LE,g=function(){function UnicodeTrie(r){var i="function"==typeof r.readUInt32BE&&"function"==typeof r.slice;if(i||r instanceof Uint8Array){var o;if(i)this.highStart=r.readUInt32LE(0),this.errorValue=r.readUInt32LE(4),o=r.readUInt32LE(8),r=r.slice(12);else{var g=new DataView(r.buffer);this.highStart=g.getUint32(0,!0),this.errorValue=g.getUint32(4,!0),o=g.getUint32(8,!0),r=r.subarray(12)}r=u(r,new Uint8Array(o)),r=u(r,new Uint8Array(o)),p(r),this.data=new Uint32Array(r.buffer)}else{var y=r;this.data=y.data,this.highStart=y.highStart,this.errorValue=y.errorValue}}return UnicodeTrie.prototype.get=function get(r){var i;return r<0||r>1114111?this.errorValue:r<55296||r>56319&&r<=65535?(i=(this.data[r>>5]<<2)+(31&r),this.data[i]):r<=65535?(i=(this.data[2048+(r-55296>>5)]<<2)+(31&r),this.data[i]):r<this.highStart?(i=this.data[2080+(r>>11)],i=((i=this.data[i+(r>>5&63)])<<2)+(31&r),this.data[i]):this.data[this.data.length-4]},UnicodeTrie}();r.exports=g},1753:function(r,i,o){"use strict";o(6992),o(1539),o(2472),o(2990),o(8927),o(3105),o(5035),o(4345),o(7174),o(2846),o(4731),o(7209),o(6319),o(8867),o(7789),o(3739),o(9368),o(4483),o(2056),o(3462),o(678),o(7462),o(3824),o(5021),o(2974),o(5016),o(9135);var u=18===new Uint8Array(new Uint32Array([305419896]).buffer)[0],p=function swap(r,i,o){var u=r[i];r[i]=r[o],r[o]=u};r.exports={swap32LE:function swap32LE(r){u&&function swap32(r){for(var i=r.length,o=0;o<i;o+=4)p(r,o,o+3),p(r,o+1,o+2)}(r)}}},8071:function(r,i,o){"use strict";var u=o(8519);r.exports=u},8519:function(r,i,o){"use strict";r=o.nmd(r),o(8309),o(7941),o(3210),o(4916),o(4723),o(3123),o(3728),o(5306),o(7042),o(9653),o(2222),o(6992),o(1539),o(3948),o(9254),o(3290);r&&void 0!==r.exports&&(r.exports=function SVGtoPDF(r,i,o,u,p){var g={aliceblue:[240,248,255],antiquewhite:[250,235,215],aqua:[0,255,255],aquamarine:[127,255,212],azure:[240,255,255],beige:[245,245,220],bisque:[255,228,196],black:[0,0,0],blanchedalmond:[255,235,205],blue:[0,0,255],blueviolet:[138,43,226],brown:[165,42,42],burlywood:[222,184,135],cadetblue:[95,158,160],chartreuse:[127,255,0],chocolate:[210,105,30],coral:[255,127,80],cornflowerblue:[100,149,237],cornsilk:[255,248,220],crimson:[220,20,60],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgoldenrod:[184,134,11],darkgray:[169,169,169],darkgrey:[169,169,169],darkgreen:[0,100,0],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkseagreen:[143,188,143],darkslateblue:[72,61,139],darkslategray:[47,79,79],darkslategrey:[47,79,79],darkturquoise:[0,206,209],darkviolet:[148,0,211],deeppink:[255,20,147],deepskyblue:[0,191,255],dimgray:[105,105,105],dimgrey:[105,105,105],dodgerblue:[30,144,255],firebrick:[178,34,34],floralwhite:[255,250,240],forestgreen:[34,139,34],fuchsia:[255,0,255],gainsboro:[220,220,220],ghostwhite:[248,248,255],gold:[255,215,0],goldenrod:[218,165,32],gray:[128,128,128],grey:[128,128,128],green:[0,128,0],greenyellow:[173,255,47],honeydew:[240,255,240],hotpink:[255,105,180],indianred:[205,92,92],indigo:[75,0,130],ivory:[255,255,240],khaki:[240,230,140],lavender:[230,230,250],lavenderblush:[255,240,245],lawngreen:[124,252,0],lemonchiffon:[255,250,205],lightblue:[173,216,230],lightcoral:[240,128,128],lightcyan:[224,255,255],lightgoldenrodyellow:[250,250,210],lightgray:[211,211,211],lightgrey:[211,211,211],lightgreen:[144,238,144],lightpink:[255,182,193],lightsalmon:[255,160,122],lightseagreen:[32,178,170],lightskyblue:[135,206,250],lightslategray:[119,136,153],lightslategrey:[119,136,153],lightsteelblue:[176,196,222],lightyellow:[255,255,224],lime:[0,255,0],limegreen:[50,205,50],linen:[250,240,230],magenta:[255,0,255],maroon:[128,0,0],mediumaquamarine:[102,205,170],mediumblue:[0,0,205],mediumorchid:[186,85,211],mediumpurple:[147,112,219],mediumseagreen:[60,179,113],mediumslateblue:[123,104,238],mediumspringgreen:[0,250,154],mediumturquoise:[72,209,204],mediumvioletred:[199,21,133],midnightblue:[25,25,112],mintcream:[245,255,250],mistyrose:[255,228,225],moccasin:[255,228,181],navajowhite:[255,222,173],navy:[0,0,128],oldlace:[253,245,230],olive:[128,128,0],olivedrab:[107,142,35],orange:[255,165,0],orangered:[255,69,0],orchid:[218,112,214],palegoldenrod:[238,232,170],palegreen:[152,251,152],paleturquoise:[175,238,238],palevioletred:[219,112,147],papayawhip:[255,239,213],peachpuff:[255,218,185],peru:[205,133,63],pink:[255,192,203],plum:[221,160,221],powderblue:[176,224,230],purple:[128,0,128],rebeccapurple:[102,51,153],red:[255,0,0],rosybrown:[188,143,143],royalblue:[65,105,225],saddlebrown:[139,69,19],salmon:[250,128,114],sandybrown:[244,164,96],seagreen:[46,139,87],seashell:[255,245,238],sienna:[160,82,45],silver:[192,192,192],skyblue:[135,206,235],slateblue:[106,90,205],slategray:[112,128,144],slategrey:[112,128,144],snow:[255,250,250],springgreen:[0,255,127],steelblue:[70,130,180],tan:[210,180,140],teal:[0,128,128],thistle:[216,191,216],tomato:[255,99,71],turquoise:[64,224,208],violet:[238,130,238],wheat:[245,222,179],white:[255,255,255],whitesmoke:[245,245,245],yellow:[255,255,0]},y={black:[g.black,1],white:[g.white,1],transparent:[g.black,0]},w={quot:34,amp:38,lt:60,gt:62,apos:39,OElig:338,oelig:339,Scaron:352,scaron:353,Yuml:376,circ:710,tilde:732,ensp:8194,emsp:8195,thinsp:8201,zwnj:8204,zwj:8205,lrm:8206,rlm:8207,ndash:8211,mdash:8212,lsquo:8216,rsquo:8217,sbquo:8218,ldquo:8220,rdquo:8221,bdquo:8222,dagger:8224,Dagger:8225,permil:8240,lsaquo:8249,rsaquo:8250,euro:8364,nbsp:160,iexcl:161,cent:162,pound:163,curren:164,yen:165,brvbar:166,sect:167,uml:168,copy:169,ordf:170,laquo:171,not:172,shy:173,reg:174,macr:175,deg:176,plusmn:177,sup2:178,sup3:179,acute:180,micro:181,para:182,middot:183,cedil:184,sup1:185,ordm:186,raquo:187,frac14:188,frac12:189,frac34:190,iquest:191,Agrave:192,Aacute:193,Acirc:194,Atilde:195,Auml:196,Aring:197,AElig:198,Ccedil:199,Egrave:200,Eacute:201,Ecirc:202,Euml:203,Igrave:204,Iacute:205,Icirc:206,Iuml:207,ETH:208,Ntilde:209,Ograve:210,Oacute:211,Ocirc:212,Otilde:213,Ouml:214,times:215,Oslash:216,Ugrave:217,Uacute:218,Ucirc:219,Uuml:220,Yacute:221,THORN:222,szlig:223,agrave:224,aacute:225,acirc:226,atilde:227,auml:228,aring:229,aelig:230,ccedil:231,egrave:232,eacute:233,ecirc:234,euml:235,igrave:236,iacute:237,icirc:238,iuml:239,eth:240,ntilde:241,ograve:242,oacute:243,ocirc:244,otilde:245,ouml:246,divide:247,oslash:248,ugrave:249,uacute:250,ucirc:251,uuml:252,yacute:253,thorn:254,yuml:255,fnof:402,Alpha:913,Beta:914,Gamma:915,Delta:916,Epsilon:917,Zeta:918,Eta:919,Theta:920,Iota:921,Kappa:922,Lambda:923,Mu:924,Nu:925,Xi:926,Omicron:927,Pi:928,Rho:929,Sigma:931,Tau:932,Upsilon:933,Phi:934,Chi:935,Psi:936,Omega:937,alpha:945,beta:946,gamma:947,delta:948,epsilon:949,zeta:950,eta:951,theta:952,iota:953,kappa:954,lambda:955,mu:956,nu:957,xi:958,omicron:959,pi:960,rho:961,sigmaf:962,sigma:963,tau:964,upsilon:965,phi:966,chi:967,psi:968,omega:969,thetasym:977,upsih:978,piv:982,bull:8226,hellip:8230,prime:8242,Prime:8243,oline:8254,frasl:8260,weierp:8472,image:8465,real:8476,trade:8482,alefsym:8501,larr:8592,uarr:8593,rarr:8594,darr:8595,harr:8596,crarr:8629,lArr:8656,uArr:8657,rArr:8658,dArr:8659,hArr:8660,forall:8704,part:8706,exist:8707,empty:8709,nabla:8711,isin:8712,notin:8713,ni:8715,prod:8719,sum:8721,minus:8722,lowast:8727,radic:8730,prop:8733,infin:8734,ang:8736,and:8743,or:8744,cap:8745,cup:8746,int:8747,there4:8756,sim:8764,cong:8773,asymp:8776,ne:8800,equiv:8801,le:8804,ge:8805,sub:8834,sup:8835,nsub:8836,sube:8838,supe:8839,oplus:8853,otimes:8855,perp:8869,sdot:8901,lceil:8968,rceil:8969,lfloor:8970,rfloor:8971,lang:9001,rang:9002,loz:9674,spades:9824,clubs:9827,hearts:9829,diams:9830},_={A:7,a:7,C:6,c:6,H:1,h:1,L:2,l:2,M:2,m:2,Q:4,q:4,S:4,s:4,T:2,t:2,V:1,v:1,Z:0,z:0},x={A3:!0,A4:!0,a3:!0,a4:!0},k={color:{inherit:!0,initial:void 0},visibility:{inherit:!0,initial:"visible",values:{hidden:"hidden",collapse:"hidden",visible:"visible"}},fill:{inherit:!0,initial:y.black},stroke:{inherit:!0,initial:"none"},"stop-color":{inherit:!1,initial:y.black},"fill-opacity":{inherit:!0,initial:1},"stroke-opacity":{inherit:!0,initial:1},"stop-opacity":{inherit:!1,initial:1},"fill-rule":{inherit:!0,initial:"nonzero",values:{nonzero:"nonzero",evenodd:"evenodd"}},"clip-rule":{inherit:!0,initial:"nonzero",values:{nonzero:"nonzero",evenodd:"evenodd"}},"stroke-width":{inherit:!0,initial:1},"stroke-dasharray":{inherit:!0,initial:[]},"stroke-dashoffset":{inherit:!0,initial:0},"stroke-miterlimit":{inherit:!0,initial:4},"stroke-linejoin":{inherit:!0,initial:"miter",values:{miter:"miter",round:"round",bevel:"bevel"}},"stroke-linecap":{inherit:!0,initial:"butt",values:{butt:"butt",round:"round",square:"square"}},"font-size":{inherit:!0,initial:16,values:{"xx-small":9,"x-small":10,small:13,medium:16,large:18,"x-large":24,"xx-large":32}},"font-family":{inherit:!0,initial:"sans-serif"},"font-weight":{inherit:!0,initial:"normal",values:{600:"bold",700:"bold",800:"bold",900:"bold",bold:"bold",bolder:"bold",500:"normal",400:"normal",300:"normal",200:"normal",100:"normal",normal:"normal",lighter:"normal"}},"font-style":{inherit:!0,initial:"normal",values:{italic:"italic",oblique:"italic",normal:"normal"}},"text-anchor":{inherit:!0,initial:"start",values:{start:"start",middle:"middle",end:"end"}},direction:{inherit:!0,initial:"ltr",values:{ltr:"ltr",rtl:"rtl"}},"dominant-baseline":{inherit:!0,initial:"baseline",values:{auto:"baseline",baseline:"baseline","before-edge":"before-edge","text-before-edge":"before-edge",middle:"middle",central:"central","after-edge":"after-edge","text-after-edge":"after-edge",ideographic:"ideographic",alphabetic:"alphabetic",hanging:"hanging",mathematical:"mathematical"}},"alignment-baseline":{inherit:!1,initial:void 0,values:{auto:"baseline",baseline:"baseline","before-edge":"before-edge","text-before-edge":"before-edge",middle:"middle",central:"central","after-edge":"after-edge","text-after-edge":"after-edge",ideographic:"ideographic",alphabetic:"alphabetic",hanging:"hanging",mathematical:"mathematical"}},"baseline-shift":{inherit:!0,initial:"baseline",values:{baseline:"baseline",sub:"sub",super:"super"}},"word-spacing":{inherit:!0,initial:0,values:{normal:0}},"letter-spacing":{inherit:!0,initial:0,values:{normal:0}},"text-decoration":{inherit:!1,initial:"none",values:{none:"none",underline:"underline",overline:"overline","line-through":"line-through"}},"xml:space":{inherit:!0,initial:"default",css:"white-space",values:{preserve:"preserve",default:"default",pre:"preserve","pre-line":"preserve","pre-wrap":"preserve",nowrap:"default"}},"marker-start":{inherit:!0,initial:"none"},"marker-mid":{inherit:!0,initial:"none"},"marker-end":{inherit:!0,initial:"none"},opacity:{inherit:!1,initial:1},transform:{inherit:!1,initial:[1,0,0,1,0,0]},display:{inherit:!1,initial:"inline",values:{none:"none",inline:"inline",block:"inline"}},"clip-path":{inherit:!1,initial:"none"},mask:{inherit:!1,initial:"none"},overflow:{inherit:!1,initial:"hidden",values:{hidden:"hidden",scroll:"hidden",visible:"visible"}}};function docBeginGroup(i){var o=new function PDFGroup(){};return o.name="G"+(r._groupCount=(r._groupCount||0)+1),o.resources=r.ref(),o.xobj=r.ref({Type:"XObject",Subtype:"Form",FormType:1,BBox:i,Group:{S:"Transparency",CS:"DeviceRGB",I:!0,K:!1},Resources:o.resources}),o.xobj.write(""),o.savedMatrix=r._ctm,o.savedPage=r.page,Ae.push(o),r._ctm=[1,0,0,1,0,0],r.page={width:r.page.width,height:r.page.height,write:function write(r){o.xobj.write(r)},fonts:{},xobjects:{},ext_gstates:{},patterns:{}},o}function docEndGroup(i){if(i!==Ae.pop())throw"Group not matching";Object.keys(r.page.fonts).length&&(i.resources.data.Font=r.page.fonts),Object.keys(r.page.xobjects).length&&(i.resources.data.XObject=r.page.xobjects),Object.keys(r.page.ext_gstates).length&&(i.resources.data.ExtGState=r.page.ext_gstates),Object.keys(r.page.patterns).length&&(i.resources.data.Pattern=r.page.patterns),i.resources.end(),i.xobj.end(),r._ctm=i.savedMatrix,r.page=i.savedPage}function docInsertGroup(i){r.page.xobjects[i.name]=i.xobj,r.addContent("/"+i.name+" Do")}function docApplyMask(i,o){var u="M"+(r._maskCount=(r._maskCount||0)+1),p=r.ref({Type:"ExtGState",CA:1,ca:1,BM:"Normal",SMask:{S:"Luminosity",G:i.xobj,BC:o?[0,0,0]:[1,1,1]}});p.end(),r.page.ext_gstates[u]=p,r.addContent("/"+u+" gs")}function docCreatePattern(r,i,o,u){var p=new function PDFPattern(){};return p.group=r,p.dx=i,p.dy=o,p.matrix=u||[1,0,0,1,0,0],p}function docUsePattern(i,o){var u,p="P"+(r._patternCount=(r._patternCount||0)+1),g=r.ref({Type:"Pattern",PatternType:1,PaintType:1,TilingType:2,BBox:[0,0,i.dx,i.dy],XStep:i.dx,YStep:i.dy,Matrix:multiplyMatrix(r._ctm,i.matrix),Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],XObject:(u={},u[i.group.name]=i.group.xobj,u)}});g.write("/"+i.group.name+" Do"),g.end(),r.page.patterns[p]=g,o?(r.addContent("/Pattern CS"),r.addContent("/"+p+" SCN")):(r.addContent("/Pattern cs"),r.addContent("/"+p+" scn"))}function docSetTextMode(i,o){var u=i&&o?2:o?1:i?0:3;r.addContent(u+" Tr")}function docFillColor(i){"PDFPattern"===i[0].constructor.name?(r.fillOpacity(i[1]),docUsePattern(i[0],!1)):r.fillColor(i[0],i[1])}function docStrokeColor(i){"PDFPattern"===i[0].constructor.name?(r.strokeOpacity(i[1]),docUsePattern(i[0],!0)):r.strokeColor(i[0],i[1])}function parseXml(r){var i=function SvgNode(r,i,o,u){this.error=u,this.nodeName=r,this.nodeValue=o,this.nodeType=i,this.attributes=Object.create(null),this.childNodes=[],this.parentNode=null,this.id="",this.textContent="",this.classList=[]};i.prototype.getAttribute=function(r){return null!=this.attributes[r]?this.attributes[r]:null},i.prototype.getElementById=function(r){var i=null;return function recursive(o){if(!i&&1===o.nodeType){o.id===r&&(i=o);for(var u=0;u<o.childNodes.length;u++)recursive(o.childNodes[u])}}(this),i},i.prototype.getElementsByTagName=function(r){var i=[];return function recursive(o){if(1===o.nodeType){o.nodeName===r&&i.push(o);for(var u=0;u<o.childNodes.length;u++)recursive(o.childNodes[u])}}(this),i};for(var o,u,p=new P(r.trim()),g=!1,y=function recursive(){var r,o;if(r=p.match(/^<([\w:.-]+)\s*/,!0)){for(var u=new i(r[1],1,null,g);r=p.match(/^([\w:.-]+)(?:\s*=\s*"([^"]*)"|\s*=\s*'([^']*)')?\s*/,!0);){var y=r[1],w=decodeEntities(r[2]||r[3]||"");u.attributes[y]?(_e('parseXml: duplicate attribute "'+y+'"'),g=!0):(u.attributes[y]=w,"id"===y&&(u.id=w),"class"===y&&(u.classList=w.split(" ")))}if(p.match(/^>/)){for(;o=recursive();)u.childNodes.push(o),o.parentNode=u,u.textContent+=3===o.nodeType||4===o.nodeType?o.nodeValue:o.textContent;return(r=p.match(/^<\/([\w:.-]+)\s*>/,!0))?(r[1]===u.nodeName||(_e('parseXml: tag not matching, opening "'+u.nodeName+'" & closing "'+r[1]+'"'),g=!0),u):(_e('parseXml: tag not matching, opening "'+u.nodeName+'" & not closing'),g=!0,u)}if(p.match(/^\/>/))return u;_e('parseXml: tag could not be parsed "'+u.nodeName+'"'),g=!0}else{if(r=p.match(/^<!--[\s\S]*?-->/))return new i(null,8,r,g);if(r=p.match(/^<\?[\s\S]*?\?>/))return new i(null,7,r,g);if(r=p.match(/^<!DOCTYPE\s*([\s\S]*?)>/))return new i(null,10,r,g);if(r=p.match(/^<!\[CDATA\[([\s\S]*?)\]\]>/,!0))return new i("#cdata-section",4,r[1],g);if(r=p.match(/^([^<]+)/,!0))return new i("#text",3,decodeEntities(r[1]),g)}};u=y();)1!==u.nodeType||o?(1===u.nodeType||3===u.nodeType&&""!==u.nodeValue.trim())&&_e("parseXml: data after document end has been discarded"):o=u;return p.matchAll()&&_e("parseXml: parsing error"),o}function decodeEntities(r){return r.replace(/&(?:#([0-9]+)|#[xX]([0-9A-Fa-f]+)|([0-9A-Za-z]+));/g,(function(r,i,o,u){return i?String.fromCharCode(parseInt(i,10)):o?String.fromCharCode(parseInt(o,16)):u&&w[u]?String.fromCharCode(w[u]):r}))}function parseColor(r){var i,o;return r=(r||"").trim(),(i=g[r])?o=[i.slice(),1]:(i=r.match(/^rgba\(\s*([0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9.]+)\s*\)$/i))?(i[1]=parseInt(i[1]),i[2]=parseInt(i[2]),i[3]=parseInt(i[3]),i[4]=parseFloat(i[4]),i[1]<256&&i[2]<256&&i[3]<256&&i[4]<=1&&(o=[i.slice(1,4),i[4]])):(i=r.match(/^rgb\(\s*([0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9]+)\s*\)$/i))?(i[1]=parseInt(i[1]),i[2]=parseInt(i[2]),i[3]=parseInt(i[3]),i[1]<256&&i[2]<256&&i[3]<256&&(o=[i.slice(1,4),1])):(i=r.match(/^rgb\(\s*([0-9.]+)%\s*,\s*([0-9.]+)%\s*,\s*([0-9.]+)%\s*\)$/i))?(i[1]=2.55*parseFloat(i[1]),i[2]=2.55*parseFloat(i[2]),i[3]=2.55*parseFloat(i[3]),i[1]<256&&i[2]<256&&i[3]<256&&(o=[i.slice(1,4),1])):(i=r.match(/^#([0-9a-f]{2})([0-9a-f]{2})([0-9a-f]{2})$/i))?o=[[parseInt(i[1],16),parseInt(i[2],16),parseInt(i[3],16)],1]:(i=r.match(/^#([0-9a-f])([0-9a-f])([0-9a-f])$/i))&&(o=[[17*parseInt(i[1],16),17*parseInt(i[2],16),17*parseInt(i[3],16)],1]),Ce?Ce(o,r):o}function opacityToColor(r,i,o){var u=r[0].slice(),p=r[1]*i;if(o){for(var g=0;g<r.length;g++)u[g]*=p;return[u,1]}return[u,p]}function multiplyMatrix(){function multiply(r,i){return[r[0]*i[0]+r[2]*i[1],r[1]*i[0]+r[3]*i[1],r[0]*i[2]+r[2]*i[3],r[1]*i[2]+r[3]*i[3],r[0]*i[4]+r[2]*i[5]+r[4],r[1]*i[4]+r[3]*i[5]+r[5]]}for(var r=arguments[0],i=1;i<arguments.length;i++)r=multiply(r,arguments[i]);return r}function transformPoint(r,i){return[i[0]*r[0]+i[2]*r[1]+i[4],i[1]*r[0]+i[3]*r[1]+i[5]]}function getGlobalMatrix(){for(var i=r._ctm,o=Ae.length-1;o>=0;o--)i=multiplyMatrix(Ae[o].savedMatrix,i);return i}function getPageBBox(){return(new B).M(0,0).L(r.page.width,0).L(r.page.width,r.page.height).L(0,r.page.height).transform(inverseMatrix(getGlobalMatrix())).getBoundingBox()}function inverseMatrix(r){var i=r[0]*r[3]-r[1]*r[2];return[r[3]/i,-r[1]/i,-r[2]/i,r[0]/i,(r[2]*r[5]-r[3]*r[4])/i,(r[1]*r[4]-r[0]*r[5])/i]}function validateMatrix(r){var i=validateNumber(r[0]),o=validateNumber(r[1]),u=validateNumber(r[2]),p=validateNumber(r[3]),g=validateNumber(r[4]),y=validateNumber(r[5]);if(isNotEqual(i*p-o*u,0))return[i,o,u,p,g,y]}function solveEquation(r){var i=r[2]||0,o=r[1]||0,u=r[0]||0;if(isEqual(i,0)&&isEqual(o,0))return[];if(isEqual(i,0))return[-u/o];var p=o*o-4*i*u;return isNotEqual(p,0)&&p>0?[(-o+Math.sqrt(p))/(2*i),(-o-Math.sqrt(p))/(2*i)]:isEqual(p,0)?[-o/(2*i)]:[]}function getCurveValue(r,i){return(i[0]||0)+(i[1]||0)*r+(i[2]||0)*r*r+(i[3]||0)*r*r*r}function isEqual(r,i){return Math.abs(r-i)<1e-10}function isNotEqual(r,i){return Math.abs(r-i)>=1e-10}function validateNumber(r){return r>-1e21&&r<1e21?Math.round(1e6*r)/1e6:0}function parseTranform(r){for(var i,o=new P((r||"").trim()),u=[1,0,0,1,0,0];i=o.match(/^([A-Za-z]+)\s*[(]([^(]+)[)]/,!0);){for(var p=i[1],g=[],y=new P(i[2].trim()),w=void 0;w=y.matchNumber();)g.push(Number(w)),y.matchSeparator();if("matrix"===p&&6===g.length)u=multiplyMatrix(u,[g[0],g[1],g[2],g[3],g[4],g[5]]);else if("translate"===p&&2===g.length)u=multiplyMatrix(u,[1,0,0,1,g[0],g[1]]);else if("translate"===p&&1===g.length)u=multiplyMatrix(u,[1,0,0,1,g[0],0]);else if("scale"===p&&2===g.length)u=multiplyMatrix(u,[g[0],0,0,g[1],0,0]);else if("scale"===p&&1===g.length)u=multiplyMatrix(u,[g[0],0,0,g[0],0,0]);else if("rotate"===p&&3===g.length){var _=g[0]*Math.PI/180;u=multiplyMatrix(u,[1,0,0,1,g[1],g[2]],[Math.cos(_),Math.sin(_),-Math.sin(_),Math.cos(_),0,0],[1,0,0,1,-g[1],-g[2]])}else if("rotate"===p&&1===g.length){var x=g[0]*Math.PI/180;u=multiplyMatrix(u,[Math.cos(x),Math.sin(x),-Math.sin(x),Math.cos(x),0,0])}else if("skewX"===p&&1===g.length){var k=g[0]*Math.PI/180;u=multiplyMatrix(u,[1,0,Math.tan(k),1,0,0])}else{if("skewY"!==p||1!==g.length)return;var O=g[0]*Math.PI/180;u=multiplyMatrix(u,[1,Math.tan(O),0,1,0,0])}o.matchSeparator()}if(!o.matchAll())return u}function parseAspectRatio(r,i,o,u,p,g){var y=(r||"").trim().match(/^(none)$|^x(Min|Mid|Max)Y(Min|Mid|Max)(?:\s+(meet|slice))?$/)||[],w=y[1]||y[4]||"meet",_=y[2]||"Mid",x=y[3]||"Mid",k=i/u,P=o/p,O={Min:0,Mid:.5,Max:1}[_]-(g||0),I={Min:0,Mid:.5,Max:1}[x]-(g||0);return"slice"===w?P=k=Math.max(k,P):"meet"===w&&(P=k=Math.min(k,P)),[k,0,0,P,O*(i-u*k),I*(o-p*P)]}function parseStyleAttr(r){var i=Object.create(null);r=(r||"").trim().split(/;/);for(var o=0;o<r.length;o++){var u=(r[o].split(":")[0]||"").trim(),p=(r[o].split(":")[1]||"").trim();u&&(i[u]=p)}if(i.marker&&(i["marker-start"]||(i["marker-start"]=i.marker),i["marker-mid"]||(i["marker-mid"]=i.marker),i["marker-end"]||(i["marker-end"]=i.marker)),i.font){for(var g=null,y=null,w="normal",_="normal",x="normal",k=i.font.split(/\s+/),P=0;P<k.length;P++)switch(k[P]){case"normal":break;case"italic":case"oblique":w=k[P];break;case"small-caps":x=k[P];break;case"bold":case"bolder":case"lighter":case"100":case"200":case"300":case"400":case"500":case"600":case"700":case"800":case"900":_=k[P];break;default:y?g?g+=" "+k[P]:g=k[P]:y=k[P].split("/")[0]}i["font-style"]||(i["font-style"]=w),i["font-variant"]||(i["font-variant"]=x),i["font-weight"]||(i["font-weight"]=_),i["font-size"]||(i["font-size"]=y),i["font-family"]||(i["font-family"]=g)}return i}function parseSelector(r){for(var i,o=r.split(/(?=[.#])/g),u=[],p=[],g=[],y=0;y<o.length;y++)if(i=o[y].match(/^[#]([_A-Za-z0-9-]+)$/))u.push(i[1]);else if(i=o[y].match(/^[.]([_A-Za-z0-9-]+)$/))p.push(i[1]);else if(i=o[y].match(/^([_A-Za-z0-9-]+)$/))g.push(i[1]);else if("*"!==o[y])return;return{tags:g,ids:u,classes:p,specificity:1e4*u.length+100*p.length+g.length}}function parseStyleSheet(r){for(var i,o=new P(r.trim()),u=[];i=o.match(/^\s*([^\{\}]*?)\s*\{([^\{\}]*?)\}/,!0);)for(var p=i[1].split(/\s*,\s*/g),g=parseStyleAttr(i[2]),y=0;y<p.length;y++){var w=parseSelector(p[y]);w&&u.push({selector:w,css:g})}return u}function matchesSelector(r,i){if(1!==r.nodeType)return!1;for(var o=0;o<i.tags.length;o++)if(i.tags[o]!==r.nodeName)return!1;for(var u=0;u<i.ids.length;u++)if(i.ids[u]!==r.id)return!1;for(var p=0;p<i.classes.length;p++)if(-1===r.classList.indexOf(i.classes[p]))return!1;return!0}function combineArrays(r,i){return r.concat(i.slice(r.length))}function getAscent(r,i){return Math.max(r.ascender,(r.bbox[3]||r.bbox.maxY)*(r.scale||1))*i/1e3}function getDescent(r,i){return Math.min(r.descender,(r.bbox[1]||r.bbox.minY)*(r.scale||1))*i/1e3}function getTextPos(r,i,o){for(var u=r.encode(""+o),p=u[0],g=u[1],y=[],w=0;w<p.length;w++){var _=r.unicode?r.unicode[parseInt(p[w],16)]:[o.charCodeAt(w)];y.push({glyph:p[w],unicode:_,width:g[w].advanceWidth*i/1e3,xOffset:g[w].xOffset*i/1e3,yOffset:g[w].yOffset*i/1e3,xAdvance:g[w].xAdvance*i/1e3,yAdvance:g[w].yAdvance*i/1e3})}return y}function createSVGElement(r,i){switch(r.nodeName){case"use":return new U(r,i);case"symbol":return new j(r,i);case"g":return new G(r,i);case"a":return new X(r,i);case"svg":return new K(r,i);case"image":return new Y(r,i);case"rect":return new te(r,i);case"circle":return new re(r,i);case"ellipse":return new ne(r,i);case"line":return new ie(r,i);case"polyline":return new oe(r,i);case"polygon":return new ae(r,i);case"path":return new se(r,i);case"text":return new ge(r,i);case"tspan":return new de(r,i);case"textPath":return new pe(r,i);case"#text":case"#cdata-section":return new he(r,i);default:return new R(r,i)}}var P=function StringParser(r){this.match=function(i,o){var u=r.match(i);if(u&&0===u.index)return r=r.substring(u[0].length),o?u:u[0]},this.matchSeparator=function(){return this.match(/^(?:\s*,\s*|\s*|)/)},this.matchSpace=function(){return this.match(/^(?:\s*)/)},this.matchLengthUnit=function(){return this.match(/^(?:px|pt|cm|mm|in|pc|em|ex|%|)/)},this.matchNumber=function(){return this.match(/^(?:[-+]?(?:[0-9]+[.][0-9]+|[0-9]+[.]|[.][0-9]+|[0-9]+)(?:[eE][-+]?[0-9]+)?)/)},this.matchAll=function(){return this.match(/^[\s\S]+/)}},O=function BezierSegment(r,i,o,u,p,g,y,w){for(var _=6*Pe,x=[r,-3*r+3*o,3*r-6*o+3*p,3*o-r-3*p+y],k=[i,-3*i+3*u,3*i-6*u+3*g,3*u-i-3*g+w],P=[-3*r+3*o,6*r-12*o+6*p,-3*r+9*o-9*p+3*y],O=[-3*i+3*u,6*i-12*u+6*g,-3*i+9*u-9*g+3*w],I=[0],B=1;B<=_;B++){var R=(B-.5)/_,D=getCurveValue(R,P)/_,N=getCurveValue(R,O)/_,W=Math.sqrt(D*D+N*N);I[B]=I[B-1]+W}this.totalLength=I[_],this.startPoint=[r,i,isEqual(r,o)&&isEqual(i,u)?Math.atan2(g-u,p-o):Math.atan2(u-i,o-r)],this.endPoint=[y,w,isEqual(p,y)&&isEqual(g,w)?Math.atan2(g-u,p-o):Math.atan2(w-g,y-p)],this.getBoundingBox=function(){var r,i=getCurveValue(0,x),o=getCurveValue(0,k),u=getCurveValue(1,x),p=getCurveValue(1,k);i>u&&(r=u,u=i,i=r),o>p&&(r=p,p=o,o=r);for(var g=solveEquation(P),y=0;y<g.length;y++)if(g[y]>=0&&g[y]<=1){var w=getCurveValue(g[y],x);w<i&&(i=w),w>u&&(u=w)}for(var _=solveEquation(O),I=0;I<_.length;I++)if(_[I]>=0&&_[I]<=1){var B=getCurveValue(_[I],k);B<o&&(o=B),B>p&&(p=B)}return[i,o,u,p]},this.getPointAtLength=function(r){if(isEqual(r,0))return this.startPoint;if(isEqual(r,this.totalLength))return this.endPoint;if(!(r<0||r>this.totalLength))for(var i=1;i<=_;i++){var o=I[i-1],u=I[i];if(o<=r&&r<=u){var p=(i-(u-r)/(u-o))/_,g=getCurveValue(p,x),y=getCurveValue(p,k),w=getCurveValue(p,P),B=getCurveValue(p,O);return[g,y,Math.atan2(B,w)]}}}},I=function LineSegment(r,i,o,u){this.totalLength=Math.sqrt((o-r)*(o-r)+(u-i)*(u-i)),this.startPoint=[r,i,Math.atan2(u-i,o-r)],this.endPoint=[o,u,Math.atan2(u-i,o-r)],this.getBoundingBox=function(){return[Math.min(this.startPoint[0],this.endPoint[0]),Math.min(this.startPoint[1],this.endPoint[1]),Math.max(this.startPoint[0],this.endPoint[0]),Math.max(this.startPoint[1],this.endPoint[1])]},this.getPointAtLength=function(r){if(r>=0&&r<=this.totalLength){var i=r/this.totalLength||0;return[this.startPoint[0]+i*(this.endPoint[0]-this.startPoint[0]),this.startPoint[1]+i*(this.endPoint[1]-this.startPoint[1]),this.startPoint[2]]}}},B=function SvgShape(){this.pathCommands=[],this.pathSegments=[],this.startPoint=null,this.endPoint=null,this.totalLength=0;var i,o,u,p=0,g=0,y=0,w=0;this.move=function(r,i){return p=y=r,g=w=i,null},this.line=function(r,i){var o=new I(y,w,r,i);return y=r,w=i,o},this.curve=function(r,i,o,u,p,g){var _=new O(y,w,r,i,o,u,p,g);return y=p,w=g,_},this.close=function(){var r=new I(y,w,p,g);return y=p,w=g,r},this.addCommand=function(r){this.pathCommands.push(r);var i=this[r[0]].apply(this,r.slice(3));i&&(i.hasStart=r[1],i.hasEnd=r[2],this.startPoint=this.startPoint||i.startPoint,this.endPoint=i.endPoint,this.pathSegments.push(i),this.totalLength+=i.totalLength)},this.M=function(r,o){return this.addCommand(["move",!0,!0,r,o]),i="M",this},this.m=function(r,i){return this.M(y+r,w+i)},this.Z=this.z=function(){return this.addCommand(["close",!0,!0]),i="Z",this},this.L=function(r,o){return this.addCommand(["line",!0,!0,r,o]),i="L",this},this.l=function(r,i){return this.L(y+r,w+i)},this.H=function(r){return this.L(r,w)},this.h=function(r){return this.L(y+r,w)},this.V=function(r){return this.L(y,r)},this.v=function(r){return this.L(y,w+r)},this.C=function(r,p,g,y,w,_){return this.addCommand(["curve",!0,!0,r,p,g,y,w,_]),i="C",o=g,u=y,this},this.c=function(r,i,o,u,p,g){return this.C(y+r,w+i,y+o,w+u,y+p,w+g)},this.S=function(r,p,g,_){return this.C(y+("C"===i?y-o:0),w+("C"===i?w-u:0),r,p,g,_)},this.s=function(r,p,g,_){return this.C(y+("C"===i?y-o:0),w+("C"===i?w-u:0),y+r,w+p,y+g,w+_)},this.Q=function(r,p,g,_){var x=y+2/3*(r-y),k=w+2/3*(p-w),P=g+2/3*(r-g),O=_+2/3*(p-_);return this.addCommand(["curve",!0,!0,x,k,P,O,g,_]),i="Q",o=r,u=p,this},this.q=function(r,i,o,u){return this.Q(y+r,w+i,y+o,w+u)},this.T=function(r,p){return this.Q(y+("Q"===i?y-o:0),w+("Q"===i?w-u:0),r,p)},this.t=function(r,p){return this.Q(y+("Q"===i?y-o:0),w+("Q"===i?w-u:0),y+r,w+p)},this.A=function(r,o,u,p,g,_,x){if(isEqual(r,0)||isEqual(o,0))this.addCommand(["line",!0,!0,_,x]);else{u*=Math.PI/180,r=Math.abs(r),o=Math.abs(o),p=1*!!p,g=1*!!g;var k=Math.cos(u)*(y-_)/2+Math.sin(u)*(w-x)/2,P=Math.cos(u)*(w-x)/2-Math.sin(u)*(y-_)/2,O=k*k/(r*r)+P*P/(o*o);O>1&&(r*=Math.sqrt(O),o*=Math.sqrt(O));var I=Math.sqrt(Math.max(0,r*r*o*o-r*r*P*P-o*o*k*k)/(r*r*P*P+o*o*k*k)),B=(p===g?-1:1)*I*r*P/o,R=(p===g?1:-1)*I*o*k/r,D=Math.cos(u)*B-Math.sin(u)*R+(y+_)/2,N=Math.sin(u)*B+Math.cos(u)*R+(w+x)/2,W=Math.atan2((P-R)/o,(k-B)/r),U=Math.atan2((-P-R)/o,(-k-B)/r);0===g&&U-W>0?U-=2*Math.PI:1===g&&U-W<0&&(U+=2*Math.PI);for(var j=Math.ceil(Math.abs(U-W)/(Math.PI/Pe)),G=0;G<j;G++){var X=W+G*(U-W)/j,K=W+(G+1)*(U-W)/j,Y=4/3*Math.tan((K-X)/4),J=D+Math.cos(u)*r*(Math.cos(X)-Y*Math.sin(X))-Math.sin(u)*o*(Math.sin(X)+Y*Math.cos(X)),$=N+Math.sin(u)*r*(Math.cos(X)-Y*Math.sin(X))+Math.cos(u)*o*(Math.sin(X)+Y*Math.cos(X)),ee=D+Math.cos(u)*r*(Math.cos(K)+Y*Math.sin(K))-Math.sin(u)*o*(Math.sin(K)-Y*Math.cos(K)),te=N+Math.sin(u)*r*(Math.cos(K)+Y*Math.sin(K))+Math.cos(u)*o*(Math.sin(K)-Y*Math.cos(K)),re=D+Math.cos(u)*r*Math.cos(K)-Math.sin(u)*o*Math.sin(K),ne=N+Math.sin(u)*r*Math.cos(K)+Math.cos(u)*o*Math.sin(K);this.addCommand(["curve",0===G,G===j-1,J,$,ee,te,re,ne])}}return i="A",this},this.a=function(r,i,o,u,p,g,_){return this.A(r,i,o,u,p,y+g,w+_)},this.path=function(r){for(var i,o,u,p=new P((r||"").trim());i=p.match(/^[astvzqmhlcASTVZQMHLC]/);){p.matchSeparator();for(var g=[];o=x[i+g.length]?p.match(/^[01]/):p.matchNumber();)p.matchSeparator(),g.length===_[i]&&(this[i].apply(this,g),g=[],"M"===i?i="L":"m"===i&&(i="l")),g.push(Number(o));if(g.length!==_[i])return void _e("SvgPath: command "+i+" with "+g.length+" numbers");this[i].apply(this,g)}return(u=p.matchAll())&&_e("SvgPath: unexpected string "+u),this},this.getBoundingBox=function(){for(var r,i=[1/0,1/0,-1/0,-1/0],o=0;o<this.pathSegments.length;o++)(r=this.pathSegments[o].getBoundingBox())[0]<i[0]&&(i[0]=r[0]),r[2]>i[2]&&(i[2]=r[2]),r[1]<i[1]&&(i[1]=r[1]),r[3]>i[3]&&(i[3]=r[3]);return i[0]===1/0&&(i[0]=0),i[1]===1/0&&(i[1]=0),i[2]===-1/0&&(i[2]=0),i[3]===-1/0&&(i[3]=0),i},this.getPointAtLength=function(r){if(r>=0&&r<=this.totalLength){for(var i,o=0;o<this.pathSegments.length;o++){if(i=this.pathSegments[o].getPointAtLength(r))return i;r-=this.pathSegments[o].totalLength}return this.endPoint}},this.transform=function(r){this.pathSegments=[],this.startPoint=null,this.endPoint=null,this.totalLength=0;for(var i=0;i<this.pathCommands.length;i++){for(var o=this.pathCommands.shift(),u=3;u<o.length;u+=2){var p=transformPoint([o[u],o[u+1]],r);o[u]=p[0],o[u+1]=p[1]}this.addCommand(o)}return this},this.mergeShape=function(r){for(var i=0;i<r.pathCommands.length;i++)this.addCommand(r.pathCommands[i].slice());return this},this.clone=function(){return(new SvgShape).mergeShape(this)},this.insertInDocument=function(){for(var i=0;i<this.pathCommands.length;i++){var o=this.pathCommands[i][0],u=this.pathCommands[i].slice(3);switch(o){case"move":r.moveTo(u[0],u[1]);break;case"line":r.lineTo(u[0],u[1]);break;case"curve":r.bezierCurveTo(u[0],u[1],u[2],u[3],u[4],u[5]);break;case"close":r.closePath()}}},this.getSubPaths=function(){for(var r=[],i=new SvgShape,o=0;o<this.pathCommands.length;o++){var u=this.pathCommands[o];"move"===this.pathCommands[o][0]&&0!==o&&(r.push(i),i=new SvgShape),i.addCommand(u)}return r.push(i),r},this.getMarkers=function(){for(var r=[],i=this.getSubPaths(),o=0;o<i.length;o++){for(var u=i[o],p=[],g=0;g<u.pathSegments.length;g++){var y=u.pathSegments[g];if(isNotEqual(y.totalLength,0)||0===g||g===u.pathSegments.length-1){if(y.hasStart){var w=y.getPointAtLength(0),_=p.pop();_&&(w[2]=.5*(_[2]+w[2])),p.push(w)}if(y.hasEnd){var x=y.getPointAtLength(y.totalLength);p.push(x)}}}r=r.concat(p)}return r}},R=function SvgElem(r,o){var u=Object.create(null),p=null;this.name=r.nodeName,this.isOuterElement=r===i||!r.parentNode,this.inherits=o||(this.isOuterElement?null:createSVGElement(r.parentNode,null)),this.stack=this.inherits?this.inherits.stack.concat(r):[r],this.style=parseStyleAttr("function"==typeof r.getAttribute&&r.getAttribute("style")),this.css=we?getComputedStyle(r):function getStyle(r){for(var i=Object.create(null),o=Object.create(null),u=0;u<Oe.length;u++){var p=Oe[u];if(matchesSelector(r,p.selector))for(var g in p.css)o[g]>p.selector.specificity||(i[g]=p.css[g],o[g]=p.selector.specificity)}return i}(r),this.allowedChildren=[],this.attr=function(i){if("function"==typeof r.getAttribute)return r.getAttribute(i)},this.resolveUrl=function(r){var o=(r||"").match(/^\s*(?:url\("(.*)#(.*)"\)|url\('(.*)#(.*)'\)|url\((.*)#(.*)\)|(.*)#(.*))\s*$/)||[],u=o[1]||o[3]||o[5]||o[7],p=o[2]||o[4]||o[6]||o[8];if(p){if(!u){var g=i.getElementById(p);if(g)return-1===this.stack.indexOf(g)?g:void _e('SVGtoPDF: loop of circular references for id "'+p+'"')}if(ke){var y=Ee[u];if(!y){(function isArrayLike(r){return"object"==typeof r&&null!==r&&"number"==typeof r.length})(y=ke(u))||(y=[y]);for(var w=0;w<y.length;w++)"string"==typeof y[w]&&(y[w]=parseXml(y[w]));Ee[u]=y}for(var _=0;_<y.length;_++){var x=y[_].getElementById(p);if(x)return-1===this.stack.indexOf(x)?x:void _e('SVGtoPDF: loop of circular references for id "'+u+"#"+p+'"')}}}},this.computeUnits=function(r,i,o,u){return"%"===i?parseFloat(r)/100*(u||null!=o?o:this.getViewport()):"ex"===i||"em"===i?r*{em:1,ex:.5}[i]*(u?o:this.get("font-size")):r*{"":1,px:1,pt:96/72,cm:96/2.54,mm:96/25.4,in:96,pc:16}[i]},this.computeLength=function(r,i,o,u){var p,g,y=new P((r||"").trim());return"string"!=typeof(p=y.matchNumber())||"string"!=typeof(g=y.matchLengthUnit())||y.matchAll()?o:this.computeUnits(p,g,i,u)},this.computeLengthList=function(r,i,o){for(var u,p,g=new P((r||"").trim()),y=[];"string"==typeof(u=g.matchNumber())&&"string"==typeof(p=g.matchLengthUnit());)y.push(this.computeUnits(u,p,i)),g.matchSeparator();if(!o||!g.matchAll())return y},this.getLength=function(r,i,o){return this.computeLength(this.attr(r),i,o)},this.getLengthList=function(r,i){return this.computeLengthList(this.attr(r),i)},this.getUrl=function(r){return this.resolveUrl(this.attr(r))},this.getNumberList=function(r){for(var i,o=new P((this.attr(r)||"").trim()),u=[];i=o.matchNumber();)u.push(Number(i)),o.matchSeparator();return u.error=o.matchAll(),u},this.getViewbox=function(r,i){var o=this.getNumberList(r);return 4===o.length&&o[2]>=0&&o[3]>=0?o:i},this.getPercent=function(r,i){var o=this.attr(r),u=new P((o||"").trim()),p=u.matchNumber();return p?(u.match("%")&&(p*=.01),u.matchAll()?i:Math.max(0,Math.min(1,p))):i},this.chooseValue=function(r){for(var i=0;i<arguments.length;i++)if(null!=arguments[i]&&arguments[i]==arguments[i])return arguments[i];return arguments[arguments.length-1]},this.get=function(r){if(void 0!==u[r])return u[r];for(var i,o,p=k[r]||{},g=0;g<3;g++){switch(g){case 0:"transform"!==r&&(i=this.css[p.css||r]);break;case 1:i=this.style[r];break;case 2:i=this.attr(r)}if("inherit"===i&&null!=(o=this.inherits?this.inherits.get(r):p.initial))return u[r]=o;if(null!=p.values&&null!=(o=p.values[i]))return u[r]=o;if(null!=i){var y=void 0;switch(r){case"font-size":o=this.computeLength(i,this.inherits?this.inherits.get(r):p.initial,void 0,!0);break;case"baseline-shift":o=this.computeLength(i,this.get("font-size"));break;case"font-family":o=i||void 0;break;case"opacity":case"stroke-opacity":case"fill-opacity":case"stop-opacity":y=parseFloat(i),isNaN(y)||(o=Math.max(0,Math.min(1,y)));break;case"transform":o=parseTranform(i);break;case"stroke-dasharray":if("none"===i)o=[];else if(y=this.computeLengthList(i,this.getViewport(),!0)){for(var w=0,_=!1,x=0;x<y.length;x++)y[x]<0&&(_=!0),w+=y[x];_||(y.length%2==1&&(y=y.concat(y)),o=0===w?[]:y)}break;case"color":o="none"===i||"transparent"===i?"none":parseColor(i);break;case"fill":case"stroke":if("none"===i||"transparent"===i)o="none";else if("currentColor"===i)o=this.get("color");else{if(y=parseColor(i))return y;if(y=(i||"").split(" ")){var P=this.resolveUrl(y[0]),O=parseColor(y[1]);o=null==P?O:"linearGradient"===P.nodeName||"radialGradient"===P.nodeName?new $(P,null,O):"pattern"===P.nodeName?new J(P,null,O):O}}break;case"stop-color":o="none"===i||"transparent"===i?"none":"currentColor"===i?this.get("color"):parseColor(i);break;case"marker-start":case"marker-mid":case"marker-end":case"clip-path":case"mask":o="none"===i?"none":this.resolveUrl(i);break;case"stroke-width":null!=(y=this.computeLength(i,this.getViewport()))&&y>=0&&(o=y);break;case"stroke-miterlimit":null!=(y=parseFloat(i))&&y>=1&&(o=y);break;case"word-spacing":case"letter-spacing":o=this.computeLength(i,this.getViewport());break;case"stroke-dashoffset":if(null!=(o=this.computeLength(i,this.getViewport()))&&o<0)for(var I=this.get("stroke-dasharray"),B=0;B<I.length;B++)o+=I[B]}if(null!=o)return u[r]=o}}return u[r]=p.inherit&&this.inherits?this.inherits.get(r):p.initial},this.getChildren=function(){if(null!=p)return p;for(var i=[],o=0;o<r.childNodes.length;o++){var u=r.childNodes[o];u.error||-1===this.allowedChildren.indexOf(u.nodeName)||i.push(createSVGElement(u,this))}return p=i},this.getParentVWidth=function(){return this.inherits?this.inherits.getVWidth():ve},this.getParentVHeight=function(){return this.inherits?this.inherits.getVHeight():me},this.getParentViewport=function(){return Math.sqrt(.5*this.getParentVWidth()*this.getParentVWidth()+.5*this.getParentVHeight()*this.getParentVHeight())},this.getVWidth=function(){return this.getParentVWidth()},this.getVHeight=function(){return this.getParentVHeight()},this.getViewport=function(){return Math.sqrt(.5*this.getVWidth()*this.getVWidth()+.5*this.getVHeight()*this.getVHeight())},this.getBoundingBox=function(){return this.getBoundingShape().getBoundingBox()}},D=function SvgElemStylable(i,o){R.call(this,i,o),this.transform=function(){r.transform.apply(r,this.getTransformation())},this.clip=function(){if("none"!==this.get("clip-path"))return new le(this.get("clip-path"),null).useMask(this.getBoundingBox()),!0},this.mask=function(){if("none"!==this.get("mask"))return new ue(this.get("mask"),null).useMask(this.getBoundingBox()),!0},this.getFill=function(r,i){var o=this.get("opacity"),u=this.get("fill"),p=this.get("fill-opacity");return r?y.white:"none"!==u&&o&&p?u instanceof $||u instanceof J?u.getPaint(this.getBoundingBox(),p*o,r,i):opacityToColor(u,p*o,i):void 0},this.getStroke=function(r,i){var o=this.get("opacity"),u=this.get("stroke"),p=this.get("stroke-opacity");if(!r&&!isEqual(this.get("stroke-width"),0))return"none"!==u&&o&&p?u instanceof $||u instanceof J?u.getPaint(this.getBoundingBox(),p*o,r,i):opacityToColor(u,p*o,i):void 0}},N=function SvgElemHasChildren(r,i){D.call(this,r,i),this.allowedChildren=["use","g","a","svg","image","rect","circle","ellipse","line","polyline","polygon","path","text"],this.getBoundingShape=function(){for(var r=new B,i=this.getChildren(),o=0;o<i.length;o++)if("none"!==i[o].get("display")&&"function"==typeof i[o].getBoundingShape){var u=i[o].getBoundingShape().clone();"function"==typeof i[o].getTransformation&&u.transform(i[o].getTransformation()),r.mergeShape(u)}return r},this.drawChildren=function(r,i){for(var o=this.getChildren(),u=0;u<o.length;u++)"none"!==o[u].get("display")&&"function"==typeof o[u].drawInDocument&&o[u].drawInDocument(r,i)}},W=function SvgElemContainer(i,o){N.call(this,i,o),this.drawContent=function(i,o){this.transform();var u,p=this.clip(),g=this.mask();(this.get("opacity")<1||p||g)&&!i&&(u=docBeginGroup(getPageBBox())),this.drawChildren(i,o),u&&(docEndGroup(u),r.fillOpacity(this.get("opacity")),docInsertGroup(u))}},U=function SvgElemUse(i,o){W.call(this,i,o);var u=this.getLength("x",this.getVWidth(),0),p=this.getLength("y",this.getVHeight(),0),g=this.getUrl("href")||this.getUrl("xlink:href");g&&(g=createSVGElement(g,this)),this.getChildren=function(){return g?[g]:[]},this.drawInDocument=function(i,o){r.save(),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return multiplyMatrix(this.get("transform"),[1,0,0,1,u,p])}},j=function SvgElemSymbol(i,o){W.call(this,i,o);var u=this.getLength("width",this.getParentVWidth(),this.getParentVWidth()),p=this.getLength("height",this.getParentVHeight(),this.getParentVHeight());o instanceof U&&(u=o.getLength("width",o.getParentVWidth(),u),p=o.getLength("height",o.getParentVHeight(),p));var g=(this.attr("preserveAspectRatio")||"").trim(),y=this.getViewbox("viewBox",[0,0,u,p]);this.getVWidth=function(){return y[2]},this.getVHeight=function(){return y[3]},this.drawInDocument=function(i,o){r.save(),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return multiplyMatrix(parseAspectRatio(g,u,p,y[2],y[3]),[1,0,0,1,-y[0],-y[1]])}},G=function SvgElemGroup(i,o){W.call(this,i,o),this.drawInDocument=function(i,o){r.save(),!this.link||i||o||this.addLink(),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return this.get("transform")}},X=function SvgElemLink(i,o){o&&o.isText?(de.call(this,i,o),this.allowedChildren=["textPath","tspan","#text","#cdata-section","a"]):G.call(this,i,o),this.link=this.attr("href")||this.attr("xlink:href"),this.addLink=function(){if(this.link.match(/^(?:[a-z][a-z0-9+.-]*:|\/\/)?/i)&&this.getChildren().length){var i=this.getBoundingShape().transform(getGlobalMatrix()).getBoundingBox();!function docInsertLink(i,o,u,p,g){var y=r.ref({Type:"Annot",Subtype:"Link",Rect:[i,o,u,p],Border:[0,0,0],A:{S:"URI",URI:new String(g)}});y.end(),Te.push(y)}(i[0],i[1],i[2],i[3],this.link)}}},K=function SvgElemSvg(i,o){W.call(this,i,o);var u=this.getLength("width",this.getParentVWidth(),this.getParentVWidth()),p=this.getLength("height",this.getParentVHeight(),this.getParentVHeight()),g=this.getLength("x",this.getParentVWidth(),0),y=this.getLength("y",this.getParentVHeight(),0);o instanceof U&&(u=o.getLength("width",o.getParentVWidth(),u),p=o.getLength("height",o.getParentVHeight(),p));var w=this.attr("preserveAspectRatio"),_=this.getViewbox("viewBox",[0,0,u,p]);this.isOuterElement&&be&&(g=y=0,u=ve,p=me,w=be),this.getVWidth=function(){return _[2]},this.getVHeight=function(){return _[3]},this.drawInDocument=function(i,o){r.save(),"hidden"===this.get("overflow")&&((new B).M(g,y).L(g+u,y).L(g+u,y+p).L(g,y+p).Z().transform(this.get("transform")).insertInDocument(),r.clip()),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return multiplyMatrix(this.get("transform"),[1,0,0,1,g,y],parseAspectRatio(w,u,p,_[2],_[3]),[1,0,0,1,-_[0],-_[1]])}},Y=function SVGElemImage(i,o){D.call(this,i,o);var u,p=xe(this.attr("href")||this.attr("xlink:href")||""),g=this.getLength("x",this.getVWidth(),0),w=this.getLength("y",this.getVHeight(),0),_=this.getLength("width",this.getVWidth(),"auto"),x=this.getLength("height",this.getVHeight(),"auto");try{u=r.openImage(p)}catch(r){_e('SVGElemImage: failed to open image "'+p+'" in PDFKit')}u&&("auto"===_&&"auto"!==x?_=x*u.width/u.height:"auto"===x&&"auto"!==_?x=_*u.height/u.width:"auto"===_&&"auto"===x&&(_=u.width,x=u.height)),("auto"===_||_<0)&&(_=0),("auto"===x||x<0)&&(x=0),this.getTransformation=function(){return this.get("transform")},this.getBoundingShape=function(){return(new B).M(g,w).L(g+_,w).M(g+_,w+x).L(g,w+x)},this.drawInDocument=function(i,o){"hidden"!==this.get("visibility")&&u&&(r.save(),this.transform(),"hidden"===this.get("overflow")&&r.rect(g,w,_,x).clip(),this.clip(),this.mask(),r.translate(g,w),r.transform.apply(r,parseAspectRatio(this.attr("preserveAspectRatio"),_,x,u?u.width:_,u?u.height:x)),i?(r.rect(0,0,u.width,u.height),docFillColor(y.white).fill()):(r.fillOpacity(this.get("opacity")),r.image(u,0,0)),r.restore())}},J=function SvgElemPattern(i,o,u){N.call(this,i,o),this.ref=function(){var r=this.getUrl("href")||this.getUrl("xlink:href");if(r&&r.nodeName===i.nodeName)return new SvgElemPattern(r,o,u)}.call(this);var p=this.attr;this.attr=function(r){var i=p.call(this,r);return null!=i||"href"===r||"xlink:href"===r?i:this.ref?this.ref.attr(r):null};var g=this.getChildren;this.getChildren=function(){var r=g.call(this);return r.length>0?r:this.ref?this.ref.getChildren():[]},this.getPaint=function(i,o,p,g){var y="userSpaceOnUse"!==this.attr("patternUnits"),w="objectBoundingBox"===this.attr("patternContentUnits"),_=this.getLength("x",y?1:this.getParentVWidth(),0),x=this.getLength("y",y?1:this.getParentVHeight(),0),k=this.getLength("width",y?1:this.getParentVWidth(),0),P=this.getLength("height",y?1:this.getParentVHeight(),0);w&&!y?(_=(_-i[0])/(i[2]-i[0])||0,x=(x-i[1])/(i[3]-i[1])||0,k=k/(i[2]-i[0])||0,P=P/(i[3]-i[1])||0):!w&&y&&(_=i[0]+_*(i[2]-i[0]),x=i[1]+x*(i[3]-i[1]),k*=i[2]-i[0],P*=i[3]-i[1]);var O=this.getViewbox("viewBox",[0,0,k,P]),I=multiplyMatrix(parseAspectRatio((this.attr("preserveAspectRatio")||"").trim(),k,P,O[2],O[3],0),[1,0,0,1,-O[0],-O[1]]),B=parseTranform(this.attr("patternTransform"));if(w&&(B=multiplyMatrix([i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]],B)),(B=validateMatrix(B=multiplyMatrix(B,[1,0,0,1,_,x])))&&(I=validateMatrix(I))&&(k=validateNumber(k))&&(P=validateNumber(P))){var R=docBeginGroup([0,0,k,P]);return r.transform.apply(r,I),this.drawChildren(p,g),docEndGroup(R),[docCreatePattern(R,k,P,B),o]}return u?[u[0],u[1]*o]:void 0},this.getVWidth=function(){var r="userSpaceOnUse"!==this.attr("patternUnits"),i=this.getLength("width",r?1:this.getParentVWidth(),0);return this.getViewbox("viewBox",[0,0,i,0])[2]},this.getVHeight=function(){var r="userSpaceOnUse"!==this.attr("patternUnits"),i=this.getLength("height",r?1:this.getParentVHeight(),0);return this.getViewbox("viewBox",[0,0,0,i])[3]}},$=function SvgElemGradient(i,o,u){R.call(this,i,o),this.allowedChildren=["stop"],this.ref=function(){var r=this.getUrl("href")||this.getUrl("xlink:href");if(r&&r.nodeName===i.nodeName)return new SvgElemGradient(r,o,u)}.call(this);var p=this.attr;this.attr=function(r){var i=p.call(this,r);return null!=i||"href"===r||"xlink:href"===r?i:this.ref?this.ref.attr(r):null};var g=this.getChildren;this.getChildren=function(){var r=g.call(this);return r.length>0?r:this.ref?this.ref.getChildren():[]},this.getPaint=function(i,o,p,g){var w=this.getChildren();if(0!==w.length){if(1===w.length){var _=w[0],x=_.get("stop-color");if("none"===x)return;return opacityToColor(x,_.get("stop-opacity")*o,g)}var k,P,O,I,B,R,D="userSpaceOnUse"!==this.attr("gradientUnits"),N=parseTranform(this.attr("gradientTransform")),W=this.attr("spreadMethod"),U=0,j=0,G=1;if(D&&(N=multiplyMatrix([i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]],N)),N=validateMatrix(N)){if("linearGradient"===this.name)P=this.getLength("x1",D?1:this.getVWidth(),0),O=this.getLength("x2",D?1:this.getVWidth(),D?1:this.getVWidth()),I=this.getLength("y1",D?1:this.getVHeight(),0),B=this.getLength("y2",D?1:this.getVHeight(),0);else{O=this.getLength("cx",D?1:this.getVWidth(),D?.5:.5*this.getVWidth()),B=this.getLength("cy",D?1:this.getVHeight(),D?.5:.5*this.getVHeight()),R=this.getLength("r",D?1:this.getViewport(),D?.5:.5*this.getViewport()),P=this.getLength("fx",D?1:this.getVWidth(),O),I=this.getLength("fy",D?1:this.getVHeight(),B),R<0&&_e("SvgElemGradient: negative r value");var X=Math.sqrt(Math.pow(O-P,2)+Math.pow(B-I,2)),K=1;X>R&&(P=O+(P-O)*(K=R/X),I=B+(I-B)*K),R=Math.max(R,X*K*1.000001)}if("reflect"===W||"repeat"===W){var Y=inverseMatrix(N),J=transformPoint([i[0],i[1]],Y),$=transformPoint([i[2],i[1]],Y),ee=transformPoint([i[2],i[3]],Y),te=transformPoint([i[0],i[3]],Y);"linearGradient"===this.name?(U=Math.max((J[0]-O)*(O-P)+(J[1]-B)*(B-I),($[0]-O)*(O-P)+($[1]-B)*(B-I),(ee[0]-O)*(O-P)+(ee[1]-B)*(B-I),(te[0]-O)*(O-P)+(te[1]-B)*(B-I))/(Math.pow(O-P,2)+Math.pow(B-I,2)),j=Math.max((J[0]-P)*(P-O)+(J[1]-I)*(I-B),($[0]-P)*(P-O)+($[1]-I)*(I-B),(ee[0]-P)*(P-O)+(ee[1]-I)*(I-B),(te[0]-P)*(P-O)+(te[1]-I)*(I-B))/(Math.pow(O-P,2)+Math.pow(B-I,2))):U=Math.sqrt(Math.max(Math.pow(J[0]-O,2)+Math.pow(J[1]-B,2),Math.pow($[0]-O,2)+Math.pow($[1]-B,2),Math.pow(ee[0]-O,2)+Math.pow(ee[1]-B,2),Math.pow(te[0]-O,2)+Math.pow(te[1]-B,2)))/R-1,U=Math.ceil(U+.5),G=(j=Math.ceil(j+.5))+1+U}k="linearGradient"===this.name?r.linearGradient(P-j*(O-P),I-j*(B-I),O+U*(O-P),B+U*(B-I)):r.radialGradient(P,I,0,O,B,R+U*R);for(var re=0;re<G;re++)for(var ne=0,ie="reflect"!==W||(re-j)%2==0,oe=0;oe<w.length;oe++){var ae=w[ie?oe:w.length-1-oe],se=ae.get("stop-color");"none"===se&&(se=y.transparent),se=opacityToColor(se,ae.get("stop-opacity")*o,g),ne=Math.max(ne,ie?ae.getPercent("offset",0):1-ae.getPercent("offset",0)),0===oe&&4===se[0].length&&(k._colorSpace="DeviceCMYK"),0===oe&&ne>0&&k.stop((re+0)/G,se[0],se[1]),k.stop((re+ne)/(U+j+1),se[0],se[1]),oe===w.length-1&&ne<1&&k.stop((re+1)/G,se[0],se[1])}return k.setTransform.apply(k,N),[k,1]}return u?[u[0],u[1]*o]:void 0}}},ee=function SvgElemBasicShape(i,o){D.call(this,i,o),this.dashScale=1,this.getBoundingShape=function(){return this.shape},this.getTransformation=function(){return this.get("transform")},this.drawInDocument=function(i,o){if("hidden"!==this.get("visibility")&&this.shape){if(r.save(),this.transform(),this.clip(),i)this.shape.insertInDocument(),docFillColor(y.white),r.fill(this.get("clip-rule"));else{var u;this.mask()&&(u=docBeginGroup(getPageBBox()));var p=this.shape.getSubPaths(),g=this.getFill(i,o),w=this.getStroke(i,o),_=this.get("stroke-width"),x=this.get("stroke-linecap");if(g||w){if(g&&docFillColor(g),w){for(var k=0;k<p.length;k++)if(isEqual(p[k].totalLength,0)&&("square"===x||"round"===x)&&_>0&&p[k].startPoint&&p[k].startPoint.length>1){var P=p[k].startPoint[0],O=p[k].startPoint[1];docFillColor(w),"square"===x?r.rect(P-.5*_,O-.5*_,_,_):"round"===x&&r.circle(P,O,.5*_),r.fill()}var I=this.get("stroke-dasharray"),B=this.get("stroke-dashoffset");if(isNotEqual(this.dashScale,1)){for(var R=0;R<I.length;R++)I[R]*=this.dashScale;B*=this.dashScale}docStrokeColor(w),r.lineWidth(_).miterLimit(this.get("stroke-miterlimit")).lineJoin(this.get("stroke-linejoin")).lineCap(x).dash(I,{phase:B})}for(var D=0;D<p.length;D++)p[D].totalLength>0&&p[D].insertInDocument();g&&w?r.fillAndStroke(this.get("fill-rule")):g?r.fill(this.get("fill-rule")):w&&r.stroke()}var N=this.get("marker-start"),W=this.get("marker-mid"),U=this.get("marker-end");if("none"!==N||"none"!==W||"none"!==U){var j=this.shape.getMarkers();if("none"!==N&&new ce(N,null).drawMarker(!1,o,j[0],_),"none"!==W)for(var G=1;G<j.length-1;G++)new ce(W,null).drawMarker(!1,o,j[G],_);"none"!==U&&new ce(U,null).drawMarker(!1,o,j[j.length-1],_)}u&&(docEndGroup(u),docInsertGroup(u))}r.restore()}}},te=function SvgElemRect(r,i){ee.call(this,r,i);var o=this.getLength("x",this.getVWidth(),0),u=this.getLength("y",this.getVHeight(),0),p=this.getLength("width",this.getVWidth(),0),g=this.getLength("height",this.getVHeight(),0),y=this.getLength("rx",this.getVWidth()),w=this.getLength("ry",this.getVHeight());void 0===y&&void 0===w?y=w=0:void 0===y&&void 0!==w?y=w:void 0!==y&&void 0===w&&(w=y),p>0&&g>0?y&&w?(y=Math.min(y,.5*p),w=Math.min(w,.5*g),this.shape=(new B).M(o+y,u).L(o+p-y,u).A(y,w,0,0,1,o+p,u+w).L(o+p,u+g-w).A(y,w,0,0,1,o+p-y,u+g).L(o+y,u+g).A(y,w,0,0,1,o,u+g-w).L(o,u+w).A(y,w,0,0,1,o+y,u).Z()):this.shape=(new B).M(o,u).L(o+p,u).L(o+p,u+g).L(o,u+g).Z():this.shape=new B},re=function SvgElemCircle(r,i){ee.call(this,r,i);var o=this.getLength("cx",this.getVWidth(),0),u=this.getLength("cy",this.getVHeight(),0),p=this.getLength("r",this.getViewport(),0);this.shape=p>0?(new B).M(o+p,u).A(p,p,0,0,1,o-p,u).A(p,p,0,0,1,o+p,u).Z():new B},ne=function SvgElemEllipse(r,i){ee.call(this,r,i);var o=this.getLength("cx",this.getVWidth(),0),u=this.getLength("cy",this.getVHeight(),0),p=this.getLength("rx",this.getVWidth(),0),g=this.getLength("ry",this.getVHeight(),0);this.shape=p>0&&g>0?(new B).M(o+p,u).A(p,g,0,0,1,o-p,u).A(p,g,0,0,1,o+p,u).Z():new B},ie=function SvgElemLine(r,i){ee.call(this,r,i);var o=this.getLength("x1",this.getVWidth(),0),u=this.getLength("y1",this.getVHeight(),0),p=this.getLength("x2",this.getVWidth(),0),g=this.getLength("y2",this.getVHeight(),0);this.shape=(new B).M(o,u).L(p,g)},oe=function SvgElemPolyline(r,i){ee.call(this,r,i);var o=this.getNumberList("points");this.shape=new B;for(var u=0;u<o.length-1;u+=2)0===u?this.shape.M(o[u],o[u+1]):this.shape.L(o[u],o[u+1]);o.error&&_e("SvgElemPolygon: unexpected string "+o.error),o.length%2==1&&_e("SvgElemPolyline: uneven number of coordinates")},ae=function SvgElemPolygon(r,i){ee.call(this,r,i);var o=this.getNumberList("points");this.shape=new B;for(var u=0;u<o.length-1;u+=2)0===u?this.shape.M(o[u],o[u+1]):this.shape.L(o[u],o[u+1]);this.shape.Z(),o.error&&_e("SvgElemPolygon: unexpected string "+o.error),o.length%2==1&&_e("SvgElemPolygon: uneven number of coordinates")},se=function SvgElemPath(r,i){ee.call(this,r,i),this.shape=(new B).path(this.attr("d"));var o=this.getLength("pathLength",this.getViewport());this.pathLength=o>0?o:void 0,this.dashScale=void 0!==this.pathLength?this.shape.totalLength/this.pathLength:1},ce=function SvgElemMarker(i,o){N.call(this,i,o);var u=this.getLength("markerWidth",this.getParentVWidth(),3),p=this.getLength("markerHeight",this.getParentVHeight(),3),g=this.getViewbox("viewBox",[0,0,u,p]);this.getVWidth=function(){return g[2]},this.getVHeight=function(){return g[3]},this.drawMarker=function(i,o,y,w){r.save();var _=this.attr("orient"),x=this.attr("markerUnits"),k="auto"===_?y[2]:(parseFloat(_)||0)*Math.PI/180,P="userSpaceOnUse"===x?1:w;r.transform(Math.cos(k)*P,Math.sin(k)*P,-Math.sin(k)*P,Math.cos(k)*P,y[0],y[1]);var O,I=this.getLength("refX",this.getVWidth(),0),B=this.getLength("refY",this.getVHeight(),0),R=parseAspectRatio(this.attr("preserveAspectRatio"),u,p,g[2],g[3],.5);"hidden"===this.get("overflow")&&r.rect(R[0]*(g[0]+g[2]/2-I)-u/2,R[3]*(g[1]+g[3]/2-B)-p/2,u,p).clip(),r.transform.apply(r,R),r.translate(-I,-B),this.get("opacity")<1&&!i&&(O=docBeginGroup(getPageBBox())),this.drawChildren(i,o),O&&(docEndGroup(O),r.fillOpacity(this.get("opacity")),docInsertGroup(O)),r.restore()}},le=function SvgElemClipPath(i,o){N.call(this,i,o),this.useMask=function(i){var o=docBeginGroup(getPageBBox());r.save(),"objectBoundingBox"===this.attr("clipPathUnits")&&r.transform(i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]),this.clip(),this.drawChildren(!0,!1),r.restore(),docEndGroup(o),docApplyMask(o,!0)}},ue=function SvgElemMask(i,o){N.call(this,i,o),this.useMask=function(i){var o,u,p,g,y=docBeginGroup(getPageBBox());r.save(),"userSpaceOnUse"===this.attr("maskUnits")?(o=this.getLength("x",this.getVWidth(),-.1*(i[2]-i[0])+i[0]),u=this.getLength("y",this.getVHeight(),-.1*(i[3]-i[1])+i[1]),p=this.getLength("width",this.getVWidth(),1.2*(i[2]-i[0])),g=this.getLength("height",this.getVHeight(),1.2*(i[3]-i[1]))):(o=this.getLength("x",this.getVWidth(),-.1)*(i[2]-i[0])+i[0],u=this.getLength("y",this.getVHeight(),-.1)*(i[3]-i[1])+i[1],p=this.getLength("width",this.getVWidth(),1.2)*(i[2]-i[0]),g=this.getLength("height",this.getVHeight(),1.2)*(i[3]-i[1])),r.rect(o,u,p,g).clip(),"objectBoundingBox"===this.attr("maskContentUnits")&&r.transform(i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]),this.clip(),this.drawChildren(!1,!0),r.restore(),docEndGroup(y),docApplyMask(y,!0)}},fe=function SvgElemTextContainer(i,o){D.call(this,i,o),this.allowedChildren=["tspan","#text","#cdata-section","a"],this.isText=!0,this.getBoundingShape=function(){for(var r=new B,i=0;i<this._pos.length;i++){var o=this._pos[i];if(!o.hidden){var u=o.ascent*Math.sin(o.rotate),p=-o.ascent*Math.cos(o.rotate),g=o.descent*Math.sin(o.rotate),y=-o.descent*Math.cos(o.rotate),w=o.width*Math.cos(o.rotate),_=o.width*Math.sin(o.rotate);r.M(o.x+u,o.y+p).L(o.x+u+w,o.y+p+_).M(o.x+g+w,o.y+y+_).L(o.x+g,o.y+y)}}return r},this.drawTextInDocument=function(i,o){!this.link||i||o||this.addLink(),"underline"===this.get("text-decoration")&&this.decorate(.05*this._font.size,-.075*this._font.size,i,o),"overline"===this.get("text-decoration")&&this.decorate(.05*this._font.size,getAscent(this._font.font,this._font.size)+.075*this._font.size,i,o);var u=this.getFill(i,o),p=this.getStroke(i,o),g=this.get("stroke-width");this._font.fauxBold&&(p?g+=.03*this._font.size:(p=u,g=.03*this._font.size));for(var y,w,_,x,k,P,O,I,B,R=this.getChildren(),D=0;D<R.length;D++){var N=R[D];switch(N.name){case"tspan":case"textPath":case"a":"none"!==N.get("display")&&N.drawTextInDocument(i,o);break;case"#text":case"#cdata-section":if("hidden"===this.get("visibility"))continue;if(u||p||i){u&&docFillColor(u),p&&g&&(docStrokeColor(p),r.lineWidth(g).miterLimit(this.get("stroke-miterlimit")).lineJoin(this.get("stroke-linejoin")).lineCap(this.get("stroke-linecap")).dash(this.get("stroke-dasharray"),{phase:this.get("stroke-dashoffset")})),I=this._font.font,B=this._font.size,r.page.fonts[I.id]||(r.page.fonts[I.id]=I.ref()),r.addContent("BT").addContent("/"+I.id+" "+B+" Tf"),docSetTextMode(!!u,!!p);for(var W=0,U=N._pos;W<U.length;W++)if(!U[W].hidden&&isNotEqual(U[W].width,0)){var j=Math.cos(U[W].rotate),G=Math.sin(U[W].rotate),X=this._font.fauxItalic?-.25:0;w=j*U[W].scale,_=G*U[W].scale,x=j*X-G,k=G*X+j,P=U[W].x,O=U[W].y,r.addContent(validateNumber(w)+" "+validateNumber(_)+" "+validateNumber(-x)+" "+validateNumber(-k)+" "+validateNumber(P)+" "+validateNumber(O)+" Tm"),y=U[W].glyph,r.addContent("<"+y+"> Tj")}r.addContent("ET")}}}"line-through"===this.get("text-decoration")&&this.decorate(.05*this._font.size,.5*(getAscent(this._font.font,this._font.size)+getDescent(this._font.font,this._font.size)),i,o)},this.decorate=function(i,o,u,p){var g=this.getFill(u,p),y=this.getStroke(u,p);g&&docFillColor(g),y&&(docStrokeColor(y),r.lineWidth(this.get("stroke-width")).miterLimit(this.get("stroke-miterlimit")).lineJoin(this.get("stroke-linejoin")).lineCap(this.get("stroke-linecap")).dash(this.get("stroke-dasharray"),{phase:this.get("stroke-dashoffset")}));for(var w=0,_=this._pos;w<_.length;w++)if(!_[w].hidden&&isNotEqual(_[w].width,0)){var x=(o+i/2)*Math.sin(_[w].rotate),k=-(o+i/2)*Math.cos(_[w].rotate),P=(o-i/2)*Math.sin(_[w].rotate),O=-(o-i/2)*Math.cos(_[w].rotate),I=_[w].width*Math.cos(_[w].rotate),R=_[w].width*Math.sin(_[w].rotate);(new B).M(_[w].x+x,_[w].y+k).L(_[w].x+x+I,_[w].y+k+R).L(_[w].x+P+I,_[w].y+O+R).L(_[w].x+P,_[w].y+O).Z().insertInDocument(),g&&y?r.fillAndStroke():g?r.fill():y&&r.stroke()}}},he=function SvgElemTextNode(r,i){this.name=r.nodeName,this.textContent=r.nodeValue},de=function SvgElemTspan(r,i){fe.call(this,r,i)},pe=function SvgElemTextPath(r,i){var o;if(fe.call(this,r,i),(o=this.attr("path"))&&""!==o.trim()){var u=this.getLength("pathLength",this.getViewport());this.pathObject=(new B).path(o),this.pathLength=u>0?u:this.pathObject.totalLength,this.pathScale=this.pathObject.totalLength/this.pathLength}else if((o=this.getUrl("href")||this.getUrl("xlink:href"))&&"path"===o.nodeName){var p=new se(o,this);this.pathObject=p.shape.clone().transform(p.get("transform")),this.pathLength=this.chooseValue(p.pathLength,this.pathObject.totalLength),this.pathScale=this.pathObject.totalLength/this.pathLength}},ge=function SvgElemText(i,o){fe.call(this,i,o),this.allowedChildren=["textPath","tspan","#text","#cdata-section","a"],function(o){var u,p,g="",y=i.textContent,w=[],_=[],x=0,k=0;function doAnchoring(){if(_.length)for(var r=_[_.length-1],i=_[0],o=r.x+r.width-i.x,g={startltr:0,middleltr:.5,endltr:1,startrtl:1,middlertl:.5,endrtl:0}[u+p]*o||0,y=0;y<_.length;y++)_[y].x-=g;_=[]}function textOnPath(r){var i=r.pathObject,o=r.pathLength,u=r.pathScale;if(i)for(var p=r.getLength("startOffset",o,0),g=0;g<r._pos.length;g++){var y=p+r._pos[g].x+.5*r._pos[g].width;if(y>o||y<0)r._pos[g].hidden=!0;else{var w=i.getPointAtLength(y*u);isNotEqual(u,1)&&(r._pos[g].scale*=u,r._pos[g].width*=u),r._pos[g].x=w[0]-.5*r._pos[g].width*Math.cos(w[2])-r._pos[g].y*Math.sin(w[2]),r._pos[g].y=w[1]-.5*r._pos[g].width*Math.sin(w[2])+r._pos[g].y*Math.cos(w[2]),r._pos[g].rotate=w[2]+r._pos[g].rotate,r._pos[g].continuous=!1}}else for(var _=0;_<r._pos.length;_++)r._pos[_].hidden=!0}!function recursive(i,o){i._x=combineArrays(i.getLengthList("x",i.getVWidth()),o?o._x.slice(o._pos.length):[]),i._y=combineArrays(i.getLengthList("y",i.getVHeight()),o?o._y.slice(o._pos.length):[]),i._dx=combineArrays(i.getLengthList("dx",i.getVWidth()),o?o._dx.slice(o._pos.length):[]),i._dy=combineArrays(i.getLengthList("dy",i.getVHeight()),o?o._dy.slice(o._pos.length):[]),i._rot=combineArrays(i.getNumberList("rotate"),o?o._rot.slice(o._pos.length):[]),i._defRot=i.chooseValue(i._rot[i._rot.length-1],o&&o._defRot,0),"textPath"===i.name&&(i._y=[]);var P={fauxItalic:!1,fauxBold:!1},O=Se(i.get("font-family"),"bold"===i.get("font-weight"),"italic"===i.get("font-style"),P);try{r.font(O)}catch(r){_e('SVGElemText: failed to open font "'+O+'" in PDFKit')}i._pos=[],i._index=0,i._font={font:r._font,size:i.get("font-size"),fauxItalic:P.fauxItalic,fauxBold:P.fauxBold};var I=i.getLength("textLength",i.getVWidth(),void 0),B="spacingAndGlyphs"===i.attr("lengthAdjust"),R=i.get("word-spacing"),D=i.get("letter-spacing"),N=i.get("text-anchor"),W=i.get("direction"),U=function getBaseline(r,i,o,u){var p,g;switch(o){case"middle":p=.5*function getXHeight(r,i){return(r.xHeight||.5*(r.ascender-r.descender))*i/1e3}(r,i);break;case"central":p=.5*(getDescent(r,i)+getAscent(r,i));break;case"after-edge":case"text-after-edge":p=getDescent(r,i);break;case"alphabetic":case"auto":case"baseline":default:p=0;break;case"mathematical":p=.5*getAscent(r,i);break;case"hanging":p=.8*getAscent(r,i);break;case"before-edge":case"text-before-edge":p=getAscent(r,i)}switch(u){case"baseline":g=0;break;case"super":g=.6*i;break;case"sub":g=-.6*i;break;default:g=u}return p-g}(i._font.font,i._font.size,i.get("alignment-baseline")||i.get("dominant-baseline"),i.get("baseline-shift"));"textPath"===i.name&&(doAnchoring(),x=k=0);for(var j=i.getChildren(),G=0;G<j.length;G++){var X=j[G];switch(X.name){case"tspan":case"textPath":case"a":recursive(X,i);break;case"#text":case"#cdata-section":var K=X.textContent,Y=K,J=void 0;X._font=i._font,X._pos=[],y=y.substring(K.length),"preserve"===i.get("xml:space")?Y=Y.replace(/[\s]/g," "):(Y=Y.replace(/[\s]+/g," "),g.match(/[\s]$|^$/)&&(Y=Y.replace(/^[\s]/,"")),y.match(/^[\s]*$/)&&(Y=Y.replace(/[\s]$/,""))),g+=K,J=0===R?[Y]:Y.split(/(\s)/);for(var $=0;$<J.length;$++){for(var ee=getTextPos(i._font.font,i._font.size,J[$]),te=0;te<ee.length;te++){var re=i._index,ne=i._x[re],ie=i._y[re],oe=i._dx[re],ae=i._dy[re],se=i._rot[re],ce=!(0===$&&0===te);void 0!==ne&&(ce=!1,doAnchoring(),x=ne),void 0!==ie&&(ce=!1,doAnchoring(),k=ie),void 0!==oe&&(ce=!1,x+=oe),void 0!==ae&&(ce=!1,k+=ae),void 0===se&&0===i._defRot||(ce=!1);var le={glyph:ee[te].glyph,rotate:Math.PI/180*i.chooseValue(se,i._defRot),x:x+ee[te].xOffset,y:k+U+ee[te].yOffset,width:ee[te].width,ascent:getAscent(i._font.font,i._font.size),descent:getDescent(i._font.font,i._font.size),scale:1,hidden:!1,continuous:ce};_.push(le),X._pos.push(le),i._pos.push(le),i._index+=ee[te].unicode.length,1===_.length&&(u=N,p=W),x+=ee[te].xAdvance+D,k+=ee[te].yAdvance}" "===J[$]&&(x+=R)}break;default:y=y.substring(X.textContent.length)}}if(I&&i._pos.length&&function adjustLength(r,i,o){var u=r[0],p=r[r.length-1],g=u.x,y=p.x+p.width;if(o){var w=i/(y-g);if(w>0&&w<1/0)for(var _=0;_<r.length;_++)r[_].x=g+w*(r[_].x-g),r[_].scale*=w,r[_].width*=w}else if(r.length>=2)for(var k=(i-(y-g))/(r.length-1),P=0;P<r.length;P++)r[P].x+=P*k;x+=i-(y-g)}(i._pos,I,B),"textPath"!==i.name&&"text"!==i.name||doAnchoring(),"textPath"===i.name){w.push(i);var ue=i.pathObject;ue&&(x=ue.endPoint[0],k=ue.endPoint[1])}o&&(o._pos=o._pos.concat(i._pos),o._index+=i._index)}(o,null);for(var P=0;P<w.length;P++)textOnPath(w[P])}(this),this.getTransformation=function(){return this.get("transform")},this.drawInDocument=function(i,o){var u;r.save(),this.transform(),this.clip(),this.mask()&&(u=docBeginGroup(getPageBBox())),this.drawTextInDocument(i,o),u&&(docEndGroup(u),docInsertGroup(u)),r.restore()}},ye=(p=p||{}).assumePt?1:.75,ve=(p.width||r.page.width)/ye,me=(p.height||r.page.height)/ye,be=p.preserveAspectRatio||null,we=p.useCSS&&"undefined"!=typeof SVGElement&&i instanceof SVGElement&&"function"==typeof getComputedStyle,_e=p.warningCallback,Se=p.fontCallback,xe=p.imageCallback,Ce=p.colorCallback,ke=p.documentCallback,Pe=Math.ceil(Math.max(1,p.precision))||3,Ae=[],Ee={},Te=[],Oe=[];if("function"!=typeof _e&&(_e=function warningCallback(r){void 0!==typeof console&&console.warn}),"function"!=typeof Se&&(Se=function fontCallback(i,o,u,p){if(o&&u){if(r._registeredFonts.hasOwnProperty(i+"-BoldItalic"))return i+"-BoldItalic";if(r._registeredFonts.hasOwnProperty(i+"-Italic"))return p.fauxBold=!0,i+"-Italic";if(r._registeredFonts.hasOwnProperty(i+"-Bold"))return p.fauxItalic=!0,i+"-Bold";if(r._registeredFonts.hasOwnProperty(i))return p.fauxBold=!0,p.fauxItalic=!0,i}if(o&&!u){if(r._registeredFonts.hasOwnProperty(i+"-Bold"))return i+"-Bold";if(r._registeredFonts.hasOwnProperty(i))return p.fauxBold=!0,i}if(!o&&u){if(r._registeredFonts.hasOwnProperty(i+"-Italic"))return i+"-Italic";if(r._registeredFonts.hasOwnProperty(i))return p.fauxItalic=!0,i}if(!o&&!u&&r._registeredFonts.hasOwnProperty(i))return i;if(i.match(/(?:^|,)\s*serif\s*$/)){if(o&&u)return"Times-BoldItalic";if(o&&!u)return"Times-Bold";if(!o&&u)return"Times-Italic";if(!o&&!u)return"Times-Roman"}else if(i.match(/(?:^|,)\s*monospace\s*$/)){if(o&&u)return"Courier-BoldOblique";if(o&&!u)return"Courier-Bold";if(!o&&u)return"Courier-Oblique";if(!o&&!u)return"Courier"}else{if(i.match(/(?:^|,)\s*sans-serif\s*$/),o&&u)return"Helvetica-BoldOblique";if(o&&!u)return"Helvetica-Bold";if(!o&&u)return"Helvetica-Oblique";if(!o&&!u)return"Helvetica"}}),"function"!=typeof xe&&(xe=function imageCallback(r){return r.replace(/\s+/g,"")}),"function"!=typeof Ce)Ce=null;else for(var Ie in y){var Be=Ce(y[Ie]);y[Ie][0]=Be[0],y[Ie][1]=Be[1]}if("function"!=typeof ke&&(ke=null),"string"==typeof i&&(i=parseXml(i)),i){for(var Fe=i.getElementsByTagName("style"),Le=0;Le<Fe.length;Le++)Oe=Oe.concat(parseStyleSheet(Fe[Le].textContent));var Re=createSVGElement(i,null);if("function"==typeof Re.drawInDocument){p.useCSS&&!we&&_e("SVGtoPDF: useCSS option can only be used for SVG *elements* in compatible browsers");var De=r._fillColor;r.save().translate(o||0,u||0).scale(ye),Re.drawInDocument();for(var Me=0;Me<Te.length;Me++)r.page.annotations.push(Te[Me]);r.restore(),r._fillColor=De}else _e("SVGtoPDF: this element can't be rendered directly: "+i.nodeName)}else _e("SVGtoPDF: the input does not look like a valid SVG")})},9742:function(r,i){"use strict";i.byteLength=function byteLength(r){var i=getLens(r),o=i[0],u=i[1];return 3*(o+u)/4-u},i.toByteArray=function toByteArray(r){var i,o,g=getLens(r),y=g[0],w=g[1],_=new p(function _byteLength(r,i,o){return 3*(i+o)/4-o}(0,y,w)),x=0,k=w>0?y-4:y;for(o=0;o<k;o+=4)i=u[r.charCodeAt(o)]<<18|u[r.charCodeAt(o+1)]<<12|u[r.charCodeAt(o+2)]<<6|u[r.charCodeAt(o+3)],_[x++]=i>>16&255,_[x++]=i>>8&255,_[x++]=255&i;2===w&&(i=u[r.charCodeAt(o)]<<2|u[r.charCodeAt(o+1)]>>4,_[x++]=255&i);1===w&&(i=u[r.charCodeAt(o)]<<10|u[r.charCodeAt(o+1)]<<4|u[r.charCodeAt(o+2)]>>2,_[x++]=i>>8&255,_[x++]=255&i);return _},i.fromByteArray=function fromByteArray(r){for(var i,u=r.length,p=u%3,g=[],y=16383,w=0,_=u-p;w<_;w+=y)g.push(encodeChunk(r,w,w+y>_?_:w+y));1===p?(i=r[u-1],g.push(o[i>>2]+o[i<<4&63]+"==")):2===p&&(i=(r[u-2]<<8)+r[u-1],g.push(o[i>>10]+o[i>>4&63]+o[i<<2&63]+"="));return g.join("")};for(var o=[],u=[],p="undefined"!=typeof Uint8Array?Uint8Array:Array,g="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",y=0,w=g.length;y<w;++y)o[y]=g[y],u[g.charCodeAt(y)]=y;function getLens(r){var i=r.length;if(i%4>0)throw new Error("Invalid string. Length must be a multiple of 4");var o=r.indexOf("=");return-1===o&&(o=i),[o,o===i?0:4-o%4]}function encodeChunk(r,i,u){for(var p,g,y=[],w=i;w<u;w+=3)p=(r[w]<<16&16711680)+(r[w+1]<<8&65280)+(255&r[w+2]),y.push(o[(g=p)>>18&63]+o[g>>12&63]+o[g>>6&63]+o[63&g]);return y.join("")}u["-".charCodeAt(0)]=62,u["_".charCodeAt(0)]=63},4181:function(r){var i=4096,o=new Uint32Array([0,1,3,7,15,31,63,127,255,511,1023,2047,4095,8191,16383,32767,65535,131071,262143,524287,1048575,2097151,4194303,8388607,16777215]);function BrotliBitReader(r){this.buf_=new Uint8Array(8224),this.input_=r,this.reset()}BrotliBitReader.READ_SIZE=i,BrotliBitReader.IBUF_MASK=8191,BrotliBitReader.prototype.reset=function(){this.buf_ptr_=0,this.val_=0,this.pos_=0,this.bit_pos_=0,this.bit_end_pos_=0,this.eos_=0,this.readMoreInput();for(var r=0;r<4;r++)this.val_|=this.buf_[this.pos_]<<8*r,++this.pos_;return this.bit_end_pos_>0},BrotliBitReader.prototype.readMoreInput=function(){if(!(this.bit_end_pos_>256))if(this.eos_){if(this.bit_pos_>this.bit_end_pos_)throw new Error("Unexpected end of input "+this.bit_pos_+" "+this.bit_end_pos_)}else{var r=this.buf_ptr_,o=this.input_.read(this.buf_,r,i);if(o<0)throw new Error("Unexpected end of input");if(o<i){this.eos_=1;for(var u=0;u<32;u++)this.buf_[r+o+u]=0}if(0===r){for(u=0;u<32;u++)this.buf_[8192+u]=this.buf_[u];this.buf_ptr_=i}else this.buf_ptr_=0;this.bit_end_pos_+=o<<3}},BrotliBitReader.prototype.fillBitWindow=function(){for(;this.bit_pos_>=8;)this.val_>>>=8,this.val_|=this.buf_[8191&this.pos_]<<24,++this.pos_,this.bit_pos_=this.bit_pos_-8>>>0,this.bit_end_pos_=this.bit_end_pos_-8>>>0},BrotliBitReader.prototype.readBits=function(r){32-this.bit_pos_<r&&this.fillBitWindow();var i=this.val_>>>this.bit_pos_&o[r];return this.bit_pos_+=r,i},r.exports=BrotliBitReader},7080:function(r,i){i.lookup=new Uint8Array([0,0,0,0,0,0,0,0,0,4,4,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,12,16,12,12,20,12,16,24,28,12,12,32,12,36,12,44,44,44,44,44,44,44,44,44,44,32,32,24,40,28,12,12,48,52,52,52,48,52,52,52,48,52,52,52,52,52,48,52,52,52,52,52,48,52,52,52,52,52,24,12,28,12,12,12,56,60,60,60,56,60,60,60,56,60,60,60,60,60,56,60,60,60,60,60,56,60,60,60,60,60,24,12,28,12,0,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,0,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,48,48,48,48,48,48,48,48,48,48,48,48,48,48,48,56,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,6,6,6,6,7,7,7,7,8,8,8,8,9,9,9,9,10,10,10,10,11,11,11,11,12,12,12,12,13,13,13,13,14,14,14,14,15,15,15,15,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,22,22,22,22,23,23,23,23,24,24,24,24,25,25,25,25,26,26,26,26,27,27,27,27,28,28,28,28,29,29,29,29,30,30,30,30,31,31,31,31,32,32,32,32,33,33,33,33,34,34,34,34,35,35,35,35,36,36,36,36,37,37,37,37,38,38,38,38,39,39,39,39,40,40,40,40,41,41,41,41,42,42,42,42,43,43,43,43,44,44,44,44,45,45,45,45,46,46,46,46,47,47,47,47,48,48,48,48,49,49,49,49,50,50,50,50,51,51,51,51,52,52,52,52,53,53,53,53,54,54,54,54,55,55,55,55,56,56,56,56,57,57,57,57,58,58,58,58,59,59,59,59,60,60,60,60,61,61,61,61,62,62,62,62,63,63,63,63,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]),i.lookupOffsets=new Uint16Array([1024,1536,1280,1536,0,256,768,512])},6450:function(r,i,o){var u=o(6154).g,p=o(6154).j,g=o(4181),y=o(5139),w=o(966).h,_=o(966).g,x=o(7080),k=o(8435),P=o(2973),O=1080,I=new Uint8Array([1,2,3,4,0,5,17,6,16,7,8,9,10,11,12,13,14,15]),B=new Uint8Array([3,2,1,0,3,3,3,3,3,3,2,2,2,2,2,2]),R=new Int8Array([0,0,0,0,-1,1,-2,2,-3,3,-1,1,-2,2,-3,3]),D=new Uint16Array([256,402,436,468,500,534,566,598,630,662,694,726,758,790,822,854,886,920,952,984,1016,1048,1080]);function DecodeWindowBits(r){var i;return 0===r.readBits(1)?16:(i=r.readBits(3))>0?17+i:(i=r.readBits(3))>0?8+i:17}function DecodeVarLenUint8(r){if(r.readBits(1)){var i=r.readBits(3);return 0===i?1:r.readBits(i)+(1<<i)}return 0}function MetaBlockLength(){this.meta_block_length=0,this.input_end=0,this.is_uncompressed=0,this.is_metadata=!1}function DecodeMetaBlockLength(r){var i,o,u,p=new MetaBlockLength;if(p.input_end=r.readBits(1),p.input_end&&r.readBits(1))return p;if(7===(i=r.readBits(2)+4)){if(p.is_metadata=!0,0!==r.readBits(1))throw new Error("Invalid reserved bit");if(0===(o=r.readBits(2)))return p;for(u=0;u<o;u++){var g=r.readBits(8);if(u+1===o&&o>1&&0===g)throw new Error("Invalid size byte");p.meta_block_length|=g<<8*u}}else for(u=0;u<i;++u){var y=r.readBits(4);if(u+1===i&&i>4&&0===y)throw new Error("Invalid size nibble");p.meta_block_length|=y<<4*u}return++p.meta_block_length,p.input_end||p.is_metadata||(p.is_uncompressed=r.readBits(1)),p}function ReadSymbol(r,i,o){var u;return o.fillBitWindow(),(u=r[i+=o.val_>>>o.bit_pos_&255].bits-8)>0&&(o.bit_pos_+=8,i+=r[i].value,i+=o.val_>>>o.bit_pos_&(1<<u)-1),o.bit_pos_+=r[i].bits,r[i].value}function ReadHuffmanCode(r,i,o,u){var p,g,y=new Uint8Array(r);if(u.readMoreInput(),1===(g=u.readBits(2))){for(var x=r-1,k=0,P=new Int32Array(4),O=u.readBits(2)+1;x;)x>>=1,++k;for(B=0;B<O;++B)P[B]=u.readBits(k)%r,y[P[B]]=2;switch(y[P[0]]=1,O){case 1:break;case 3:if(P[0]===P[1]||P[0]===P[2]||P[1]===P[2])throw new Error("[ReadHuffmanCode] invalid symbols");break;case 2:if(P[0]===P[1])throw new Error("[ReadHuffmanCode] invalid symbols");y[P[1]]=1;break;case 4:if(P[0]===P[1]||P[0]===P[2]||P[0]===P[3]||P[1]===P[2]||P[1]===P[3]||P[2]===P[3])throw new Error("[ReadHuffmanCode] invalid symbols");u.readBits(1)?(y[P[2]]=3,y[P[3]]=3):y[P[0]]=2}}else{var B,R=new Uint8Array(18),D=32,N=0,W=[new w(2,0),new w(2,4),new w(2,3),new w(3,2),new w(2,0),new w(2,4),new w(2,3),new w(4,1),new w(2,0),new w(2,4),new w(2,3),new w(3,2),new w(2,0),new w(2,4),new w(2,3),new w(4,5)];for(B=g;B<18&&D>0;++B){var U,j=I[B],G=0;u.fillBitWindow(),G+=u.val_>>>u.bit_pos_&15,u.bit_pos_+=W[G].bits,U=W[G].value,R[j]=U,0!==U&&(D-=32>>U,++N)}if(1!==N&&0!==D)throw new Error("[ReadHuffmanCode] invalid num_codes or space");!function ReadHuffmanCodeLengths(r,i,o,u){for(var p=0,g=8,y=0,x=0,k=32768,P=[],O=0;O<32;O++)P.push(new w(0,0));for(_(P,0,5,r,18);p<i&&k>0;){var I,B=0;if(u.readMoreInput(),u.fillBitWindow(),B+=u.val_>>>u.bit_pos_&31,u.bit_pos_+=P[B].bits,(I=255&P[B].value)<16)y=0,o[p++]=I,0!==I&&(g=I,k-=32768>>I);else{var R,D,N=I-14,W=0;if(16===I&&(W=g),x!==W&&(y=0,x=W),R=y,y>0&&(y-=2,y<<=N),p+(D=(y+=u.readBits(N)+3)-R)>i)throw new Error("[ReadHuffmanCodeLengths] symbol + repeat_delta > num_symbols");for(var U=0;U<D;U++)o[p+U]=x;p+=D,0!==x&&(k-=D<<15-x)}}if(0!==k)throw new Error("[ReadHuffmanCodeLengths] space = "+k);for(;p<i;p++)o[p]=0}(R,r,y,u)}if(0===(p=_(i,o,8,y,r)))throw new Error("[ReadHuffmanCode] BuildHuffmanTable failed: ");return p}function ReadBlockLength(r,i,o){var u,p;return u=ReadSymbol(r,i,o),p=k.kBlockLengthPrefixCode[u].nbits,k.kBlockLengthPrefixCode[u].offset+o.readBits(p)}function TranslateShortCodes(r,i,o){var u;return r<16?(o+=B[r],u=i[o&=3]+R[r]):u=r-16+1,u}function MoveToFront(r,i){for(var o=r[i],u=i;u;--u)r[u]=r[u-1];r[0]=o}function HuffmanTreeGroup(r,i){this.alphabet_size=r,this.num_htrees=i,this.codes=new Array(i+i*D[r+31>>>5]),this.htrees=new Uint32Array(i)}function DecodeContextMap(r,i){var o,u,p={num_htrees:null,context_map:null},g=0;i.readMoreInput();var y=p.num_htrees=DecodeVarLenUint8(i)+1,_=p.context_map=new Uint8Array(r);if(y<=1)return p;for(i.readBits(1)&&(g=i.readBits(4)+1),o=[],u=0;u<O;u++)o[u]=new w(0,0);for(ReadHuffmanCode(y+g,o,0,i),u=0;u<r;){var x;if(i.readMoreInput(),0===(x=ReadSymbol(o,0,i)))_[u]=0,++u;else if(x<=g)for(var k=1+(1<<x)+i.readBits(x);--k;){if(u>=r)throw new Error("[DecodeContextMap] i >= context_map_size");_[u]=0,++u}else _[u]=x-g,++u}return i.readBits(1)&&function InverseMoveToFrontTransform(r,i){var o,u=new Uint8Array(256);for(o=0;o<256;++o)u[o]=o;for(o=0;o<i;++o){var p=r[o];r[o]=u[p],p&&MoveToFront(u,p)}}(_,r),p}function DecodeBlockType(r,i,o,u,p,g,y){var w,_=2*o,x=o,k=ReadSymbol(i,o*O,y);(w=0===k?p[_+(1&g[x])]:1===k?p[_+(g[x]-1&1)]+1:k-2)>=r&&(w-=r),u[o]=w,p[_+(1&g[x])]=w,++g[x]}function CopyUncompressedBlockToOutput(r,i,o,u,p,y){var w,_=p+1,x=o&p,k=y.pos_&g.IBUF_MASK;if(i<8||y.bit_pos_+(i<<3)<y.bit_end_pos_)for(;i-- >0;)y.readMoreInput(),u[x++]=y.readBits(8),x===_&&(r.write(u,_),x=0);else{if(y.bit_end_pos_<32)throw new Error("[CopyUncompressedBlockToOutput] br.bit_end_pos_ < 32");for(;y.bit_pos_<32;)u[x]=y.val_>>>y.bit_pos_,y.bit_pos_+=8,++x,--i;if(k+(w=y.bit_end_pos_-y.bit_pos_>>3)>g.IBUF_MASK){for(var P=g.IBUF_MASK+1-k,O=0;O<P;O++)u[x+O]=y.buf_[k+O];w-=P,x+=P,i-=P,k=0}for(O=0;O<w;O++)u[x+O]=y.buf_[k+O];if(i-=w,(x+=w)>=_){r.write(u,_),x-=_;for(O=0;O<x;O++)u[O]=u[_+O]}for(;x+i>=_;){if(w=_-x,y.input_.read(u,x,w)<w)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");r.write(u,_),i-=w,x=0}if(y.input_.read(u,x,i)<i)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");y.reset()}}function JumpToByteBoundary(r){var i=r.bit_pos_+7&-8;return 0==r.readBits(i-r.bit_pos_)}function BrotliDecompressedSize(r){var i=new u(r),o=new g(i);return DecodeWindowBits(o),DecodeMetaBlockLength(o).meta_block_length}function BrotliDecompress(r,i){var o,u,p,_,I,B,R,D,N,W,U=0,j=0,G=0,X=[16,15,11,4],K=0,Y=0,J=0,$=[new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0)],ee=128+g.READ_SIZE;p=(1<<(u=DecodeWindowBits(W=new g(r))))-16,I=(_=1<<u)-1,B=new Uint8Array(_+ee+y.maxDictionaryWordLength),R=_,D=[],N=[];for(var te=0;te<3240;te++)D[te]=new w(0,0),N[te]=new w(0,0);for(;!j;){var re,ne,ie,oe,ae,se,ce,le,ue,fe,he,de=0,pe=[1<<28,1<<28,1<<28],ge=[0],ye=[1,1,1],ve=[0,1,0,1,0,1],me=[0],be=null,we=null,_e=0,Se=null,xe=0,Ce=0,ke=0;for(o=0;o<3;++o)$[o].codes=null,$[o].htrees=null;W.readMoreInput();var Pe=DecodeMetaBlockLength(W);if(U+(de=Pe.meta_block_length)>i.buffer.length){var Ae=new Uint8Array(U+de);Ae.set(i.buffer),i.buffer=Ae}if(j=Pe.input_end,re=Pe.is_uncompressed,Pe.is_metadata)for(JumpToByteBoundary(W);de>0;--de)W.readMoreInput(),W.readBits(8);else if(0!==de)if(re)W.bit_pos_=W.bit_pos_+7&-8,CopyUncompressedBlockToOutput(i,de,U,B,I,W),U+=de;else{for(o=0;o<3;++o)ye[o]=DecodeVarLenUint8(W)+1,ye[o]>=2&&(ReadHuffmanCode(ye[o]+2,D,o*O,W),ReadHuffmanCode(26,N,o*O,W),pe[o]=ReadBlockLength(N,o*O,W),me[o]=1);for(W.readMoreInput(),oe=(1<<(ne=W.readBits(2)))-1,ae=(ie=16+(W.readBits(4)<<ne))+(48<<ne),be=new Uint8Array(ye[0]),o=0;o<ye[0];++o)W.readMoreInput(),be[o]=W.readBits(2)<<1;var Ee=DecodeContextMap(ye[0]<<6,W);ce=Ee.num_htrees,se=Ee.context_map;var Te=DecodeContextMap(ye[2]<<2,W);for(ue=Te.num_htrees,le=Te.context_map,$[0]=new HuffmanTreeGroup(256,ce),$[1]=new HuffmanTreeGroup(704,ye[1]),$[2]=new HuffmanTreeGroup(ae,ue),o=0;o<3;++o)$[o].decode(W);for(we=0,Se=0,fe=be[ge[0]],Ce=x.lookupOffsets[fe],ke=x.lookupOffsets[fe+1],he=$[1].htrees[0];de>0;){var Oe,Ie,Be,Fe,Le,Re,De,Me,Ne,We,Ue,ze;for(W.readMoreInput(),0===pe[1]&&(DecodeBlockType(ye[1],D,1,ge,ve,me,W),pe[1]=ReadBlockLength(N,O,W),he=$[1].htrees[ge[1]]),--pe[1],(Ie=(Oe=ReadSymbol($[1].codes,he,W))>>6)>=2?(Ie-=2,De=-1):De=0,Be=k.kInsertRangeLut[Ie]+(Oe>>3&7),Fe=k.kCopyRangeLut[Ie]+(7&Oe),Le=k.kInsertLengthPrefixCode[Be].offset+W.readBits(k.kInsertLengthPrefixCode[Be].nbits),Re=k.kCopyLengthPrefixCode[Fe].offset+W.readBits(k.kCopyLengthPrefixCode[Fe].nbits),Y=B[U-1&I],J=B[U-2&I],Ne=0;Ne<Le;++Ne)W.readMoreInput(),0===pe[0]&&(DecodeBlockType(ye[0],D,0,ge,ve,me,W),pe[0]=ReadBlockLength(N,0,W),we=ge[0]<<6,fe=be[ge[0]],Ce=x.lookupOffsets[fe],ke=x.lookupOffsets[fe+1]),_e=se[we+(x.lookup[Ce+Y]|x.lookup[ke+J])],--pe[0],J=Y,Y=ReadSymbol($[0].codes,$[0].htrees[_e],W),B[U&I]=Y,(U&I)===I&&i.write(B,_),++U;if((de-=Le)<=0)break;if(De<0)if(W.readMoreInput(),0===pe[2]&&(DecodeBlockType(ye[2],D,2,ge,ve,me,W),pe[2]=ReadBlockLength(N,2160,W),Se=ge[2]<<2),--pe[2],xe=le[Se+(255&(Re>4?3:Re-2))],(De=ReadSymbol($[2].codes,$[2].htrees[xe],W))>=ie)ze=(De-=ie)&oe,De=ie+((je=(2+(1&(De>>=ne))<<(Ue=1+(De>>1)))-4)+W.readBits(Ue)<<ne)+ze;if((Me=TranslateShortCodes(De,X,K))<0)throw new Error("[BrotliDecompress] invalid distance");if(We=U&I,Me>(G=U<p&&G!==p?U:p)){if(!(Re>=y.minDictionaryWordLength&&Re<=y.maxDictionaryWordLength))throw new Error("Invalid backward reference. pos: "+U+" distance: "+Me+" len: "+Re+" bytes left: "+de);var je=y.offsetsByLength[Re],Ge=Me-G-1,Ve=y.sizeBitsByLength[Re],He=Ge>>Ve;if(je+=(Ge&(1<<Ve)-1)*Re,!(He<P.kNumTransforms))throw new Error("Invalid backward reference. pos: "+U+" distance: "+Me+" len: "+Re+" bytes left: "+de);var qe=P.transformDictionaryWord(B,We,je,Re,He);if(U+=qe,de-=qe,(We+=qe)>=R){i.write(B,_);for(var Xe=0;Xe<We-R;Xe++)B[Xe]=B[R+Xe]}}else{if(De>0&&(X[3&K]=Me,++K),Re>de)throw new Error("Invalid backward reference. pos: "+U+" distance: "+Me+" len: "+Re+" bytes left: "+de);for(Ne=0;Ne<Re;++Ne)B[U&I]=B[U-Me&I],(U&I)===I&&i.write(B,_),++U,--de}Y=B[U-1&I],J=B[U-2&I]}U&=1073741823}}i.write(B,U&I)}HuffmanTreeGroup.prototype.decode=function(r){var i,o=0;for(i=0;i<this.num_htrees;++i)this.htrees[i]=o,o+=ReadHuffmanCode(this.alphabet_size,this.codes,o,r)},i.BrotliDecompressBuffer=function BrotliDecompressBuffer(r,i){var o=new u(r);null==i&&(i=BrotliDecompressedSize(r));var g=new Uint8Array(i),y=new p(g);return BrotliDecompress(o,y),y.pos<y.buffer.length&&(y.buffer=y.buffer.subarray(0,y.pos)),y.buffer},y.init()},5340:function(r,i,o){var u=o(9742);o(3857);i.init=function(){return(0,o(6450).BrotliDecompressBuffer)(u.toByteArray(o(2722)))}},2722:function(r){r.exports="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"},5139:function(r,i,o){var u=o(5340);i.init=function(){i.dictionary=u.init()},i.offsetsByLength=new Uint32Array([0,0,0,0,0,4096,9216,21504,35840,44032,53248,63488,74752,87040,93696,100864,104704,106752,108928,113536,115968,118528,119872,121280,122016]),i.sizeBitsByLength=new Uint8Array([0,0,0,0,10,10,11,11,10,10,10,10,10,9,9,8,7,7,8,7,7,6,6,5,5]),i.minDictionaryWordLength=4,i.maxDictionaryWordLength=24},966:function(r,i){function HuffmanCode(r,i){this.bits=r,this.value=i}i.h=HuffmanCode;var o=15;function GetNextKey(r,i){for(var o=1<<i-1;r&o;)o>>=1;return(r&o-1)+o}function ReplicateValue(r,i,o,u,p){do{r[i+(u-=o)]=new HuffmanCode(p.bits,p.value)}while(u>0)}function NextTableBitSize(r,i,u){for(var p=1<<i-u;i<o&&!((p-=r[i])<=0);)++i,p<<=1;return i-u}i.g=function(r,i,u,p,g){var y,w,_,x,k,P,O,I,B,R,D=i,N=new Int32Array(16),W=new Int32Array(16);for(R=new Int32Array(g),w=0;w<g;w++)N[p[w]]++;for(W[1]=0,y=1;y<o;y++)W[y+1]=W[y]+N[y];for(w=0;w<g;w++)0!==p[w]&&(R[W[p[w]]++]=w);if(B=I=1<<(O=u),1===